-- External dependencies
local PZAPI        = PZAPI
local ISContextMenu  = ISContextMenu
local ISToolTip      = ISToolTip

-- Game classes
local IsoPlayer       = IsoPlayer
local IsoZombie       = IsoZombie

-- Module configuration
local ZomboLewdConfig = ZomboLewdConfig

-- Localization
local getText         = getText

-- Cached client reference
local zombolewdClient = nil

-- ===== Action Dispatchers =====

--- Send or execute the "SpawnIsoZombieSurvivor" action
local function spawnIsoZombieSurvivor(worldObjs, player)
    if isClient() then
        zombolewdClient.Callbacks:sendClientCommand("SpawnIsoZombieSurvivor")
    else
        ZomboLewdConfig.Modules.SpawnIsoZombieSurvivor(worldObjs, player)
    end
end

--- Send or execute the "SpawnIsoPlayerSurvivor" action
local function spawnIsoPlayerSurvivor(worldObjs, player)
    if isClient() then
        zombolewdClient.Callbacks:sendClientCommand("SpawnIsoPlayerSurvivor")
    else
        ZomboLewdConfig.Modules.SpawnIsoPlayerSurvivor(worldObjs, player)
    end
end

--- Send or execute the "SpawnIsoSurvivor" action
local function spawnIsoSurvivor(worldObjs, player)
    if isClient() then
        zombolewdClient.Callbacks:sendClientCommand("SpawnIsoSurvivor")
    else
        ZomboLewdConfig.Modules.SpawnIsoSurvivor(worldObjs, player)
    end
end

--- Send or execute the "spawnZombie" action
local function spawnZombie(worldObjs, player)
    if isClient() then
        zombolewdClient.Callbacks:sendClientCommand("spawnZombie")
    else
        ZomboLewdConfig.Modules.SpawnZombie(worldObjs, player)
    end
end


-- ===== Context Menu Builder =====

--- Builds the debug submenu under the world-object context menu
-- Triggers only if the "box_debug" option is enabled
return function(contextMenuAPI, player, context, worldObjs)
    local options = PZAPI.ModOptions:getOptions("ZomboLewdFramework")
    if not options:getOption("box_debug"):getValue() then
        return
    end

    -- Cache ZomboLewd client on first invocation
    if zombolewdClient == nil then
        zombolewdClient = contextMenuAPI.Client
    end

    -- Main debug menu entry
    local mainOpt  = context:addOption(getText("ContextMenu_Debug_Option"), worldObjs)
    local subMenu  = ISContextMenu:getNew(context)
    context:addSubMenu(mainOpt, subMenu)

    -- Helper to add a submenu entry with tooltip
    local function addDebugOption(textKey, tooltipKey, callback)
        local tip = ISToolTip:new()
        tip.description = getText(tooltipKey)
        tip:initialise()
        tip:setVisible(false)

        local opt = subMenu:addOption(getText(textKey), worldObjs, callback, player)
        opt.toolTip = tip
    end

    addDebugOption(
        "ContextMenu_Debug_Spawn_IsoZombie_Survivor",
        "ContextMenu_Debug_Spawn_IsoZombie_Survivor_Description",
        spawnIsoZombieSurvivor
    )
    addDebugOption(
        "ContextMenu_Debug_Spawn_IsoPlayer_Survivor",
        "ContextMenu_Debug_Spawn_IsoPlayer_Survivor_Description",
        spawnIsoPlayerSurvivor
    )
    addDebugOption(
        "ContextMenu_Debug_Spawn_IsoSurvivor",
        "ContextMenu_Debug_Spawn_IsoSurvivor_Description",
        spawnIsoSurvivor
    )
    addDebugOption(
        "ContextMenu_Debug_Spawn_Zombie",
        "ContextMenu_Debug_Spawn_Zombie_Description",
        spawnZombie
    )
end