local Pane = require("InvMatrixPane")
local Config = require("InvMatrixConfig")
local Utils = require("InvMatrixUtils")

local M = ISCollapsableWindow:derive("InvMatrixPopup")
M.__index = M

M.new = function(self, x, y, width, height, parent)
    local o = ISCollapsableWindow.new(self, x, y, width, height)
    setmetatable(o, M)
    local cellSize = Config.getCellSize()
    o.minimumWidth = cellSize * 6
    o.minimumHeight = cellSize * 6
    o.resizable = true
    o.parent = parent
    o.cell = nil
    return o
end

M.close = function(self)
    ISCollapsableWindow.close(self)
    Utils.handleAiming()
end

M.setCell = function(self, cell)
    self.cell = cell
end

M.createChildren = function(self)
    ISCollapsableWindow.createChildren(self)
    self.collapseButton:setVisible(false)
    self.gridPane = Pane:new(0, 0, 0, 0, self)
    self.gridPane:setContext({
        getInventory = function()
            return self.parent.context:getInventory()
        end,
        getInventoryPane = function()
            return self.parent.context:getInventoryPane()
        end,
        getItems = function()
            for _, cell in ipairs(self.parent.cells) do
                if cell and cell.type == self.cell.type then
                    return cell.items
                end
            end
            self:close()
            return {}
        end,
        isOnCharacter = function()
            return self.parent.context:isOnCharacter()
        end,
    })
    self.gridPane:initialise()
    self.gridPane.canStackItems = false
    self:addChild(self.gridPane)
end

M.update = function(self)
    local x, y = self:getX(), self:getY()
    if not self:isPointOver(x, y) then
        self:removeFromUIManager()
        return
    end
    if not self:isVisible() then
        self:removeFromUIManager()
        return
    end
    local items = self.gridPane.context:getItems()
    local title = string.format("%s (%s)", self.cell:getName(), #items)
    self:setTitle(title)
end

M.prerender = function(self)
    ISCollapsableWindow.prerender(self)
    local titleBarHeight = self:titleBarHeight()
    self.gridPane:setX(0)
    self.gridPane:setY(titleBarHeight)
    self.gridPane:setWidth(self.width)
    self.gridPane:setHeight(self.height - titleBarHeight - 16)
end

M.onMouseMove = function(self, x, y)
    ISCollapsableWindow.onMouseMove(self, x, y)
    if ISMouseDrag.dragging and ISMouseDrag.draggingFocus == self.gridPane then
        self:bringToTop()
    end
    Utils.handleAiming()
end

M.onMouseMoveOutside = function(self, x, y)
    ISCollapsableWindow.onMouseMoveOutside(self, x, y)
    Utils.handleAiming()
end

M.onMouseDownOutside = function(self, x, y)
    ISCollapsableWindow.onMouseDownOutside(self, x, y)
    if self:isMouseOver() then
        return
    end
    local playerInventory = getPlayerInventory(0)
    if playerInventory then
        if playerInventory:isMouseOver() then
            return
        end
        local popup = playerInventory.gridPane and playerInventory.gridPane.popup
        if popup and popup:isMouseOver() then
            return
        end
    end
    local playerLoot = getPlayerLoot(0)
    if playerLoot then
        if playerLoot:isMouseOver() then
            return
        end
        local popup = playerLoot.gridPane and playerLoot.gridPane.popup
        if popup and popup:isMouseOver() then
            return
        end
    end
    self:close()
end

M.onMouseUp = function(self, x, y)
    ISCollapsableWindow.onMouseUp(self, x, y)
    local cellSize = Config.getCellSize()
    local paneWidth = math.floor(self.width / cellSize) * cellSize
    if paneWidth ~= self.width then
        self:setWidth(paneWidth + 1)
    end
end

M.onMouseUpOutside = function(self, x, y)
    ISCollapsableWindow.onMouseUpOutside(self, x, y)
    self:onMouseUp(x, y)
end

return M
