local Config = require("InvMatrixConfig")
local Constants = require("InvMatrixConstants")
local Cell = require("InvMatrixCell")

local M = ISPanel:derive("InvMatrixPane")
M.__index = M

M.new = function(self, x, y, width, height, parent)
    local o = ISPanel.new(self, x, y, width, height)
    setmetatable(o, M)
    o.parent = parent
    o.context = nil
    o.canStackItems = true
    o.cache = nil
    o.items = nil
    o.cells = nil
    o.tooltip = nil
    o.selected = nil
    o.dragging = nil
    o.highlighted = nil
    o.mouseDownX = nil
    o.mouseDownY = nil
    o.selStartX = nil
    o.selStartY = nil
    o.scrollY = 0
    return o
end

M.setContext = function(self, context)
    self.context = context
end

M.invalidate = function(self)
    self.items = nil
    self.cells = nil
    if self.popup then
        self.popup.gridPane:invalidate()
    end
end

M.getItems = function(self)
    if self.items then
        return self.items
    end
    local items = self.context:getItems()
    local filtered = {}
    local onCharacter = self.context:isOnCharacter()
    local equippedMode = Constants.EQUIPPED_MODES[Config.getOptionValue("CharacterEquippedItemsMode")]
    local hideEquipped = onCharacter and equippedMode == "Hidden"
    for _, item in ipairs(items) do
        if not item:isHidden() then
            if not hideEquipped or not item:isEquipped() then
                table.insert(filtered, item)
            end
        end
    end
    local sortingMethod = Constants.SORTING_METHODS[Config.getOptionValue("SortingMethod")]
    timSort(filtered, function(a, b)
        if onCharacter and equippedMode == "Sorted Last" then
            local c, d = a:isEquipped(), b:isEquipped()
            if c ~= d then
                return not c
            end
        end
        if sortingMethod == "Alphabetical" then
            return a:getName() < b:getName()
        elseif sortingMethod == "Category" then
            return a:getType() < b:getType()
        elseif sortingMethod == "Weight" then
            return a:getWeight() > b:getWeight()
        elseif sortingMethod == "Condition" then
            return a:getCondition() > b:getCondition()
        else
            return a:getID() < b:getID()
        end
    end)
    self.items = filtered
    return self.items
end

M.getStackingKey = function(self, item)
    if not self.canStackItems or item:isEquipped() then
        return item:getID()
    end
    if item:getLootType() == "Ammo" and item:getMaxAmmo() > 0 then
        return item:getID()
    end
    return item:getName()
end

M.getCells = function(self)
    if self.cells then
        return self.cells
    end
    local items = self:getItems()
    local map = {}
    for _, item in ipairs(items) do
        local type = self:getStackingKey(item)
        if not map[type] then
            local cell = Cell:new(self, type)
            map[type] = cell
        end
        local cell = map[type]
        table.insert(cell.items, item)
    end
    local cells = {}
    for _, cell in pairs(map) do
        table.insert(cells, cell)
    end
    local equippedMode = Constants.EQUIPPED_MODES[Config.getOptionValue("CharacterEquippedItemsMode")]
    if self.context:isOnCharacter() and equippedMode == "Separated" then
        local groupped = {}
        for _, cell in ipairs(cells) do
            if not cell:isEquipped() then
                table.insert(groupped, cell)
            end
        end
        local cols = self:getCols()
        local remaining = #groupped % cols
        if remaining ~= 0 then
            for _ = 1, cols - remaining do
                table.insert(groupped, false)
            end
        end
        for _, cell in ipairs(cells) do
            if cell:isEquipped() then
                table.insert(groupped, cell)
            end
        end
        cells = groupped
    end
    self.cells = cells
    return self.cells
end

M.getCols = function(self)
    local cellSize = Config.getCellSize()
    return math.floor((self.width + 1) / cellSize)
end

M.getCellAt = function(self, x, y)
    if x < 0 or y < 0 or x >= self.width or y >= self.height then
        return
    end
    local cells = self:getCells()
    local cellSize = Config.getCellSize()
    local col = math.floor(x / cellSize)
    local row = math.floor((y + self.scrollY) / cellSize)
    local cols = self:getCols()
    local index = row * cols + col
    return cells[index + 1]
end

M.updateCache = function(self)
    local inventory = self.context:getInventory()
    local items = inventory:getItems()
    local cache = 0
    for i = 0, items:size() - 1 do
        local item = items:get(i)
        cache = cache + item:getID()
        if item:isEquipped() then
            cache = cache + item:getID()
        end
    end
    cache = cache + Config.getOptionValue("SortingMethod")
    cache = cache + Config.getOptionValue("CharacterEquippedItemsMode")
    if not self.cache or self.cache ~= cache then
        self.cache = cache
        self:invalidate()
    end
end

M.updateSelection = function(self)
    local cells = self:getCells()
    if self:isMouseOver() and isCtrlKeyDown() and isKeyDown(Keyboard.KEY_A) then
        local selected = {}
        for _, cell in ipairs(cells) do
            if cell then
                table.insert(selected, cell)
            end
        end
        self.selected = selected
        return
    end
    local selected = {}
    for _, selection in ipairs(self.selected or {}) do
        for _, cell in ipairs(cells) do
            if cell and cell.type == selection.type then
                table.insert(selected, cell)
            end
        end
    end
    self.selected = selected
end

M.updateDetails = function(self)
    local cells = self:getCells()
    for _, cell in ipairs(cells) do
        if cell then
            for _, item in ipairs(cell.items) do
                item:updateAge()
                if instanceof(item, "Clothing") then
                    item:updateWetness()
                end
            end
        end
    end
end

M.updateTooltip = function(self)
    local hideTooltip = function()
        if self.tooltip then
            self.tooltip:removeFromUIManager()
            self.tooltip:setVisible(false)
        end
    end
    local menu = getPlayerContextMenu(0)
    if not self:isMouseOver() or menu:isAnyVisible() then
        hideTooltip()
        return
    end
    if self.popup and self.popup:isVisible() then
        hideTooltip()
        return
    end
    local relMouseX, relMouseY = self:getMouseX(), self:getMouseY()
    local cell = self:getCellAt(relMouseX, relMouseY)
    if not cell then
        hideTooltip()
        return
    end
    local first = cell.items[1]
    if not self.tooltip then
        self.tooltip = ISToolTipInv:new(first)
        self.tooltip:initialise()
        self.tooltip:setVisible(true)
        self.tooltip:addToUIManager()
        self.tooltip:setOwner(self)
        local player = getPlayer()
        self.tooltip:setCharacter(player)
    else
        self.tooltip:setItem(first)
        self.tooltip:setVisible(true)
        self.tooltip:addToUIManager()
        self.tooltip:bringToTop()
    end
    local weight = 0
    if #cell.items > 1 then
        for _, item in ipairs(cell.items) do
            weight = weight + item:getUnequippedWeight()
        end
    end
    self.tooltip.tooltip:setWeightOfStack(weight)
end

M.updateHighlighted = function(self)
    local relMouseX, relMouseY = self:getMouseX(), self:getMouseY()
    local hovered = self:getCellAt(relMouseX, relMouseY)
    if not hovered then
        for _, worldItem in ipairs(self.highlighted or {}) do
            worldItem:setHighlighted(0, false)
        end
        self.highlighted = nil
        return
    end
    local highlighted = {}
    for _, item in ipairs(hovered.items) do
        local worldItem = item:getWorldItem()
        if worldItem then
            table.insert(highlighted, worldItem)
            if #highlighted > 500 then
                break
            end
        end
    end
    for _, worldItem in ipairs(self.highlighted or {}) do
        local highlight = luautils.tableContains(highlighted, worldItem)
        worldItem:setHighlighted(0, highlight, true)
    end
    self.highlighted = highlighted
end

M.updateDragging = function(self)
    local getDropElement = function()
        local elements = UIManager.getUI()
        local mouseX, mouseY = getMouseX(), getMouseY()
        for i = 0, elements:size() - 1 do
            local element = elements:get(i)
            if element:isPointOver(mouseX, mouseY) then
                return element
            end
        end
    end
    if not ISMouseDrag.dragging or ISMouseDrag.draggingFocus ~= self or isMouseButtonDown(0) then
        return
    end
    local inventory = self.context:getInventory()
    local dropElement = getDropElement()
    if inventory:getType() ~= "floor" and not dropElement then
        for _, item in ipairs(ISMouseDrag.dragging) do
            ISInventoryPaneContextMenu.dropItem(item, 0)
        end
    end
    if ISMouseDrag.draggingFocus then
        ISMouseDrag.draggingFocus:onMouseUp(0, 0)
    end
    ISMouseDrag.draggingFocus = nil
    ISMouseDrag.dragging = nil
end

M.update = function(self)
    self:updateCache()
    self:updateDetails()
    self:updateSelection()
    self:updateTooltip()
    self:updateHighlighted()
    self:updateDragging()
end

M.prerender = function(self)
    if not Config.getOptionValue("DrawGridlines") then
        return
    end
    self:setStencilRect(0, 0, self.width, self.height)
    local cellSize = Config.getCellSize()
    local cols = self:getCols()
    for col = 0, cols do
        local x = col * cellSize
        self:drawRect(x - 1, 0, 2, self.height, 1.0, 0.2, 0.2, 0.2)
    end
    local rows = math.floor(self.height / cols)
    for row = 0, rows do
        local y = row * cellSize - self.scrollY
        self:drawRect(0, y - 1, self.width, 2, 1.0, 0.2, 0.2, 0.2)
    end
    local inventory = self.context:getInventory()
    if not self.parent.cell and inventory:getType() == "none" then
        local equippedMode = Constants.EQUIPPED_MODES[Config.getOptionValue("CharacterEquippedItemsMode")]
        if self.context:isOnCharacter() and equippedMode == "Separated" then
            local cells = self:getCells()
            for i, cell in ipairs(cells) do
                if cell and cell:isEquipped() then
                    local y = math.floor(i / cols) * cellSize - self.scrollY
                    self:drawRect(0, y - 1, self.width, 2, 1.0, 0.4, 0.4, 0.4)
                    break
                end
            end
        end
    end
    self:clearStencilRect()
end

M.render = function(self)
    self:setStencilRect(0, 0, self.width, self.height)
    local cells = self:getCells()
    local cellSize = Config.getCellSize()
    local cols = self:getCols()
    local relMouseX, relMouseY = self:getMouseX(), self:getMouseY()
    for i, cell in ipairs(cells) do
        if cell then
            local index = i - 1
            local col, row = index % cols, math.floor(index / cols)
            local x, y = col * cellSize, row * cellSize - self.scrollY
            if y + cellSize >= 0 and y < self.height then
                cell:render(x, y)
            end
        end
    end
    if self.dragging then
        self:suspendStencil()
        for i, cell in ipairs(cells) do
            if cell and cell:isDragging() then
                local index = i - 1
                local col, row = index % cols, math.floor(index / cols)
                local x, y = col * cellSize, row * cellSize - self.scrollY
                local offsetX = relMouseX - self.mouseDownX
                local offsetY = relMouseY - self.mouseDownY
                if y + cellSize >= 0 and y < self.height then
                    cell:render(x + offsetX, y + offsetY, true)
                end
            end
        end
        self:resumeStencil()
    end
    if isMouseButtonDown(0) and self.selStartX and self.selStartY then
        local x = math.min(relMouseX, self.selStartX)
        local y = math.min(relMouseY, self.selStartY)
        local width = math.abs(relMouseX - self.selStartX)
        local height = math.abs(relMouseY - self.selStartY)
        self:drawRectBorder(x, y, width, height, 1.0, 0.0, 1.0, 0.0)
    end
    self:clearStencilRect()
end

M.onMouseDown = function(self, x, y)
    self.mouseDownX = x
    self.mouseDownY = y
    if self.popup then
        self.popup:close()
    end
    local hovered = self:getCellAt(x, y)
    if not hovered then
        self.selStartX = x
        self.selStartY = y
        self.selected = nil
        return
    end
    if isCtrlKeyDown() then
        if hovered:isSelected() then
            for i, selected in ipairs(self.selected) do
                if selected.type == hovered.type then
                    table.remove(self.selected, i)
                    break
                end
            end
        else
            if not self.selected then
                self.selected = {}
            end
            table.insert(self.selected, hovered)
        end
    elseif isShiftKeyDown() then
        if not self.selected or #self.selected == 0 then
            self.selected = { hovered }
            return
        end
        local cells = self:getCells()
        local lastSelected = self.selected[#self.selected]
        local lastIndex, currentIndex
        for i, cell in ipairs(cells) do
            if cell then
                if cell.type == lastSelected.type then
                    lastIndex = i
                end
                if cell.type == hovered.type then
                    currentIndex = i
                end
            end
        end
        local startIndex = math.min(lastIndex, currentIndex)
        local endIndex = math.max(lastIndex, currentIndex)
        for i = startIndex, endIndex do
            local cell = cells[i]
            if cell and not cell:isSelected() then
                table.insert(self.selected, cell)
            end
        end
    elseif isAltKeyDown() then
        if #hovered.items > 1 then
            hovered:openPopup(x, y)
        end
    elseif not hovered:isSelected() then
        self.selected = { hovered }
    end
end

M.onMouseMove = function(self, _, _)
    if not isMouseButtonDown(0) then
        return
    end
    if self.popup and self.popup.resizeWidget.resizing then
        return
    end
    if self.selStartX and self.selStartY then
        local cellSize = Config.getCellSize()
        local startCol = math.floor(self.selStartX / cellSize)
        local startRow = math.floor(self.selStartY / cellSize)
        local relMouseX, relMouseY = self:getMouseX(), self:getMouseY()
        local endCol = math.floor(relMouseX / cellSize)
        local endRow = math.floor(relMouseY / cellSize)
        local minCol = math.min(startCol, endCol)
        local maxCol = math.max(startCol, endCol)
        local minRow = math.min(startRow, endRow)
        local maxRow = math.max(startRow, endRow)
        local cols = self:getCols()
        local cells = self:getCells()
        local selected = {}
        for row = minRow, maxRow do
            for col = minCol, maxCol do
                local index = row * cols + col
                local cell = cells[index + 1]
                if cell then
                    table.insert(selected, cell)
                end
            end
        end
        self.selected = selected
        return
    end
    if ISMouseDrag.dragging or not self.mouseDownX or not self.mouseDownY then
        return
    end
    local relMouseX, relMouseY = self:getMouseX(), self:getMouseY()
    local hovered = self:getCellAt(relMouseX, relMouseY)
    local dragDistance = math.abs(relMouseX - self.mouseDownX) + math.abs(relMouseY - self.mouseDownY)
    if hovered and dragDistance > 2 then
        self.dragging = self.selected
        ISMouseDrag.dragging = {}
        for _, cell in ipairs(self.dragging) do
            for _, item in ipairs(cell.items) do
                table.insert(ISMouseDrag.dragging, item)
            end
        end
        ISMouseDrag.draggingFocus = self
        if self.popup and self.popup:isVisible() then
            self.popup:close()
        end
    end
end

M.onMouseUp = function(self, x, y)
    if not isCtrlKeyDown() and not isShiftKeyDown() and x == self.mouseDownX and y == self.mouseDownY then
        local cell = self:getCellAt(x, y)
        if cell then
            self.selected = { cell }
        else
            self.selected = nil
        end
    end
    self.selStartX = nil
    self.selStartY = nil
    if ISMouseDrag.dragging then
        if ISMouseDrag.draggingFocus and ISMouseDrag.draggingFocus ~= self then
            local player = getPlayer()
            local target = self.context:getInventory()
            for _, item in ipairs(ISMouseDrag.dragging) do
                if not self.parent.cell or item:getFullType() == self.parent.cell.type then
                    local source = item:getContainer()
                    local action = ISInventoryTransferAction:new(player, item, source, target)
                    ISTimedActionQueue.add(action)
                end
            end
            if ISMouseDrag.draggingFocus then
                ISMouseDrag.draggingFocus:onMouseUp(0, 0)
            end
            ISMouseDrag.dragging = nil
            ISMouseDrag.draggingFocus = nil
            return
        end
        self.dragging = nil
        self.selected = nil
    end
end

M.onRightMouseUp = function(self, x, y)
    if self.popup then
        self.popup:close()
    end
    local hovered = self:getCellAt(x, y)
    if hovered then
        if not hovered:isSelected() then
            self.selected = { hovered }
        end
        local items = {}
        for _, cell in ipairs(self.selected or {}) do
            for _, item in ipairs(cell.items) do
                table.insert(items, item)
            end
        end
        local onCharacter = self.context:isOnCharacter()
        local mouseX, mouseY = getMouseX(), getMouseY()
        local menu = ISInventoryPaneContextMenu.createMenu(0, onCharacter, items, mouseX, mouseY)
        if not menu:isEmpty() then
            if #items > 1 then
                local option = menu:addOption("Open Stack", self, function()
                    hovered:openPopup(x, y)
                end)
                table.remove(menu.options, #menu.options)
                table.insert(menu.options, 1, option)
            end
            menu:setVisible(true)
        end
        return
    end
    local mouseX, mouseY = getMouseX(), getMouseY()
    local menu = ISContextMenu.get(0, mouseX, mouseY)
    local sortingOption = menu:addOption("Sorting Method", nil, nil)
    local sortingMenu = ISContextMenu:getNew(menu)
    menu:addSubMenu(sortingOption, sortingMenu)
    local activeSortingMethodIndex = Config.getOptionValue("SortingMethod")
    for i, method in ipairs(Constants.SORTING_METHODS) do
        local option = sortingMenu:addOption(method, self, function(_)
            Config.setOptionValue("SortingMethod", i)
        end)
        option.checkMark = i == activeSortingMethodIndex
    end
    local equippedOption = menu:addOption("Character Equipped Items", nil, nil)
    local equippedMenu = ISContextMenu:getNew(menu)
    menu:addSubMenu(equippedOption, equippedMenu)
    local activeEquippedModeIndex = Config.getOptionValue("CharacterEquippedItemsMode")
    for i, mode in ipairs(Constants.EQUIPPED_MODES) do
        local option = equippedMenu:addOption(mode, self, function(_)
            Config.setOptionValue("CharacterEquippedItemsMode", i)
        end)
        option.checkMark = i == activeEquippedModeIndex
    end
    local drawGridlines = Config.getOptionValue("DrawGridlines")
    local drawGridlinesOption = menu:addOption("Draw Gridlines", self, function(_)
        Config.setOptionValue("DrawGridlines", not drawGridlines)
    end)
    drawGridlinesOption.checkMark = drawGridlines
end

M.onMouseDoubleClick = function(self, x, y)
    if self.popup then
        self.popup:close()
    end
    local cell = self:getCellAt(x, y)
    if not cell then
        return
    end
    if self.context:isOnCharacter() then
        local item = cell.items[1] or nil
        if item then
            local inventoryPane = self.context:getInventoryPane()
            inventoryPane:doContextualDblClick(item)
        end
        return
    end
    local player = getPlayer()
    local playerInventory = getPlayerInventory(0)
    local target = playerInventory.inventory
    for _, item in ipairs(cell.items) do
        local source = item:getContainer()
        local action = ISInventoryTransferAction:new(player, item, source, target)
        ISTimedActionQueue.add(action)
    end
end

M.onMouseWheel = function(self, delta)
    if isShiftKeyDown() then
        return false
    end
    local cellSize = Config.getCellSize()
    self.scrollY = self.scrollY + delta * cellSize / 3
    self:clampScroll()
    return true
end

M.clampScroll = function(self)
    local cellSize = Config.getCellSize()
    local cols = self:getCols()
    local cells = self:getCells()
    local rows = math.ceil(#cells / cols)
    local contentHeight = rows * cellSize
    local bottomGap = math.min(cellSize * 2, self.height * 0.5)
    local maxScroll = contentHeight + bottomGap - self.height
    self.scrollY = math.max(math.min(self.scrollY, maxScroll), 0)
end

M.onResize = function(self, width, height)
    local valid = self.width and self.height
    ISPanel.onResize(self, width, height)
    if valid then
        self:invalidate()
    end
    self:clampScroll()
end

return M
