local Config = require("InvMatrixConfig")

local M = {}
M.__index = M

M.new = function(_, parent, type, items)
    local o = {}
    setmetatable(o, M)
    o.parent = parent
    o.type = type
    o.items = items or {}
    o.attachedIcon = getTexture("media/ui/iconInHotbar.png")
    o.brokenIcon = getTexture("media/ui/icon_broken.png")
    o.equippedIcon = getTexture("media/ui/icon.png")
    o.favoriteIcon = getTexture("media/ui/FavoriteStar.png")
    o.frozenIcon = getTexture("media/ui/icon_frozen.png")
    o.poisonIcon = getTexture("media/ui/SkullPoison.png")
    o.tickIcon = getTexture("media/ui/Tick_Mark-10.png")
    return o
end

M.getName = function(self)
    local item = self.items[1] or nil
    return item and item:getName()
end

M.isHovered = function(self)
    if not self.parent:isMouseOver() then
        return
    end
    local relMouseX, relMouseY = self.parent:getMouseX(), self.parent:getMouseY()
    local hovered = self.parent:getCellAt(relMouseX, relMouseY)
    return hovered and hovered.type == self.type
end

M.isSelected = function(self)
    for _, cell in ipairs(self.parent.selected or {}) do
        if cell.type == self.type then
            return true
        end
    end
    return false
end

M.isDragging = function(self)
    for _, cell in ipairs(self.parent.dragging or {}) do
        if cell.type == self.type then
            return true
        end
    end
    return false
end

M.isFavorite = function(self)
    for _, item in ipairs(self.items) do
        if not item:isFavorite() then
            return false
        end
    end
    return true
end

M.isEquipped = function(self)
    for _, item in ipairs(self.items) do
        if not item:isEquipped() then
            return false
        end
    end
    return true
end

M.isAttached = function(self)
    local hotbar = getPlayerHotbar(0)
    for _, item in ipairs(self.items) do
        if not hotbar:isInHotbar(item) then
            return false
        end
    end
    return true
end

M.isBroken = function(self)
    for _, item in ipairs(self.items) do
        if not item:isBroken() then
            return false
        end
    end
    return true
end

M.isFrozen = function(self)
    for _, item in ipairs(self.items) do
        if not instanceof(item, "Food") or not item:isFrozen() then
            return false
        end
    end
    return true
end

M.isPoison = function(self)
    local isPoison = function(item)
        local player = getPlayer()
        if player:isKnownPoison(item) then
            return true
        end
        local options = getSandboxOptions()
        local option = options:getOptionByName("EnableTaintedWaterText")
        local warnTainted = option:getValue()
        if warnTainted and instanceof(item, "Food") and item:isTainted() then
            return true
        end
        local fluidContainer = item:getFluidContainer()
        if not fluidContainer then
            local worldItem = item:getWorldItem()
            fluidContainer = worldItem and worldItem:getFluidContainer()
        end
        if fluidContainer then
            if fluidContainer:contains(Fluid.Bleach) then
                return true
            end
            if fluidContainer:contains(Fluid.TaintedWater) and fluidContainer:getPoisonRatio() > 0.1 then
                return true
            end
        end
        return false
    end
    for _, item in ipairs(self.items) do
        if not isPoison(item) then
            return false
        end
    end
    return true
end

M.isCompleted = function(self)
    local isCompleted = function(item)
        local player = getPlayer()
        if item:hasBeenSeen(player) or item:hasBeenHeard(player) or player:hasReadMap(item) then
            return true
        end
        if ISInventoryPane.isLiteratureRead(self.parent, player, item) then
            return true
        end
        return false
    end
    for _, item in ipairs(self.items) do
        if not isCompleted(item) then
            return false
        end
    end
    return true
end

M.isUnwanted = function(self)
    local player = getPlayer()
    for _, item in ipairs(self.items) do
        if not item:isUnwanted(player) then
            return false
        end
    end
    return true
end

M.drawBackground = function(self, x, y, alpha, red, green, blue)
    local cellSize = Config.getCellSize()
    local left, top = x + 1, y + 1
    local width, height = cellSize - 2, cellSize - 2
    self.parent:drawRect(left, top, width, height, alpha, red, green, blue)
end

M.drawBorder = function(self, x, y, alpha, red, green, blue)
    local cellSize = Config.getCellSize()
    self.parent:drawRectBorder(x, y, cellSize, cellSize, alpha, red, green, blue)
end

M.drawIcon = function(self, x, y, dragging)
    local item = self.items[1] or nil
    if not item then
        return
    end
    local texture = item:getTexture()
    if not texture then
        return
    end
    local width, height = texture:getWidth(), texture:getHeight()
    local cellSize = Config.getCellSize()
    local size = cellSize / 1.5
    local scale = size / math.max(width, height)
    width, height = width * scale, height * scale
    local left = x + cellSize / 2 - width / 2
    local top = y + cellSize / 2 - height / 2
    local red, green, blue = item:getR(), item:getG(), item:getB()
    local colorMask = item:getTextureColorMask()
    if colorMask then
        red, green, blue = 1.0, 1.0, 1.0
    end
    if self:isUnwanted() then
        red, green, blue = 1.0, 0.0, 0.0
    end
    local alpha = not dragging and self:isDragging() and 0.4 or 1.0
    self.parent:drawTextureScaled(texture, left, top, width, height, alpha, red, green, blue)
    if colorMask and self.parent.drawTextureIconMask then
        self.parent:drawTextureIconMask(colorMask, 1.0, left, top, width, height, red, green, blue, alpha)
    end
end

M.drawIndicator = function(self, texture, x, y, offsetX, offsetY)
    local cellSize = Config.getCellSize()
    local left = x + cellSize * offsetX - texture:getWidth() / 2
    local top = y + cellSize * offsetY - texture:getHeight() / 2
    self.parent:drawTexture(texture, left, top, 1, 1, 1, 1)
end

M.drawStackCount = function(self, x, y)
    local cellSize = Config.getCellSize()
    local text = tostring(#self.items)
    local textManager = getTextManager()
    local textWidth = textManager:MeasureStringX(UIFont.Small, text)
    local left, top = x + cellSize - textWidth - 4, y
    local offsets = {
        { x = -1, y = -1 },
        { x = -1, y = 1 },
        { x = 1, y = 1 },
        { x = 1, y = -1 },
    }
    for _, offset in ipairs(offsets) do
        self.parent:drawText(text, left + offset.x, top + offset.y, 0.0, 0.0, 0.0, 1.0, UIFont.Small, true)
    end
    self.parent:drawText(text, left, top, 1.0, 1.0, 1.0, 1.0, UIFont.Small, true)
end

M.render = function(self, x, y, dragging)
    if self:isSelected() then
        self:drawBackground(x, y, 1.0, 0.7, 0.7, 0.7)
    elseif self:isHovered() then
        self:drawBackground(x, y, 1.0, 0.5, 0.5, 0.5)
    end
    self:drawIcon(x, y, dragging)
    if not dragging then
        if self:isFavorite() then
            self:drawIndicator(self.favoriteIcon, x, y, 0.2, 0.8)
        end
        if self:isEquipped() then
            self:drawIndicator(self.equippedIcon, x, y, 0.8, 0.8)
        elseif self:isAttached() then
            self:drawIndicator(self.attachedIcon, x, y, 0.8, 0.8)
        elseif self:isBroken() then
            self:drawIndicator(self.brokenIcon, x, y, 0.8, 0.8)
        elseif self:isFrozen() then
            self:drawIndicator(self.frozenIcon, x, y, 0.8, 0.8)
        elseif self:isPoison() then
            self:drawIndicator(self.poisonIcon, x, y, 0.8, 0.8)
        elseif self:isCompleted() then
            self:drawIndicator(self.tickIcon, x, y, 0.8, 0.8)
        end
    end
    if #self.items > 1 then
        self:drawStackCount(x, y)
    end
end

M.openPopup = function(self, x, y)
    local cellSize = Config.getCellSize()
    local col = math.floor(x / cellSize)
    local row = math.floor((y + self.parent.scrollY) / cellSize)
    local parentAbsX, parentAbsY = self.parent:getAbsoluteX(), self.parent:getAbsoluteY()
    local left = parentAbsX + col * cellSize
    local top = parentAbsY + row * cellSize + cellSize - self.parent.scrollY
    local width, height = cellSize * 6, cellSize * 6
    local Popup = require("InvMatrixPopup")
    local popup = Popup:new(left, top, width, height, self.parent)
    popup:setVisible(true)
    popup:addToUIManager()
    popup:bringToTop()
    popup:setCell(self)
    self.parent.popup = popup
end

return M
