-- Proximity Inventory 42.13 Compatibility
-- Client-side helper functions
-- Original mod by mxswat: https://steamcommunity.com/sharedfiles/filedetails/?id=2847184718

-- GetLocalContainer - creates a virtual container to hold proximity items
function ISInventoryPage.GetLocalContainer(playerNum)
    if ISInventoryPage.localContainer == nil then
        ISInventoryPage.localContainer = {}
    end
    if ISInventoryPage.localContainer[playerNum+1] == nil then
        -- Build 42 ItemContainer constructors:
        -- ItemContainer() or ItemContainer(String name, IsoGridSquare, IsoObject)
        local container = ItemContainer.new("local", nil, nil)

        if container then
            ISInventoryPage.localContainer[playerNum+1] = container
            if container.setExplored then
                container:setExplored(true)
            end
            if container.setOnlyAcceptCategory then
                container:setOnlyAcceptCategory("none")
            end
        end
    end
    return ISInventoryPage.localContainer[playerNum+1]
end

-- Right-click context menu handler
local old_ISInventoryPage_onBackpackRightMouseDown = ISInventoryPage.onBackpackRightMouseDown
function ISInventoryPage:onBackpackRightMouseDown(x, y)
    local result

    -- Only call original if we have a valid inventory with a parent object
    if old_ISInventoryPage_onBackpackRightMouseDown then
        local inv = self.inventory
        -- Skip calling original for "local" type containers (our virtual container)
        if inv and inv.getType and inv:getType() ~= "local" then
            local success, res = pcall(old_ISInventoryPage_onBackpackRightMouseDown, self, x, y)
            if success then
                result = res
            end
        end
    end

    if not self.parent then return result end

    local page = self.parent
    local container = self.inventory

    if container and container.getType and container:getType() == "local" then
        -- Get player number safely - use 0 as default for single player
        local playerNum = 0
        if page and page.player then
            playerNum = page.player
        elseif self.player then
            playerNum = self.player
        end

        -- Create context menu using the simpler constructor
        local context = ISContextMenu.get(playerNum, getMouseX(), getMouseY())
        if context and ProxInv and ProxInv.populateContextMenuOptions then
            ProxInv.populateContextMenuOptions(context, self)
        end
    end

    return result
end

-- Update function for highlighting containers
local old_ISInventoryPage_update = ISInventoryPage.update
function ISInventoryPage:update()
    local result
    if old_ISInventoryPage_update then
        local success, res = pcall(old_ISInventoryPage_update, self)
        if success then
            result = res
        end
    end

    if self.onCharacter then
        return result
    end

    self.coloredProxInventories = self.coloredProxInventories or {}

    -- Clear previous highlighting
    for _, container in ipairs(self.coloredProxInventories) do
        if container and container.getParent then
            local parent = container:getParent()
            if parent and parent.setHighlighted then
                pcall(function() parent:setHighlighted(false) end)
            end
        end
    end

    -- Clear the cache table
    self.coloredProxInventories = {}

    -- Apply highlighting if enabled
    if ProxInv and ProxInv.isHighlightEnable and not self.isCollapsed and self.inventory and self.inventory.getType and self.inventory:getType() == "local" then
        for _, container in ipairs(ProxInv.containerCache or {}) do
            if container and container.getParent then
                local parent = container:getParent()
                if parent and (instanceof(parent, "IsoObject") or instanceof(parent, "IsoDeadBody")) then
                    if parent.setHighlighted then
                        pcall(function() parent:setHighlighted(true, false) end)
                    end
                    if parent.setHighlightColor and getCore then
                        pcall(function()
                            local core = getCore()
                            if core and core.getObjectHighlitedColor then
                                parent:setHighlightColor(core:getObjectHighlitedColor())
                            end
                        end)
                    end
                    table.insert(self.coloredProxInventories, container)
                end
            end
        end
    end

    return result
end
