-- Proximity Inventory 42.13 Compatibility
-- Original mod by mxswat: https://steamcommunity.com/sharedfiles/filedetails/?id=2847184718
-- Updated for Project Zomboid 42.13 compatibility

ProxInv = ProxInv or {}
ProxInv.isToggled = true
ProxInv.isHighlightEnable = true
ProxInv.isForceSelected = false
ProxInv.inventoryIcon = getTexture("media/ui/ProximityInventory.png")
ProxInv.isLocalContainerSelected = false
ProxInv.buttonCache = nil
ProxInv.containerCache = {}

ProxInv.toggleState = function()
    ProxInv.isToggled = not ProxInv.isToggled
    if ISInventoryPage and ISInventoryPage.dirtyUI then
        ISInventoryPage.dirtyUI()
    end
end

ProxInv.setForceSelected = function()
    ProxInv.isForceSelected = not ProxInv.isForceSelected
    if ISInventoryPage and ISInventoryPage.dirtyUI then
        ISInventoryPage.dirtyUI()
    end
end

ProxInv.setHighlightEnable = function()
    ProxInv.isHighlightEnable = not ProxInv.isHighlightEnable
    if ISInventoryPage and ISInventoryPage.dirtyUI then
        ISInventoryPage.dirtyUI()
    end
end

ProxInv.resetContainerCache = function()
    ProxInv.containerCache = {}
end

ProxInv.getTooltip = function()
    local text = getText("IGUI_ProxInv_RightClickSettings") or "Right click for settings"
    text = not ProxInv.isToggled and (getText("IGUI_ProxInv_Disabled") or "Disabled") .. " - " .. text or text
    return text
end

ProxInv.zombieTypes = {
    inventoryfemale = true,
    inventorymale = true,
}

ProxInv.canBeAdded = function(container, playerObj)
    if not container then return false end

    -- Do not allow if it's a stove or washer or similar "Active things"
    -- It can cause issues like the item stops cooking or stops drying
    -- Also don't allow to see inside containers locked to you
    local object = container:getParent()

    if SandboxVars and SandboxVars.ProxInv and SandboxVars.ProxInv.ZombieOnly then
        return ProxInv.zombieTypes[container:getType()]
    end

    if object and instanceof(object, "IsoThumpable") and object:isLockedToCharacter(playerObj) then
        return false
    end

    return true
end

ProxInv.populateContextMenuOptions = function(context)
    local toggleText = ProxInv.isToggled and "OFF" or "ON"
    local optToggle = context:addOption(getText("IGUI_ProxInv_Toggle") or "Toggle " .. toggleText, nil, ProxInv.toggleState)
    if ProxInv.inventoryIcon then
        optToggle.iconTexture = ProxInv.inventoryIcon
    end

    local forceSelectedText = ProxInv.isForceSelected and (getText("IGUI_ProxInv_Disable") or "Disable") or (getText("IGUI_ProxInv_Enable") or "Enable")
    local optForce = context:addOption(forceSelectedText .. " " .. (getText("IGUI_ProxInv_ForceSelected") or "Force Selected"), nil, ProxInv.setForceSelected)
    local pinIcon = getTexture("media/ui/Panel_Icon_Pin.png")
    if pinIcon then
        optForce.iconTexture = pinIcon
    end

    local highlightText = ProxInv.isHighlightEnable and (getText("IGUI_ProxInv_Disable") or "Disable") or (getText("IGUI_ProxInv_Enable") or "Enable")
    context:addOption(highlightText .. " " .. (getText("IGUI_ProxInv_Highlight") or "Highlight"), nil, ProxInv.setHighlightEnable)
end

ProxInv.OnButtonsAdded = function(invSelf)
    if not invSelf then return end

    local playerObj = getSpecificPlayer(invSelf.player)
    if not playerObj then return end

    local localContainer = ISInventoryPage.GetLocalContainer(invSelf.player)
    if not localContainer then return end

    -- Clear the local container safely
    pcall(function()
        if localContainer.clear then
            localContainer:clear()
        elseif localContainer.getItems then
            local items = localContainer:getItems()
            if items and items.clear then
                items:clear()
            end
        end
    end)

    local title = getText("IGUI_ProxInv_Title") or "Proximity Inv"

    -- Try to add the container button with error handling
    local proxInvButton
    if invSelf.addContainerButton then
        local success, btn = pcall(function()
            return invSelf:addContainerButton(localContainer, ProxInv.inventoryIcon, title, ProxInv.getTooltip())
        end)
        if success then
            proxInvButton = btn
        end
    end

    if not proxInvButton then return end

    proxInvButton.capacity = 0
    -- Note: Let Build 42 handle button positioning automatically
    ProxInv.buttonCache = proxInvButton
    ProxInv.resetContainerCache()

    -- Add all backpacks content except last which is proxInv
    if invSelf.backpacks then
        for i = 1, (#invSelf.backpacks - 1) do
            local buttonToPatch = invSelf.backpacks[i]
            if buttonToPatch and buttonToPatch.inventory then
                local invToAdd = buttonToPatch.inventory
                if ProxInv.canBeAdded(invToAdd, playerObj) then
                    local items = invToAdd:getItems()
                    if items and proxInvButton.inventory and proxInvButton.inventory.getItems then
                        local proxItems = proxInvButton.inventory:getItems()
                        if proxItems and proxItems.addAll then
                            proxItems:addAll(items)
                        end
                    end
                    table.insert(ProxInv.containerCache, invToAdd)
                end
                -- Note: Button Y positioning removed - Build 42 handles this automatically
            end
        end
    end

    if not ProxInv.isToggled then
        -- Remove the backpack from the list
        if invSelf.backpacks then
            table.remove(invSelf.backpacks, #invSelf.backpacks)
        end
        return
    end

    if ProxInv.isToggled and ProxInv.isForceSelected then
        if invSelf.setForceSelectedContainer then
            invSelf:setForceSelectedContainer(ISInventoryPage.GetLocalContainer(invSelf.player))
        end
    end

    if ProxInv.isAsFirst and invSelf.backpacks then
        -- Remove from last
        local removedButton = table.remove(invSelf.backpacks, #invSelf.backpacks)
        -- Rebuild table and put it as first
        if removedButton then
            invSelf.backpacks = { removedButton, table.unpack(invSelf.backpacks) }
        end
    end
end

ProxInv.OnBeginRefresh = function(invSelf)
    -- This avoids the generation of multiple buttons when it's off
    -- Since children gets removed via #invSelf.backpacks, and when it's toggled off the button does not appear
    -- in the #invSelf.backpacks
    if ProxInv.isToggled then
        return
    end
    if invSelf and invSelf.removeChild and ProxInv.buttonCache then
        invSelf:removeChild(ProxInv.buttonCache)
    end
end

ProxInv.OnRefreshInventoryWindowContainers = function(invSelf, state)
    if not invSelf then return end

    local playerObj = getSpecificPlayer(invSelf.player)
    if not playerObj then return end

    if invSelf.onCharacter or playerObj:getVehicle() then
        -- Ignore character containers, as usual
        -- Ignore in vehicles
        return
    end

    if state == "begin" then
        return ProxInv.OnBeginRefresh(invSelf)
    end

    if state == "buttonsAdded" then
        return ProxInv.OnButtonsAdded(invSelf)
    end
end

Events.OnRefreshInventoryWindowContainers.Add(ProxInv.OnRefreshInventoryWindowContainers)

-- Key binding setup
local KEY_ForceSelected = {
    name = "ProxInv_Force_Selected",
    key = Keyboard.KEY_NUMPAD0,
}

if ModOptions and ModOptions.AddKeyBinding then
    ModOptions:AddKeyBinding("[ProximityInventory]", KEY_ForceSelected)
end

local function OnKeyPressed(keynum)
    local player = getSpecificPlayer(0)
    if not player then
        return
    end

    if keynum == KEY_ForceSelected.key then
        ProxInv.setForceSelected()
        local text = (getText("IGUI_ProxInv_Force_Selected") or "Prox Inv Force Selected: ") .. (ProxInv.isForceSelected and "ON" or "OFF")
        if HaloTextHelper and HaloTextHelper.addText and HaloTextHelper.getColorWhite then
            HaloTextHelper.addText(player, text, HaloTextHelper.getColorWhite())
        end
        return
    end
end

Events.OnKeyPressed.Add(OnKeyPressed)

-- ISCraftingUI compatibility
if ISCraftingUI then
    local old_ISCraftingUI_getContainers = ISCraftingUI.getContainers
    function ISCraftingUI:getContainers()
        local result = old_ISCraftingUI_getContainers(self)
        if not self.character or not ProxInv.isToggled then
            return result
        end

        -- If ProxInv is enabled:
        local localContainer = ISInventoryPage.GetLocalContainer(self.playerNum)
        if localContainer and self.containerList and self.containerList.remove then
            self.containerList:remove(localContainer)
        end
        return result
    end
end

-- ISInventoryPaneContextMenu compatibility
if ISInventoryPaneContextMenu then
    local old_ISInventoryPaneContextMenu_getContainers = ISInventoryPaneContextMenu.getContainers
    ISInventoryPaneContextMenu.getContainers = function(character)
        if not character or not ProxInv.isToggled then
            return old_ISInventoryPaneContextMenu_getContainers(character)
        end

        local containerList = old_ISInventoryPaneContextMenu_getContainers(character)
        if not containerList then
            return containerList
        end

        local localContainer = ISInventoryPage.GetLocalContainer(character:getPlayerNum())
        -- If ProxInv is enabled:
        if localContainer and containerList.remove then
            containerList:remove(localContainer)
        end

        return containerList
    end
end
