require "EventSystem/EventSystem"


local checkPlayerDist = 250
local convertDist = 75
local spawnRadius = 5

local allowedZoneTypes = {
	"Nav",
	"ForagingNav",
	"Farm",
	"FarmLand",
	"Forest",
	"DeepForest",
	"BirchForest",
	"BirchMixForest",
	"FarmForest",
	"FarmMixForest",
	"PRForest",
	"PHForest",
	"PHMixForest",
	"OrganicForest",
	"Vegitation",
	"TownZone",
	"TrailerPark",
	"Region",
}

Horde = {
	speed = 10,
	dir = "",
	posXold = -1,
	posYold = -1,
	targetZone = nil,
	targetZoneOld = nil,
	targetX = nil,
	targetY = nil,
	changeX = 0,
	changeY = 0,
 
	leader = nil,
	leaderHasTarget = false,

	isVirtual = true,
	isRepositioning = false,
	isStuck = false,

	sfx = nil,

	leader_timeoutCounter = 0,
	collect_timeoutCounter = 0,
	wait_timeoutCounter = 0,
	return_timeoutCounter = 0,

	growth_daysCounter = 0,
	virtualDisbandAt = nil,
}


function Horde:new(o)
	setmetatable(o, self)
	self.__index = self
	o:init()
	return o
end


function Horde:init()
	self.targetX = self.posX
	self.targetY = self.posY 
	self.zombieList = {}
	self.modData = {
		name = self.name,
		logName = self.logName,
		type = self.type,
		size = self.size,
		dir = self.dir,
		targetX = self.targetX,
		targetY = self.targetY,
		targetZoneX = self.targetZone and self.targetZone:getX() or nil,
		targetZoneY = self.targetZone and self.targetZone:getY() or nil,
		targetZoneType = self.targetZone and self.targetZone:getType() or nil,
	}
	
	if not self.posX or not self.posX then
		self:log("STUCK FROM NO POSITION!")
		self:disband()
		return
	end
	local p = getPlayer()
	if IsoUtils.DistanceManhattenSquare(self.posX, self.posY, p:getX(), p:getY()) < convertDist then
		self:log("INIT")
		self.isVirtual = false
		self:collectZombiesStart(25, self.virtualToReal2)
		self.virtualDisbandAt = nil
	elseif self.name == "Temp" then
		local hours = tonumber(SandboxVars.Hordes.TempHordeVirtualHours) or 24
		if hours > 0 then
			self.virtualDisbandAt = GameTime:getInstance():getWorldAgeHours() + hours
		else
			self.virtualDisbandAt = nil
		end
	end
end




--------------------------------
-- UPDATE
--------------------------------



function Horde:update()
	if self.modData.size < 10 then
		self:log("DISBAND SMALL HORDE")
		self:disband()
		return
	end
	if self.collect_timeoutCounter > 0 then return end

	-- restore persisted navigation state if available
	if self.targetZone == nil and self.modData.targetZoneX and self.modData.targetZoneY then
		local zone = getWorld():getMetaGrid():getZoneAt(self.modData.targetZoneX, self.modData.targetZoneY, 0)
		if zone and zone:getType() == self.modData.targetZoneType then
			self.targetZone = zone
		end
	end
	if self.dir == "" and self.modData.dir then
		self.dir = self.modData.dir
	end
	if not self.targetX and self.modData.targetX then
		self.targetX = self.modData.targetX
	end
	if not self.targetY and self.modData.targetY then
		self.targetY = self.modData.targetY
	end

	-- auto-disband temp hordes that stay virtual too long
	if self.isVirtual and self.name == "Temp" and self.virtualDisbandAt and GameTime:getInstance():getWorldAgeHours() >= self.virtualDisbandAt then
		self:log("DISBAND VIRTUAL TEMP HORDE")
		self:disband()
		return
	end

	if not self.isVirtual then
		if self.leader == nil then
			self.leader = self.zombieList[0]
		end
		-- if GameTime.getInstance():getMultiplier() > 10 then return end
	end

	local p = getPlayer()

	local dist = IsoUtils.DistanceManhattenSquare(self.posX, self.posY, p:getX(), p:getY())
	if dist < checkPlayerDist then
		EventSystem.addEventListener(self, "OnPlayerUpdate",  "checkPlayerDistance")		
	end

	if self.leader then
		-- self.leader:Say(self.name, 1.0, 1.0, 1.0, UIFont.Dialogue, 30.0, "radio")

	-- SPECIAL LEADER ACTIONS --

		-- is leader targeting something?
		if self.leader:getTarget() or self.leader:getPathFindBehavior2():isGoalSound() then
			if self.leader:getTarget() then
				self:log("leader has PLAYER target!")
			else
				self:log("leader has SOUND target!")
			end

			for _,zomb in ipairs(self.zombieList) do
				zomb:pathToLocation(p:getX(), p:getY(), p:getZ())
			end

			self.leaderHasTarget = true
			if self.posXold == -1 then
				--remember pos where leader went off course
				self.posXold = self.posX
				self.posYold = self.posY
			end
			self.posX = self.leader:getX()
			self.posY = self.leader:getY()
			return
		elseif self.leaderHasTarget then
			-- leader had target, but lost it
			self:log("LOST TARGET")
			-- try to navigate from current pos 
			self.targetZone = nil
			self.targetZoneOld = nil
			self.posX = self.leader:getX()
			self.posY = self.leader:getY()
			local roads = self:findRoads(self.posX, self.posY, true)
			if roads and #roads > 0 then 
				--done!
				self.leaderHasTarget = false
				self.posXold = -1
				self.posYold = -1
				self:navigateRoads(roads, true)
				return
			else
				-- if we can't navigate from current pos, keep going back to old pos
				self.return_timeoutCounter = self.return_timeoutCounter+1
				if self.return_timeoutCounter == 5 then
					self.return_timeoutCounter = 0
					self:log("FAILED: RETURNING TO LAST POS")
					self:disband()
					return
				end
				self:log("RETURNING TO LAST POS",self.return_timeoutCounter)
				self:moveTo(self.posXold, self.posYold)
				return
			end
		else
			-- does group need to catch up?
			local distance = 0
			local zombCounter = 0
			for _,zomb in ipairs(self.zombieList) do
				distance = IsoUtils.DistanceManhattenSquare(zomb:getX(), zomb:getY(), self.posX, self.posY)
				if distance > 12 then
					zombCounter = zombCounter+1
				elseif distance < 2 then
					self.leader = zomb
				end
			end
			if zombCounter > #self.zombieList/4 then
				self.wait_timeoutCounter = self.wait_timeoutCounter+1
				self:log("WAIT FOR GROUP",self.wait_timeoutCounter)
				self:moveTo()
				return
			else
				self.wait_timeoutCounter = 0
			end
		end		
	end

	-- NORMAL MOVEMENT ACTIONS --
	if self.parent ~= nil then
		if self.parent.posX ~= self.posX or self.parent.posY ~= self.posY then
			self.posX = self.parent.posX
			self.posY = self.parent.posY
		end
	else
		if self.targetZone == nil then
			self:findRoadsAndNavigate(true)--usually freshly spawned horde
		else
			local distance = IsoUtils.DistanceManhattenSquare(self.posX, self.posY, self.targetX, self.targetY)
			if distance == 0 then-- arrived at target?
				self:findRoadsAndNavigate(true)
				self.isRepositioning = true
			elseif distance <= self.speed then-- near target?
				self.posX = self.targetX
				self.posY = self.targetY
			else
				if p:getVehicle() or GameTime.getInstance():getMultiplier() > 10 then -- if things are moving too fast...
					if (self.isVirtual and dist > convertDist+self.speed) or (not self.isVirtual and dist < convertDist+10-self.speed) then -- ...don't jump over convertDist
						if self.isRepositioning == true then -- don't navigate right after choosing a new target
							self.posX = self.posX + self.changeX
							self.posY = self.posY + self.changeY
							self.isRepositioning = false
						elseif self:findRoadsAndNavigate(false) == false then-- if no new target...
							-- ...then keep moving
							self.posX = self.posX + self.changeX
							self.posY = self.posY + self.changeY
						end
					end
				else
					if self.isRepositioning == true then -- don't navigate right after choosing a new target
						self.posX = self.posX + self.changeX
						self.posY = self.posY + self.changeY
						self.isRepositioning = false
					elseif self:findRoadsAndNavigate(false) == false then-- if no new target...
						-- ...then keep moving
						self.posX = self.posX + self.changeX
						self.posY = self.posY + self.changeY
					end
				end
			end
		end
	end

	self:moveTo()
end


function Horde:moveTo(x,y)
	if x then self.posX = x end
	if y then self.posY = y end
	self.modData.posX = self.posX
	self.modData.posY = self.posY
	if self.isVirtual == true or self.leader == nil then return end

	for _,zomb in ipairs(self.zombieList) do
		if zomb:getModData().offsetX then 
			zomb:pathToLocation(self.posX+zomb:getModData().offsetX, self.posY+zomb:getModData().offsetY, 0)
		end
	end
end



function Horde:checkPlayerDistance()

	-- make horde virtual/real depending on player distance
	local p = getPlayer()	
	local distance = IsoUtils.DistanceManhattenSquare(self.posX, self.posY, p:getX(), p:getY())

	if self.isVirtual == true then
		if distance < convertDist then
			-- spawnHorde(x1,y1,x2,y2,z, size)
			-- createHordeFromTo(x,y,xTO,yTO, size)
			-- createHordeInAreaTo(x,y,z?,radius?,xTO,yTO, size)
			-- addVirtualZombie
			spawnHorde(self.posX+spawnRadius,self.posY+spawnRadius,self.posX-spawnRadius,self.posY-spawnRadius,0, self.modData.size)
			self:virtualToReal()
		elseif distance > checkPlayerDist+10 then
			EventSystem.removeEventListener(self, "OnPlayerUpdate",  "checkPlayerDistance")
		end
	else
		if distance > convertDist+5 then
			self:realToVirtual()
			if self.sfx then self.sfx:stop() end
		else
			if self.leader and self.modData.size >= 20 then
				if not self.sfx then 
					self.sfx = SoundFX:new() 
					self.sfx:setVolume(0.1)
				end
				self.sfx:setPos(self.leader:getX(),self.leader:getY(),self.leader:getZ())
				if not self.sfx:isPlaying() then self.sfx:play("horde"..ZombRand(1,5)) end
			end
		end
	end
end




--------------------------------
-- NAVIGATION
--------------------------------



function Horde:findRoadsAndNavigate(arrivedAtTarget)
	local choices = {}
	if arrivedAtTarget or ZombRand(0, 10) == 0 then
		choices = self:findRoads(self.posX, self.posY, true)
		-- if not choices or #choices == 0 then
		-- 	print("No zones found near " .. self.posX .. "," .. self.posY)
		-- end
		return self:navigateRoads(choices, arrivedAtTarget)
	elseif ZombRand(0, 5) == 0 then
		choices = self:findRoads(self.posX, self.posY, false)
		-- if not choices or #choices == 0 then
		-- 	print("No zones found near " .. self.posX .. "," .. self.posY)
		-- end
		return self:navigateRoads(choices, arrivedAtTarget)
	end
	return false
end

function Horde:findRoads(wx, wy, checkAdjacent)
	local metaChunk = nil
	local zones = {}

	if checkAdjacent then
		local dist = 10--checks all adjacent chunks
		for x = wx-dist,wx+dist,dist do
			for y = wy-dist,wy+dist,dist do
				metaChunk = getWorld():getMetaChunkFromTile(x, y)
				self:addZonesFromChunk(metaChunk, zones)
			end
		end	
	else
		metaChunk = getWorld():getMetaChunkFromTile(wx, wy)
		self:addZonesFromChunk(metaChunk, zones)
	end
	-- if not zones or #zones == 0 then
	-- 	print("No zones found from metaChunk at " .. wx .. "," .. wy)
	-- end
	return self:checkZones(zones)
end

function Horde:addZonesFromChunk(chunk, zones)
	if not chunk then return end
	for i = 0, chunk:getZonesSize()-1 do
		local zone = chunk:getZone(i)
		local duplicate = false
		for _,z in pairs(zones) do
			if z == zone then 
				duplicate = true
				break
			end
		end
		if duplicate == false then 
			zones[#zones+1] = zone
		end
	end
end

function Horde:checkZones(zones)
	local choices = {}
	for _,z in pairs(zones) do
		local choice = self:checkZone(z)
		if choice then choices[#choices+1] = choice end
	end
	-- if not choices or #choices == 0 then
	-- 	print("No valid choices found for zones at " .. self.posX .. "," .. self.posY)
	-- 	for _,z in pairs(zones) do
	-- 		print(" - zone type: " .. z:getType() .. " size: " .. z:getWidth() .. "x" .. z:getHeight())
	-- 	end
	-- end
	return choices
end	

function Horde:checkZone(z)
	local choice = nil
	if SandboxVars.Hordes.NavigateOutsideHighways then
		choice = Horde:checkZoneNavAndOther(z)
	else
		choice = Horde:checkZoneNavOnly(z)
	end
	return choice
end

function Horde:checkZoneNavOnly(z)
	local choice = nil
	if (z:getType() == "Nav") and z ~= self.targetZone and z ~= self.targetZoneOld then
		local W = z:getWidth()
		local H = z:getHeight()
		if z:getType() == "Nav" and W >= 20 and H >= 20 then -- avoid large areas like parking lots
			return nil
		elseif W <= 5 and H <= 5 then
			return nil
		else
			local chance
			if W < H then
				chance = W*30 --prefer wide roads
			else
				chance = H*30
			end
			if W > H then
				-- W<->E
				if self.dir=="E" and z:getX()+z:getWidth()*0.5 > self.posX then
					choice = {z, chance*2} -- prefer going straight ahead
				elseif self.dir=="W" and z:getX()+z:getWidth()*0.5 < self.posX then
					choice = {z, chance*2} -- prefer going straight ahead
				elseif self.dir == "N" or self.dir == "S" then
					choice = {z, chance} -- making a turn
				elseif self.dir == "" then
					choice = {z, chance}
				end
			else
				-- N<->S
				if self.dir=="S" and z:getY()+z:getHeight()*0.5 > self.posY then
					choice = {z, chance*2}
				elseif self.dir=="N" and z:getY()+z:getHeight()*0.5 < self.posY then
					choice = {z, chance*2}
				elseif self.dir=="W" or self.dir=="E" then
					choice = {z, chance}
				elseif self.dir == "" then
					if z:getType() == "Forest" then chance = chance/20 end
					choice = {z, chance}
				end
			end
		end
	end
	return choice
end

function Horde:checkZoneNavAndOther(z)
	local choice = nil
	local zoneType = z:getType()
	local isInAllowedZone = false
	for _,allowedType in ipairs(allowedZoneTypes) do
		if zoneType == allowedType then
			isInAllowedZone = true
			break
		end
	end
	if not isInAllowedZone then return nil end

	if z ~= self.targetZone and z ~= self.targetZoneOld then
		local W = z:getWidth()
		local H = z:getHeight()
		if W <= 0 or H <= 0 then return nil end
		-- if zoneType == "Nav" and W >= 20 and H >= 20 then return nil end -- avoid giant parking areas
		-- if zoneType == "Nav" and W <= 5 and H <= 5 then return nil end -- avoid tiny stubs

		local chance
		if W < H then
			chance = W*30 --prefer wide roads
		else
			chance = H*30
		end
		if W > H then
			-- W<->E
			if self.dir=="E" and z:getX()+z:getWidth()*0.5 > self.posX then
				choice = {z, chance*2} -- prefer going straight ahead
			elseif self.dir=="W" and z:getX()+z:getWidth()*0.5 < self.posX then
				choice = {z, chance*2} -- prefer going straight ahead
			elseif self.dir == "N" or self.dir == "S" then
				choice = {z, chance} -- making a turn
			elseif self.dir == "" then
				choice = {z, chance}
			end
		else
			-- N<->S
			if self.dir=="S" and z:getY()+z:getHeight()*0.5 > self.posY then
				choice = {z, chance*2}
			elseif self.dir=="N" and z:getY()+z:getHeight()*0.5 < self.posY then
				choice = {z, chance*2}
			elseif self.dir=="W" or self.dir=="E" then
				choice = {z, chance}
			elseif self.dir == "" then
				choice = {z, chance}
			end
		end
	end
	return choice
end

function Horde:navigateRoads(roads, arrivedAtTarget)
	if not roads or #roads == 0 then
		if self.targetZone ~= nil then
			if arrivedAtTarget then
				roads[1] = {self.targetZone, 1000}
				-- self:log("going back...") 
			else
				if self:tryFallbackNavigate() then return true end
				return false -- no new navigations
			end
		else
			if self:tryFallbackNavigate() then return true end
			self:log("STUCK FROM NAVIGATING ROADS! NO TARGET ZONE")
			self:disband()
			return false
		end
	end

	local z = nil

	--random choice
	local counter = 900
	while z == nil and counter ~= 0 do
		local i = ZombRand(1,#roads+1)
		local r = ZombRand(900) --max is road width*2*30==900
		if r <= roads[i][2] then
			z = roads[i][1]
		end
		counter = counter-1
	end
	if not z then 
		if self.targetZone ~= nil then
			if arrivedAtTarget then
				z = self.targetZone
				-- self:log("going back...") 
			else
				if self:tryFallbackNavigate() then return true end
				return false -- no new navigations
			end
		else
			if self:tryFallbackNavigate() then return true end
			self:log("STUCK FROM NAVIGATING ROADS2!")
			self:disband()
			return false
		end
	end
	if z == self.targetZone and arrivedAtTarget == false then
		return false -- direction hasnt changed
	end

	--set new target
	self.targetZoneOld = self.targetZone
	self.targetZone = z
	self.modData.targetZoneX = z:getX()
	self.modData.targetZoneY = z:getY()
	self.modData.targetZoneType = z:getType()
	self.modData.dir = self.dir
	if z:getWidth() > z:getHeight() then
		-- west <-> east
		if z:getType() == "Nav" then
			self.targetY = z:getY() + z:getHeight()/2
			self.changeY = 0
			self.posY = self.targetY -- snap into position
		else
			self.targetY = z:getY() + ZombRand(0, z:getHeight()+1)
			local dx = z:getX()+z:getWidth()-self.posX
			if math.abs(dx) < 0.001 then
				self.changeY = 0
			else
				self.changeY = (self.targetY-self.posY) / dx * self.speed
			end
		end
		if z == self.targetZoneOld and arrivedAtTarget then
			if self.dir == "W" then self.dir = "E" else self.dir = "W" end --go back
		elseif self.dir == "N" or self.dir == "S" or self.dir == "" then
			if z:getX()+5 > self.posX and z:getX()+z:getWidth()-5 > self.posX then
				self.dir = "E"
			elseif z:getX()+z:getWidth()-5 < self.posX and z:getX()+5 < self.posX then
				self.dir = "W"
			else
				if ZombRand(0,2) == 0 then self.dir = "W" else self.dir = "E" end --random turn
			end
		end
		if self.dir == "E" then
			self.targetX = z:getX() + z:getWidth()
			self.changeX = self.speed
		elseif self.dir == "W" then
			self.targetX = z:getX()
			self.changeX = -self.speed
		end
	else
		-- north <-> south
		if z:getType() == "Nav" then
			self.targetX = z:getX() + z:getWidth()/2
			self.changeX = 0
			self.posX = self.targetX
		else
			self.targetX = z:getX() +  ZombRand(0, z:getWidth()+1)
			local dy = z:getY()+z:getHeight()-self.posY
			if math.abs(dy) < 0.001 then
				self.changeX = 0
			else
				self.changeX = (self.targetX-self.posX) / dy * self.speed
			end
		end
		if z == self.targetZoneOld and arrivedAtTarget then
			if self.dir == "N" then self.dir = "S" else self.dir = "N" end --go back
		elseif self.dir == "W" or self.dir == "E" or self.dir == "" then
			if z:getY()+5 > self.posY and z:getY()+z:getHeight()-5 > self.posY then
				self.dir = "S"
			elseif z:getY()+z:getHeight()-5 < self.posY and z:getY()+5 < self.posY then
				self.dir = "N"
			else
				if ZombRand(0,2) == 0 then self.dir = "N" else self.dir = "S" end --random turn
			end
		end
		if self.dir == "S" then
			self.targetY = z:getY() + z:getHeight()
			self.changeY = self.speed
		elseif self.dir == "N" then
			self.targetY = z:getY()
			self.changeY = -self.speed
		end
	end
	return true
end

-- Fallback navigation when no zones are available and off-road navigation is allowed.
-- Picks a simple step in the current/random direction within world bounds.
function Horde:tryFallbackNavigate()
	self:log("FALLBACK NAVIGATION")
	if not SandboxVars.Hordes.NavigateOutsideHighways then return false end

	local minX, maxX = 0, 15000
	local minY, maxY = 900, 15500

	-- reuse existing heading when possible; otherwise pick a random cardinal
	local dir = self.dir
	if dir == "" then
		local dirs = {"N","S","E","W"}
		dir = dirs[ ZombRand(1, #dirs+1) ]
	end

	local targetX, targetY = self.posX, self.posY
	if dir == "N" then
		targetY = self.posY - self.speed
	elseif dir == "S" then
		targetY = self.posY + self.speed
	elseif dir == "E" then
		targetX = self.posX + self.speed
	elseif dir == "W" then
		targetX = self.posX - self.speed
	end

	-- clamp to world bounds
	if targetX < minX then targetX = minX end
	if targetX > maxX then targetX = maxX end
	if targetY < minY then targetY = minY end
	if targetY > maxY then targetY = maxY end

	-- if clamping forced us to stay in place, pick a random small offset
	if targetX == self.posX and targetY == self.posY then
		targetX = math.min(maxX, math.max(minX, self.posX + ZombRand(-self.speed, self.speed+1)))
		targetY = math.min(maxY, math.max(minY, self.posY + ZombRand(-self.speed, self.speed+1)))
	end

	self.targetZoneOld = self.targetZone
	self.targetZone = nil
	self.modData.targetZoneX = nil
	self.modData.targetZoneY = nil
	self.modData.targetZoneType = nil
	self.dir = dir
	self.modData.dir = dir
	self.targetX = targetX
	self.targetY = targetY
	self.modData.targetX = targetX
	self.modData.targetY = targetY
	self.changeX = targetX > self.posX and self.speed or (targetX < self.posX and -self.speed or 0)
	self.changeY = targetY > self.posY and self.speed or (targetY < self.posY and -self.speed or 0)
	return true
end




--------------------------------
-- CONVERSION
--------------------------------



--MAKE REAL
function Horde:virtualToReal()
	self:log("ToReal")
	if not self.isVirtual then return end

	self.isVirtual = false
	self:collectZombiesStart(10, self.virtualToReal2)
end
function Horde:virtualToReal2()
	self.leader = self.zombieList[1]
	self.virtualDisbandAt = nil
	for _,zomb in ipairs(self.zombieList) do
		zomb:getModData().offsetX = ZombRand(-5,5)
		zomb:getModData().offsetY = ZombRand(-5,5)
	end
	self:dressHorde()
end

function Horde:dressHorde()
	local outfits = HordeData.outfits[self.type]
	for _,zomb in ipairs(self.zombieList) do
		if outfits.ALL then
			zomb:dressInNamedOutfit( outfits.ALL[ ZombRand(1,#outfits.ALL+1) ] )
		elseif zomb:isFemale() then
			zomb:dressInNamedOutfit( outfits.F[ ZombRand(1,#outfits.F+1) ] )
		else
			zomb:dressInNamedOutfit( outfits.M[ ZombRand(1,#outfits.M+1) ] )
		end
	end
end


--MAKE VIRTUAL
function Horde:realToVirtual()
	self:log("ToVirtual")
	if self.isVirtual then return end

	if self.name == "Temp" then
		local hours = tonumber(SandboxVars.Hordes.TempHordeVirtualHours) or 24
		if hours > 0 then
			self.virtualDisbandAt = GameTime:getInstance():getWorldAgeHours() + hours
		else
			self.virtualDisbandAt = nil
		end
	end

	if self.collect_timeoutCounter == 0 then 
		-- delete data
		local modData
		for _,zomb in ipairs(self.zombieList) do
			modData = zomb:getModData()
			modData.offsetX = nil
			modData.offsetY = nil
		end

		self:removeZombies()
	end

	self.isVirtual = true
	self.leader = nil
end




--------------------------------
-- ZOMBIE COLLECTION
--------------------------------

-- zombies are just dots on the map. dots get loaded at a X/Y distance of ~90-100 tiles. 
-- getCell():getZombieList() is NOT a list of all zombies in a cell. It's a list of "real" zombies only.
-- Old zombies are "recycled" into new zombies when despawning and respawning - including their modData!
-- tiered zombie updates have an effect on OnZombieUpdate events. OnZombieUpdate is bad for timing.
-- OnTick is also bad for timing, it's based on FPS.



function Horde:collectZombiesStart(radius, callback)
	self.radius = radius
	self.collect_timeoutCounter = 200
	self.collect_callback = callback
	self.zombieList = {}
	EventSystem.addEventListener(self, "OnTick",  "collectZombies") -- need to wait a couple of ticks until zombies exist
end

function Horde:collectZombies()
	local counter = #self.zombieList
	local zList = getCell():getZombieList()
	local size = zList:size()-1
	local zomb = nil
	local modData = nil 
	for i= 0,size do 
		zomb = zList:get(i)
		modData = zomb:getModData()
		if modData.horde == nil then
			if self:distTest(zomb:getX(), zomb:getY(), self.posX, self.posY) < self.radius then
				modData.horde = self
				table.insert(self.zombieList, zomb)
				counter = counter +1
				if #self.zombieList == self.modData.size then break end
			end
		end
	end

	self.collect_timeoutCounter = self.collect_timeoutCounter - 1

	if #self.zombieList == self.modData.size or self.collect_timeoutCounter == 0 then
		self:log("collectZombies(): found "..#self.zombieList.."/"..self.modData.size)
		self.modData.size = #self.zombieList
		self.collect_timeoutCounter = 0
		EventSystem.removeEventListener(self, "OnTick",  "collectZombies")
		EventSystem.addEventListener(self, "OnTick",  "collectZombiesEnd") -- need to wait 1 tick again before dressing zombies?
	end
end
function Horde:distTest(x1,y1, x2,y2)
	if math.abs(x1-x2) > math.abs(y1-y2) then
		return math.abs(x1-x2)
	else
		return math.abs(y1-y2)
	end
end

function Horde:collectZombiesEnd()
	EventSystem.removeEventListener(self, "OnTick",  "collectZombiesEnd")
	if self.collect_callback then
		self.collect_callback(self)
	end
end

function Horde:removeZombies()
	local counter = 0
	local pX = getPlayer():getX()
	local pY = getPlayer():getY()
	local zList = getCell():getZombieList()
	local size = zList:size()-1
	local zomb = nil
	local modData = nil
	--gotta copy zombie list to table because accessing list while modifing it seems to screw it up
	local zTable = {}
	--remove horde members
	for i = 0,size do
		zomb = zList:get(i)
		modData = zomb:getModData()
		if modData.horde == self then
			if IsoUtils.DistanceManhattenSquare(zomb:getX(), zomb:getY(), pX, pY) > convertDist-20 then 
				table.insert(zTable, zomb)
				counter = counter +1
			end
		end
	end	
	--if we couldn't get all, try to remove the remaining # of zombies near pos
	if counter < self.modData.size then
		for i = 0,size do
			zomb = zList:get(i)
			modData = zomb:getModData()
			if modData.horde == nil then
				if IsoUtils.DistanceManhattenSquare(zomb:getX(), zomb:getY(), self.posX, self.posY) < 20 then
					table.insert(zTable, zomb)
					counter = counter +1
					if counter == self.modData.size then break end
				end
			end
		end	
	end

	for _, z in ipairs(self.zombieList) do
		z:getModData().horde = nil
	end

	--remove all collected zombies
	for _, z in ipairs(zTable) do
		z:removeFromWorld()
	end
	self:log("removeZombies(): found "..counter.."/"..self.modData.size)
	-- self.modData.size = counter
	self.zombieList = {}
end

function Horde:doVictoryOrDeath(x,y)
	self:log("-------------------------")
	self:log("-- VICTORY OR DEATH!!! --")
	self:log("-------------------------")
	self.victoryOrDeath = true
	self.targetX = x
	self.targetY = y
	self:moveTo(x,y)
	self:disband()
end

function Horde:disband()
	self:log("DISBANDED")
	self.disbanded = true
	self.name = ""
	if self.sfx then self.sfx:stop() end
	EventSystem.removeEventListener(self, "OnPlayerUpdate",  "checkPlayerDistance")
	EventSystem.removeEventListener(self, "OnTick",  "collectZombies")
end

function Horde:log(v)
	-- if self.logName then
	-- 	print(self.logName..": "..v)
	-- 	if self.leader then
	-- 		self.leader:Say(self.logName..": "..v, 1.0, 1.0, 1.0, UIFont.Dialogue, 30.0, "radio")
	-- 	end
	-- elseif v then
	-- 	print("no name: "..v)
	-- end
end
