local InspectPlugins = require "InspectSystem/Core/Inspect_Plugins"
local M = {}

local function containsIgnoreCase(s, sub)
    if not s or not sub then return false end
    s   = string.lower(tostring(s))
    sub = string.lower(tostring(sub))
    return string.find(s, sub, 1, true) ~= nil
end

function M.describe(item, statMap, child, parent)
    local subtype = parent -- FluidContainer
    local b = InspectDescription.newBuilder()

    --------------------------------------------------------
    -- 1) Filtering: Only run for true fluid containers
    --------------------------------------------------------
    local lightDist = statMap["lightDistance"]
    local lightStr  = statMap["lightStrength"]

    local displayCategoryStr = ""
    if item and item.getDisplayCategory then
        local ok, dc = pcall(function() return item:getDisplayCategory() end)
        if ok and dc then
            displayCategoryStr = tostring(dc)
        end
    end

    local looksLikeLight = (lightDist ~= nil or lightStr ~= nil)
        or containsIgnoreCase(displayCategoryStr, "light")

    if subtype ~= "FluidContainer" or looksLikeLight then
        return nil
    end

    --------------------------------------------------------
    -- 2) BASE LINE
    --------------------------------------------------------
    InspectDescription.addLine(
        b,
        InspectUI_tr(
            "IGUI_Inspect_Fluid_Base_Container",
            "Container designed to store liquids such as water, fuel, or chemicals."
        )
    )

    --------------------------------------------------------
    -- 3) STATS
    --------------------------------------------------------
    local function fmt(n, d)
        return InspectUI_fmtNumber(n, d or 1)
    end

    local cap = statMap["fluidCapacity"]
    if cap then
        InspectDescription.addFormatted(
            b,
            "IGUI_Inspect_Fluid_Stat_Capacity",
            "Can hold around %s units of liquid.",
            fmt(cap.value or cap.raw, 2)
        )
    end

    local uses = statMap["usesLeft"]
    if uses then
        InspectDescription.addFormatted(
            b,
            "IGUI_Inspect_Fluid_Stat_Uses",
            "Divided into roughly %s separate uses or portions.",
            fmt(uses.value or uses.raw, 0)
        )
    end

    local rain = statMap["rainFactor"]
    if rain then
        InspectDescription.addFormatted(
            b,
            "IGUI_Inspect_Fluid_Stat_Rain",
            "Can collect rainwater with an efficiency factor of about %s.",
            fmt(rain.value or rain.raw, 2)
        )
    end

    local w = statMap["weight"]
    if w then
        InspectDescription.addFormatted(
            b,
            "IGUI_Inspect_Fluid_Stat_Weight",
            "Weight: %s units.",
            fmt(w.value or w.raw, 2)
        )
    end

    local cond = statMap["condition"]
    if cond then
        InspectDescription.addFormatted(
            b,
            "IGUI_Inspect_Fluid_Stat_Condition",
            "Condition: %s%%.",
            fmt(cond.value or cond.raw, 0)
        )
    end

    --------------------------------------------------------
    -- 4) PLUGINS
    --------------------------------------------------------
    local extra = InspectPlugins.appendFluidLines
        and InspectPlugins.appendFluidLines(item, statMap, subtype)

    if extra then
        if type(extra) == "string" then
            InspectDescription.addLine(b, extra)
        elseif type(extra) == "table" then
            for _, line in ipairs(extra) do
                InspectDescription.addLine(b, line)
            end
        end
    end

    --------------------------------------------------------
    -- 5) RETURN
    --------------------------------------------------------
    return InspectDescription.finish(b)
end

return M
