local InspectPlugins = require "InspectSystem/Core/Inspect_Plugins"
local M = {}

function M.describe(item, statMap, child, parent)
    local b = InspectDescription.newBuilder()
    local matched = false

    local function addStats()
        local function fmt(n, d) return InspectUI_fmtNumber(n, d or 1) end
        local st = statMap or {}

        local w = st["weight"]
        if w then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Weight",
                "Weight: %s units.",
                fmt(w.value or w.raw, 2)
            )
        end

        local c = st["condition"]
        if c then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Condition",
                "Condition: %s%%.",
                fmt(c.value or c.raw, 0)
            )
        end

        local u = st["usesLeft"]
        if u then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Uses",
                "Remaining uses: %s.",
                fmt(u.value or u.raw, 0)
            )
        end

        local cap = st["capacity"]
        if cap then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Capacity",
                "Capacity: %s units.",
                fmt(cap.value or cap.raw, 1)
            )
        end

        local temp = st["temperature"]
        if temp then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Temperature",
                "Temperature: %s°C.",
                fmt(temp.value or temp.raw, 1)
            )
        end

        local integ = st["integrity"]
        if integ then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Integrity",
                "Structural integrity: %s%%.",
                fmt(integ.value or integ.raw, 0)
            )
        end

        local sig = st["signalStrength"]
        if sig then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Signal",
                "Signal strength: %s.",
                fmt(sig.value or sig.raw, 1)
            )
        end

        local energy = st["energyLevel"]
        if energy then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Energy",
                "Energy level: %s.",
                fmt(energy.value or energy.raw, 1)
            )
        end

        local rad = st["radiation"]
        if rad then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Radiation",
                "Radiation level: %s.",
                fmt(rad.value or rad.raw, 1)
            )
        end

        local fluid = st["fluidCapacity"]
        if fluid then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Fallback_Stat_Fluid",
                "Fluid capacity: %s units.",
                fmt(fluid.value or fluid.raw, 1)
            )
        end
    end

    --------------------------------------------------------
    -- DYNAMIC / SCRIPTED ITEMS
    --------------------------------------------------------
    local isDynamic = item and (
        item.CustomName or
        (item.getFullType and (item:getFullType():find("Script") or item:getFullType():find("Dynamic")))
    )

    if isDynamic then
        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Fallback_Dynamic_Base",
                "Dynamically generated item with properties defined by scripts or external systems.")
        )

        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Fallback_Dynamic_Usage",
                "Behavior and functionality may vary depending on active mods or events.")
        )

        matched = true
    end

    --------------------------------------------------------
    -- STAT-ONLY OBJECTS
    --------------------------------------------------------
    local hasStats = statMap and next(statMap) ~= nil
    if not matched and hasStats then
        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Fallback_StatOnly_Base",
                "Item with detectable properties but no assigned category.")
        )

        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Fallback_StatOnly_Usage",
                "Technical details extracted from available internal statistics.")
        )

        matched = true
    end

    --------------------------------------------------------
    -- PRIMARY FALLBACK
    --------------------------------------------------------
    if not matched then
        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Fallback_Primary",
                "General item with no specific category assignment.")
        )
    end

    --------------------------------------------------------
    -- STATS
    --------------------------------------------------------
    addStats()

    --------------------------------------------------------
    -- PLUGINS
    --------------------------------------------------------
    local extra =
        InspectPlugins.appendFallbackLines and
        InspectPlugins.appendFallbackLines(item, statMap, child, parent)

    if extra then
        if type(extra) == "string" then
            InspectDescription.addLine(b, extra)
        else
            for _, line in ipairs(extra) do
                InspectDescription.addLine(b, line)
            end
        end
    end

    return InspectDescription.finish(b)
end

return M
