----------------------------------------------------------
-- Category_Electronics.lua
-- Token-based rules (v3.8)
----------------------------------------------------------

local C = {}
C.parent = "Electronics"

-- token tables
local TOKENS = {
    LightSource       = { "light", "lamp", "flashlight", "torch" },
    LightBulb         = { "light bulb", "lightbulb", "bulb" },
    Radio             = { "radio", "walkie", "communication" },
    Battery           = { "battery", "bateria" },
    ElectronicPart    = { "electro" },
    ElectronicTool    = { "welding", "welder", "multimeter", "tester", "solder" },
    EntertainmentDevice = { "player", "vhs", "mp3", "cd", "game", "console" },
    RemoteController  = { "remote", "control", "controlador" },
    AudioDevice       = { "speaker", "headset", "earbud", "microphone" },
    Generator         = { "generator" },
}

-- Universal token matcher for all categories
local function matchTokens(text, tokens)
    for _, token in ipairs(tokens) do
        if text:find(token, 1, true) then
            return true
        end
    end
    return false
end

local function lower(val)
    if not val then return "" end
    return string.lower(tostring(val))
end

local function getDisplayCategory(item)
    return lower(item.getDisplayCategory and item:getDisplayCategory())
end

local function getText(item)
    local name = lower(item.getDisplayName and item:getDisplayName())
    local dc   = getDisplayCategory(item)
    return (name or "") .. " " .. (dc or "")
end

local function isElectronics(item)
    local dc = getDisplayCategory(item)
    if dc and (dc:find("electro", 1, true) or dc:find("firesource", 1, true)) then
        return true
    end
    -- Fallback: some batteries or cells may not flag electronics display category
    return matchTokens(getText(item), TOKENS.Battery)
end

C.rules = {
    -- Base
    {
        id = "Electronics.BaseCheck",
        priority = 50,
        match = function(item, statMap)
            return isElectronics(item)
        end,
        category = "Electronics",
    },

    -- Subcategories (normalized priorities 90–120)
    {
        id = "Electronics.LightSource",
        priority = 100,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            statMap = statMap or {}
            if statMap["lightDistance"] or statMap["lightStrength"] then return true end
            return matchTokens(getText(item), TOKENS.LightSource)
        end,
        category = "Electronics/LightSource",
    },

    {
        id = "Electronics.LightBulb",
        priority = 100,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            return matchTokens(getText(item), TOKENS.LightBulb)
        end,
        category = "Electronics/LightBulb",
    },

    {
        id = "Electronics.Radio",
        priority = 100,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            return matchTokens(getText(item), TOKENS.Radio)
        end,
        category = "Electronics/Radio",
    },

    {
        id = "Electronics.Battery",
        priority = 100,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            return matchTokens(getText(item), TOKENS.Battery)
        end,
        category = "Electronics/Battery",
    },

    {
        id = "Electronics.ElectronicPart",
        priority = 95,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            statMap = statMap or {}
            local cat = getDisplayCategory(item)
            return cat:find("electro") ~= nil
                and not statMap["lightDistance"]
                and not statMap["usesLeft"]
                and not cat:find("radio")
        end,
        category = "Electronics/ElectronicPart",
    },

    {
        id = "Electronics.ElectronicTool",
        priority = 100,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            return matchTokens(getText(item), TOKENS.ElectronicTool)
        end,
        category = "Electronics/ElectronicTool",
    },

    {
        id = "Electronics.EntertainmentDevice",
        priority = 95,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            return matchTokens(getText(item), TOKENS.EntertainmentDevice)
        end,
        category = "Electronics/EntertainmentDevice",
    },

    {
        id = "Electronics.RemoteController",
        priority = 95,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            return matchTokens(getText(item), TOKENS.RemoteController)
        end,
        category = "Electronics/RemoteController",
    },

    {
        id = "Electronics.AudioDevice",
        priority = 95,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            if item.getDisplayCategory and lower(item:getDisplayCategory()):find("radio") then return false end
            return matchTokens(getText(item), TOKENS.AudioDevice)
        end,
        category = "Electronics/AudioDevice",
    },

    {
        id = "Electronics.Generator",
        priority = 100,
        match = function(item, statMap)
            if not isElectronics(item) then return false end
            return matchTokens(getText(item), TOKENS.Generator)
        end,
        category = "Electronics/Generator",
    },
}

return C
