local InspectPlugins = require "InspectSystem/Core/Inspect_Plugins"
local M = {}

-- Detect if this tool should be treated as a hybrid weapon (delegate to other modules)
local function isHybridWeaponTool(item)
    if not item then return false end

    if item.isWeapon and item:isWeapon() then
        return true
    end

    if item.getMaxDamage and item:getMaxDamage() > 0 then
        return true
    end

    if item.getDisplayCategory then
        local cat = item:getDisplayCategory()
        if cat == "WeaponUtility" or cat == "Melee" then
            return true
        end
    end

    return false
end

function M.describe(item, statMap, child, parent)
    local b = InspectDescription.newBuilder()
    local matched = false

    local function addStats()
        local function fmt(n, d) return InspectUI_fmtNumber(n, d or 1) end

        local w = statMap["weight"]
        if w then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Tool_Stat_Weight",
                "Weight: %s units.",
                fmt(w.value or w.raw, 2)
            )
        end

        local c = statMap["condition"]
        if c then
            InspectDescription.addFormatted(
                b,
                "IGUI_Inspect_Tool_Stat_Condition",
                "Condition: %s%%.",
                fmt(c.value or c.raw, 0)
            )
        end
    end

    --------------------------------------------------------
    -- HYBRID TOOL – Delegate to Melee or WeaponUtility
    --------------------------------------------------------
    if isHybridWeaponTool(item) then
        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Tool_Hybrid_Note",
                "This tool is also suitable for close-quarters impact use.")
        )

        if InspectDescriptions and InspectDescriptions.WeaponUtility then
            return InspectDescriptions.WeaponUtility.describe(
                item, statMap, "HybridTool", parent)
        end

        if InspectDescriptions and InspectDescriptions.Melee then
            return InspectDescriptions.Melee.describe(
                item, statMap, "HybridTool", parent)
        end
    end


    --------------------------------------------------------
    -- UTILITY TOOL
    --------------------------------------------------------
    if child == "Tool" or parent == "Tool" then
        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Tool_Utility_Base",
                "Tool designed for construction, repair, or general utility tasks.")
        )

        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Tool_Utility_Usage",
                "Used to perform practical or crafting-related actions.")
        )

        matched = true
    end


    --------------------------------------------------------
    -- FALLBACK
    --------------------------------------------------------
    if not matched then
        InspectDescription.addLine(b,
            InspectUI_tr("IGUI_Inspect_Tool_Fallback",
                "General-purpose tool with basic functional use.")
        )
    end

    --------------------------------------------------------
    -- STATS
    --------------------------------------------------------
    addStats()

    --------------------------------------------------------
    -- PLUGINS
    --------------------------------------------------------
    local extra =
        InspectPlugins.appendToolLines and
        InspectPlugins.appendToolLines(item, statMap, child, parent)

    if extra then
        if type(extra) == "string" then
            InspectDescription.addLine(b, extra)
        else
            for _, line in ipairs(extra) do
                InspectDescription.addLine(b, line)
            end
        end
    end

    return InspectDescription.finish(b)
end

return M
