----------------------------------------------------------
-- Category_Food.lua
-- Token-based rules (v3.8)
----------------------------------------------------------

local C = {}
C.parent = "Food"

-- token tables
local TOKENS = {
    Liquor = { "beer", "wine", "whiskey", "vodka", "rum", "bourbon", "gin", "liqueur", "tequila" },
}

-- Universal token matcher for all categories
local function matchTokens(text, tokens)
    for _, token in ipairs(tokens) do
        if text:find(token, 1, true) then
            return true
        end
    end
    return false
end

local function lower(val)
    if not val then return "" end
    return string.lower(tostring(val))
end

local function getText(item)
    local name = lower(item.getDisplayName and item:getDisplayName())
    local dc   = lower(item.getDisplayCategory and item:getDisplayCategory())
    local ft   = item.getFoodType and item:getFoodType()
    if ft then ft = lower(ft) end
    return (name or "") .. " " .. (dc or "") .. " " .. (ft or "")
end

local function isFood(item, statMap)
    if not (instanceof and instanceof(item, "Food")) then return false end
    local dc = lower(item.getDisplayCategory and item:getDisplayCategory())
    if dc:find("medical") or dc:find("first") then return false end
    if dc:find("animalpart") then return false end
    return true
end

C.rules = {
    {
        id = "Food.BaseCheck",
        priority = 50,
        match = function(item, statMap)
            return isFood(item, statMap)
        end,
        category = "Food",
    },

    {
        id = "Food.Drink",
        priority = 100,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            return item.isDrink and item:isDrink()
        end,
        category = "Food/Drink",
    },

    {
        id = "Food.Fruit",
        priority = 100,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            local t = item.getFoodType and item:getFoodType()
            return t == "Fruit"
        end,
        category = "Food/Fruit",
    },

    {
        id = "Food.Vegetable",
        priority = 100,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            local t = item.getFoodType and item:getFoodType()
            return t == "Vegetables" or t == "Vegetable"
        end,
        category = "Food/Vegetable",
    },

    {
        id = "Food.Meat",
        priority = 100,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            local t = item.getFoodType and item:getFoodType()
            return t == "Meat" or t == "Fish" or t == "Poultry"
        end,
        category = "Food/Meat",
    },

    {
        id = "Food.Spice",
        priority = 95,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            local t = item.getFoodType and item:getFoodType()
            return t == "Spice" or (item.isSpice and item:isSpice())
        end,
        category = "Food/Spice",
    },

    {
        id = "Food.Liquor",
        priority = 95,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            local text = getText(item)
            return matchTokens(text, TOKENS.Liquor) or text:find("alcohol") or text:find("liquor")
        end,
        category = "Food/Liquor",
    },

    {
        id = "Food.Prepared",
        priority = 90,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            return (item.getReplaceOnCooked and item:getReplaceOnCooked())
                or (item.isCookable and item:isCookable())
        end,
        category = "Food/Prepared",
    },

    {
        id = "Food.Perishable",
        priority = 90,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            if type(statMap) ~= "table" then return false end
            local fresh = statMap["daysFresh"]
            return fresh ~= nil
        end,
        category = "Food/Perishable",
    },

    {
        id = "Food.NonPerishable",
        priority = 90,
        match = function(item, statMap)
            if not isFood(item, statMap) then return false end
            if type(statMap) ~= "table" then return true end
            local fresh = statMap["daysFresh"]
            return fresh == nil
        end,
        category = "Food/NonPerishable",
    },
}

return C
