-- ===================================================================================================== --
-- Woody's Rain & Wet Snow Wash - Wringing Hook
-- ===================================================================================================== --
-- Hooks into vanilla/Better Wringing to add dirt/blood removal
-- Compatible with "Better Wringing" mod (Workshop ID: 3408740647)
-- ===================================================================================================== --

require "TimedActions/ISWringClothing"

local WoodysRainSnow_Config = require("WoodysRainSnow_Config")

-- Store original functions
local originalStart = ISWringClothing.start
local originalComplete = ISWringClothing.complete
local originalGetDuration = ISWringClothing.getDuration

-- ===================================================================================================== --
-- HOOK: START - Track Original Wetness
-- ===================================================================================================== --

function ISWringClothing:start()
    -- Store original wetness before wringing
    self.originalWetness = self.item:getWetness()
    
    -- Store original dirt/blood levels
    if instanceof(self.item, "Clothing") then
        self.originalDirtLevels = {}
        self.originalBloodLevels = {}
        
        local coveredParts = BloodClothingType.getCoveredParts(self.item:getBloodClothingType())
        if coveredParts then
            for j = 0, coveredParts:size() - 1 do
                local part = coveredParts:get(j)
                if part then
                    self.originalDirtLevels[part] = self.item:getDirt(part)
                    self.originalBloodLevels[part] = self.item:getBlood(part)
                end
            end
        end
    end
    
    -- Call original
    return originalStart(self)
end

-- ===================================================================================================== --
-- HOOK: COMPLETE - Add Dirt/Blood Removal
-- ===================================================================================================== --

function ISWringClothing:complete()
    -- Call original function first
    local result = originalComplete(self)
    
    -- Check if wringing cleaning is enabled
    if not WoodysRainSnow_Config.isWringCleaningEnabled() then
        return result
    end
    
    -- Add our dirt/blood removal logic
    if instanceof(self.item, "Clothing") then
        local originalWetness = self.originalWetness or self.item:getWetness()
        local currentWetness = self.item:getWetness()
        local waterRemoved = math.max(0, originalWetness - currentWetness)
        
        if waterRemoved > 0 then
            -- Get config values
            local dirtEfficiency = WoodysRainSnow_Config.getWringDirtEfficiency()
            local bloodRemoval = WoodysRainSnow_Config.getWringBloodRemoval()
            local wetnessExponent = WoodysRainSnow_Config.getWringWetnessBonus()
            
            -- Calculate wetness bonus (nonlinear: x^exponent)
            -- Higher original wetness = better cleaning efficiency
            local wetnessRatio = originalWetness / 100  -- 0.0 to 1.0
            local wetnessBonus = wetnessRatio ^ wetnessExponent
            
            -- Calculate dirt removal
            -- Formula: (waterRemoved / 100) × dirtEfficiency × wetnessBonus
            local dirtRemovalAmount = (waterRemoved / 100) * dirtEfficiency * wetnessBonus
            
            -- Get covered body parts
            local coveredParts = BloodClothingType.getCoveredParts(self.item:getBloodClothingType())
            if coveredParts then
                local totalDirt = 0
                local totalBlood = 0
                local totalDirtBefore = 0
                local totalBloodBefore = 0
                
                for j = 0, coveredParts:size() - 1 do
                    local part = coveredParts:get(j)
                    if part then
                        -- Store before values for debug
                        local dirtBefore = self.item:getDirt(part)
                        local bloodBefore = self.item:getBlood(part)
                        totalDirtBefore = totalDirtBefore + dirtBefore
                        totalBloodBefore = totalBloodBefore + bloodBefore
                        
                        -- Remove dirt (nonlinear based on wetness)
                        if dirtBefore > 0 then
                            local newDirt = math.max(0, dirtBefore - dirtRemovalAmount)
                            self.item:setDirt(part, newDirt)
                        end
                        
                        -- Remove blood (fixed percentage)
                        if bloodBefore > 0 then
                            local newBlood = math.max(0, bloodBefore - bloodRemoval)
                            self.item:setBlood(part, newBlood)
                        end
                        
                        -- Track totals for overall levels
                        totalDirt = totalDirt + self.item:getDirt(part)
                        totalBlood = totalBlood + self.item:getBlood(part)
                    end
                end
                
                -- Update item's overall blood/dirt levels
                local partCount = coveredParts:size()
                if partCount > 0 then
                    self.item:setDirtyness(totalDirt * 100 / partCount)
                    self.item:setBloodLevel(totalBlood * 100 / partCount)
                    
                    -- Debug: log wringing results
                    local avgDirtBefore = math.floor((totalDirtBefore / partCount) * 100)
                    local avgBloodBefore = math.floor((totalBloodBefore / partCount) * 100)
                    local avgDirtAfter = math.floor((totalDirt / partCount) * 100)
                    local avgBloodAfter = math.floor((totalBlood / partCount) * 100)
                    
                    print("[WOODYDEBUG]_RainSnow:_Wring_" .. self.item:getDisplayName() .. "_wet:" .. math.floor(originalWetness) .. "%->" .. math.floor(currentWetness) .. "%_dirt:" .. avgDirtBefore .. "%->" .. avgDirtAfter .. "%_blood:" .. avgBloodBefore .. "%->" .. avgBloodAfter .. "%")
                end
            end
            
            -- Sync changes
            syncItemFields(self.character, self.item)
        end
    end
    
    return result
end

-- ===================================================================================================== --
-- HOOK: GET DURATION - Add Time Penalty For Dirty Clothes
-- ===================================================================================================== --

function ISWringClothing:getDuration()
    if self.character:isTimedActionInstant() then
        return 1
    end
    
    -- Get base time (vanilla/Better Wringing calculation)
    local baseTime = math.ceil(self.item:getWetness() * 5)
    
    -- Check if wringing cleaning is enabled
    if not WoodysRainSnow_Config.isWringCleaningEnabled() then
        return baseTime
    end
    
    -- Add dirt/blood penalty (cleaning takes extra time!)
    local dirtPenalty = 0
    if instanceof(self.item, "Clothing") then
        local totalDirt = 0
        local totalBlood = 0
        local partCount = 0
        
        local coveredParts = BloodClothingType.getCoveredParts(self.item:getBloodClothingType())
        if coveredParts then
            for j = 0, coveredParts:size() - 1 do
                local part = coveredParts:get(j)
                if part then
                    totalDirt = totalDirt + (self.item:getDirt(part) * 100)
                    totalBlood = totalBlood + (self.item:getBlood(part) * 100)
                    partCount = partCount + 1
                end
            end
            
            if partCount > 0 then
                -- Average dirt/blood across parts
                local avgDirt = totalDirt / partCount
                local avgBlood = totalBlood / partCount
                local avgDirtyness = avgDirt + avgBlood
                
                -- Add progressive time penalty for dirty clothes (>10% threshold)
                -- Example: 40% dirty = (40/100) * 250 = 100 ticks = +2 seconds
                -- Example: 80% dirty = (80/100) * 250 = 200 ticks = +4 seconds
                -- Maximum: 100% dirty = 250 ticks = +5 seconds total
                if avgDirtyness > 10 then
                    dirtPenalty = math.ceil((avgDirtyness / 100) * 250)
                end
            end
        end
    end
    
    return baseTime + dirtPenalty
end

-- ===================================================================================================== --
-- INITIALIZATION
-- ===================================================================================================== --

print("[WOODYDEBUG]_RainSnow:_Wringing_hook_installed_successfully")
print("[WOODYDEBUG]_RainSnow:_Dirt_efficiency_" .. WoodysRainSnow_Config.getWringDirtEfficiency())
print("[WOODYDEBUG]_RainSnow:_Blood_removal_" .. (WoodysRainSnow_Config.getWringBloodRemoval() * 100) .. "%")
print("[WOODYDEBUG]_RainSnow:_Wetness_bonus_curve_x^" .. WoodysRainSnow_Config.getWringWetnessBonus())

return true
