-- ===================================================================================================
-- Woody's Rain & Wet Snow Wash - Manual Washing Actions
-- ===================================================================================================
-- Manual washing at water sources (sink, river, rain barrel, etc.)
-- with Fear of Blood trait bonuses!
-- ===================================================================================================

require "TimedActions/ISBaseTimedAction"

-- Debug removed - use [WOODYDEBUG] in functions if needed

WoodysRainSnow_ManualWashAction = ISBaseTimedAction:derive("WoodysRainWash_ManualWashAction")

-- ===================================================================================================
-- INITIALIZATION
-- ===================================================================================================

function WoodysRainSnow_ManualWashAction:new(character, waterSource)
    if not character then return nil end
    local o = ISBaseTimedAction and ISBaseTimedAction.new and ISBaseTimedAction.new(self, character) or nil
    if not o then return nil end
    o.waterSource = waterSource
    o.stopOnWalk = true
    o.stopOnRun = true
    
    -- Calculate washing time based on blood/dirt amount
    local totalBlood = 0
    local totalDirt = 0
    local visual = character.getHumanVisual and character:getHumanVisual() or nil
    if visual then
        for i=0, (BloodBodyPartType and BloodBodyPartType.MAX and BloodBodyPartType.MAX.index and BloodBodyPartType.MAX:index()-1 or -1) do
            local part = BloodBodyPartType and BloodBodyPartType.FromIndex and BloodBodyPartType.FromIndex(i) or nil
            if part then
                totalBlood = totalBlood + (visual.getBlood and visual:getBlood(part) or 0)
                totalDirt = totalDirt + (visual.getDirt and visual:getDirt(part) or 0)
            end
        end
    end
    local wornItems = character.getWornItems and character:getWornItems() or nil
    if wornItems and wornItems.size and wornItems.getItemByIndex then
        for i=0, wornItems:size()-1 do
            local item = wornItems:getItemByIndex(i); item = item and item.getItem and item:getItem() or nil
            if item then
                totalBlood = totalBlood + (item.getBloodLevel and item:getBloodLevel() or 0)
                totalDirt = totalDirt + (item.getDirtLevel and item:getDirtLevel() or 0)
            end
        end
    end
    local baseTime = 60 + (totalBlood + totalDirt) * 30
    local fearOfBloodBoost = WoodysRainSnow and WoodysRainSnow.getManualWashBoost and WoodysRainSnow.getManualWashBoost(character, totalBlood) or 1
    o.maxTime = math.floor(baseTime / fearOfBloodBoost)
    
    -- Debug log
    print("[WOODYDEBUG]_RainSnow:_Manual_wash_started_blood:" .. math.floor(totalBlood * 100) .. "%_dirt:" .. math.floor(totalDirt * 100) .. "%_time:" .. o.maxTime .. "s_boost:" .. fearOfBloodBoost .. "x")
    
    o.totalBloodStart = totalBlood
    o.totalDirtStart = totalDirt
    
    return o
end

-- ===================================================================================================
-- VALIDATION
-- ===================================================================================================

function WoodysRainSnow_ManualWashAction:isValid()
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : isValid called')
    return self.character and self.character.isAlive and self.character:isAlive()
end

-- ===================================================================================================
-- EXECUTION
-- ===================================================================================================

function WoodysRainSnow_ManualWashAction:start()
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : start')
    if self.setActionAnim then self:setActionAnim("Wash") end
    if self.setOverrideHandModels then self:setOverrideHandModels(nil, nil) end
    if self.character and self.character.getEmitter and self.character:getEmitter() then
        print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : start playing sound')
        self.sound = self.character:getEmitter().playSound and self.character:getEmitter():playSound("WashClothing") or nil
    end
end

function WoodysRainSnow_ManualWashAction:update()
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : update')
    local progress = self.getJobDelta and self:getJobDelta() or 0
    local cleaningRate = 0.02 * progress
    local fearOfBloodBoost = WoodysRainSnow and WoodysRainSnow.getManualWashBoost and WoodysRainSnow.getManualWashBoost(self.character, self.totalBloodStart) or 1
    cleaningRate = cleaningRate * fearOfBloodBoost
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : update cleaningRate:', cleaningRate)
    if WoodysRainSnow and WoodysRainSnow.manualCleanBody then
        WoodysRainSnow.manualCleanBody(self.character, cleaningRate)
    end
    if WoodysRainSnow and WoodysRainSnow.manualCleanClothing then
        WoodysRainSnow.manualCleanClothing(self.character, cleaningRate)
    end
end

function WoodysRainSnow_ManualWashAction:complete()
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : complete')
    local finalCleanRate = 0.1
    local fearOfBloodBoost = WoodysRainSnow and WoodysRainSnow.getManualWashBoost and WoodysRainSnow.getManualWashBoost(self.character, self.totalBloodStart) or 1
    finalCleanRate = finalCleanRate * fearOfBloodBoost
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : complete finalCleanRate:', finalCleanRate)
    if WoodysRainSnow and WoodysRainSnow.manualCleanBody then
        WoodysRainSnow.manualCleanBody(self.character, finalCleanRate)
    end
    if WoodysRainSnow and WoodysRainSnow.manualCleanClothing then
        WoodysRainSnow.manualCleanClothing(self.character, finalCleanRate)
    end
    return true
end

function WoodysRainSnow_ManualWashAction:perform()
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : perform')
    if self.complete then self:complete() end
    if ISBaseTimedAction and ISBaseTimedAction.perform then ISBaseTimedAction.perform(self) end
end

function WoodysRainSnow_ManualWashAction:stop()
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : stop')
    if self.sound and self.character and self.character.getEmitter and self.character:getEmitter() and self.character:getEmitter().isPlaying and self.character:getEmitter():isPlaying(self.sound) then
        print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : stop sound')
        if self.character:getEmitter().stopSound then self.character:getEmitter():stopSound(self.sound) end
    end
    if ISBaseTimedAction and ISBaseTimedAction.stop then ISBaseTimedAction.stop(self) end
end

-- ===================================================================================================
-- DISPLAY
-- ===================================================================================================

function WoodysRainSnow_ManualWashAction:getDuration()
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : getDuration')
    if self.character and self.character.isTimedActionInstant and self.character:isTimedActionInstant() then
        return 1
    end
    return self.maxTime or 1
end

function WoodysRainSnow_ManualWashAction:getJobType()
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : getJobType')
    return "Washing Blood & Dirt"
end

-- ===================================================================================================
-- MANUAL CLEANING FUNCTIONS (in WoodysRainWash namespace)
-- ===================================================================================================

--- Get Fear of Blood boost for MANUAL washing (not auto-rain!)
--- @param character IsoPlayer
--- @param totalBlood number Total blood amount on character
--- @return number Boost multiplier
function WoodysRainSnow.getManualWashBoost(character, totalBlood)
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : getManualWashBoost for', tostring(character), 'totalBlood:', totalBlood)
    if not (character and character.HasTrait and character:HasTrait("Hemophobic")) then
        return 1.0
    end
    local config = WoodysRainSnow and WoodysRainSnow.CONFIG and WoodysRainSnow.CONFIG.FEAR_OF_BLOOD_MODE or nil
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : getManualWashBoost config:', tostring(config))
    if config == "OFF" then return 1.0
    elseif config == "FIXED_1.5X" then return 1.5
    elseif config == "FIXED_2X" then return 2.0
    elseif config == "FIXED_3X" then return 3.0
    elseif config == "PROGRESSIVE" then
        -- Progressive: little blood = 1× (normal), full blood = 3×
        -- Blood range: 0 (clean) to ~5.0 (completely covered)
        local bloodRatio = math.min(totalBlood / 5.0, 1.0)
        print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : getManualWashBoost progressive bloodRatio:', bloodRatio)
        return 1.0 + (bloodRatio * 2.0)
    end
    return 1.0
end

--- Manual clean character body parts
--- @param character IsoPlayer
--- @param cleanRate number Amount to clean (0.0-1.0)
function WoodysRainSnow.manualCleanBody(character, cleanRate)
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : manualCleanBody', tostring(character), 'cleanRate:', cleanRate)
    if not character then print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : manualCleanBody NIL character!'); return end
    local visual = character.getHumanVisual and character:getHumanVisual() or nil
    if not visual then print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : manualCleanBody NIL visual!'); return end
    for i=0, (BloodBodyPartType and BloodBodyPartType.MAX and BloodBodyPartType.MAX.index and BloodBodyPartType.MAX:index()-1 or -1) do
        local part = BloodBodyPartType and BloodBodyPartType.FromIndex and BloodBodyPartType.FromIndex(i) or nil
        if part then
            local blood = visual.getBlood and visual:getBlood(part) or 0
            local dirt  = visual.getDirt and visual:getDirt(part) or 0
            print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : cleaning part', i, 'blood:', blood, 'dirt:', dirt)
            if visual.setBlood and blood > 0 then
                visual:setBlood(part, math.max(0, blood - cleanRate))
            end
            if visual.setDirt and dirt > 0 then
                visual:setDirt(part, math.max(0, dirt - cleanRate))
            end
        end
    end
end

--- Manual clean clothing items
--- @param character IsoPlayer
--- @param cleanRate number Amount to clean (0.0-1.0)
function WoodysRainSnow.manualCleanClothing(character, cleanRate)
    print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : manualCleanClothing', tostring(character), 'cleanRate:', cleanRate)
    if not character then print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : manualCleanClothing NIL character!'); return end
    local wornItems = character.getWornItems and character:getWornItems() or nil
    if wornItems and wornItems.size and wornItems.getItemByIndex then
        for i=0, wornItems:size()-1 do
            local wornItem = wornItems:getItemByIndex(i)
            local item = wornItem and wornItem.getItem and wornItem:getItem() or nil
            if item then
                local blood = item.getBloodLevel and item:getBloodLevel() or 0
                local dirt  = item.getDirtLevel and item:getDirtLevel() or 0
                print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : cleaning item', i, 'blood:', blood, 'dirt:', dirt)
                if item.setBloodLevel and blood > 0 then
                    item:setBloodLevel(math.max(0, blood - cleanRate))
                end
                if item.setDirtLevel and dirt > 0 then
                    item:setDirtLevel(math.max(0, dirt - cleanRate))
                end
            end
        end
    else
        print('[WOODYDEBUG] RainSnow:[RainAndWetSnowCleaning] : manualCleanClothing NIL wornItems!')
    end
end

-- ===================================================================================================
-- CONTEXT MENU (Right-click water source)
-- ===================================================================================================

local function safe_onFillWorldObjectContextMenu(player, context, worldobjects, test)
    local success, err = pcall(function()
        print('[WOODYDEBUG] RainSnow: onFillWorldObjectContextMenu: player:', tostring(player), 'context:', tostring(context), 'worldobjects:', tostring(worldobjects), 'test:', tostring(test))
        local playerObj = getSpecificPlayer and getSpecificPlayer(player) or nil
        if not playerObj then print('[WOODYDEBUG] RainSnow: playerObj NIL'); return end

        -- Check if clicking on water source
        local waterSource = nil
        if worldobjects and type(worldobjects) == 'table' then
            for i, obj in ipairs(worldobjects) do
                print('[WOODYDEBUG] RainSnow: worldobject', i, tostring(obj))
                if obj and instanceof and instanceof(obj, "IsoObject") then
                    local sprite = obj.getSprite and obj:getSprite() or nil
                    if not sprite then print('[WOODYDEBUG] RainSnow: missing sprite for object', tostring(obj)) end
                    if sprite and sprite.getName then
                        local spriteName = tostring(sprite:getName()):lower()
                        print('[WOODYDEBUG] RainSnow: spriteName:', spriteName)
                        if spriteName:find('sink') or spriteName:find('tap') or spriteName:find('barrel') or spriteName:find('well') then
                            waterSource = obj
                            break
                        end
                    else
                        print('[WOODYDEBUG] RainSnow: sprite missing getName')
                    end
                end
            end
        else
            print('[WOODYDEBUG] RainSnow: worldobjects is nil or not table')
        end

        if waterSource then
            local hasBlood = false
            local hasDirt = false
            local visual = playerObj.getHumanVisual and playerObj:getHumanVisual() or nil
            if not visual then print('[WOODYDEBUG] RainSnow: playerObj:getHumanVisual() NIL') end
            if visual and BloodBodyPartType and BloodBodyPartType.MAX and BloodBodyPartType.MAX.index and BloodBodyPartType.FromIndex then
                for i=0, BloodBodyPartType.MAX:index()-1 do
                    local part = BloodBodyPartType.FromIndex(i)
                    if visual.getBlood and visual:getBlood(part) > 0 then hasBlood = true end
                    if visual.getDirt and visual:getDirt(part) > 0 then hasDirt = true end
                end
            else
                print('[WOODYDEBUG] RainSnow: visual or BloodBodyPartType methods missing')
            end
            local wornItems = playerObj.getWornItems and playerObj:getWornItems() or nil
            if wornItems and wornItems.size and wornItems.getItemByIndex then
                for i=0, wornItems:size()-1 do
                    local wornItem = wornItems:getItemByIndex(i)
                    local item = wornItem and wornItem.getItem and wornItem:getItem() or nil
                    if item then
                        if item.getBloodLevel and (item:getBloodLevel() or 0) > 0 then hasBlood = true end
                        if item.getDirtLevel and (item:getDirtLevel() or 0) > 0 then hasDirt = true end
                    end
                end
            else
                print('[WOODYDEBUG] RainSnow: wornItems is nil or missing methods')
            end
            if hasBlood or hasDirt then
                local option = context and context.addOption and context:addOption("Wash Blood & Dirt", playerObj, function(player)
                    if ISTimedActionQueue and ISTimedActionQueue.add and WoodysRainSnow_ManualWashAction and WoodysRainSnow_ManualWashAction.new then
                        ISTimedActionQueue.add(WoodysRainSnow_ManualWashAction:new(player, waterSource))
                    else
                        print('[WOODYDEBUG] RainSnow: ISTimedActionQueue or WoodysRainSnow_ManualWashAction missing!')
                    end
                end) or nil
                if playerObj and playerObj.HasTrait and playerObj:HasTrait("Hemophobic") then
                    local mode = (WoodysRainSnow and WoodysRainSnow.CONFIG) and WoodysRainSnow.CONFIG.FEAR_OF_BLOOD_MODE or nil
                    if mode and mode ~= "OFF" then
                        if ISWorldObjectContextMenu and ISWorldObjectContextMenu.addToolTip and option then
                            local tooltip = ISWorldObjectContextMenu.addToolTip and ISWorldObjectContextMenu.addToolTip() or nil
                            if tooltip then
                                tooltip.description = "Fear of Blood trait: faster blood removal!"
                                option.toolTip = tooltip
                            else
                                print('[WOODYDEBUG] RainSnow: addToolTip returned nil')
                            end
                        else
                            print('[WOODYDEBUG] RainSnow: ISWorldObjectContextMenu, addToolTip, or option missing!')
                        end
                    end
                end
            end
        else
            print('[WOODYDEBUG] RainSnow: no waterSource detected')
        end
    end)
    if not success then print('[WOODYDEBUG] RainSnow:[EXCEPTION] in onFillWorldObjectContextMenu:', tostring(err)) end
end

Events.OnFillWorldObjectContextMenu.Add(safe_onFillWorldObjectContextMenu)

return WoodysRainSnow_ManualWashAction
