-- ===================================================================================================== --
-- Woody's Rain & Wet Snow Wash - Core System
-- ===================================================================================================== --
-- Author: Woody
-- Philosophy: Rain washes blood and dirt SEPARATELY and DIRECTLY (no weird conversions!)
-- Approach: Linear modifiers for predictable, realistic behavior

local WoodysRainSnow_Config = require("WoodysRainSnow_Config")
-- 
-- FEATURES:
-- ✅ Auto-rain washing - stand outside in rain/wet snow → gradual cleaning
-- ✅ Manual washing - right-click water source → controlled, faster cleaning
-- ✅ Fear of Blood trait bonus - MANUAL washing only! (rain doesn't care about fear!)
--    - Fixed boost: 1.5×, 2×, 3× (configurable)
--    - Progressive boost: little blood = normal speed, lots = 3× faster!
-- ✅ Wet snow support - temperature-based (40% as effective as rain)
-- ✅ Separate tracking - blood and dirt wash independently (no weird conversions!)
-- ✅ Linear modifiers - predictable, easy to understand behavior
-- ✅ Water resistance penalty - waterproof clothes harder to clean (less water absorption)
-- ✅ Realistic timing - AI-calculated wash times
--
-- Design Goals:
-- - Separate blood and dirt tracking (no blood-to-dirt conversion!)
-- - Linear curves (wetness/rain affect cleaning in straightforward way)
-- - Fear of Blood trait support (MANUAL washing only! Rain doesn't care about fear!)
-- - Wet snow support (temperature-based detection)
-- - Configurable rates (easy to tune for playtesting)
-- ===================================================================================================== --

WoodysRainSnow = WoodysRainSnow or {}

-- ===================================================================================================== --
-- CONFIGURATION CONSTANTS
-- ===================================================================================================== --
-- These values control how fast blood/dirt wash off in rain/snow
-- Higher = faster washing, Lower = slower washing
-- 
-- REALISTIC TIMING TABLE (90% removal, 100% wetness):
-- 
-- | Scenario            | Blood     | Dirt      | Notes                        |
-- |---------------------|-----------|-----------|------------------------------|
-- | Light rain (40%)    | ~9h 12m   | ~5h 45m   | Slow but steady             |
-- | Strong rain (80%)   | ~2h 18m   | ~1h 26m   | Most common scenario        |
-- | Storm (100%)        | ~1h 32m   | ~57m      | Very fast                   |
-- | Wet snow (80%)      | ~8h       | ~5h       | 40% as effective as rain    |
-- | Fear of Blood + Storm  | ~46m      | ~57m      | Panic-driven fast cleaning! |
-- 
-- DESIGN PHILOSOPHY:
-- - Blood washes SLOWER than dirt (organic stains are stubborn!)
-- - Dirt washes FASTER (non-organic particles)
-- - Fear of Blood trait gives 2× boost for blood (extreme motivation!)
-- - Wet snow is 40% as effective as rain (realistic)
-- ===================================================================================================== --

WoodysRainSnow.CONFIG = {
    -- Base cleaning rates (how much blood/dirt removed per minute)
    -- REALISTIC TIMING: Based on real-world rain washing
    -- Target: 90% removal in strong rain (80% intensity, 100% wet)
    --   Blood: ~138 minutes (2h 18min) - organic stains persist longer
    --   Dirt: ~86 minutes (1h 26min) - non-organic washes faster
    BLOOD_CLEAN_RATE = 0.015,       -- Blood washes slowly (realistic organic stains)
    DIRT_CLEAN_RATE = 0.024,        -- Dirt washes faster (non-organic particles)
    
    -- Weather modifiers
    RAIN_MULTIPLIER = 1.5,          -- Rain effectiveness multiplier
    WET_SNOW_MULTIPLIER = 0.4,      -- Wet snow = 40% as effective as rain (realistic!)
    WET_SNOW_TEMP_MIN = -1,         -- Wet snow temp range (°C)
    WET_SNOW_TEMP_MAX = 2,          -- Above 2°C = rain, below -1°C = dry snow
    
    -- Trait bonuses (for MANUAL washing only, not auto-rain!)
    -- Note: Fear of Blood boost is configured via ModOptions
    -- Rain doesn't care how scared you are - boost only applies to manual scrubbing!
    
    -- Minimum thresholds
    MIN_WEATHER_INTENSITY = 0.1,    -- Minimum rain/snow to start washing
    MIN_WETNESS = 0.1,              -- Minimum wetness to start washing (0-100 scale)
}

-- ===================================================================================================== --
-- STATE TRACKING
-- ===================================================================================================== --
-- We track blood and dirt levels separately for each body part and clothing item
-- This allows independent cleaning rates without weird conversions
-- ===================================================================================================== --

WoodysRainSnow.character = nil
WoodysRainSnow.bodyStats = {}      -- { [BloodBodyPartType] = { blood = 0.5, dirt = 0.3 } }
WoodysRainSnow.clothStats = {}     -- { [itemID] = { [BloodBodyPartType] = { blood = 0.4, dirt = 0.2 } } }

local lastUpdateTime = 0
local updateModel = false
local debugLogCounter = 0  -- Log clothing status every 5 minutes

-- ===================================================================================================== --
-- WEATHER DETECTION
-- ===================================================================================================== --

--- Get current weather washing intensity
--- @return number intensity (0.0-1.0), string type ("rain" or "wet_snow")
function WoodysRainSnow.getWeatherWashIntensity()
    local climate = getClimateManager()
    local rainIntensity = climate:getRainIntensity()
    local snowIntensity = climate:getSnowIntensity()
    local temperature = climate:getTemperature()
    
    -- Check rain first
    if rainIntensity >= WoodysRainSnow.CONFIG.MIN_WEATHER_INTENSITY then
        return rainIntensity, "rain"
    end
    
    -- Check wet snow (temperature-based)
    if snowIntensity >= WoodysRainSnow.CONFIG.MIN_WEATHER_INTENSITY then
        if temperature >= WoodysRainSnow.CONFIG.WET_SNOW_TEMP_MIN 
           and temperature <= WoodysRainSnow.CONFIG.WET_SNOW_TEMP_MAX then
            return snowIntensity, "wet_snow"
        end
    end
    
    return 0.0, "none"
end

-- ===================================================================================================== --
-- MODIFIER CALCULATIONS (LINEAR APPROACH)
-- ===================================================================================================== --
-- These use LINEAR curves for predictable, realistic behavior
-- 
-- WHY LINEAR?
-- - Predictable: 50% wet = 50% effective (easy to understand)
-- - Realistic: Washing scales proportionally with wetness/rain
-- - No sudden jumps: Smooth washing experience
-- 
-- COMPARISON WITH QUADRATIC (x²):
-- - Linear (x):     60% wet = 60% effective
-- - Quadratic (x²): 60% wet = 36% effective (much slower!)
-- ===================================================================================================== --

--- Calculate wetness modifier (how wet the item/body part is)
--- @param wetness number Wetness level (0-100)
--- @return number modifier (0.0-1.0)
function WoodysRainSnow.getWetnessMod(wetness)
    -- Linear: direct proportion
    -- 0% wet = 0% effective
    -- 50% wet = 50% effective
    -- 100% wet = 100% effective
    return wetness / 100
end

--- Calculate weather modifier (how intense the rain/snow is)
--- @param intensity number Weather intensity (0.0-1.0)
--- @param weatherType string "rain" or "wet_snow"
--- @return number modifier
function WoodysRainSnow.getWeatherMod(intensity, weatherType)
    if weatherType == "wet_snow" then
        -- Wet snow is less effective than rain
        return intensity * WoodysRainSnow.CONFIG.WET_SNOW_MULTIPLIER
    else
        -- Regular rain with multiplier
        return intensity * WoodysRainSnow.CONFIG.RAIN_MULTIPLIER
    end
end

--- Get trait-based boost for character (AUTO-RAIN ONLY)
--- Note: Fear of Blood boost is NOT applied to auto-rain washing!
--- Rain doesn't care how scared you are - boost only for manual scrubbing!
--- @param character IsoPlayer
--- @return number bloodBoost, number dirtBoost
function WoodysRainSnow.getTraitBoosts(character)
    -- No trait boosts for auto-rain washing!
    -- Fear of Blood boost only applies to manual washing actions
    return 1.0, 1.0
end

-- ===================================================================================================== --
-- BODY PART CLEANING
-- ===================================================================================================== --
-- Clean blood and dirt from character's body parts
-- Each stat (blood, dirt) is tracked and cleaned SEPARATELY
-- ===================================================================================================== --

--- Update all body parts for character
--- @param character IsoPlayer
--- @param weatherIntensity number
--- @param weatherType string
function WoodysRainSnow.updateBodyParts(character, weatherIntensity, weatherType)
    local visual = character:getHumanVisual()
    local bodyDamage = character:getBodyDamage()
    local bodyParts = bodyDamage:getBodyParts()
    
    local weatherMod = WoodysRainSnow.getWeatherMod(weatherIntensity, weatherType)
    local bloodBoost, dirtBoost = WoodysRainSnow.getTraitBoosts(character)
    
    -- Process each body part
    for i = 0, bodyParts:size() - 1 do
        local bodyPart = bodyParts:get(i)
        local bloodBodyPartType = BloodBodyPartType.FromIndex(bodyPart:getIndex())
        
        -- Only wash if wet enough
        local wetness = bodyPart:getWetness()
        if wetness >= WoodysRainSnow.CONFIG.MIN_WETNESS then
            local wetnessMod = WoodysRainSnow.getWetnessMod(wetness)
            
            -- Get current blood/dirt levels
            local blood = visual:getBlood(bloodBodyPartType)
            local dirt = visual:getDirt(bloodBodyPartType)
            
            -- Clean blood SEPARATELY
            if blood > 0 then
                local bloodClean = WoodysRainSnow.CONFIG.BLOOD_CLEAN_RATE 
                                   * weatherMod 
                                   * wetnessMod 
                                   * bloodBoost
                
                local newBlood = math.max(0, blood - bloodClean)
                visual:setBlood(bloodBodyPartType, newBlood)
                
                -- Track change
                if not WoodysRainSnow.bodyStats[bloodBodyPartType] then
                    WoodysRainSnow.bodyStats[bloodBodyPartType] = {}
                end
                WoodysRainSnow.bodyStats[bloodBodyPartType].blood = newBlood
                updateModel = true
            end
            
            -- Clean dirt SEPARATELY (no dependency on blood!)
            if dirt > 0 then
                -- Apply progressive dirt penalty (heavy dirt slows cleaning)
                local dirtPenalty = 1.0
                if WoodysRainSnow_Config and WoodysRainSnow_Config.isRainPenaltyEnabled 
                   and WoodysRainSnow_Config.isRainPenaltyEnabled() then
                    local penaltyFactor = WoodysRainSnow_Config.getRainPenaltyFactor()
                    dirtPenalty = 1.0 - (dirt * penaltyFactor)
                    -- Ensure minimum 40% speed (never completely stops)
                    dirtPenalty = math.max(0.4, dirtPenalty)
                end
                
                local dirtClean = WoodysRainSnow.CONFIG.DIRT_CLEAN_RATE 
                                  * weatherMod 
                                  * wetnessMod 
                                  * dirtBoost
                                  * dirtPenalty
                
                local newDirt = math.max(0, dirt - dirtClean)
                visual:setDirt(bloodBodyPartType, newDirt)
                
                -- Track change
                if not WoodysRainSnow.bodyStats[bloodBodyPartType] then
                    WoodysRainSnow.bodyStats[bloodBodyPartType] = {}
                end
                WoodysRainSnow.bodyStats[bloodBodyPartType].dirt = newDirt
                updateModel = true
            end
        end
        
        -- Special case: Back (torso affects back too)
        if bodyPart:getType() == BodyPartType.Torso_Upper then
            if wetness >= WoodysRainSnow.CONFIG.MIN_WETNESS then
                local bloodBodyPartTypeBack = BloodBodyPartType.Back
                local bloodBack = visual:getBlood(bloodBodyPartTypeBack)
                local dirtBack = visual:getDirt(bloodBodyPartTypeBack)
                
                if bloodBack > 0 or dirtBack > 0 then
                    local wetnessMod = WoodysRainSnow.getWetnessMod(wetness)
                    
                    -- Clean back blood
                    if bloodBack > 0 then
                        local bloodClean = WoodysRainSnow.CONFIG.BLOOD_CLEAN_RATE 
                                           * weatherMod 
                                           * wetnessMod 
                                           * bloodBoost
                        visual:setBlood(bloodBodyPartTypeBack, math.max(0, bloodBack - bloodClean))
                        updateModel = true
                    end
                    
                    -- Clean back dirt
                    if dirtBack > 0 then
                        -- Apply progressive dirt penalty
                        local dirtPenalty = 1.0
                        if WoodysRainSnow_Config and WoodysRainSnow_Config.isRainPenaltyEnabled 
                           and WoodysRainSnow_Config.isRainPenaltyEnabled() then
                            local penaltyFactor = WoodysRainSnow_Config.getRainPenaltyFactor()
                            dirtPenalty = 1.0 - (dirtBack * penaltyFactor)
                            dirtPenalty = math.max(0.4, dirtPenalty)
                        end
                        
                        local dirtClean = WoodysRainSnow.CONFIG.DIRT_CLEAN_RATE 
                                          * weatherMod 
                                          * wetnessMod 
                                          * dirtBoost
                                          * dirtPenalty
                        visual:setDirt(bloodBodyPartTypeBack, math.max(0, dirtBack - dirtClean))
                        updateModel = true
                    end
                end
            end
        end
    end
end

-- ===================================================================================================== --
-- CLOTHING CLEANING
-- ===================================================================================================== --
-- Clean blood and dirt from worn clothing items
-- ===================================================================================================== --

--- Update all worn clothing for character
--- @param character IsoPlayer
--- @param weatherIntensity number
--- @param weatherType string
function WoodysRainSnow.updateClothing(character, weatherIntensity, weatherType)
    local wornItems = character:getWornItems()
    local weatherMod = WoodysRainSnow.getWeatherMod(weatherIntensity, weatherType)
    local bloodBoost, dirtBoost = WoodysRainSnow.getTraitBoosts(character)
    
    for i = 0, wornItems:size() - 1 do
        local item = wornItems:get(i):getItem()
        
        if item:IsClothing() and not item:isHidden() then
            local wetness = item:getWetness()
            
            -- Only wash if wet enough
            if wetness >= WoodysRainSnow.CONFIG.MIN_WETNESS then
                local wetnessMod = WoodysRainSnow.getWetnessMod(wetness)
                
                -- Get covered body parts
                local coveredParts = BloodClothingType.getCoveredParts(item:getBloodClothingType())
                if coveredParts then
                    for j = 0, coveredParts:size() - 1 do
                        local part = coveredParts:get(j)
                        if part then
                            local blood = item:getBlood(part)
                            local dirt = item:getDirt(part)
                            
                            -- Clean blood SEPARATELY
                            if blood > 0 then
                                -- Water resistance affects cleaning (waterproof = less water absorption = slower cleaning)
                                -- WR 0.0 = normal fabric → penalty 1.0 (no penalty)
                                -- WR 0.5 = semi-waterproof → penalty 1.5 (slower)
                                -- WR 1.0 = raincoat → penalty 2.0 (much slower)
                                local waterResistancePenalty = 1.0 + item:getWaterResistance()
                                
                                local bloodClean = (WoodysRainSnow.CONFIG.BLOOD_CLEAN_RATE 
                                                   * weatherMod 
                                                   * wetnessMod 
                                                   * bloodBoost) / waterResistancePenalty
                                
                                local newBlood = math.max(0, blood - bloodClean)
                                item:setBlood(part, newBlood)
                                updateModel = true
                            end
                            
                            -- Clean dirt SEPARATELY
                            if dirt > 0 then
                                local waterResistancePenalty = 1.0 + item:getWaterResistance()
                                
                                -- Apply progressive dirt penalty
                                local dirtPenalty = 1.0
                                if WoodysRainSnow_Config and WoodysRainSnow_Config.isRainPenaltyEnabled 
                                   and WoodysRainSnow_Config.isRainPenaltyEnabled() then
                                    local penaltyFactor = WoodysRainSnow_Config.getRainPenaltyFactor()
                                    dirtPenalty = 1.0 - (dirt * penaltyFactor)
                                    dirtPenalty = math.max(0.4, dirtPenalty)
                                end
                                
                                local dirtClean = (WoodysRainSnow.CONFIG.DIRT_CLEAN_RATE 
                                                  * weatherMod 
                                                  * wetnessMod 
                                                  * dirtBoost
                                                  * dirtPenalty) / waterResistancePenalty
                                
                                local newDirt = math.max(0, dirt - dirtClean)
                                item:setDirt(part, newDirt)
                                updateModel = true
                            end
                        end
                    end
                    
                    -- Update item's overall blood/dirt levels
                    local totalBlood = 0
                    local totalDirt = 0
                    for j = 0, coveredParts:size() - 1 do
                        local part = coveredParts:get(j)
                        if part then
                            totalBlood = totalBlood + item:getBlood(part)
                            totalDirt = totalDirt + item:getDirt(part)
                        end
                    end
                    item:setBloodLevel(totalBlood * 100 / coveredParts:size())
                    item:setDirtyness(totalDirt * 100 / coveredParts:size())
                end
            end
        end
    end
end

-- ===================================================================================================== --
-- BAGS & WEAPONS CLEANING
-- ===================================================================================================== --
-- Clean blood from equipped bags and weapons (they only have blood, no dirt)
-- ===================================================================================================== --

--- Update equipped bags and weapons
--- @param character IsoPlayer
--- @param weatherIntensity number
--- @param weatherType string
function WoodysRainSnow.updateEquipment(character, weatherIntensity, weatherType)
    local inventory = character:getInventory()
    local bodyDamage = character:getBodyDamage()
    local handPart = bodyDamage:getBodyPart(BodyPartType.Hand_R)
    local wetness = handPart:getWetness()
    
    -- Only wash if hands are wet enough
    if wetness < WoodysRainSnow.CONFIG.MIN_WETNESS then
        return
    end
    
    local weatherMod = WoodysRainSnow.getWeatherMod(weatherIntensity, weatherType)
    local wetnessMod = WoodysRainSnow.getWetnessMod(wetness)
    local bloodBoost, _ = WoodysRainSnow.getTraitBoosts(character)
    
    -- Clean equipped bags
    local bags = inventory:getItemsFromCategory("Container")
    for i = 0, bags:size() - 1 do
        local bag = bags:get(i)
        if bag:isEquipped() and not bag:isHidden() then
            local blood = bag:getBloodLevel()
            if blood > 0 then
                local bloodClean = WoodysRainSnow.CONFIG.BLOOD_CLEAN_RATE 
                                   * weatherMod 
                                   * wetnessMod 
                                   * bloodBoost
                
                bag:setBloodLevel(math.max(0, blood - bloodClean))
                updateModel = true
            end
        end
    end
    
    -- Clean equipped/attached weapons
    local weapons = inventory:getItemsFromCategory("Weapon")
    for i = 0, weapons:size() - 1 do
        local weapon = weapons:get(i)
        if weapon:isEquipped() or weapon:getAttachedSlot() > 0 then
            local blood = weapon:getBloodLevel()
            if blood > 0 then
                -- Attached weapons wash slower (not directly in hands)
                local attachedPenalty = 1.0
                if weapon:getAttachedSlot() > 0 and not character:isHandItem(weapon) then
                    attachedPenalty = 2.0
                end
                
                local bloodClean = (WoodysRainSnow.CONFIG.BLOOD_CLEAN_RATE 
                                   * weatherMod 
                                   * wetnessMod 
                                   * bloodBoost) / attachedPenalty
                
                weapon:setBloodLevel(math.max(0, blood - bloodClean))
                updateModel = true
            end
        end
    end
end

-- ===================================================================================================== --
-- MAIN UPDATE LOOP
-- ===================================================================================================== --

function WoodysRainSnow.onEveryMinute()
    if not WoodysRainSnow.character then return end
    
    local character = WoodysRainSnow.character
    
    -- Check if character is outside (no washing indoors!)
    if not character:isOutside() then return end
    
    -- Check if in vehicle (no washing in cars!)
    if character:getVehicle() then return end
    
    -- Get current weather
    local weatherIntensity, weatherType = WoodysRainSnow.getWeatherWashIntensity()
    if weatherIntensity < WoodysRainSnow.CONFIG.MIN_WEATHER_INTENSITY then
        return  -- Not raining/snowing enough
    end
    
    -- Reset update flag
    updateModel = false
    
    -- Debug: log clothing status every 5 minutes
    debugLogCounter = debugLogCounter + 1
    local shouldDebugLog = (debugLogCounter >= 5)
    
    if shouldDebugLog then
        print("[WOODYDEBUG]_RainSnow:_Rain_washing_active_weather:" .. weatherType .. "_intensity:" .. math.floor(weatherIntensity * 100) .. "%")
        debugLogCounter = 0
    end
    
    -- Clean everything!
    WoodysRainSnow.updateBodyParts(character, weatherIntensity, weatherType)
    WoodysRainSnow.updateClothing(character, weatherIntensity, weatherType)
    WoodysRainSnow.updateEquipment(character, weatherIntensity, weatherType)
    
    -- Debug: log some clothing status (every 5 minutes)
    if shouldDebugLog then
        local wornItems = character:getWornItems()
        local itemCount = 0
        for i = 0, wornItems:size() - 1 do
            local item = wornItems:get(i):getItem()
            if item:IsClothing() and not item:isHidden() and itemCount < 3 then  -- Log first 3 items only
                local wetness = math.floor(item:getWetness())
                local dirt = math.floor(item:getDirtyness())
                local blood = math.floor(item:getBloodLevel())
                print("[WOODYDEBUG]_RainSnow:_Item_" .. item:getDisplayName() .. "_wet:" .. wetness .. "%_dirt:" .. dirt .. "%_blood:" .. blood .. "%")
                itemCount = itemCount + 1
            end
        end
    end
    
    -- Update character model if anything changed
    if updateModel then
        character:resetModelNextFrame()
        triggerEvent("OnClothingUpdated", character)
    end
end

-- ===================================================================================================== --
-- INITIALIZATION
-- ===================================================================================================== --

local function onGameStart()
    WoodysRainSnow.character = getPlayer()
    
    -- Start update loop
    Events.EveryOneMinute.Add(WoodysRainSnow.onEveryMinute)
    
    print("[WOODYDEBUG]_RainSnow:_Core_module_loaded_successfully")
    print("[WOODYDEBUG]_RainSnow:_Blood_clean_rate_" .. WoodysRainSnow.CONFIG.BLOOD_CLEAN_RATE)
    print("[WOODYDEBUG]_RainSnow:_Dirt_clean_rate_" .. WoodysRainSnow.CONFIG.DIRT_CLEAN_RATE)
end

Events.OnGameStart.Add(onGameStart)

return WoodysRainWash
