--***********************************************************
--**                    WOODY'S MODS                       **
--**     Rain & Wet Snow Cleaning - Config               **
--***********************************************************

local WoodysRainSnow_Config = {
    fearOfBloodModeOption = nil,
    manualWashSpeedOption = nil,
    enableWetSnowOption = nil,
    
    -- NEW: Wringing options
    enableWringCleaningOption = nil,
    wringDirtEfficiencyOption = nil,
    wringBloodRemovalOption = nil,
    wringWetnessBonusOption = nil,
    
    -- NEW: Rain penalty options
    enableRainPenaltyOption = nil,
    rainPenaltyFactorOption = nil,
}

-- ===================================================================================================== --
-- OPTION ACCESS FUNCTIONS
-- ===================================================================================================== --

---Get Fear of Blood mode
---@return number mode (1-5)
function WoodysRainSnow_Config.getFearOfBloodMode()
    if WoodysRainSnow_Config.fearOfBloodModeOption then
        return WoodysRainSnow_Config.fearOfBloodModeOption:getValue()
    else
        return 1  -- Default: Off
    end
end

---Get manual wash speed multiplier
---@return number speed
function WoodysRainSnow_Config.getManualWashSpeed()
    if WoodysRainSnow_Config.manualWashSpeedOption then
        return WoodysRainSnow_Config.manualWashSpeedOption:getValue()
    else
        return 1.0  -- Default: realistic speed
    end
end

---Get whether wet snow is enabled
---@return boolean enabled
function WoodysRainSnow_Config.isWetSnowEnabled()
    if WoodysRainSnow_Config.enableWetSnowOption then
        return WoodysRainSnow_Config.enableWetSnowOption:getValue()
    else
        return true  -- Default: enabled
    end
end

-- ===================================================================================================== --
-- NEW: WRINGING CONFIG OPTIONS
-- ===================================================================================================== --

---Get whether wringing cleaning is enabled
---@return boolean enabled
function WoodysRainSnow_Config.isWringCleaningEnabled()
    if WoodysRainSnow_Config.enableWringCleaningOption then
        return WoodysRainSnow_Config.enableWringCleaningOption:getValue()
    else
        return true  -- Default: enabled
    end
end

---Get wringing dirt efficiency (base multiplier)
---@return number efficiency (0.1-0.5, default 0.3 = 30%)
function WoodysRainSnow_Config.getWringDirtEfficiency()
    if WoodysRainSnow_Config.wringDirtEfficiencyOption then
        return WoodysRainSnow_Config.wringDirtEfficiencyOption:getValue()
    else
        return 0.3  -- Default: 30% base efficiency
    end
end

---Get wringing blood removal percentage
---@return number removal (0.0-0.1, default 0.05 = 5%)
function WoodysRainSnow_Config.getWringBloodRemoval()
    if WoodysRainSnow_Config.wringBloodRemovalOption then
        return WoodysRainSnow_Config.wringBloodRemovalOption:getValue()
    else
        return 0.05  -- Default: 5% blood removal
    end
end

---Get wringing wetness bonus exponent
---@return number exponent (1.0-2.0, default 1.5)
function WoodysRainSnow_Config.getWringWetnessBonus()
    if WoodysRainSnow_Config.wringWetnessBonusOption then
        return WoodysRainSnow_Config.wringWetnessBonusOption:getValue()
    else
        return 1.5  -- Default: quadratic bonus (x^1.5)
    end
end

-- ===================================================================================================== --
-- NEW: RAIN PENALTY CONFIG OPTIONS
-- ===================================================================================================== --

---Get whether rain dirt penalty is enabled
---@return boolean enabled
function WoodysRainSnow_Config.isRainPenaltyEnabled()
    if WoodysRainSnow_Config.enableRainPenaltyOption then
        return WoodysRainSnow_Config.enableRainPenaltyOption:getValue()
    else
        return true  -- Default: enabled
    end
end

---Get rain penalty factor (how much dirt slows cleaning)
---@return number factor (0.0-1.0, default 0.6 = 60% max penalty)
function WoodysRainSnow_Config.getRainPenaltyFactor()
    if WoodysRainSnow_Config.rainPenaltyFactorOption then
        return WoodysRainSnow_Config.rainPenaltyFactorOption:getValue()
    else
        return 0.6  -- Default: 60% maximum penalty at 100% dirt
    end
end

---Calculate Fear of Blood boost for manual washing
---@param character IsoPlayer
---@param mode number (1-5: Off, 1.5x, 2x, 3x, Auto)
---@return number boost
function WoodysRainSnow_Config.getFearOfBloodBoost(character, mode)
    -- Check if character has Fear of Blood trait
    if not character:HasTrait("Hemophobic") then
        return 1.0  -- No boost if not fearOfBlood
    end
    
    -- Mode 1: Disabled
    if mode == 1 then
        return 1.0
    end
    
    -- Mode 2: Fixed 1.5×
    if mode == 2 then
        return 1.5
    end
    
    -- Mode 3: Fixed 2×
    if mode == 3 then
        return 2.0
    end
    
    -- Mode 4: Fixed 3×
    if mode == 4 then
        return 3.0
    end
    
    -- Mode 5: Progressive (based on blood amount)
    if mode == 5 then
        return WoodysRainSnow_Config.calculateProgressiveBoost(character)
    end
    
    -- Fallback
    return 1.0
end

--- Calculate progressive boost based on total blood amount on character
--- Logic: More blood = more panic = faster scrubbing!
--- @param character IsoPlayer
--- @return number boost (1.0-3.0)
function WoodysRainSnow_Config.calculateProgressiveBoost(character)
    local totalBlood = 0
    local totalParts = 0
    
    -- Sum blood from all body parts
    local visual = character:getHumanVisual()
    for i = 0, BloodBodyPartType.MAX:index() - 1 do
        local bloodBodyPartType = BloodBodyPartType.FromIndex(i)
        local blood = visual:getBlood(bloodBodyPartType)
        totalBlood = totalBlood + blood
        totalParts = totalParts + 1
    end
    
    -- Calculate average blood level (0.0-1.0)
    local avgBlood = totalBlood / totalParts
    
    -- Progressive scaling:
    -- 0-30% blood: 1.0× (little blood, normal washing)
    -- 30-60% blood: 1.0-2.0× (medium blood, getting worried)
    -- 60-100% blood: 2.0-3.0× (heavy blood, EXTREME PANIC!)
    
    if avgBlood < 0.3 then
        -- Little blood: no boost yet
        return 1.0
    elseif avgBlood < 0.6 then
        -- Medium blood: linear scale from 1.0 to 2.0
        local progress = (avgBlood - 0.3) / 0.3  -- 0.0-1.0
        return 1.0 + (progress * 1.0)  -- 1.0-2.0
    else
        -- Heavy blood: linear scale from 2.0 to 3.0
        local progress = (avgBlood - 0.6) / 0.4  -- 0.0-1.0
        return 2.0 + (progress * 1.0)  -- 2.0-3.0
    end
end

-- ===================================================================================================== --
-- MOD OPTIONS INITIALIZATION
-- ===================================================================================================== --

function WoodysRainSnow_Config.initialize()
    if not PZAPI or not PZAPI.ModOptions then
        print("[WOODYDEBUG]_RainSnow:_PZAPI_ModOptions_not_available_using_defaults")
        return
    end

    print("[WOODYDEBUG]_RainSnow:_Initializing_mod_options")
    local options = PZAPI.ModOptions:create("WoodysRainSnow", "Woody's Rain & Wet Snow Cleaning")

    -- Description
    options:addDescription("Configure Rain & Wet Snow Cleaning options below.")

    -- ===== GENERAL OPTIONS =====
    options:addDescription(" ")
    options:addDescription("[GENERAL]")
    
    -- Wet Snow (tickbox)
    WoodysRainSnow_Config.enableWetSnowOption = options:addTickBox(
        "enableWetSnow",
        "Wet Snow Cleaning",
        true
    )

    -- Fear of Blood Boost (dropdown)
    WoodysRainSnow_Config.fearOfBloodModeOption = options:addComboBox(
        "fearOfBloodMode",
        "Fear of Blood Boost",
        "Choose speed boost (manual wash only)"
    )
    WoodysRainSnow_Config.fearOfBloodModeOption:addItem("Off", true)
    WoodysRainSnow_Config.fearOfBloodModeOption:addItem("1.5x", false)
    WoodysRainSnow_Config.fearOfBloodModeOption:addItem("2x", false)
    WoodysRainSnow_Config.fearOfBloodModeOption:addItem("3x", false)
    WoodysRainSnow_Config.fearOfBloodModeOption:addItem("Auto", false)

    -- ===== RAIN WASHING OPTIONS =====
    options:addDescription(" ")
    options:addDescription("[RAIN WASHING]")
    options:addDescription("Progressive dirt penalty: heavy dirt slows rain cleaning")
    
    WoodysRainSnow_Config.enableRainPenaltyOption = options:addTickBox(
        "enableRainPenalty",
        "Enable Dirt Penalty",
        true
    )
    
    WoodysRainSnow_Config.rainPenaltyFactorOption = options:addSlider(
        "rainPenaltyFactor",
        "Penalty Strength (0% = none, 100% = max)",
        0.0,
        1.0,
        0.05,
        0.6
    )

    -- ===== WRINGING OPTIONS =====
    options:addDescription(" ")
    options:addDescription("[WRINGING CLOTHING]")
    options:addDescription("Wringing removes dirt/blood mechanically (realistic!)")
    
    WoodysRainSnow_Config.enableWringCleaningOption = options:addTickBox(
        "enableWringCleaning",
        "Enable Wringing Cleaning",
        true
    )
    
    options:addDescription("Dirt efficiency: how much dirt wringing removes")
    WoodysRainSnow_Config.wringDirtEfficiencyOption = options:addSlider(
        "wringDirtEfficiency",
        "Dirt Removal Efficiency (10%-50%)",
        0.1,
        0.5,
        0.05,
        0.3
    )
    
    options:addDescription("Blood removal: fixed amount per wring")
    WoodysRainSnow_Config.wringBloodRemovalOption = options:addSlider(
        "wringBloodRemoval",
        "Blood Removal (0%-10%)",
        0.0,
        0.1,
        0.01,
        0.05
    )
    
    options:addDescription("Wetness bonus: wet clothes clean better (nonlinear)")
    WoodysRainSnow_Config.wringWetnessBonusOption = options:addSlider(
        "wringWetnessBonus",
        "Wetness Bonus Curve (1.0=linear, 2.0=strong)",
        1.0,
        2.0,
        0.1,
        1.5
    )
    
    print("[WOODYDEBUG]_RainSnow:_ModOptions_initialized_successfully")
    return true
end

-- ===================================================================================================== --
-- GETTERS
-- ===================================================================================================== --

function WoodysRainSnow_Config.getFearOfBloodMode()
    if WoodysRainSnow_Config.fearOfBloodModeOption then
        return WoodysRainSnow_Config.fearOfBloodModeOption:getValue()
    else
        return 3  -- Default: Fixed 2x
    end
end

function WoodysRainSnow_Config.getManualWashSpeed()
    if WoodysRainSnow_Config.manualWashSpeedOption then
        return WoodysRainSnow_Config.manualWashSpeedOption:getValue()
    else
        return 1.0  -- Default: realistic speed
    end
end

function WoodysRainSnow_Config.isWetSnowEnabled()
    if WoodysRainSnow_Config.enableWetSnowOption then
        return WoodysRainSnow_Config.enableWetSnowOption:getValue()
    else
        return true  -- Default: enabled
    end
end

return WoodysRainSnow_Config
