--===================================================================
-- RWC_Utility.lua
-- Helpers, all inline-debug via RWC.DEBUG.
--===================================================================
if not RWC then RWC = {} end

function RWC_Clamp(val,lo,up)
	if val < lo then return lo end
    if val > up then return up end
	return val
end 
--=====================================================================
-- Determine cold pack storage context
--=====================================================================
function RWC_GetStorageTypeForItem(item, containerOverride)
    if not item or not item.getContainer then
        if RWC.DEBUG then print("[RWC] RWC_GetStorageTypeForItem: no item or no getContainer") end
        return RWC.Storage.AMBIENT
    end
    if RWC.DEBUG then print(string.format("[RWC] Checking storage for %s id=%s", item:getFullType(), tostring(item:getID()))) end
	
    -- Prefer the explicit container passed from ISInventoryTransferAction,
    local container = containerOverride or item:getContainer()
	if not container then
        if RWC.DEBUG then print("[RWC] ➜ No container: ambient") end
        return RWC.Storage.AMBIENT
    end
	local cType = container.getType and container:getType() or "unknown"
    if not container then
        if RWC.DEBUG then print("[RWC] ➜ No container: ambient") end
        return RWC.Storage.AMBIENT
    end

    -- direct cooler check
    if container.getContainingItem then
        local ownerItem = container:getContainingItem()
        if ownerItem and ownerItem.getFullType and RWC.CoolerTypes then
            local oFull = ownerItem:getFullType()
            if RWC.CoolerTypes[oFull] then
                if RWC.DEBUG then print(string.format("[RWC] ➜ Cooler container detected via owner %s",tostring(oFull))) end
                return RWC.Storage.COOLER
            end
        end
    end

    -- Build 42 fridge/freezer detection
    local powered = container.isPowered and container:isPowered() or false
    if RWC.DEBUG then print(string.format("[RWC] ➜ Raw container type=%s | powered=%s",tostring(cType),tostring(powered))) end

    if powered then
        if cType == "freezer" then
            if RWC.DEBUG then print("[RWC] ➜ Classified as 'freezer'") end
            return RWC.Storage.FREEZER
        elseif cType == "fridge" then
            if RWC.DEBUG then print("[RWC] ➜ Classified as 'fridge'") end
            return RWC.Storage.FRIDGE
        end
    end

    if RWC.DEBUG then print("[RWC] ➜ Default: ambient") end
    return RWC.Storage.AMBIENT
end

-----------------------------------------------------------------------
-- Register a cold pack 
-- item is tested before function is called
-----------------------------------------------------------------------
function RWC_RegisterColdPack(item)
	local reg = RWC.ColdPackRegistry
	local DEBUG = RWC.DEBUG
    
	-- Check if this exact object is already in the list
    for i = 1, RWC.ColdPackRegistryCount do
        if reg[i] == item then
            if DEBUG then print(string.format("[RWC] Cooler already registered at index %d: ID=%s (%s)",i, tostring(item:getID()), item:getFullType())) end
            return
        end
    end
	
	-- append a new entry
	RWC.ColdPackRegistryCount = RWC.ColdPackRegistryCount + 1
	RWC.ColdPackRegistry[RWC.ColdPackRegistryCount] = item
    if DEBUG then print(string.format("[RWC] Registered cold pack index=%d ID=%s (%s)",RWC.ColdPackRegistryCount, tostring(item:getID()), item:getFullType())) end
end

-----------------------------------------------------------------------
-- Register a cooler item 
-- item is tested before function is called
-----------------------------------------------------------------------
function RWC_RegisterCoolerItem(item)
	local reg = RWC.ColdPackRegistry
	local DEBUG = RWC.DEBUG
	
    -- Check if this exact object is already in the list
    for i = 1, RWC.CoolerRegistryCount do
        if reg[i] == item then
            if DEBUG then print(string.format("[RWC] Cooler already registered at index %d: ID=%s (%s)",i, tostring(item:getID()), item:getFullType())) end
            return
        end
    end
	
	-- append a new entry
	RWC.CoolerRegistryCount = RWC.CoolerRegistryCount + 1
	RWC.CoolerRegistry[RWC.CoolerRegistryCount] = item
    if DEBUG then print(string.format("[RWC] Registered cooler index=%d ID=%s (%s)",RWC.CoolerRegistryCount, tostring(item:getID()), item:getFullType())) end
end

-----------------------------------------------------------------------
-- Remove a cooler at a specific index by swapping with last.
-- Keeps the array dense and O(1).
-----------------------------------------------------------------------
local function RWC_RemoveCoolerAtIndex(idx)
	-- index out of range safety
    --if idx < 1 or idx > RWC.CoolerRegistryCount then return end

	-- pull the item to be removed fo debug print
    --local removed = RWC.CoolerRegistry[idx]

    -- Move last element into idx, should also work it idx is last item
    local lastItem = RWC.CoolerRegistry[RWC.CoolerRegistryCount]
    RWC.CoolerRegistry[idx] = lastItem
    RWC.CoolerRegistry[RWC.CoolerRegistryCount] = nil
    RWC.CoolerRegistryCount = RWC.CoolerRegistryCount - 1

    --if RWC.DEBUG then print(string.format("[RWC] Removed cooler at index=%d; new count=%d (removed ID=%s)",idx, RWC.CoolerRegistryCount, removed and tostring(removed:getID()) or "nil")) end
end

-------------------------------------------------------------------
-- recursively scan an ItemContainer for RWC items
-------------------------------------------------------------------
function RWC_scanContainerRecursive(container)
	if not container then return end

	local items = container:getItems()
	if not items then return end

	for i = 0, items:size() - 1 do
		local item = items:get(i)
		if item and item.getFullType then
			local fullType = item:getFullType()

			-- Cold packs
			if RWC.ColdPackTypes and RWC.ColdPackTypes[fullType] then
				local md = item:getModData()
                md.RWC_StoredIn = RWC_GetStorageTypeForItem(item, container)
                RWC_RegisterColdPack(item)
				if RWC.DEBUG then print(string.format("[RWC] OnGameStart: found cold pack (%s)",tostring(fullType))) end
			end

			-- Coolers
			if RWC.CoolerTypes and RWC.CoolerTypes[fullType] then
				RWC_RegisterCoolerItem(item)
				if RWC.DEBUG then print(string.format("[RWC] OnGameStart: found cooler item (%s)",tostring(fullType))) end
			end
		end

		-- Recurse into nested containers (bags, boxes, etc.)
		if item and item.getInventory then
			local rec = item:getInventory()
			if rec then
				RWC_scanContainerRecursive(rec)
			end
		end
	end
end

-----------------------------------------------------------------------
-- Map a cold pack battery value (0–100) to a state name
-- using RWC.ColdPackBatteryStates.
--
-- Returns one of: "Frozen", "Icy", "Chilled", "Cool", "Warm"
-----------------------------------------------------------------------
function RWC_GetColdPackStateName(battery)
    if not battery then battery = 0 end
    -- clamp to 0–100 just in case
    if battery < 0 then battery = 0 end
    if battery > 100 then battery = 100 end

    for _, entry in ipairs(RWC.ColdPackBatteryStates) do
        local threshold = entry[1]
        local stateName = entry[2]
        if battery >= threshold then
            return stateName
        end
    end

    -- Fallback; should never really hit because last threshold is 0.0
    return "Warm"
end

-----------------------------------------------------------------------
-- Update a cold pack's display name based on its battery level.
-- Uses:
--   • md.RWC_ColdBattery (0–100)
--   • RWC_GetColdPackStateName(battery)
--   • RWC.Suffix.ColdPack[stateName]
--
-- Stores the original base name in md.RWC_BaseName the first time.
-----------------------------------------------------------------------
function RWC_UpdateColdPackName(item)
    if not item then return end

    local md = item:getModData()

    -- Remember the original base name (without any suffixes) once
    if not md.RWC_BaseName or md.RWC_BaseName == "" then
        -- Try to use the script item's display name first
        local scriptItem = item.getScriptItem and item:getScriptItem() or nil
        local baseName = nil

        if scriptItem and scriptItem.getDisplayName then
            baseName = scriptItem:getDisplayName()
        elseif item.getDisplayName then
            baseName = item:getDisplayName()
        else
            baseName = item:getName()
        end

        md.RWC_BaseName = baseName or item:getName()
    end

	-- figure out suffix name
	local stateName = "Warm"
    for _, entry in ipairs(RWC.ColdPackBatteryStates) do
        local threshold = entry[1]
        local sn = entry[2]
        if md.RWC_ColdBattery >= threshold then
            stateName = sn
			break
        end
    end

	-- stop here if it the state is unchanged
	if md.RWC_LastShownState == stateName then return end
	md.RWC_LastShownState = stateName
	
    local suffix    = RWC.Suffix and RWC.Suffix.ColdPack and RWC.Suffix.ColdPack[stateName] or nil

    local newName = md.RWC_BaseName
    if suffix then
        newName = md.RWC_BaseName .. suffix
    end

    -- Only rename if changed, to avoid spamming updates
    if item.getName and item.setName then
        if item:getName() ~= newName then
            item:setName(newName)
			item:setCustomName(true)
			local c = item:getContainer() 
			if c and c.setDrawDirty then c:setDrawDirty(true) end
            if RWC.DEBUG then print(string.format("[RWC] Updated cold pack name to '%s' (battery=%.1f, state=%s)", newName, md.RWC_ColdBattery, tostring(stateName))) end
        end
    end
end

-----------------------------------------------------------------------
-- Update a cooler's display name based on its "battery" level (0–100).
-- Uses:
--   • md.RWC_CoolerBattery (0–100)
--   • RWC.CoolerStateByTemp (thresholds)
--   • RWC.Suffix.Cooler[stateName]
--
-- Stores original name in md.RWC_CoolerBaseName the first time.
-----------------------------------------------------------------------
function RWC_UpdateCoolerName(item)
    if not item then return end
    local md = item:getModData()

    -- Remember original base name (without suffix) once
    if not md.RWC_CoolerBaseName or md.RWC_CoolerBaseName == "" then
        local scriptItem = item.getScriptItem and item:getScriptItem() or nil
        local baseName

        if scriptItem and scriptItem.getDisplayName then
            baseName = scriptItem:getDisplayName()
        elseif item.getDisplayName then
            baseName = item:getDisplayName()
        else
            baseName = item:getName()
        end

        md.RWC_CoolerBaseName = baseName or item:getName()
    end

    -- Map battery -> stateName using your CoolerStateByTemp config
    local stateName = "Warm"
    if RWC.CoolerStateByTemp then
        for _, entry in ipairs(RWC.CoolerStateByTemp) do
            local threshold = entry[1]
            local name      = entry[2]
            if md.RWC_ColdBattery >= threshold then
                stateName = name
                break
            end
        end
    end

	-- stop here if it the state is unchanged
	if md.RWC_LastShownState == stateName then return end
	md.RWC_LastShownState = stateName

    local suffixTable = RWC.Suffix and RWC.Suffix.Cooler or nil
    local suffix      = suffixTable and suffixTable[stateName] or nil
    local newName     = md.RWC_CoolerBaseName

    if suffix then
        newName = md.RWC_CoolerBaseName .. suffix
    end
    if item.getName and item.setName and item:getName() ~= newName then
        item:setName(newName)
		item:setCustomName(true)
		local c = item:getContainer() 
		if c and c.setDrawDirty then c:setDrawDirty(true) end
        if RWC.DEBUG then print(string.format("[RWC] Updated cooler name to '%s' (battery=%.1f, state=%s)",newName, md.RWC_ColdBattery, tostring(stateName))) end
    end
end

-- debug helper
function RWC_StorageToLabel(v)
    if v == nil then return "Unknown" end

    -- Prefer enum → label
    if type(v) == "number" then
        -- Build a safe mapping even if RWC.Storage is missing
        local S = (RWC and RWC.Storage) or {}
        local names = {
            [S.AMBIENT or 0] = "Ambient",
            [S.COOLER  or 1] = "Cooler",
            [S.FRIDGE  or 2] = "Fridge",
            [S.FREEZER or 3] = "Freezer",
        }
        return names[v] or ("Unknown(" .. tostring(v) .. ")")
    end

    -- Fallback: legacy strings → title-cased labels
    if type(v) == "string" then
        local s = v:lower()
        if s == "cooler"  then return "Cooler"  end
        if s == "fridge"  then return "Fridge"  end
        if s == "freezer" then return "Freezer" end
        if s == "ambient" then return "Ambient" end
        return v
    end

    return "Unknown"
end



