----------------------------------------------------------------------
-- Real Working Cooler - Tooltip Debug (Build 42)
-- Mirrors Realistic Shoes / Realistic Clothes ISToolTipInv:render hook.
-- Adds debug lines with heat values when RWC.DEBUG is true.
----------------------------------------------------------------------

if not RWC then RWC = {} end

local _RWC_old_ISToolTipInv_render = nil
local _RWC_TooltipHookInstalled    = false

----------------------------------------------------------------------
-- Helper: build debug lines for the hovered item.
-- Returns an array of strings (lines) or nil if nothing to show.
----------------------------------------------------------------------

local function RWC_buildTooltipDebugLines(item)
    if not item then return nil end
    if not RWC.TOOLTIP_DEBUG then return nil end

    local lines = {}

    ------------------------------------------------------------------
    -- Case 1: hovered item IS a Food → show its own heats
    ------------------------------------------------------------------
    if instanceof(item, "Food") then
        local heat    = item.getHeat     and item:getHeat()     or 0
        local invHeat = item.getInvHeat  and item:getInvHeat()  or 0
        local iHeat   = item.getItemHeat and item:getItemHeat() or 0

        table.insert(lines, string.format(
            "<RWC DEBUG> Food Heat=%.2f | InvHeat=%.2f | ItemHeat=%.2f",
            heat, invHeat, iHeat
        ))

        -- Spoilage-related
        local age    = item.getAge      and item:getAge()      or 0
        local off    = item.getOffAge   and item:getOffAge()   or 0
        local offMax = item.getOffAgeMax and item:getOffAgeMax() or 0

        table.insert(lines, string.format(
            "<RWC DEBUG> Age=%.2f | OffAge=%.2f / %.2f ",
            age, off,  offMax
        ))

    else
        -- Case 2: Cold Pack → battery, state, storage context
        local fullType = item.getFullType and item:getFullType() or nil
        if fullType and RWC.ColdPackTypes and RWC.ColdPackTypes[fullType] then
            local md = item:getModData()
            local batt = md and md.RWC_ColdBattery or 0
            -- Clamp defensively
            if batt < 0 then batt = 0 elseif batt > 100 then batt = 100 end

            -- Get state label (e.g., Frozen/Icy/Chilled/Cool/Warm)
            local state = RWC_GetColdPackStateName and RWC_GetColdPackStateName(batt) or "Unknown"

			-- Storage context (prefer cached md)
			local storedIn = md and md.RWC_StoredIn or nil
			local storedLabel = (RWC_StorageToLabel and RWC_StorageToLabel(storedIn)) or "Unknown"

			table.insert(lines, string.format(
				"<RWC DEBUG> Cold Pack • Battery=%.1f%% • State=%s • StoredIn=%s",
				batt, tostring(state), storedLabel
			))
        end
    end
	
	------------------------------------------------------------------
    -- If the hovered item lives inside a Cooler (inventory item),
    -- append a second line with that cooler's battery + state.
    ------------------------------------------------------------------
    do
        local cont  = item.getContainer and item:getContainer() or nil
        local owner = cont and cont.getContainingItem and cont:getContainingItem() or nil
        local oft   = owner and owner.getFullType and owner:getFullType() or nil
        local isCooler = owner and oft and RWC.CoolerTypes and RWC.CoolerTypes[oft] == true

        if isCooler then
            local cmd   = owner:getModData()
            local cbatt = (cmd and cmd.RWC_ColdBattery) or 0
            if cbatt < 0 then cbatt = 0 elseif cbatt > 100 then cbatt = 100 end

            -- Map cooler battery → state label
            local cstate
            if RWC_GetCoolerStateName then
                cstate = RWC_GetCoolerStateName(cbatt)
            else
                -- Fallback thresholds aligned with your Suffix.Cooler
                if     cbatt >= 75 then cstate = "VeryCold"
                elseif cbatt >= 50 then cstate = "Cold"
                elseif cbatt >= 25 then cstate = "Cool"
                else                     cstate = "Warm"
                end
            end

            table.insert(lines, string.format(
                "<RWC DEBUG> Cooler     • Battery=%.1f%% • State=%s",
                cbatt, tostring(cstate)
            ))
        end
    end
	
    if #lines == 0 then
        return nil
    end

    return lines
end

----------------------------------------------------------------------
-- Our wrapper for ISToolTipInv:render (RealisticShoes-style)
----------------------------------------------------------------------

local function RWC_ISToolTipInv_render(self)
    -- If we don't have RWC or debug off, just pass through.
    if not RWC.TOOLTIP_DEBUG then
        return _RWC_old_ISToolTipInv_render(self)
    end

    local item = self and self.item or nil
    if not item then
        return _RWC_old_ISToolTipInv_render(self)
    end

    local lines = RWC_buildTooltipDebugLines(item)
    if not lines or #lines == 0 then
        return _RWC_old_ISToolTipInv_render(self)
    end

    -- Number of extra lines, 16px each (same as RealisticShoes)
    local lineCount   = #lines
    local heightDelta = 16 * lineCount

    --if RWC.REALTIME_DEBUG then
    --    print(string.format(
    --        "[RWC] Tooltip: injecting %d debug line(s) for %s",
    --        lineCount, item:getFullType()
    --    ))
    --end

    local injectionStage = 1
    local originalHeight

    -- 1) Override setHeight to extend tooltip height once
    local oldSetHeight = self.setHeight
    self.setHeight = function(tool, height, ...)
        if injectionStage == 1 then
            injectionStage = 2
            originalHeight = height
            height = height + heightDelta
            --if RWC.REALTIME_DEBUG then print(string.format("[RWC] Tooltip: extend height %d → %d", originalHeight or -1, height))end
        end
        return oldSetHeight(tool, height, ...)
    end

	-- 2) Override drawRectBorder to draw our extra lines
	local oldDrawRectBorder = self.drawRectBorder
	self.drawRectBorder = function(tool, ...)
		if injectionStage == 2 then
			injectionStage = 3

			local tooltipObj = tool.tooltip or self.tooltip
			if tooltipObj then
				local font = UIFont[getCore():getOptionTooltipFont()]

				-- Use the *final* tooltip height (after extension)
				local baseHeight = tool:getHeight()

				-- Draw lines stacked at the bottom, going up
				local firstY = baseHeight - 4 - lineCount * 16
				local x = 7

				for idx, text in ipairs(lines) do
					local y = firstY + (idx - 1) * 16
					tooltipObj:DrawText(font, text, x, y, 1, 1, 1, 1)
				end
			end
		end
		return oldDrawRectBorder(tool, ...)
	end

    -- 3) Call the original render
    local result = _RWC_old_ISToolTipInv_render(self)

    -- 4) Restore methods (avoid global side effects)
    self.setHeight       = oldSetHeight
    self.drawRectBorder  = oldDrawRectBorder

    return result
end

----------------------------------------------------------------------
-- One-time installer
----------------------------------------------------------------------

local function RWC_installTooltipHook()
    if _RWC_TooltipHookInstalled then return end
    if not ISToolTipInv or not ISToolTipInv.render then
        if RWC.TOOLTIP_DEBUG then
            print("[RWC] Tooltip: ISToolTipInv not ready; will retry.")
        end
        return
    end

    _RWC_old_ISToolTipInv_render = ISToolTipInv.render
    ISToolTipInv.render          = RWC_ISToolTipInv_render
    _RWC_TooltipHookInstalled    = true

    if RWC.TOOLTIP_DEBUG then
        print("[RWC] Tooltip: Hook installed (ISToolTipInv:render).")
    end
end

Events.OnGameStart.Add(RWC_installTooltipHook)
Events.OnCreatePlayer.Add(RWC_installTooltipHook)

if RWC.TOOLTIP_DEBUG then
    print("[RWC] RWC_Tooltip debug hook loaded.")
end
