--=====================================================================
-- RWC_Hook.lua v3
--
-- PURPOSE
--   Central hook module for Real Working Cooler.
--
--   Responsibilities:
--     • Drive the per-minute update tick for RWC systems
--       (cold pack / cooler registries).
--     • Rebuild RWC registries on load by scanning:
--         - loaded world gridsquares (containers on world objects)
--         - player inventory (including nested bags)
--         - all containers on all vehicles currently loaded in the cell
--     • Capture inventory transfer actions as a cheap edge-case hook
--       for newly spawned or moved cold packs / coolers.
--
-- OVERVIEW
--   1) RWC_OnTick
--        - Hooked via Events.OnTick.
--        - Runs once per in-game minute (minutesStamp check).
--        - Calls RWC_TickColdPackRegistry() and logs when
--          RWC.REALTIME_DEBUG is enabled.
--
--   2) RWC_OnLoadGridsquare
--        - Hooked via Events.LoadGridsquare.
--        - For each loaded square, walks all IsoObjects and their
--          containers; registers any cold packs / coolers found into
--          the global registries.
--
--   3) RWC_OnGameStart
--        - Hooked via Events.OnGameStart.
--        - One-time scan at game start:
--            * player inventory (recursively via RWC_scanContainerRecursive)
--            * all containers on all currently loaded vehicles
--        - Seeds RWC.ColdPackRegistry and RWC.CoolerRegistry so the
--          minute tick has live references to work with.
--
--   4) ISInventoryTransferAction:perform
--        - Post-transfer hook (calls oldPerform first).
--        - When a cold pack moves:
--            * Updates md.RWC_StoredIn via RWC_GetStorageTypeForItem(item)
--            * Ensures the item is present in RWC.ColdPackRegistry
--        - When a cooler item moves:
--            * Ensures the item is present in RWC.CoolerRegistry
--        - Primarily used to catch edge cases (e.g. newly spawned
--          vehicles or containers discovered while the UI is closed),
--          without needing heavy world scans on every tick.
--
--=====================================================================

if not RWC then RWC = {} end

-----------------------------------------------------------------------
-- HOOK: Slow Tick
-- We check the global game time and only tick when minute changes
-- spent a lot of time trying to make a system to throttle updates for performance
-- there are too many issues with time scale because we can't turn off vanilla heat changes 
-- alternatively, we are just going to tick every cold pack and cooler every frame 
-----------------------------------------------------------------------
--local RWC_lastColdPackMinute = -1
--local RWC_CoolerTickDelay = 1      -- throttle the updates
--local RWC_CoolerTickCounter = 0 

local function RWC_OnTick()
    --local gt = getGameTime()
    --if not gt then return end

    --local nowMinutes = gt:getMinutesStamp()

    --------------------------------------------------------------------
    -- Cold packs: every X in-game minutes
    --------------------------------------------------------------------
    --local coldInterval = RWC.ColdPackTickMinutes or 1
    --if nowMinutes >= RWC_lastColdPackMinute + coldInterval then
        --RWC_lastColdPackMinute = nowMinutes

        --if RWC.REALTIME_DEBUG and RWC.DEBUG_CP then print(string.format("[RWC] ColdPack tick (every %d in-game minute(s))", coldInterval)) end
        RWC_TickColdPackRegistry()

    --end

    --------------------------------------------------------------------
    -- Coolers: high-frequency, but only a few per step
    --------------------------------------------------------------------
	--if RWC_CoolerTickCounter == 0 then 
		RWC_TickCoolerRegistry()
		--RWC_CoolerTickCounter = RWC_CoolerTickDelay
	--else 
		--RWC_CoolerTickCounter = RWC_CoolerTickCounter - 1
	--end
end
Events.OnTick.Add(RWC_OnTick)

-----------------------------------------------------------------------
-- repopulate the registry on load
-- does not check vehicles or player inventory
-----------------------------------------------------------------------
local function RWC_OnLoadGridsquare(square)
    if not square then return end

    -- Iterate world objects on this square
    local objects = square:getObjects()
    for i = 0, objects:size() - 1 do
        local obj = objects:get(i)

        -- Does this object have an inventory container?
        local containerCount = obj:getContainerCount() or 0
        for c = 0, containerCount - 1 do
            local container = obj:getContainerByIndex(c)
            if container then
                local items = container:getItems()
                for n = 0, items:size() - 1 do
                    local item = items:get(n)
					if item and item.getFullType then
						local fullType = item:getFullType()

						-- Cold packs
						if RWC.ColdPackTypes and RWC.ColdPackTypes[fullType] then
							RWC_RegisterColdPack(item)
						end

						-- Coolers
						if RWC.CoolerTypes and RWC.CoolerTypes[fullType] then
							RWC_RegisterCoolerItem(item)
						end
					end
                end
            end
        end
    end
end
Events.LoadGridsquare.Add(RWC_OnLoadGridsquare)

-----------------------------------------------------------------------
-- OnGameStart:
--   • Scan player inventory (including nested bags)
--   • Scan all containers on all loaded vehicles
--   • Register any cold packs / coolers found
-----------------------------------------------------------------------
local function RWC_OnGameStart()
    if RWC.DEBUG then print("[RWC] OnGameStart: scanning player inventory + loaded vehicles for RWC items") end

    -------------------------------------------------------------------
    -- 1) Player inventory (including nested containers)
    -------------------------------------------------------------------
    local player = getSpecificPlayer(0)
    if player then
        local inv = player:getInventory()
        if inv then
            RWC_scanContainerRecursive(inv)
        end
    end

    -------------------------------------------------------------------
    -- 2) All loaded vehicles in the current cell
    --    getCell():getVehicles() returns vehicles in all loaded chunks.
    -------------------------------------------------------------------
    local cell = getCell()
    if cell then
        local vehicles = cell:getVehicles()
        if vehicles then
            if RWC.DEBUG then print(string.format("[RWC] OnGameStart: scanning %d loaded vehicles",vehicles:size())) end

			--loop all vehicles
            for vi = 0, vehicles:size() - 1 do
                local vehicle = vehicles:get(vi)
                if vehicle then
					--loop all parts
                    local partCount = vehicle:getPartCount() or 0
                    for pi = 0, partCount - 1 do
						-- see if part is a container
                        local part = vehicle:getPartByIndex(pi)
                        if part then
                            local container = part:getItemContainer()
                            if container then
								--scan for coolers and cold packs
                                RWC_scanContainerRecursive(container)
                            end
                        end
                    end
                end
            end
        end
    end
end
Events.OnGameStart.Add(RWC_OnGameStart)

-----------------------------------------------------------------------
-- catch inventory item transfers
-- only needed for some edge cases like newly spawned vehicles that might have coolers or cold packs inside
-- Patch ISInventoryTransferAction.perform to handle single + batch transfers
-- 0.8 redesign the idea to just look at dest container instead of the item, hard to catch all batch items
-----------------------------------------------------------------------
local _RWC_oldPerform = ISInventoryTransferAction.perform
function ISInventoryTransferAction:perform()
    -- do the vanilla move first so destinations are final
    _RWC_oldPerform(self)

    -- post-scan the destination container so every moved item is caught
    local dest = self.destContainer
    if dest then
        RWC_scanContainerRecursive(dest)
        return
    end

    -- fallback: if no dest container (drop-to-ground), at least tag the single item
    if self.item and self.item.getFullType then
        local ft = self.item:getFullType()
        if RWC.ColdPackTypes and RWC.ColdPackTypes[ft] then
            local md = self.item:getModData()
            md.RWC_StoredIn = RWC_GetStorageTypeForItem(self.item,dest)
            RWC_RegisterColdPack(self.item)
            if RWC.DEBUG_CP then print(("[RWC] POST-SCAN: id=%s StoredIn=Ambient (no dest container)") :format(tostring(self.item:getID()))) end
        elseif RWC.CoolerTypes and RWC.CoolerTypes[ft] then
            RWC_RegisterCoolerItem(self.item)
        end
    end
end


-----------------------------------------------------------------------
-- EARLY RESET: clear runtime registries when a world is initialized
-- (fires before OnGameStart and before LoadGridsquare spam)
-----------------------------------------------------------------------
local function RWC_OnInitWorld()
    RWC.ColdPackRegistry = {}
    RWC.CoolerRegistry   = {}
    RWC.CoolerRegistryCount = 0
    --RWC.CoolerIterPos       = 1
	
    if RWC.DEBUG then print("[RWC] OnInitWorld: RWC registries reset for new/load world") end
end
Events.OnInitWorld.Add(RWC_OnInitWorld)

