--=====================================================================
-- RWC_Cooler.lua
-- Phase 1: ONLY cooler detection + registry (no thermal logic yet)
--
-- PURPOSE:
--   Maintain a lightweight registry of cooler items the player has
--   interacted with, similar to cold packs.
--
--   No spoilage or temperature calculations here yet.
--   Just detection, registration, and pointer cleanup.
--=====================================================================

if not RWC then RWC = {} end

-----------------------------------------------------------------------
-- TEST: For a given cooler item, set all FOOD contents to super cold
-----------------------------------------------------------------------
local function RWC_TestCoolerSetFoodsCold(coolerItem)
    if not coolerItem or not coolerItem.getInventory then return end

    local container = coolerItem:getInventory()
    if not container then return end

    local items = container:getItems()
    if not items then return end

    for i = 0, items:size() - 1 do
        local inner = items:get(i)

        if inner and instanceof(inner, "Food") then
            -- Drive the Food heat field (what getInvHeat uses)
            if inner.setHeat then
                inner:setHeat(0.0)  -- absurdly “cold” so it’s obvious
            end

            -- (Optional) also touch itemHeat for completeness
            if inner.setItemHeat then
                --inner:setItemHeat(0.0)
            end

            if RWC.DEBUG and inner.getHeat and inner.getInvHeat then print(string.format("[RWC] Cooler test: %s Heat=%.2f InvHeat=%.2f",tostring(inner:getFullType()),inner:getHeat(),inner:getInvHeat())) end
        end
    end
end

-----------------------------------------------------------------------
-- RWC_TickCooler(coolerItem)
--
-- Per-frame cooler simulation for a *single* cooler item.
--
-- Responsibilities:
--   • Maintain an internal cooler battery (md.RWC_ColdBattery, 0–100).
--   • Scan the cooler’s ItemContainer:
--       - Sum cold pack batteries and lightly bleed each pack.
--       - Compute total food mass and a mass-weighted average food heat.
--   • Compute an average “system battery” from:
--       - Cooler internal battery
--       - All cold packs inside (each counts as one “pack” with weight
--         in the average via CPCount / totalBatt).
--   • Convert average battery → target food Heat:
--       - 100 battery ≈ 0.2 Heat (very cold)
--       -   0 battery ≈ 1.0 Heat (ambient)
--   • Compute a battery drain rate based on:
--       - ΔHeat between avg food heat and targetHeat
--       - Total food mass
--       - RWC.Rates.ColdBatteryDrainPerHeat
--       - Slight refund when food is colder than target.
--   • Nudge cooler + cold pack batteries toward the average battery
--       (RWC.Rates.ColdBatteryNudgeRate) and then apply drain.
--   • Adjust each food item’s Heat toward targetHeat
--       (RWC.Rates.FoodNudgeRate), overpowering vanilla’s slow change.
--   • Clamp all batteries to [0, 100] and update cooler / cold pack
--       display names (suffixes) accordingly.
--
-- Notes:
--   • dt is derived from getMinutesStamp() via md.RWC_LastMinuteStamp.
--   • Assumes most contents are Food or ColdPacks for performance.
--   • Called from RWC_TickCoolerRegistry(), which handles batching and
--     iteration across all registered coolers.
-----------------------------------------------------------------------

function RWC_TickCooler(coolerItem)
    local md = coolerItem:getModData()

    -- Initialize battery on first use
	if md.RWC_ColdBattery == nil then
		-- determing a starting temp based on container type
			md.RWC_ColdBattery = 0
 		if RWC.DEBUG then print("[RWC] Init cooler battery:", md.RWC_ColdBattery) end
    end
    
	-- Initialize last minute stamp
	if md.RWC_LastMinuteStamp == nil then
        md.RWC_LastMinuteStamp = getGameTime():getWorldAgeHours() * 60.0  -- in-game minutes (float)
		if RWC.REALTIME_DEBUG then print(string.format("[RWC] Cooler init id=%s set LastMinute=%d",tostring(coolerItem:getID()), md.RWC_LastMinuteStamp)) end
        return -- done for this tick
    end

    -- Compute time delta since last tick
    local now = getGameTime():getWorldAgeHours() * 60.0  -- in-game minutes (float)
    local dt  = math.max(0, now - md.RWC_LastMinuteStamp)
    if dt <= 0 then -- no time passed
 		if RWC.DEBUG then print("[RWC] cooler tick: no time has passed")  end
		return 
	end 
    if RWC.REALTIME_DEBUG then print(string.format("[RWC] Cooler tick id=%s dt(min)=%.2f batt=%.1f",tostring(coolerItem:getID()), dt, md.RWC_ColdBattery or -1)) end

	-- get the container
    local container = coolerItem:getInventory()
    if not container then
        if RWC.REALTIME_DEBUG then print("[RWC] Cooler has no container (skip)") end
        return
    end
	
	-- get items list
    local items = container:getItems()
	
	-- special case if cooler is empty
    if not items or items:isEmpty() then
		-- bleed battery slowly
		local before = md.RWC_ColdBattery
        md.RWC_ColdBattery = md.RWC_ColdBattery - RWC.Rates.ColdBatteryBleedRate * dt
        md.RWC_ColdBattery = RWC_Clamp(md.RWC_ColdBattery, 0, 100)
        if RWC.REALTIME_DEBUG then print(string.format("[RWC] Empty cooler bleed: %.2f → %.2f (dt=%.2f, bleed=%.4f)",before, md.RWC_ColdBattery, dt, RWC.Rates.ColdBatteryBleedRate)) end
        RWC_UpdateCoolerName(coolerItem)
        md.RWC_LastMinuteStamp = now
        return
	end

	-- local vars
	local totalBatt = md.RWC_ColdBattery           -- accumulator for total battery avaliable
	local CPCount = 1                              -- how many cold packs are inside, count internal cooler bat as pack
	local CPMass = coolerItem:getWeight()          -- accumulator for battery mass
	local totalFoodMass = 0          -- tracking how much cooling needs to be done this update
	local avgFoodHeat = 0            -- tracking average current heat of food

	-- scan the contents of the container to see what we are working with 
    for i = 0, items:size() - 1 do
        local it = items:get(i)
        if it and it.getFullType then
            local ft = it:getFullType()
	
			-- if its a cold pack, accumulate a total battery and pack counter
            if RWC.ColdPackTypes[ft] then
				local cpmd = it:getModData()
				cpmd.RWC_StoredIn = RWC.Storage.COOLER -- override just to be safe 
				CPCount = CPCount + 1
				CPMass = CPMass + it:getWeight()
				totalBatt = totalBatt + cpmd.RWC_ColdBattery
                if RWC.REALTIME_DEBUG then print(string.format("[RWC] cold pack in cooler ID=%d %s batt=%.1f", it:getID(), ft, cpmd.RWC_ColdBattery)) end
				-- bleed battery slowly
				local before = cpmd.RWC_ColdBattery
				cpmd.RWC_ColdBattery = cpmd.RWC_ColdBattery - RWC.Rates.ColdBatteryBleedRate * dt
                if RWC.REALTIME_DEBUG then print(string.format("[RWC]   bleed pack: %.2f -> %.2f", before, cpmd.RWC_ColdBattery)) end				
			
			-- if its food, then accumulate a total value of the mass of all food and an average heat value
			elseif instanceof(it, "Food") then
				local weight = it.getWeight and it:getWeight() or 0.0
				local heat = it.getHeat and it:getHeat() or 0.0
				if totalFoodMass == 0 then 
					avgFoodHeat = heat
				else 
					-- mass-weighted incremental average
					avgFoodHeat = (avgFoodHeat * totalFoodMass + heat * weight) / (totalFoodMass + weight)
				end
				totalFoodMass = totalFoodMass + weight
				if RWC.REALTIME_DEBUG then print(string.format("[RWC]  food %s w=%.2f heat=%.2f (avg=%.2f mass=%.2f)",ft, weight, heat, avgFoodHeat, totalFoodMass)) end
			end
		end
	end

    -- average battery of "all packs", including the cooler's own battery
    local avgBatt = totalBatt / CPCount
	avgBatt = RWC_Clamp(avgBatt,0,100)

	-- set target heat for food based on average battery 100 bat = 0.2 heat  /  0 bat = 1.0 heat
    local targetHeat = 1.0 - 0.8 * (avgBatt / 100.0) 
	targetHeat = RWC_Clamp(targetHeat,0.2,1.0)

	-- calculate battery drain rate - compare target heat and avg food heat, the greater the delta the higher battery drain
	-- if avg heat is colder (lower) than target, refund a little battery? 
	--                                scale to mass of food vs mass of batt                   tuning knob                               flip sign if food is colder than battery
	local battDrainRate = ((totalFoodMass / (CPMass*RWC.Rates.ColdPackBattMassMult)) * RWC.Rates.ColdBatteryDrainPerHeat * (avgFoodHeat >= targetHeat and 1 or -1)) 
    if RWC.REALTIME_DEBUG then print(string.format("[RWC] summary: CPs=%d totalBatt=%.2f avgBatt=%.2f foodMass=%.2f avgFoodHeat=%.2f targetHeat=%.2f battDrainRate=%.2f",CPCount, totalBatt, avgBatt, totalFoodMass, avgFoodHeat, targetHeat, battDrainRate)) end

	local beforeCooler = md.RWC_ColdBattery
	-- slow bleed cooler 
    md.RWC_ColdBattery = md.RWC_ColdBattery - RWC.Rates.ColdBatteryBleedRate * dt
	-- nudge the cooler battery to the average by the nudge rate, assume no losses 
	if CPCount > 1 then 
		md.RWC_ColdBattery = md.RWC_ColdBattery + (md.RWC_ColdBattery > avgBatt and -RWC.Rates.ColdBatteryNudgeRate*dt*CPCount or RWC.Rates.ColdBatteryNudgeRate*dt*CPCount)
	end 
	-- drain cooler battery based on total food mass and avg food heat
	md.RWC_ColdBattery = md.RWC_ColdBattery - battDrainRate*dt*coolerItem:getWeight() 
	md.RWC_ColdBattery = RWC_Clamp(md.RWC_ColdBattery,0,100)
	RWC_UpdateCoolerName(coolerItem)
    if RWC.REALTIME_DEBUG then print(string.format("[RWC] cooler batt tick: %.2f ->	%.2f (nudge=±%.4f dt=%.2f drain=%.4f cpcount=%d)", beforeCooler, md.RWC_ColdBattery, RWC.Rates.ColdBatteryNudgeRate, dt, battDrainRate, CPCount)) end

	-- scan the contents again to make changes
	-- we could look at other options, but assuming most items in the coolers will be either food or cold packs
	-- this should be efficient enough
    for i = 0, items:size() - 1 do
        local it = items:get(i)
        if it and it.getFullType then
            local ft = it:getFullType()

			-- cold pack 
            if RWC.ColdPackTypes[ft] then
				local cpmd = it:getModData()
				local before = cpmd.RWC_ColdBattery
				-- nudge the cold pack battery to average all batteries together
				cpmd.RWC_ColdBattery = cpmd.RWC_ColdBattery + (cpmd.RWC_ColdBattery > avgBatt and -RWC.Rates.ColdBatteryNudgeRate*dt or RWC.Rates.ColdBatteryNudgeRate*dt)
				-- drain batteries based on total food mass and avg food heat
				cpmd.RWC_ColdBattery = cpmd.RWC_ColdBattery - battDrainRate*dt*it:getWeight()
				cpmd.RWC_ColdBattery = RWC_Clamp(cpmd.RWC_ColdBattery,0,100)
				RWC_UpdateColdPackName(it)
                if RWC.REALTIME_DEBUG then print(string.format("[RWC]    cold pack %s batt: %.2f -> %.2f", ft, before, cpmd.RWC_ColdBattery)) end		

			-- food - nudge heat values toward target heat 
			elseif instanceof(it, "Food") then
				if it.getHeat then 
					local heat = it:getHeat() 
					local diff = math.abs(heat - targetHeat) -- scale nudge rate based on diff
					local scale = (diff > 0.1) and RWC_Clamp(1+(((diff - 0.1) / 0.7) * 2.0), 1.0, 2.0) or 1.0
					it:setHeat(heat + (heat > targetHeat and -RWC.Rates.FoodNudgeRate*dt*scale or RWC.Rates.FoodNudgeRate*dt*scale))
                    if RWC.REALTIME_DEBUG then print(string.format("[RWC]    food %s heat: %.2f -> %.2f (target=%.2f)",ft, heat, it:getHeat(), targetHeat)) end
				end
			end
		end
	end
	md.RWC_LastMinuteStamp = now
end

-----------------------------------------------------------------------
-- TICK ONE COOLER (lean hot path)
--  • Per-tick battery & heat management
--  • Precompute dt-scaled rates
--  • Only rename when visible state actually changes
-----------------------------------------------------------------------
function RWC_TickCoolerFaster(coolerItem)
    -- Locals to cut global lookups
    local clamp      = RWC_Clamp
    local Rates      = RWC.Rates
	local coldTypes  = RWC.ColdPackTypes
	local RWC_COOLER = RWC.Storage.COOLER
    local md         = coolerItem:getModData()

    -- Initialize battery once
    local batt = md.RWC_ColdBattery
    if batt == nil then
        batt = 0
        md.RWC_ColdBattery = batt
        if RWC.DEBUG then print("[RWC] Init cooler battery: 0") end
    end

	-- Initialize last minute stamp
	if md.RWC_LastWorldMinF == nil then
        md.RWC_LastWorldMinF = getGameTime():getWorldAgeHours() * 60.0  -- in-game minutes (float)
        return -- done for this tick
    end

    -- Time delta as float
	local now = getGameTime():getWorldAgeHours() * 60.0  -- in-game minutes (float)
	local dt = now - md.RWC_LastWorldMinF
	if dt <= 0 then return end

    -- Precompute dt-scaled rates once
    local battBleed_dt   = Rates.ColdBatteryBleedRate * dt
    local foodNudge_dt = Rates.FoodNudgeRate * dt

    -- bail if container not valid
	local container = coolerItem:getInventory()
	if not container then 
		md.RWC_LastWorldMinF = now
		return 
	end 
    
	-- bail if container is empty
	local items = container:getItems()
    if not items or items:isEmpty() then 
        -- idle bleed
        md.RWC_ColdBattery = clamp(batt - battBleed_dt, 0, 100)
        RWC_UpdateCoolerName(coolerItem) -- (cheap when state unchanged)
        md.RWC_LastWorldMinF = now
        return
    end

    --------------------------------------------------------------------
    -- First pass: aggregate totals (battery & food mass/heat)
    --------------------------------------------------------------------
    local totalBatt           = batt
	local totalBattMass       = coolerItem:getWeight()
    local packCount           = 1    -- include cooler’s internal "pack"
    local totalFoodMass       = 0.0
	local totalFoodHeatByMass = 0.0

	-- scan the contents of the container to see what we are working with 
    for i = 0, items:size() - 1 do
        local it = items:get(i)
        if it and it.getFullType then
            local ft = it:getFullType()
			
			-- if its a cold pack, accumulate a total battery and pack counter
            if coldTypes[ft] then
                local cpmd = it:getModData()
                local cpb  = cpmd.RWC_ColdBattery or 0
                packCount  = packCount + 1
                totalBatt  = totalBatt + cpb
				totalBattMass = totalBattMass + it:getWeight()
                -- small bleed on packs
                cpb = cpb - battBleed_dt
                if cpb < 0 then cpb = 0 end
                cpmd.RWC_ColdBattery = cpb
				-- safety check for cold pack flag
				local s = cpmd.RWC_StoredIn
				if s ~= RWC_COOLER then cpmd.RWC_StoredIn = RWC_COOLER end
			
			-- if its food, then accumulate a total value of the mass of all food and an average heat value
            elseif instanceof(it, "Food") then
                local w = (it.getWeight and it:getWeight()) or 0.0
                if w > 0 then
                    local h = (it.getHeat and it:getHeat()) or 1.0
					totalFoodMass = totalFoodMass + w
					totalFoodHeatByMass = totalFoodHeatByMass + h * w
                end
            end
        end
    end

    --------------------------------------------------------------------
    -- Derived targets (battery → heat)
    --------------------------------------------------------------------
	-- avg food heat (mass-weighted), clamped
	local avgFoodHeat = 1.0 
	if totalFoodMass > 0 then avgFoodHeat = clamp(totalFoodHeatByMass / totalFoodMass, 0.2, 1.0) end

	-- average battery across all cold packs 
    local avgBatt = clamp(totalBatt / packCount, 0, 100)

	-- target heat relative to battery avaliable
    -- 100% batt → 0.2 heat; 0% batt → 1.0 heat (clamped)
    local targetHeat = clamp(1.0 - (0.8 * (avgBatt * 0.01)), 0.2, 1.0)

    -- Nudge cooler battery toward group average, relative to distance from average
	local battNudge_dt = Rates.ColdBatteryNudgeRate * (avgBatt - batt) * dt
	
    -- Battery drain ~ (how much we need to cool) 
	-- compare avg food heat vs room temp (1.0)
	--    the colder the food the faster the drain
	-- compare total food mass vs total battery mass * ColdPackBattMassMult 
	--    the higher the food mass, the faster the batt drain
	-- compare avg food heat vs target heat 
	--    if food is colder than target, then we charge batt instead of drain
	local heatrate = 0.8 + avgFoodHeat -- range from 1 - 1.8
	local massrate = totalFoodMass / (totalBattMass * Rates.ColdPackBattMassMult)
	local heatcomp = avgFoodHeat > targetHeat and Rates.ColdBatteryDrainPerHeat or -Rates.ColdBatteryDrainPerHeat
	local battDrain_dt = heatrate * massrate * heatcomp * dt
	
	-- clamp and slow bleed cooler batt and update md
    md.RWC_ColdBattery = clamp(batt + battNudge_dt - battDrain_dt - battBleed_dt, 0, 100)

    -- Update cooler name only when display state actually changes
    RWC_UpdateCoolerName(coolerItem)

    --------------------------------------------------------------------
    -- Second pass: apply computed values (rename packs, set food heat)
    --------------------------------------------------------------------
    for i = 0, items:size() - 1 do
        local it = items:get(i)
        if it and it.getFullType then
            local ft = it:getFullType()

			-- cold pack update
            if RWC.ColdPackTypes[ft] then
                local cpmd  = it:getModData()
                local cpb   = cpmd.RWC_ColdBattery or 0
                -- nudge pack battery toward group avg
				battNudge_dt = Rates.ColdBatteryNudgeRate * (avgBatt - cpb) * dt
                cpb = clamp(cpb + battNudge_dt - battDrain_dt, 0, 100)
                cpmd.RWC_ColdBattery = cpb

                -- rename only if state changed
                RWC_UpdateColdPackName(it)

			-- food - nudge heat values toward target heat 
            elseif instanceof(it, "Food") and it.getHeat then
                local heat  = it:getHeat()
				local heatNudge_dt = (targetHeat - heat) * foodNudge_dt
				it:setHeat(clamp( heat + heatNudge_dt, 0.2, 1.0))
				
            end
        end
    end

    md.RWC_LastWorldMinF = now

    if RWC.REALTIME_DEBUG then
        print("[RWC] Cooler tick id="..tostring(coolerItem:getID())
          .." batt="..tostring(md.RWC_ColdBattery)
          .." packs="..tostring(packCount)
          .." foodMass="..tostring(totalFoodMass)
          .." tgtHeat="..string.format("%.2f", targetHeat))
    end
end
-----------------------------------------------------------------------
-- MAIN: Tick a subset of coolers each frame.
--  • Uses dense array RWC.CoolerRegistry[1..count]
--  • Processes up to RWC.CoolerTickBatchSize per call
--  • Rotates using RWC.CoolerIterPos
-----------------------------------------------------------------------
--local RWC_CoolerWaitTicks = 0  -- keep your cool-down between full cycles  RWC.CoolerCycleWaitTicks
function RWC_TickCoolerRegistry()
--[[
	-- bail if registry is empty
    if RWC.CoolerRegistryCount == 0 then return end

    -- Respect "cool-down" between full cycles (optional)
    if RWC_CoolerWaitTicks > 0 then
        RWC_CoolerWaitTicks = RWC_CoolerWaitTicks - 1
        if RWC.REALTIME_DEBUG then print(string.format("[RWC] Cooler tick: waiting (%d steps left)", RWC_CoolerWaitTicks)) end
        return
    end
	
    -- clamp batch size so we never tick more times than there are coolers
    local batchSize = RWC.CoolerTickBatchSize or 1
    if batchSize < 1 then batchSize = 1 end
    if batchSize > RWC.CoolerRegistryCount then
        batchSize = RWC.CoolerRegistryCount
    end
	
    if RWC.REALTIME_DEBUG then print(string.format("[RWC] CoolerRegistry tick: count=%d, batch=%d, iterPos=%d",RWC.CoolerRegistryCount, batchSize, RWC.CoolerIterPos)) end	

	-- counter for batch size
    local processed = 0
    while processed < batchSize and RWC.CoolerRegistryCount > 0 do
        if RWC.CoolerIterPos > RWC.CoolerRegistryCount then
            if RWC.REALTIME_DEBUG then print(string.format("[RWC] CoolerRegistry: wrap iterPos %d → 1 (count=%d)",RWC.CoolerIterPos, RWC.CoolerRegistryCount)) end
            RWC.CoolerIterPos = 1 -- wrap around
        end

        local item = RWC.CoolerRegistry[RWC.CoolerIterPos]

        -- Validate entry (stale pointer, or no longer a cooler type)
        if not item or not item.getFullType or not RWC.CoolerTypes[item:getFullType()] then
            if RWC.REALTIME_DEBUG then print(string.format("[RWC] CoolerRegistry: removing stale/invalid at index=%d",RWC.CoolerIterPos)) end
            RWC_RemoveCoolerAtIndex(RWC.CoolerIterPos)
        else
            if RWC.REALTIME_DEBUG then print(string.format("[RWC] CoolerRegistry: tick index=%d id=%s type=%s",RWC.CoolerIterPos, tostring(item:getID()),item:getFullType())) end
            RWC_TickCooler(item)
            processed = processed + 1
            RWC.CoolerIterPos = RWC.CoolerIterPos + 1
			
			-- if at end of registry then build in a pause
			if RWC.CoolerIterPos > RWC.CoolerRegistryCount then
				RWC_CoolerWaitTicks = 0 --RWC.CoolerCycleWaitTicks - RWC.CoolerRegistryCount
				if RWC_CoolerWaitTicks < 0 then RWC_CoolerWaitTicks = 0 end
				if RWC.REALTIME_DEBUG then print(string.format("[RWC] CoolerRegistry: full cycle done, size=%d, pausing for %d cooler ticks",RWC.CoolerRegistryCount, RWC_CoolerWaitTicks))	end			
			end
        end
    end
    if RWC.REALTIME_DEBUG then print(string.format("[RWC] CoolerRegistry tick done: processed=%d, nextIterPos=%d, count=%d",processed, RWC.CoolerIterPos, RWC.CoolerRegistryCount)) end
    --for i = RWC.CoolerRegistryCount, 1, -1 do
]]
	-- leaner more performant friendly version for every tick update
 	-- bail if registry is empty
    local count = RWC.CoolerRegistryCount
    if count == 0 then return end

    local reg  = RWC.CoolerRegistry
    local types = RWC.CoolerTypes

    -- i only increments when we keep the entry; when we remove,
    -- we do not increment so we can re-check the swapped-in element
    local i = 1
    while i <= count do
        local it = reg[i]
        if (not it) or (not it.getFullType) or (not types[it:getFullType()]) then
            -- remove by swapping last into i
            local last = reg[count]
            reg[i]   = last
            reg[count] = nil
            count = count - 1
            RWC.CoolerRegistryCount = count
            -- no i++ here; we must re-check the swapped-in element
            if RWC.REALTIME_DEBUG then print("[RWC] CoolerRegistry: removed invalid at index "..tostring(i)) end
        else
            -- valid entry → tick
            RWC_TickCoolerFaster(it)
            i = i + 1
        end
    end
end


