--=====================================================================
-- RWC_TickColdPack.lua
-- Phase 1: ONLY cold-pack temperature simulation (no cooler logic)
--
-- PURPOSE:
--   Maintain internal temperature for cold packs.
--   They warm toward ambient if not stored cold.
--   They cool toward fridge/freezer temps when placed inside one.
--
--   No registry logic, no container scans, no world loops here.
--   This function simply updates ONE item when asked to.
--
-- CALL SIGNATURE:
--     RWC_TickColdPack(item, square, storageType)
--
-- PARAMETERS:
--   item         = InventoryItem (cold pack)
--   square       = IsoGridSquare reference (may be nil)
--   storageType  = "freezer" | "fridge" | "ambient"
--                  Caller decides meaning. Cooler logic comes later.
--
--=====================================================================

if not RWC then RWC = {} end

-----------------------------------------------------------------------
-- MAIN FUNCTION: Update a single cold pack's internal temperature
-----------------------------------------------------------------------
function RWC_TickColdPack(item)
    local md = item:getModData()
	local Storage = RWC.Storage

    -- Initialize battery on first use
	if md.RWC_ColdBattery == nil then
		-- determing a starting temp based on container type
			md.RWC_ColdBattery = 0
 		if RWC.DEBUG_CP then print("[RWC] Init cold pack battery:", md.RWC_ColdBattery) end
    end
    
	-- Initialize last minute stamp
	if md.RWC_LastWorldMinF == nil then
        md.RWC_LastWorldMinF = getGameTime():getWorldAgeHours() * 60.0  -- in-game minutes (float)
        return -- done for this tick
    end

	-- safeguard
	if md.RWC_StoredIn == nil then md.RWC_StoredIn = Storage.AMBIENT end
	
	-- we handle draining battery in cooler in cooler logic
	if md.RWC_StoredIn == Storage.COOLER then return end 

    -- Compute time delta since last tick	
    local now = getGameTime():getWorldAgeHours() * 60.0  -- in-game minutes (float)
    local dt  = now - md.RWC_LastWorldMinF
	if dt <= 0 then return end

    -- Get the reference temperature we should move toward
    local targetBattery, changeRate
    if md.RWC_StoredIn == Storage.FREEZER then
        targetBattery     = RWC.Rates.FreezerTargetBattery
        changeRate        = RWC.Rates.FreezerApproachPerMin
    elseif md.RWC_StoredIn == Storage.FRIDGE then
        targetBattery     = RWC.Rates.FridgeTargetBattery
        changeRate        = RWC.Rates.FridgeApproachPerMin
    else -- default: ambient warming/cooling
        targetBattery     = RWC.Rates.AmbientTargetBattery
        changeRate        = RWC.Rates.AmbientApproachPerMin
    end

    -- Compute change
    local oldBat = md.RWC_ColdBattery
	local newBat = oldBat > targetBattery and oldBat - (changeRate * dt) or oldBat + (changeRate * dt)
	md.RWC_ColdBattery = RWC_Clamp(newBat,0,100)

    md.RWC_LastWorldMinF = now
	RWC_UpdateColdPackName(item)

    -- Debug output
	if RWC.DEBUG_CP then print(string.format("[RWC] ColdPackTick: id=%s type=%s dt=%d old=%d -> new=%d target=%d rate=%.2f (%s)",tostring(item:getID()),item:getFullType(),dt, oldBat, md.RWC_ColdBattery, targetBattery, changeRate, md.RWC_StoredIn)) end

end

--=====================================================================
-- RWC_ColdPackRegistry
-- Purpose:
--   Track cold packs that the player has interacted with.
--   We store a direct pointer to the item when possible, and remove it
--   if the pointer becomes invalid (chunk unload, dropped far away, etc.)
--
--   No world searching. No restoration of unloaded pointers.
--   If player picks it up again, we detect it and re-register.
--
--   Tick: Iterate registry & update only valid items
--   Called once per minute externally from main tick
-----------------------------------------------------------------------
function RWC_TickColdPackRegistry()
    local count = RWC.ColdPackRegistryCount
    if count == 0 then return end

    local reg  = RWC.ColdPackRegistry
    local types = RWC.ColdPackTypes

    -- i only increments when we keep the entry; when we remove,
    -- we do not increment so we can re-check the swapped-in element
    local i = 1
    while i <= count do
        local it = reg[i]
        if (not it) or (not it.getFullType) or (not types[it:getFullType()]) then
            -- remove by swapping last into i
            local last = reg[count]
            reg[i]   = last
            reg[count] = nil
            count = count - 1
            RWC.ColdPackRegistryCount = count
            -- no i++ here; we must re-check the swapped-in element
            if RWC.REALTIME_DEBUG then print("[RWC] Cold Pack Reg: removed invalid at index "..tostring(i)) end
        else
            -- valid entry → tick
            RWC_TickColdPack(it)
            i = i + 1
        end
    end	
end



