RCIndustrialHeaterSpawner = RCIndustrialHeaterSpawner or {}
local Spawner = RCIndustrialHeaterSpawner

Spawner.spawnedSquares = Spawner.spawnedSquares or {}
Spawner.spawnReserves = Spawner.spawnReserves or {}
Spawner.processedSquares = Spawner.processedSquares or {}
Spawner.buildingsWithHeaters = Spawner.buildingsWithHeaters or {}
Spawner.pendingBuildings = Spawner.pendingBuildings or {}
Spawner.eventsRegistered = Spawner.eventsRegistered or false

Spawner.tickCounter = 0
Spawner.checkInterval = 45
Spawner.minBuildingArea = 400
Spawner.defaultSpawnRate = 5
Spawner.roomKeywords = { "warehouse", "factory", "industrial", "hangar", "distribution" }
Spawner.buildingKeywords = { "warehouse", "factory", "industrial", "distribution", "logistic" }

local function lower(value)
    if not value then return nil end
    return string.lower(value)
end

local function ensureModData()
    Spawner.spawnedSquares = ModData.getOrCreate("RC_IndustrialHeaterSpawnedSquares") or {}
    Spawner.buildingsWithHeaters = ModData.getOrCreate("RC_IndustrialHeaterBuildings") or {}
end

local function getSquareID(square)
    if not square then return "invalid" end
    return string.format("%d,%d,%d", square:getX(), square:getY(), square:getZ())
end

local function getBuildingID(square)
    if not square or not square.getBuilding then return nil end
    local building = square:getBuilding()
    if not building or not building.getDef then return nil end
    local def = building:getDef()
    if not def then return nil end
    local x, y, w, h = def:getX(), def:getY(), def:getW(), def:getH()
    if not x or not y or not w or not h then return nil end
    return string.format("%d,%d,%d,%d", x, y, w, h)
end

local function getBuildingArea(square)
    if not square or not square.getBuilding then return 0 end
    local building = square:getBuilding()
    if not building or not building.getDef then return 0 end
    local def = building:getDef()
    if not def then return 0 end
    return (def:getW() or 0) * (def:getH() or 0)
end

local function buildingNameMatches(square)
    if not square or not square.getBuilding then return false end
    local building = square:getBuilding()
    if not building or not building.getDef then return false end
    local def = building:getDef()
    if not def or not def.getName then return false end
    local name = lower(def:getName())
    if not name or name == "" then return false end
    for _, keyword in ipairs(Spawner.buildingKeywords) do
        if string.find(name, keyword, 1, true) then
            return true
        end
    end
    return false
end

local function roomMatches(room)
    if not room or not room.getName then return false end
    local name = lower(room:getName())
    if not name then return false end
    for _, keyword in ipairs(Spawner.roomKeywords) do
        if string.find(name, keyword, 1, true) then
            return true
        end
    end
    return false
end

local function isValidSpawnLocation(square)
    if not square then return false end
    if not square:isFree(false) then return false end
    if square:isOutside() then return false end

    local x, y, z = square:getX(), square:getY(), square:getZ()
    local aboveSquare = getCell():getGridSquare(x, y, z + 1)
    if aboveSquare then
        if aboveSquare:HasStairs() then return false end
        local objects = aboveSquare:getObjects()
        for i = 0, objects:size() - 1 do
            local obj = objects:get(i)
            if obj and obj:getSprite() and obj:getSprite():getProperties() then
                local props = obj:getSprite():getProperties()
                if props:Is("FloorOverlay") or props:Is("WallOverlay") then
                    return false
                end
            end
        end
    end

    local function hasBlockingProps(squareToCheck)
        local objects = squareToCheck:getObjects()
        for i = 0, objects:size() - 1 do
            local obj = objects:get(i)
            if obj and obj:getSprite() and obj:getSprite():getProperties() then
                local props = obj:getSprite():getProperties()
                if props:Is("SolidTrans") or props:Is("Solid") or props:Is("WallN") or props:Is("WallW") then
                    return true
                end
            end
        end
        return false
    end

    if hasBlockingProps(square) then
        return false
    end

    local adjacentOpenCount = 0
    local neighbours = {
        getCell():getGridSquare(x + 1, y, z),
        getCell():getGridSquare(x - 1, y, z),
        getCell():getGridSquare(x, y + 1, z),
        getCell():getGridSquare(x, y - 1, z)
    }

    for _, adj in ipairs(neighbours) do
        if adj and adj:isFreeOrMidair(false) and not adj:isSolid() and not adj:isSolidTrans() then
            if not hasBlockingProps(adj) then
                adjacentOpenCount = adjacentOpenCount + 1
            end
        end
    end

    return adjacentOpenCount >= 2
end

local function getSpawnRate()
    if Spawner.spawnRate then
        return Spawner.spawnRate
    end
    local rate = Spawner.defaultSpawnRate
    if SandboxVars and SandboxVars.RealisticCold and SandboxVars.RealisticCold.IndustrialHeaterSpawnRate then
        rate = SandboxVars.RealisticCold.IndustrialHeaterSpawnRate
    end
    if type(rate) ~= "number" then
        rate = Spawner.defaultSpawnRate
    end
    Spawner.spawnRate = math.max(0, math.floor(rate + 0.5))
    return Spawner.spawnRate
end

function Spawner.refreshSandboxOptions()
    Spawner.spawnRate = nil
end

local function canSpawnInSquare(square)
    if getSpawnRate() <= 0 then
        return false
    end

    local room = square:getRoom()
    if not room or not roomMatches(room) then
        return false
    end

    if getBuildingArea(square) < Spawner.minBuildingArea and not buildingNameMatches(square) then
        return false
    end

    if not isValidSpawnLocation(square) then
        return false
    end

    local buildingID = getBuildingID(square)
    if buildingID and (Spawner.buildingsWithHeaters[buildingID] or Spawner.pendingBuildings[buildingID]) then
        return false
    end

    return true
end

function Spawner.onLoadGridsquare(square)
    ensureModData()
    Spawner.processedSquares = Spawner.processedSquares or {}

    if not square then return end

    local squareID = getSquareID(square)
    if Spawner.processedSquares[squareID] then return end
    Spawner.processedSquares[squareID] = true

    if not canSpawnInSquare(square) then return end

    local persistentSquareID = string.format("%s,%s", squareID, lower(square:getRoom():getName() or ""))
    if Spawner.spawnedSquares[persistentSquareID] then
        return
    end

    local spawnRate = getSpawnRate()
    if spawnRate <= 0 then return end

    local roll = ZombRand(100)
    if roll >= spawnRate then
        return
    end

    local buildingID = getBuildingID(square)
    table.insert(Spawner.spawnReserves, {
        square = square,
        squareID = persistentSquareID,
        buildingID = buildingID,
        processed = false,
        offsetX = ZombRandFloat(-0.15, 0.15),
        offsetY = ZombRandFloat(-0.15, 0.15)
    })

    if buildingID then
        Spawner.pendingBuildings[buildingID] = true
    end
end

function Spawner.processSpawnReserves()
    for i = #Spawner.spawnReserves, 1, -1 do
        local reserve = Spawner.spawnReserves[i]
        if reserve.square and not reserve.processed and isValidSpawnLocation(reserve.square) then
            reserve.processed = true
            local item = reserve.square:AddWorldInventoryItem(
                "RC_TempSimMod.Mov_RCIndustrialHeater",
                0.5 + (reserve.offsetX or 0),
                0.5 + (reserve.offsetY or 0),
                0
            )

            if item then
                Spawner.spawnedSquares[reserve.squareID] = true
                if reserve.buildingID then
                    Spawner.pendingBuildings[reserve.buildingID] = nil
                    Spawner.buildingsWithHeaters[reserve.buildingID] = true
                    ModData.add("RC_IndustrialHeaterBuildings", Spawner.buildingsWithHeaters)
                    ModData.transmit("RC_IndustrialHeaterBuildings")
                end
                ModData.add("RC_IndustrialHeaterSpawnedSquares", Spawner.spawnedSquares)
                ModData.transmit("RC_IndustrialHeaterSpawnedSquares")
                table.remove(Spawner.spawnReserves, i)
            end
        else
            if reserve and reserve.buildingID then
                Spawner.pendingBuildings[reserve.buildingID] = nil
            end
            table.remove(Spawner.spawnReserves, i)
        end
    end
end

function Spawner.onTick()
    Spawner.tickCounter = Spawner.tickCounter + 1
    if Spawner.tickCounter >= Spawner.checkInterval then
        Spawner.tickCounter = 0
        if #Spawner.spawnReserves > 0 then
            Spawner.processSpawnReserves()
        end
    end
end

function Spawner.OnGameStart()
    ensureModData()
    Spawner.spawnReserves = {}
    Spawner.processedSquares = {}
    Spawner.pendingBuildings = {}
    Spawner.tickCounter = 0
    Spawner.refreshSandboxOptions()

    if getSpawnRate() <= 0 then
        if Spawner.eventsRegistered then
            Events.LoadGridsquare.Remove(Spawner.onLoadGridsquare)
            Events.OnTick.Remove(Spawner.onTick)
            Spawner.eventsRegistered = false
        end
        return
    end

    if not Spawner.eventsRegistered then
        Events.LoadGridsquare.Add(Spawner.onLoadGridsquare)
        Events.OnTick.Add(Spawner.onTick)
        Spawner.eventsRegistered = true
    end
end

function Spawner.OnSaveGame()
    if Spawner.spawnedSquares then
        ModData.add("RC_IndustrialHeaterSpawnedSquares", Spawner.spawnedSquares)
        ModData.transmit("RC_IndustrialHeaterSpawnedSquares")
    end

    if Spawner.buildingsWithHeaters then
        ModData.add("RC_IndustrialHeaterBuildings", Spawner.buildingsWithHeaters)
        ModData.transmit("RC_IndustrialHeaterBuildings")
    end
end

Events.OnGameStart.Add(Spawner.OnGameStart)
Events.OnServerStarted.Add(Spawner.OnSaveGame)
Events.OnPreMapLoad.Add(Spawner.OnSaveGame)
