RCHeaters = require("RC_Heaters")

local MINUTE_EVENT_NAME = "OnRealisticColdMinuteUpdate"

if LuaEventManager and LuaEventManager.AddEvent then
    LuaEventManager.AddEvent(MINUTE_EVENT_NAME)
end

RC_TempSim = RC_TempSim or {}

local function _applySandboxOptions()
    local sandbox = SandboxVars and SandboxVars.RealisticCold or nil
    if not sandbox then
        return
    end

    local function assign(field, key)
        local value = sandbox[key]
        if value ~= nil then
            RC_TempSim[field] = value
        end
    end

    assign("HEAT_DIST_EXP", "HeatDistanceExponent")
    assign("HEAT_TARGET_MAX_C", "HeatTargetMaximum")
    assign("VEHICLE_HEATER_MAX_DELTA_C", "VehicleHeaterMaxDelta")

    assign("BREACH_WEIGHT_WINDOW", "BreachWeightWindow")
    assign("BREACH_WEIGHT_DOOR", "BreachWeightDoor")
    assign("BREACH_WEIGHT_GAP", "BreachWeightGap")
    assign("BREACH_WEIGHT_CURTAIN", "BreachWeightCurtain")
    assign("EXPOSURE_MAX", "ExposureCap")

    assign("COLD_WORSEN_RATE_PER_HOUR", "ColdWorsenPerHour")
    assign("COLD_RECOVER_RATE_PER_HOUR", "ColdRecoverPerHour")
    assign("COLD_MEDICINE_DECAY_PER_HOUR", "ColdMedicineDecayPerHour")
    assign("COLD_SICKNESS_MULT", "ColdSicknessMultiplier")
    assign("CATCH_COLD_THRESHOLD", "CatchColdThreshold")
    assign("CATCH_COLD_RATE_MULT", "CatchColdRateMultiplier")

    assign("CHILLY_AIR_TEMP_THRESHOLD_C", "ChillyAirThreshold")
    assign("CHILLY_COOLING_MULT_THRESHOLD", "ChillyCoolingMultiplierThreshold")
    assign("CHILLY_INSULATION_THRESHOLD", "ChillyInsulationThreshold")

    assign("CLOTHING_INSULATION_SCALE", "ClothingInsulationScale")

    assign("HYPOTHERMIA_DAMAGE_INTERVAL_MIN", "HypothermiaDamageIntervalMin")
    assign("HYPOTHERMIA_DAMAGE_PER_TICK", "HypothermiaDamagePerTick")

    assign("USE_ONE_MINUTE_UPDATES", "UseOneMinuteUpdates")
end

function RC_TempSim.refreshSandboxVars()
    _applySandboxOptions()
end

local function _shouldUseOneMinuteUpdates()
    local flag = RC_TempSim.USE_ONE_MINUTE_UPDATES
    if flag == nil then return false end
    if flag == true then return true end
    if flag == 1 then return true end
    if flag == "1" then return true end
    if flag == "true" or flag == "TRUE" or flag == "True" then return true end
    return false
end

function RC_TempSim.initializeThermoregulatorOverrides()
    if Thermoregulator and Thermoregulator.setSimulationMultiplier then
        Thermoregulator.setSimulationMultiplier(0.0)
    end

    if Thermoregulator_tryouts and Thermoregulator_tryouts.setSimulationMultiplier then
        Thermoregulator_tryouts.setSimulationMultiplier(0.0)
    end
end

RC_TempSim.refreshSandboxVars()

-- tunables
RC_TempSim.DEFAULT_UPDATE_EVERY_MINUTES = RC_TempSim.DEFAULT_UPDATE_EVERY_MINUTES or 10
RC_TempSim.UPDATE_EVERY_MINUTES      = RC_TempSim.UPDATE_EVERY_MINUTES or RC_TempSim.DEFAULT_UPDATE_EVERY_MINUTES
RC_TempSim.USE_ONE_MINUTE_UPDATES    = RC_TempSim.USE_ONE_MINUTE_UPDATES or false
RC_TempSim._usingOneMinuteUpdates    = RC_TempSim._usingOneMinuteUpdates or _shouldUseOneMinuteUpdates()
RC_TempSim._minuteEventTriggerRegistered = RC_TempSim._minuteEventTriggerRegistered or false
RC_TempSim.MINUTE_EVENT_NAME         = RC_TempSim.MINUTE_EVENT_NAME or MINUTE_EVENT_NAME
RC_TempSim.INDOOR_OFFSET_DEFAULT_C   = 2.0
RC_TempSim.BASE_BLEND_PER_STEP       = 0.05
RC_TempSim.SEARCH_STAIRS_RADIUS      = 6
RC_TempSim.HYPOTHERMIA_DAMAGE_STAGE_THRESHOLD = 4
RC_TempSim.VANILLA_BODY_TEMP_AIR_THRESHOLD_C = RC_TempSim.VANILLA_BODY_TEMP_AIR_THRESHOLD_C or 20.0
RC_TempSim.ROOM_SIZE_REF_SQ = 15

RC_TempSim.SIZE_EXP = 1
RC_TempSim.SIZE_FACTOR_MIN = 3
RC_TempSim.SIZE_FACTOR_MAX = 1.8

RC_TempSim.HEAT_SIZE_EXP = 0.5
RC_TempSim.HEAT_SIZE_FACTOR_MIN = 1
RC_TempSim.HEAT_SIZE_FACTOR_MAX = 1.8
RC_TempSim.HEAT_INFLUENCE_MAX = 50
RC_TempSim.HEAT_TARGET_MIN_C = -50

RC_TempSim.PROXIMITY_HEAT_RATE_PER_UNIT = 0.02
RC_TempSim.PROXIMITY_HEAT_RATE_MAX = 1.5
RC_TempSim.PROXIMITY_ADJACENT_WEIGHT = 0.5
RC_TempSim.PROXIMITY_DIAGONAL_WEIGHT = 0.35

RC_TempSim.INTERNAL_EQ_RATE_PER_EDGE = 1
RC_TempSim.CLOSED_DOOR_EQ_RATE_MULT = 0.05

RC_TempSim.INDOOR_LEAK_RATE_PER_HOUR = RC_TempSim.INDOOR_LEAK_RATE_PER_HOUR or 0.003

RC_TempSim.BREACH_WINDOW_BARRICADED_MULT = RC_TempSim.BREACH_WINDOW_BARRICADED_MULT or 0.9
RC_TempSim.BREACH_WINDOW_CURTAIN_CLOSED_MULT = RC_TempSim.BREACH_WINDOW_CURTAIN_CLOSED_MULT or 0.85
RC_TempSim.BREACH_WINDOW_BARRICADED_CURTAIN_MULT = RC_TempSim.BREACH_WINDOW_BARRICADED_CURTAIN_MULT or 0.88
RC_TempSim.BREACH_WEIGHT_CURTAIN = RC_TempSim.BREACH_WEIGHT_CURTAIN or 0.3

local BREACH_WINDOW_FLOW_REFERENCE = 0.8
local BREACH_WINDOW_BARRICADE_REDUCTION_FRAC = {
    [0] = 0.0,
    [1] = 0.05 / BREACH_WINDOW_FLOW_REFERENCE,
    [2] = 0.12 / BREACH_WINDOW_FLOW_REFERENCE,
    [3] = 0.25 / BREACH_WINDOW_FLOW_REFERENCE,
    [4] = 0.50 / BREACH_WINDOW_FLOW_REFERENCE,
}
local BREACH_WINDOW_CURTAIN_REDUCTION_FRAC = 0.08 / BREACH_WINDOW_FLOW_REFERENCE

-- simplified simulation controls
RC_TempSim.LARGE_BUILDING_ROOM_THRESHOLD    = RC_TempSim.LARGE_BUILDING_ROOM_THRESHOLD    or 30
RC_TempSim.LARGE_BUILDING_BREACH_THRESHOLD  = RC_TempSim.LARGE_BUILDING_BREACH_THRESHOLD  or 18
RC_TempSim.LARGE_BUILDING_EXPOSURE_FLOOR    = RC_TempSim.LARGE_BUILDING_EXPOSURE_FLOOR    or 0.05

-- size/distance scaling
RC_TempSim.DIST_EXP           = 1.0
RC_TempSim.HUGE_DISTANCE_EXP         = RC_TempSim.HUGE_DISTANCE_EXP         or 1.25
RC_TempSim.HUGE_DISTANCE_MIN_FACTOR  = RC_TempSim.HUGE_DISTANCE_MIN_FACTOR  or 0.05
RC_TempSim.HUGE_DISTANCE_MAX_FACTOR  = RC_TempSim.HUGE_DISTANCE_MAX_FACTOR  or 1.0

RC_TempSim.HEAT_BASE_BLEND_PER_STEP = 0.1
RC_TempSim.HEAT_SCAN_LIMIT          = 12000
RC_TempSim.NATIVE_HEAT_MULT         = 1.0
if RC_TempSim.INCLUDE_NATIVE_HEAT == nil then RC_TempSim.INCLUDE_NATIVE_HEAT = true end

-- huge building rework parameters
RC_TempSim.HUGE_REWORK_ENABLED          = (RC_TempSim.HUGE_REWORK_ENABLED ~= false)
RC_TempSim.HUGE_FOCUS_MAX_AREA          = RC_TempSim.HUGE_FOCUS_MAX_AREA or 120
RC_TempSim.HUGE_FOCUS_EXTRA_NEIGHBORS   = RC_TempSim.HUGE_FOCUS_EXTRA_NEIGHBORS or 1
RC_TempSim.HUGE_POWERED_MAX_C           = RC_TempSim.HUGE_POWERED_MAX_C or 22.0
RC_TempSim.HUGE_POWERED_MIN_C           = RC_TempSim.HUGE_POWERED_MIN_C or 15.0
RC_TempSim.HUGE_POWERED_WARM_OUTDOOR_C  = RC_TempSim.HUGE_POWERED_WARM_OUTDOOR_C or -10.0
RC_TempSim.HUGE_POWERED_COLD_OUTDOOR_C  = RC_TempSim.HUGE_POWERED_COLD_OUTDOOR_C or -30.0
RC_TempSim.HUGE_POWER_ON_BLEND          = RC_TempSim.HUGE_POWER_ON_BLEND or 0.5
RC_TempSim.HUGE_COOL_MIN_RATE           = RC_TempSim.HUGE_COOL_MIN_RATE or 0.002
RC_TempSim.HUGE_COOL_MAX_RATE           = RC_TempSim.HUGE_COOL_MAX_RATE or 0.02
RC_TempSim.HUGE_COOL_RAMP_HOURS         = RC_TempSim.HUGE_COOL_RAMP_HOURS or 96
RC_TempSim.HUGE_COOL_SIZE_EXP           = RC_TempSim.HUGE_COOL_SIZE_EXP or 0.75
RC_TempSim.HUGE_COOL_SIZE_MIN           = RC_TempSim.HUGE_COOL_SIZE_MIN or 0.05
RC_TempSim.HUGE_COOL_SIZE_MAX           = RC_TempSim.HUGE_COOL_SIZE_MAX or 2.5
RC_TempSim.HUGE_COOL_DISTANCE_WEIGHT    = RC_TempSim.HUGE_COOL_DISTANCE_WEIGHT or 0.6
RC_TempSim.HUGE_COOL_DISTANCE_FLOOR     = RC_TempSim.HUGE_COOL_DISTANCE_FLOOR or 0.2

RC_TempSim.HUGE_PRECISE_ROOM_MAX_AREA   = RC_TempSim.HUGE_PRECISE_ROOM_MAX_AREA or 80
RC_TempSim.HUGE_PRECISE_ROOM_MAX_ROOMS  = RC_TempSim.HUGE_PRECISE_ROOM_MAX_ROOMS or 16
RC_TempSim.HUGE_NEIGHBOR_AREA_THRESHOLD = RC_TempSim.HUGE_NEIGHBOR_AREA_THRESHOLD or 36
RC_TempSim.HUGE_NEIGHBOR_SIZE_RATIO     = RC_TempSim.HUGE_NEIGHBOR_SIZE_RATIO or 1.25

-- Huge-building offline tuner caps (vanilla buildings only)
RC_TempSim.HUGE_OFFLINE_MAX_DELTA_PER_HOUR   = RC_TempSim.HUGE_OFFLINE_MAX_DELTA_PER_HOUR or 0.25
RC_TempSim.HUGE_OFFLINE_AVG_WINDOW_HOURS     = RC_TempSim.HUGE_OFFLINE_AVG_WINDOW_HOURS or 24
RC_TempSim.HUGE_OFFLINE_MIN_ROOMS            = RC_TempSim.HUGE_OFFLINE_MIN_ROOMS or 30

-- These likely already exist in your huge baseline; listed for clarity.
RC_TempSim.HUGE_POWERED_WARM_OUT             = RC_TempSim.HUGE_POWERED_WARM_OUT or -10
RC_TempSim.HUGE_POWERED_COLD_OUT             = RC_TempSim.HUGE_POWERED_COLD_OUT or -30
RC_TempSim.HUGE_POWERED_MAX_C                = RC_TempSim.HUGE_POWERED_MAX_C or 22
RC_TempSim.HUGE_POWERED_MIN_C                = RC_TempSim.HUGE_POWERED_MIN_C or 15

-- Small-room <-> big-room HVAC vent coupling (huge buildings only)
RC_TempSim.HUGE_VENT_SMALL_MAX_SQ     = RC_TempSim.HUGE_VENT_SMALL_MAX_SQ or 60    -- small room ≤ 60 tiles
RC_TempSim.HUGE_VENT_BIG_MIN_SQ       = RC_TempSim.HUGE_VENT_BIG_MIN_SQ or 80     -- adjacent “big” room ≥ 120 tiles
RC_TempSim.HUGE_VENT_RATE_PER_HOUR    = RC_TempSim.HUGE_VENT_RATE_PER_HOUR or 1.2  -- how fast small approaches big, per hour

RC_TempSim.HUGE_SMALLROOM_BASELINE_HEAT_FACTOR_CAP = RC_TempSim.HUGE_SMALLROOM_BASELINE_HEAT_FACTOR_CAP or 1.0

RC_TempSim.HUGE_MINUTES_BETWEEN_FULL_UPDATES = RC_TempSim.HUGE_MINUTES_BETWEEN_FULL_UPDATES or 60


-- offline simulation parameters
RC_TempSim.OUTSIDE_HISTORY_MAX      = 24

-- internal equalization
RC_TempSim.CAPACITY_EXP              = 1.0
RC_TempSim.CAPACITY_MIN              = 8

RC_TempSim.OFFLINE_HEAT_REF_AREA     = RC_TempSim.OFFLINE_HEAT_REF_AREA or 80
RC_TempSim.OFFLINE_HEAT_BREACH_EXP   = RC_TempSim.OFFLINE_HEAT_BREACH_EXP or 1.25
RC_TempSim.OFFLINE_HEAT_BREACH_MIN   = RC_TempSim.OFFLINE_HEAT_BREACH_MIN or 0.05
RC_TempSim.OFFLINE_HEAT_RESPONSE_PER_DEG = RC_TempSim.OFFLINE_HEAT_RESPONSE_PER_DEG or 0.06
RC_TempSim.OFFLINE_HEAT_RESPONSE_CAP = RC_TempSim.OFFLINE_HEAT_RESPONSE_CAP or 2.0

RC_TempSim.OFFLINE_LEAK_SEALED = 0.005
RC_TempSim.OFFLINE_LEAK_WINDOW = 15

RC_TempSim.OFFLINE_LEAK_DOOR = 20
RC_TempSim.OFFLINE_DECAY_BASE = 0.05
RC_TempSim.OFFLINE_DECAY_LEAK_MULT = 0.45
RC_TempSim.OFFLINE_DECAY_HIGH_LEAK_MULT = 0.35
RC_TempSim.OFFLINE_DECAY_SEALED_EXP = RC_TempSim.OFFLINE_DECAY_SEALED_EXP or 0.3
RC_TempSim.OFFLINE_DECAY_SEALED_MIN = RC_TempSim.OFFLINE_DECAY_SEALED_MIN or 0.1
RC_TempSim.OFFLINE_HEAT_EFFICIENCY = 50
RC_TempSim.OFFLINE_HEATER_RUNTIME_CAP = 72

RC_TempSim.SEED_BIAS_C = 0.0
RC_TempSim.WATER_ON_SEED_BIAS_C = 4.0
RC_TempSim.WATER_ON_MIN_C       = 21.0
RC_TempSim.WATER_ON_MAX_C       = 26.0
RC_TempSim.CHILLY_FORCE_EPSILON = RC_TempSim.CHILLY_FORCE_EPSILON or 0.01

RC_TempSim.CLOTHING_SEVERE_COLD_START_C   = RC_TempSim.CLOTHING_SEVERE_COLD_START_C   or -5.0
RC_TempSim.CLOTHING_SEVERE_COLD_FULL_C    = RC_TempSim.CLOTHING_SEVERE_COLD_FULL_C    or -35.0
RC_TempSim.CLOTHING_DYNAMIC_MIN_FRACTION  = RC_TempSim.CLOTHING_DYNAMIC_MIN_FRACTION  or 0.2
RC_TempSim.CLOTHING_LOW_PROTECTION_SCALE  = RC_TempSim.CLOTHING_LOW_PROTECTION_SCALE  or 12.0
RC_TempSim.CLOTHING_COOLING_MULT_MIN      = 0.1
RC_TempSim.CLOTHING_COOLING_MULT_MAX      = 1
RC_TempSim.CLOTHING_SEVERE_COLD_EXPOSURE_BOOST = RC_TempSim.CLOTHING_SEVERE_COLD_EXPOSURE_BOOST or 2.0
RC_TempSim.CLOTHING_INSULATION_CURVE_EXP     = RC_TempSim.CLOTHING_INSULATION_CURVE_EXP     or 1.8
RC_TempSim.CLOTHING_INSULATION_CURVE_ANCHOR  = RC_TempSim.CLOTHING_INSULATION_CURVE_ANCHOR  or 2.9
RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_START   = RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_START   or 2.5
RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_SPAN    = RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_SPAN    or 1.0
RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_STRENGTH = RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_STRENGTH or 6.0
RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_EXP     = RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_EXP     or 2.0
RC_TempSim.CLOTHING_STAGE_FREEZE_CAP_INSULATION   = RC_TempSim.CLOTHING_STAGE_FREEZE_CAP_INSULATION   or 2.9
RC_TempSim.CLOTHING_STAGE_COLD_CAP_INSULATION     = RC_TempSim.CLOTHING_STAGE_COLD_CAP_INSULATION     or 3.4
RC_TempSim.CLOTHING_STAGE_CAP_MIN_OUTDOOR_C       = RC_TempSim.CLOTHING_STAGE_CAP_MIN_OUTDOOR_C       or -40.0
RC_TempSim.CLOTHING_WETNESS_SOAK_RATE_PER_MIN     = RC_TempSim.CLOTHING_WETNESS_SOAK_RATE_PER_MIN     or 0.18
RC_TempSim.CLOTHING_WETNESS_SOAK_DELTA_MULT       = RC_TempSim.CLOTHING_WETNESS_SOAK_DELTA_MULT       or 0.25
RC_TempSim.CLOTHING_WETNESS_RETURN_RATE_PER_MIN   = RC_TempSim.CLOTHING_WETNESS_RETURN_RATE_PER_MIN   or 0.12
RC_TempSim.CLOTHING_WETNESS_BODY_DRY_THRESHOLD    = RC_TempSim.CLOTHING_WETNESS_BODY_DRY_THRESHOLD    or 20.0
RC_TempSim.CLOTHING_WETNESS_RETURN_DELAY_MIN      = RC_TempSim.CLOTHING_WETNESS_RETURN_DELAY_MIN      or 10.0
RC_TempSim.WIND_CHILL_EFFECT_MULT                 = RC_TempSim.WIND_CHILL_EFFECT_MULT                 or 0.3

RC_TempSim.MOVEMENT_COOLING_MULT_MOVING     = RC_TempSim.MOVEMENT_COOLING_MULT_MOVING     or 0.0
RC_TempSim.MOVEMENT_COOLING_MULT_WALKING    = RC_TempSim.MOVEMENT_COOLING_MULT_WALKING    or 0.0
RC_TempSim.MOVEMENT_COOLING_MULT_RUNNING    = RC_TempSim.MOVEMENT_COOLING_MULT_RUNNING    or 0.8
RC_TempSim.MOVEMENT_COOLING_MULT_SPRINTING  = RC_TempSim.MOVEMENT_COOLING_MULT_SPRINTING  or 0.75

RC_TempSim.MOVEMENT_HEAT_FRACTION_MOVING    = RC_TempSim.MOVEMENT_HEAT_FRACTION_MOVING    or 0
RC_TempSim.MOVEMENT_HEAT_FRACTION_WALKING   = RC_TempSim.MOVEMENT_HEAT_FRACTION_WALKING   or 0
RC_TempSim.MOVEMENT_HEAT_FRACTION_RUNNING   = RC_TempSim.MOVEMENT_HEAT_FRACTION_RUNNING   or 3
RC_TempSim.MOVEMENT_HEAT_FRACTION_SPRINTING = RC_TempSim.MOVEMENT_HEAT_FRACTION_SPRINTING or 4

RC_TempSim.BODY_HEATGEN_TARGET_IDLE         = RC_TempSim.BODY_HEATGEN_TARGET_IDLE         or 0.0
RC_TempSim.BODY_HEATGEN_TARGET_SNEAK        = RC_TempSim.BODY_HEATGEN_TARGET_SNEAK        or 0.002
RC_TempSim.BODY_HEATGEN_TARGET_WALK         = RC_TempSim.BODY_HEATGEN_TARGET_WALK         or 0.005
RC_TempSim.BODY_HEATGEN_TARGET_RUN          = RC_TempSim.BODY_HEATGEN_TARGET_RUN          or 1.0
RC_TempSim.BODY_HEATGEN_TARGET_SPRINT       = RC_TempSim.BODY_HEATGEN_TARGET_SPRINT       or 1.2
RC_TempSim.BODY_HEATGEN_TARGET_COMBAT       = RC_TempSim.BODY_HEATGEN_TARGET_COMBAT       or 0.75
RC_TempSim.BODY_HEATGEN_TARGET_MAX          = RC_TempSim.BODY_HEATGEN_TARGET_MAX          or 1.25
RC_TempSim.BODY_HEATGEN_HEAVY_LOAD_BONUS_MAX = RC_TempSim.BODY_HEATGEN_HEAVY_LOAD_BONUS_MAX or 0.25
RC_TempSim.BODY_HEATGEN_WEIGHT_BONUS_MAX    = RC_TempSim.BODY_HEATGEN_WEIGHT_BONUS_MAX    or 0.15
RC_TempSim.BODY_HEATGEN_WEIGHT_REF          = RC_TempSim.BODY_HEATGEN_WEIGHT_REF          or 80.0
RC_TempSim.BODY_HEATGEN_WEIGHT_MAX          = RC_TempSim.BODY_HEATGEN_WEIGHT_MAX          or 110.0

RC_TempSim.BODY_HEATGEN_WARM_IDLE_THRESHOLD_C = RC_TempSim.BODY_HEATGEN_WARM_IDLE_THRESHOLD_C or 18.0
RC_TempSim.BODY_HEATGEN_WARM_IDLE_FULL_C      = RC_TempSim.BODY_HEATGEN_WARM_IDLE_FULL_C      or 36.0
RC_TempSim.BODY_HEATGEN_WARM_IDLE_TARGET      = RC_TempSim.BODY_HEATGEN_WARM_IDLE_TARGET      or 0.4

RC_TempSim.BODY_HEATGEN_INCREASE_RATE_BASE      = RC_TempSim.BODY_HEATGEN_INCREASE_RATE_BASE      or 0.45
RC_TempSim.BODY_HEATGEN_INCREASE_RATE_NEUTRAL_MULT = RC_TempSim.BODY_HEATGEN_INCREASE_RATE_NEUTRAL_MULT or 1.0
RC_TempSim.BODY_HEATGEN_INCREASE_RATE_COLD_MULT   = RC_TempSim.BODY_HEATGEN_INCREASE_RATE_COLD_MULT   or 0.4
RC_TempSim.BODY_HEATGEN_INCREASE_RATE_WARM_MULT   = RC_TempSim.BODY_HEATGEN_INCREASE_RATE_WARM_MULT   or 1.75
RC_TempSim.BODY_HEATGEN_COLD_AIR_THRESHOLD_C      = RC_TempSim.BODY_HEATGEN_COLD_AIR_THRESHOLD_C      or 0.0
RC_TempSim.BODY_HEATGEN_WARM_AIR_THRESHOLD_C      = RC_TempSim.BODY_HEATGEN_WARM_AIR_THRESHOLD_C      or 25.0
RC_TempSim.BODY_HEATGEN_STAMINA_RATE_MULT         = RC_TempSim.BODY_HEATGEN_STAMINA_RATE_MULT         or 0.4

RC_TempSim.BODY_HEATGEN_DECREASE_RATE_BASE      = RC_TempSim.BODY_HEATGEN_DECREASE_RATE_BASE      or 0.5
RC_TempSim.BODY_HEATGEN_DECREASE_RATE_COLD_MULT = RC_TempSim.BODY_HEATGEN_DECREASE_RATE_COLD_MULT or 1.2
RC_TempSim.BODY_HEATGEN_DECREASE_RATE_WARM_MULT = RC_TempSim.BODY_HEATGEN_DECREASE_RATE_WARM_MULT or 0.6

RC_TempSim.BODY_HEATGEN_NORMALIZATION_MAX    = RC_TempSim.BODY_HEATGEN_NORMALIZATION_MAX    or 1.0
RC_TempSim.BODY_HEATGEN_COLD_PUSH_MOVEMENT_THRESHOLD = RC_TempSim.BODY_HEATGEN_COLD_PUSH_MOVEMENT_THRESHOLD or 0.75
RC_TempSim.BODY_HEATGEN_COLD_PUSH_CLOTHING_THRESHOLD = RC_TempSim.BODY_HEATGEN_COLD_PUSH_CLOTHING_THRESHOLD or 0.55
RC_TempSim.BODY_HEATGEN_COLD_PUSH_AIR_MAX_C          = RC_TempSim.BODY_HEATGEN_COLD_PUSH_AIR_MAX_C          or 15.0
RC_TempSim.BODY_HEATGEN_COLD_PUSH_RATE_PER_MIN       = RC_TempSim.BODY_HEATGEN_COLD_PUSH_RATE_PER_MIN       or 0.05
RC_TempSim.BODY_HEATGEN_COLD_PUSH_MIN_STAGE          = RC_TempSim.BODY_HEATGEN_COLD_PUSH_MIN_STAGE          or 2
RC_TempSim.BODY_HEATGEN_COOLING_MULT         = RC_TempSim.BODY_HEATGEN_COOLING_MULT         or 1.0
RC_TempSim.BODY_HEATGEN_COOLING_MIN          = RC_TempSim.BODY_HEATGEN_COOLING_MIN          or 0.0
RC_TempSim.BODY_HEATGEN_RECOVERY_MULT        = RC_TempSim.BODY_HEATGEN_RECOVERY_MULT        or 1.0
RC_TempSim.BODY_HEATGEN_DIRECT_GAIN_THRESHOLD   = RC_TempSim.BODY_HEATGEN_DIRECT_GAIN_THRESHOLD   or 0.65
RC_TempSim.BODY_HEATGEN_DIRECT_GAIN_MAX_PER_MIN = RC_TempSim.BODY_HEATGEN_DIRECT_GAIN_MAX_PER_MIN or 0.04
RC_TempSim.BODY_HEATGEN_DIRECT_WARM_MULT        = RC_TempSim.BODY_HEATGEN_DIRECT_WARM_MULT        or 1.6
RC_TempSim.BODY_HEATGEN_DIRECT_WARM_FULL_C      = RC_TempSim.BODY_HEATGEN_DIRECT_WARM_FULL_C      or 35.0
RC_TempSim.BODY_HEATGEN_COLD_REQUIRED_INSULATION_BASE = RC_TempSim.BODY_HEATGEN_COLD_REQUIRED_INSULATION_BASE or 0.45
RC_TempSim.BODY_HEATGEN_COLD_REQUIRED_INSULATION_PER_SEVERITY = RC_TempSim.BODY_HEATGEN_COLD_REQUIRED_INSULATION_PER_SEVERITY or 1.35
RC_TempSim.BODY_HEATGEN_COLD_MIN_ACTIVITY_FRACTION = RC_TempSim.BODY_HEATGEN_COLD_MIN_ACTIVITY_FRACTION or 0.05
RC_TempSim.BODY_HEATGEN_SEVERE_COLD_TARGET_PENALTY = RC_TempSim.BODY_HEATGEN_SEVERE_COLD_TARGET_PENALTY or 0.15
RC_TempSim.BODY_HEATGEN_COLD_MOVEMENT_REFUND_SUPPRESSION = RC_TempSim.BODY_HEATGEN_COLD_MOVEMENT_REFUND_SUPPRESSION or 2.0
RC_TempSim.BODY_HEATGEN_COLD_MOVEMENT_MIN_RETENTION = RC_TempSim.BODY_HEATGEN_COLD_MOVEMENT_MIN_RETENTION or 0.1
RC_TempSim.SWEAT_COLD_ALLOWANCE_THRESHOLD = RC_TempSim.SWEAT_COLD_ALLOWANCE_THRESHOLD or 0.6
RC_TempSim.SWEAT_COLD_ALLOWANCE_SEVERITY_MULT = RC_TempSim.SWEAT_COLD_ALLOWANCE_SEVERITY_MULT or 1.0

RC_TempSim.MOVEMENT_CHILLY_INSULATION_BONUS_PER_HEAT = RC_TempSim.MOVEMENT_CHILLY_INSULATION_BONUS_PER_HEAT or 3.0
RC_TempSim.MOVEMENT_CHILLY_COOLING_BONUS_PER_HEAT    = RC_TempSim.MOVEMENT_CHILLY_COOLING_BONUS_PER_HEAT    or 0.75
RC_TempSim.MOVEMENT_CHILLY_AIR_BUFFER_PER_HEAT       = RC_TempSim.MOVEMENT_CHILLY_AIR_BUFFER_PER_HEAT       or 5.0

-- state
RC_TempSim._state = RC_TempSim._state or {
    mode = nil,
    ownerPlayer = nil,
    -- vanilla
    currentBuildingDef = nil,
    building = nil,
    rooms = nil,
    -- player
    playerBuilding = nil,
    playerRegions = nil,
    -- common
    graph = nil,
    closedDoors = nil,
    temps = nil,
    breaches = nil,
    byRoomName = nil,
    roomArea = nil,
    lastLeakInfo = nil,
    hugeInfo = nil,
    lastHugeFullUpdateHour = nil,
    lastHugeScanRevision = nil,
    lastHugeLightUpdateHour = nil,
}

RC_TempSim._vehicleHeaterState = RC_TempSim._vehicleHeaterState or setmetatable({}, { __mode = "k" })

---------------------------------------------------------------------------
-- Helpers (common)
---------------------------------------------------------------------------

local function md_get()
    if ModData and ModData.getOrCreate then
        return ModData.getOrCreate("RC_TempSim")
    end
    RC_TempSim._fallbackMD = RC_TempSim._fallbackMD or {}
    return RC_TempSim._fallbackMD
end

local function md_transmit()
    if ModData and ModData.transmit then pcall(ModData.transmit, "RC_TempSim") end
end

local function _maxMovementHeatFraction()
    local maxHeat = RC_TempSim.BODY_HEATGEN_NORMALIZATION_MAX
    if not maxHeat or maxHeat <= 0 then
        maxHeat = RC_TempSim.BODY_HEATGEN_TARGET_MAX or 1.0
    end
    if not maxHeat or maxHeat <= 0 then
        maxHeat = 1.0
    end
    return maxHeat
end

local function _normalizedMovementHeat(heat)
    local normalized = heat or 0.0
    if normalized < 0 then normalized = 0 end

    local maxHeat = _maxMovementHeatFraction()
    if maxHeat and maxHeat > 0 then
        normalized = normalized / maxHeat
    end

    if normalized < 0 then normalized = 0 end
    if normalized > 1 then normalized = 1 end

    return normalized
end

local function _currentWorldHours()
    local gt = getGameTime and getGameTime() or nil
    if gt and gt.getWorldAgeHours then
        return gt:getWorldAgeHours() or 0
    end
    return 0
end

local function _currentWorldHourFloor()
    local hours = _currentWorldHours()
    if hours <= 0 then return 0 end
    return math.floor(hours)
end

local function _ensureOutsideHistory(store)
    store = store or md_get()
    local hist = store.outsideHistory
    if type(hist) ~= "table" then
        hist = { entries = {}, lastHour = nil }
        store.outsideHistory = hist
    end
    if type(hist.entries) ~= "table" then
        hist.entries = {}
    end
    return hist
end

local function _cloneOutsideHistoryEntries()
    local store = md_get()
    local hist = _ensureOutsideHistory(store)
    local out = {}
    for i = 1, #(hist.entries or {}) do
        local rec = hist.entries[i]
        if type(rec) == "table" then
            out[#out+1] = { hour = rec.hour, temp = rec.temp }
        end
    end
    return out
end

local function recordOutsideTemperature()
    local store = md_get()
    local hist = _ensureOutsideHistory(store)
    local hour = _currentWorldHourFloor()
    if hist.lastHour == hour then return end
    local temp = 0
    local cm = getClimateManager and getClimateManager() or nil
    if cm and cm.getTemperature then
        temp = cm:getTemperature() or 0
    end
    hist.entries[#hist.entries+1] = { hour = hour, temp = temp }
    hist.lastHour = hour
    local maxEntries = tonumber(RC_TempSim.OUTSIDE_HISTORY_MAX) or 24
    if maxEntries < 1 then maxEntries = 1 end
    while #hist.entries > maxEntries do
        table.remove(hist.entries, 1)
    end
    store.outsideHistory = hist
    md_transmit()
end

local function _ensurePowerStore()
    local store = md_get()
    local rec = store.powerState
    if type(rec) ~= "table" then
        rec = {}
        store.powerState = rec
    end
    return rec
end

local function _avgOutsideOverWindow(historyEntries, outsideNow, windowHours)
    local now = _currentWorldHours and _currentWorldHours() or 0
    if not historyEntries or #historyEntries == 0 then
        return outsideNow
    end
    local span = math.max(0.01, math.min(windowHours or 0, RC_TempSim.HUGE_OFFLINE_AVG_WINDOW_HOURS or 24))
    local startT = now - span

    table.sort(historyEntries, function(a,b) return (a.hour or 0) < (b.hour or 0) end)

    local acc, covered = 0.0, 0.0
    local lastHour, lastTemp = startT, outsideNow
    for i = #historyEntries, 1, -1 do
        local h = historyEntries[i].hour or 0
        if h <= startT then
            lastTemp = historyEntries[i].temp or outsideNow
            break
        end
    end

    for _, rec in ipairs(historyEntries) do
        local h = rec.hour or 0
        local t = rec.temp or outsideNow
        if h > startT then
            local segEnd = math.min(h, now)
            if segEnd > lastHour then
                acc     = acc     + (segEnd - lastHour) * lastTemp
                covered = covered + (segEnd - lastHour)
                lastHour = segEnd
            end
            lastTemp = t
        end
        if lastHour >= now then break end
    end

    if lastHour < now then
        acc     = acc     + (now - lastHour) * lastTemp
        covered = covered + (now - lastHour)
    end

    if covered <= 0 then return outsideNow end
    return acc / covered
end

local function _trackWorldPowerStatus()
    local rec = _ensurePowerStore()
    local gm   = getGameTime()
    local sb   = getSandboxOptions()
    local world = getWorld()

    local powered = true
    local computedPowerOffHour = nil

    if gm and gm.getWorldAgeHours and sb and sb.getOptionByName then
        local opt = sb:getOptionByName("ElecShutModifier")
        local shutDays = opt and opt.getValue and opt:getValue() or nil
        if shutDays and shutDays >= 0 then
            local worldHours   = gm:getWorldAgeHours() or 0
            local timeSinceApo = (sb.getTimeSinceApo and sb:getTimeSinceApo()) or 1
            local offsetDays   = (timeSinceApo - 1) * 30
            computedPowerOffHour = (shutDays - offsetDays) * 24
            powered = worldHours < computedPowerOffHour
        end
    end

    if world and world.isHydroPowerOn then
        local ok, hydro = pcall(function() return world:isHydroPowerOn() end)
        if ok then
            powered = (hydro == true)
        end
    end

    local changed = false
    if rec.lastPowered ~= powered then
        rec.lastPowered = powered
        changed = true
    end
    if rec.powerOffHour ~= computedPowerOffHour then
        rec.powerOffHour = computedPowerOffHour
        changed = true
    end
    if changed then
        md_transmit()
    end
    if rec.lastPowered == nil then rec.lastPowered = powered end
    return powered, rec.powerOffHour
end

local function computePoweredHugeIndoorTarget(outdoorTemp)
    local warmOut = RC_TempSim.HUGE_POWERED_WARM_OUTDOOR_C or -10
    local coldOut = RC_TempSim.HUGE_POWERED_COLD_OUTDOOR_C or (warmOut - 20)
    local maxC = RC_TempSim.HUGE_POWERED_MAX_C or 22
    local minC = RC_TempSim.HUGE_POWERED_MIN_C or 15
    if outdoorTemp == nil then return maxC end
    if outdoorTemp >= warmOut then
        return maxC
    end
    if outdoorTemp <= coldOut then
        return minC
    end
    local span = warmOut - coldOut
    if span <= 0 then return maxC end
    local t = (outdoorTemp - coldOut) / span
    return minC + (maxC - minC) * t
end

local function buildFilteredGraph(graph, allowed)
    if type(graph) ~= "table" or not allowed then return graph end
    local out = {}
    for node, neighbors in pairs(graph) do
        if allowed[node] then
            local filtered = {}
            for nb,_ in pairs(neighbors or {}) do
                if allowed[nb] then
                    filtered[nb] = true
                end
            end
            out[node] = filtered
        end
    end
    return out
end

local function applyHVACVentCoupling(S, focusSet, stepHours, directHeatMap)
    if not (S and S.hugeInfo and S.temps and S.graph and S.roomArea) then return end
    if not focusSet or not stepHours or stepHours <= 0 then return end

    local smallMax  = RC_TempSim.HUGE_VENT_SMALL_MAX_SQ or 60
    local bigMin    = RC_TempSim.HUGE_VENT_BIG_MIN_SQ   or 120
    local ratePH    = RC_TempSim.HUGE_VENT_RATE_PER_HOUR or 1.2

    local blend = 1.0 - math.exp(-ratePH * stepHours)
    if blend <= 0 then return end
    if blend > 1 then blend = 1 end

    local graph = S.graph or {}
    local area  = S.roomArea or {}
    for node,_ in pairs(focusSet) do
        local aSmall = area[node]
        if aSmall and aSmall <= smallMax then
            local best, bestArea = nil, -1
            for nb,_ in pairs(graph[node] or {}) do
                local aNb = area[nb] or 0
                if aNb >= bigMin and aNb > bestArea then
                    best, bestArea = nb, aNb
                end
            end
            local hasActiveHeat = false
            if directHeatMap then
                local direct = directHeatMap[node]
                if type(direct) == "number" and direct > 0 then
                    hasActiveHeat = true
                end
            end

            if best then
                local tSmall = S.temps[node]
                local tBig   = S.temps[best] or tSmall
                local skipCoupling = false
                if hasActiveHeat then
                    if type(tSmall) == "number" then
                        if type(tBig) ~= "number" then
                            skipCoupling = true
                        elseif tSmall >= tBig - 0.1 then
                            skipCoupling = true
                        end
                    end
                end
                if tSmall and tBig and not skipCoupling then
                    S.temps[node] = tSmall + blend * (tBig - tSmall)
                end
            end
        end
    end
end

local function applyHugeBuildingBaseline(S, rooms, outdoorTemp, stepHours, powered, powerOffHour, excludeSet)
    if not S or type(S.temps) ~= "table" then return end
    rooms = rooms or {}
    if #rooms == 0 then return end

    if not stepHours or stepHours <= 0 then
        local minutes = tonumber(RC_TempSim.UPDATE_EVERY_MINUTES) or 10
        stepHours = minutes / 60
    end
    if stepHours <= 0 then return end

    local now = _currentWorldHours and _currentWorldHours() or 0
    local elapsed = 0
    if powerOffHour and now and now > powerOffHour then
        elapsed = now - powerOffHour
    end
    if elapsed < 0 then elapsed = 0 end
    local rampHours = RC_TempSim.HUGE_COOL_RAMP_HOURS or 0
    local ramp = 1.0
    if rampHours and rampHours > 0 then
        ramp = math.min(1, math.max(0, elapsed / rampHours))
    end

    -- size & distance shaping
    local ref        = RC_TempSim.ROOM_SIZE_REF_SQ or 100
    local sizeExp    = RC_TempSim.HUGE_SIZE_EXP or 0.5
    local minFactor  = RC_TempSim.HUGE_SIZE_FACTOR_MIN or 0.35
    local maxFactor  = RC_TempSim.HUGE_SIZE_FACTOR_MAX or 1.6

    local distWeight = RC_TempSim.HUGE_COOL_DISTANCE_WEIGHT or 0.5
    local distFloor  = RC_TempSim.HUGE_COOL_DISTANCE_FLOOR or 0.0
    if distFloor < 0 then distFloor = 0 end

    local baseRate   = (RC_TempSim.HUGE_COOL_MIN_RATE or 0.03) + ramp * (RC_TempSim.HUGE_COOL_MAX_RATE or 0.18)

    local areaMap = S.roomArea or {}
    local distMap = (S.hugeInfo and S.hugeInfo.distanceFromPerimeter) or {}
    local adjacency = S.graph or {}

    local neighborThreshold = RC_TempSim.HUGE_NEIGHBOR_AREA_THRESHOLD or ref
    local neighborRatio     = RC_TempSim.HUGE_NEIGHBOR_SIZE_RATIO or 1.0
    if neighborThreshold and neighborThreshold <= 0 then neighborThreshold = nil end
    if neighborRatio and neighborRatio <= 0 then neighborRatio = nil end

    local sorted, seen = {}, {}
    for _, node in ipairs(rooms) do
        if node and not seen[node] then
            seen[node] = true
            local area = areaMap[node] or ref
            sorted[#sorted+1] = { node = node, area = area }
        end
    end
    table.sort(sorted, function(a, b)
        local aa = a.area or ref
        local bb = b.area or ref
        if aa ~= bb then return aa > bb end
        return tostring(a.node) < tostring(b.node)
    end)

    local powerTarget    = computePoweredHugeIndoorTarget(outdoorTemp)
    local offlineTarget  = outdoorTemp or powerTarget
    local updated = {}

    local bigThreshold = RC_TempSim.HUGE_FOCUS_MAX_AREA
    if bigThreshold and bigThreshold <= 0 then bigThreshold = nil end
    local bigPreSum, bigPreCount = 0, 0
    if bigThreshold then
        for room, area in pairs(areaMap) do
            if area and area > bigThreshold and not (excludeSet and excludeSet[room]) then
                local temp = S.temps[room]
                if type(temp) == "number" then
                    bigPreSum = bigPreSum + temp
                    bigPreCount = bigPreCount + 1
                end
            end
        end
    end

    for _, entry in ipairs(sorted) do
        local node = entry.node
        if not (excludeSet and excludeSet[node]) then
            local area = entry.area or ref

            local ratio = math.max(1, area) / ref
            local sizeFactor = ratio ^ sizeExp
            if sizeFactor < minFactor then sizeFactor = minFactor end
            if sizeFactor > maxFactor then sizeFactor = maxFactor end

            local dist = distMap[node] or 0
            if dist < 0 then dist = 0 end
            local distFactor = 1 / (1 + dist * distWeight)
            if distFactor < distFloor then distFactor = distFloor end
            local rate = baseRate * sizeFactor * distFactor
            if rate < 0 then rate = 0 end
            local coolBlend = 1 - math.exp(-rate * stepHours)
            if coolBlend < 0 then coolBlend = 0 end
            if coolBlend > 1 then coolBlend = 1 end

            local hadExisting = type(S.temps[node]) == "number"
            local current = S.temps[node]
            if type(current) ~= "number" then
                current = outdoorTemp or offlineTarget
            end

            local target = powered and powerTarget or offlineTarget

            local largestNeighbor, largestArea = nil, nil
            for nb,_ in pairs(adjacency[node] or {}) do
                local nbArea = areaMap[nb]
                if nbArea and ((largestArea == nil) or nbArea > largestArea) then
                    largestArea = nbArea
                    largestNeighbor = nb
                end
            end
            if largestNeighbor then
                local neighborTemp = updated[largestNeighbor]
                if neighborTemp == nil then neighborTemp = S.temps[largestNeighbor] end
                if type(neighborTemp) == "number" and neighborTemp ~= current then
                    local useNeighbor = false
                    local ratioArea = (largestArea and area and area > 0) and (largestArea / area) or nil
                    if largestArea and area then
                        if largestArea > area then
                            useNeighbor = true
                        elseif neighborThreshold and largestArea >= neighborThreshold and area < neighborThreshold then
                            useNeighbor = true
                        end
                    elseif largestArea and neighborThreshold and largestArea >= neighborThreshold then
                        useNeighbor = true
                    end
                    if not useNeighbor and neighborRatio and ratioArea and ratioArea >= neighborRatio then
                        useNeighbor = true
                    end
                    if useNeighbor then
                        target = neighborTemp
                    end
                end
            end

            local snapToTarget = powered and not hadExisting

            local blend = coolBlend
            if not snapToTarget and powered and current and target then
                if current < target then
                    local heatBase = RC_TempSim.HEAT_BASE_BLEND_PER_STEP or RC_TempSim.BASE_BLEND_PER_STEP or 0.25
                    local heatRef = RC_TempSim.ROOM_SIZE_REF_SQ or ref or 100
                    if not heatRef or heatRef <= 0 then heatRef = ref or 100 end
                    local heatSizeExp = RC_TempSim.HEAT_SIZE_EXP or 0.5
                    local heatMin = RC_TempSim.HEAT_SIZE_FACTOR_MIN
                    local heatMax = RC_TempSim.HEAT_SIZE_FACTOR_MAX
                    local heaterRatio = heatRef / math.max(1, area or heatRef)
                    local heaterFactor = heaterRatio ^ heatSizeExp
                    local capFactor = RC_TempSim.HUGE_SMALLROOM_BASELINE_HEAT_FACTOR_CAP
                    if capFactor and capFactor > 0 then
                        local smallLimit = RC_TempSim.HUGE_SMALLROOM_MAX_SQ or heatRef
                        if area and area > 0 and smallLimit and smallLimit > 0 and area <= smallLimit then
                            local largeNeighbor = false
                            local neighborLimit = RC_TempSim.HUGE_VENT_BIG_MIN_SQ or neighborThreshold or smallLimit
                            if largestArea and largestArea > area then
                                if neighborLimit and neighborLimit > 0 then
                                    if largestArea >= neighborLimit then
                                        largeNeighbor = true
                                    end
                                else
                                    largeNeighbor = true
                                end
                            end
                            if largeNeighbor and heaterFactor > capFactor then
                                heaterFactor = capFactor
                            end
                        end
                    end
                    if heatMin and heaterFactor < heatMin then heaterFactor = heatMin end
                    if heatMax and heaterFactor > heatMax then heaterFactor = heatMax end
                    blend = heatBase * heaterFactor
                    if blend < 0 then
                        blend = 0
                    elseif blend > 1 then
                        blend = 1
                    end
                else
                    blend = coolBlend
                end
            end

            if snapToTarget and target then
                S.temps[node] = target
            else
                S.temps[node]  = current + (target - current) * blend
            end
            updated[node]  = S.temps[node]
        end
    end

    if bigThreshold then
        local sum, count = 0, 0
        for room, temp in pairs(updated) do
            local area = areaMap[room]
            if area and area > bigThreshold and type(temp) == "number" then
                sum = sum + temp
                count = count + 1
            end
        end
        if count == 0 and bigPreCount > 0 then
            sum = bigPreSum
            count = bigPreCount
        end
        if count > 0 then
            local shared = sum / count
            for room, area in pairs(areaMap) do
                if area and area > bigThreshold and not (excludeSet and excludeSet[room]) then
                    if S.temps[room] ~= shared then
                        S.temps[room] = shared
                        updated[room] = shared
                    end
                end
            end
        end
    end
end

local function clampPoweredHugeRooms(S, rooms, powered, stepHours)
    if not powered or not S or type(S.temps) ~= "table" then return end

    local minC = RC_TempSim.HUGE_POWERED_MIN_C or 15
    local ratePerHour = RC_TempSim.HUGE_FLOOR_RATE_PER_HOUR or 0.6

    if not stepHours or stepHours <= 0 then
        local minutes = tonumber(RC_TempSim.UPDATE_EVERY_MINUTES) or 10
        stepHours = minutes / 60
    end
    if stepHours <= 0 then return end

    local blend = 1 - math.exp(-ratePerHour * stepHours)
    if blend <= 0 then return end
    if blend > 1 then blend = 1 end

    local iter = {}
    if type(rooms) == "table" then
        for _, node in ipairs(rooms) do
            iter[#iter+1] = node
        end
    end
    if S.rooms and S.rooms.size then
        for i = 0, S.rooms:size() - 1 do
            iter[#iter+1] = S.rooms:get(i)
        end
    end
    if #iter == 0 then return end

    local seen = {}
    for _, node in ipairs(iter) do
        if node and not seen[node] then
            seen[node] = true
            local current = S.temps[node]
            if type(current) == "number" and current < minC then
                S.temps[node] = current + (minC - current) * blend
            end
        end
    end
end

local function _tableHasEntries(tbl)
    if type(tbl) ~= "table" then return false end
    for _, _ in pairs(tbl) do
        return true
    end
    return false
end

local function _breachType(rec)
    if not rec then return nil end
    if rec.window then return "window" end
    if rec.curtain then return "curtain" end
    if rec.type then return rec.type end
    if rec.door then return "door" end
    return nil
end

local function _windowBreachFlowMultiplier(rec)
    if not rec then return 1.0 end
    local baseFlow = RC_TempSim.BREACH_WEIGHT_WINDOW or BREACH_WINDOW_FLOW_REFERENCE
    if not baseFlow or baseFlow <= 0 then return 0.0 end

    local curtainClosed = rec.curtainClosed and true or false
    local win = rec.window
    if win and RC_RoomLogic and RC_RoomLogic.windowHasClosedCurtains then
        local ok, dynClosed = pcall(RC_RoomLogic.windowHasClosedCurtains, win)
        if ok and dynClosed ~= nil then
            curtainClosed = dynClosed and true or false
        end
    end

    local planks = rec.barricadePlanks
    if planks == nil and win and RC_RoomLogic and RC_RoomLogic.windowBarricadePlankCount then
        local ok, count = pcall(RC_RoomLogic.windowBarricadePlankCount, win)
        if ok then planks = count end
    end
    if planks == nil and rec.barricaded then planks = 1 end

    planks = tonumber(planks) or 0
    if planks < 0 then planks = 0 end
    if planks > 4 then planks = 4 end

    local plankFrac = BREACH_WINDOW_BARRICADE_REDUCTION_FRAC[planks] or 0.0
    local curtainFrac = curtainClosed and BREACH_WINDOW_CURTAIN_REDUCTION_FRAC or 0.0

    local flow = baseFlow * (1.0 - plankFrac - curtainFrac)
    if flow < 0 then flow = 0 end

    return flow / baseFlow
end

local function computeOpenBreachStrengthsForRooms(breaches, isOpenFn, allowedSet)
    local strengths, totalCount = {}, 0
    local counts = { window = 0, door = 0, curtain = 0, gap = 0 }
    local totalWeighted = 0.0
    if type(breaches) ~= "table" or not isOpenFn or not allowedSet or not _tableHasEntries(allowedSet) then
        return strengths, totalCount, counts, totalWeighted
    end

    for _, rec in ipairs(breaches) do
        local room = rec and rec.room
        local recType = _breachType(rec)
        if recType ~= "curtain" and room and allowedSet[room] and recType and isOpenFn(rec) then
            local w = RC_TempSim.BREACH_WEIGHT_WINDOW
            if recType == "door" then
                w = RC_TempSim.BREACH_WEIGHT_DOOR; counts.door = counts.door + 1
            elseif recType == "gap" then
                w = RC_TempSim.BREACH_WEIGHT_GAP;  counts.gap  = counts.gap + 1
            else
                counts.window = counts.window + 1
                w = w * _windowBreachFlowMultiplier(rec)
            end
            strengths[room] = (strengths[room] or 0) + w
            totalWeighted = totalWeighted + w
            totalCount = totalCount + 1
        end
    end

    return strengths, totalCount, counts, totalWeighted
end

local function breachIsOpen_vanilla(rec)
    local recType = _breachType(rec)
    if not recType then return false end
    if recType == "gap" then return true end
    if recType == "door" then
        return RC_RoomLogic.isDoorOpen(rec.door)
    elseif recType == "curtain" then
        return false
    elseif recType == "window" then
        local win = rec.window
        local open = false
        local smashed = false
        if win then
            open = RC_RoomLogic.isWindowOpen(win) or false
            smashed = RC_RoomLogic.isWindowSmashed(win) or false
            if open or smashed then return true end
        end
        if rec.open ~= nil then open = open or rec.open end
        if rec.smashed ~= nil then smashed = smashed or rec.smashed end
        return open or smashed
    end
    return false
end

local function computeOpenBreachStrengths(breaches, isOpenFn)
    local strengths, totalCount = {}, 0
    local counts = { window = 0, door = 0, curtain = 0, gap = 0 }
    local totalWeighted = 0.0

    for _, rec in ipairs(breaches or {}) do
        local recType = _breachType(rec)
        if recType ~= "curtain" and rec and rec.room and recType and isOpenFn(rec) then
            local w = RC_TempSim.BREACH_WEIGHT_WINDOW
            if recType == "door" then
                w = RC_TempSim.BREACH_WEIGHT_DOOR; counts.door = counts.door + 1
            elseif recType == "gap" then
                w = RC_TempSim.BREACH_WEIGHT_GAP;  counts.gap  = counts.gap + 1
            else
                counts.window = counts.window + 1
                w = w * _windowBreachFlowMultiplier(rec)
            end
            strengths[rec.room] = (strengths[rec.room] or 0) + w
            totalWeighted = totalWeighted + w
            totalCount = totalCount + 1
        end
    end
    return strengths, totalCount, counts, totalWeighted
end

local function _clamp01(x)
    if x ~= x then return 0 end
    if x < 0 then return 0 elseif x > 1 then return 1 end
    return x
end

local function computeLeakRatio(openCount, typeCounts, totalWeighted)
    if not openCount or openCount <= 0 then
        return RC_TempSim.OFFLINE_LEAK_SEALED
    end
    local counts = typeCounts or {}
    if (counts.door or 0) > 0 or (counts.gap or 0) > 0 then
        return RC_TempSim.OFFLINE_LEAK_DOOR
    end
    local cap = RC_TempSim.EXPOSURE_MAX
    local normalized = 0
    if totalWeighted and totalWeighted > 0 and cap > 0 then
        normalized = totalWeighted / cap
    end
    normalized = _clamp01(normalized)
    local leak = RC_TempSim.OFFLINE_LEAK_WINDOW
    return leak * normalized
end

local function leakInfoFromBreaches(breaches, isOpenFn)
    if type(breaches) ~= "table" or not isOpenFn then
        return { leakRatio = RC_TempSim.OFFLINE_LEAK_SEALED }
    end
    local _, openCount, typeCounts, totalWeighted = computeOpenBreachStrengths(breaches, isOpenFn)
    local leakRatio = computeLeakRatio(openCount, typeCounts, totalWeighted)
    return {
        leakRatio = leakRatio,
        openCount = openCount or 0,
        typeCounts = typeCounts or {},
        totalWeighted = totalWeighted or 0,
    }
end

local function applyPreciseHugeRoomSimulation(S, rooms, outdoorTemp)
    if not S or type(rooms) ~= "table" or #rooms == 0 then
        return nil
    end

    local allowed = {}
    for _, node in ipairs(rooms) do
        if node then
            allowed[node] = true
        end
    end
    if not _tableHasEntries(allowed) then return nil end

    local strengths, openCount, typeCounts, totalWeighted = computeOpenBreachStrengthsForRooms(
        S and S.breaches or {},
        breachIsOpen_vanilla,
        allowed
    )

    if openCount > 0 then
        local exposure = {}
        for room, strength in pairs(strengths) do
            exposure[room] = strength
        end

        local distMap = S.hugeInfo and S.hugeInfo.distanceFromPerimeter or nil
        if type(distMap) == "table" and _tableHasEntries(distMap) then
            local expPow = RC_TempSim.HUGE_DISTANCE_EXP or 1.0
            local minFactor = RC_TempSim.HUGE_DISTANCE_MIN_FACTOR or 0.0
            local maxFactor = RC_TempSim.HUGE_DISTANCE_MAX_FACTOR or nil
            for room, value in pairs(exposure) do
                local d = distMap[room]
                local falloff
                if d == nil then
                    falloff = minFactor
                else
                    falloff = 1.0 / ((1 + d) ^ expPow)
                    if maxFactor and falloff > maxFactor then falloff = maxFactor end
                    if minFactor and falloff < minFactor then falloff = minFactor end
                end
                exposure[room] = (value or 0) * (falloff or 0)
            end
        end

        local base = RC_TempSim.BASE_BLEND_PER_STEP
        local ref = RC_TempSim.ROOM_SIZE_REF_SQ or 40
        if not ref or ref <= 0 then ref = 40 end
        local sizeExp = RC_TempSim.SIZE_EXP or 1.0
        local minFactor = RC_TempSim.SIZE_FACTOR_MIN
        local maxFactor = RC_TempSim.SIZE_FACTOR_MAX

        for _, room in ipairs(rooms) do
            local exp = exposure[room]
            if exp and exp > 0 then
                local current = S.temps[room] or outdoorTemp
                local area = (S.roomArea and S.roomArea[room]) or ref
                if not area or area <= 0 then area = ref end
                local ratio = ref / math.max(1, area)
                local sizeFactor = ratio ^ sizeExp
                if sizeFactor < minFactor then sizeFactor = minFactor end
                if sizeFactor > maxFactor then sizeFactor = maxFactor end
                if exp > RC_TempSim.EXPOSURE_MAX then exp = RC_TempSim.EXPOSURE_MAX end
                local alpha = base * sizeFactor * exp
                S.temps[room] = current + alpha * (outdoorTemp - current)
            end
        end
    end

    local leakInfo = {
        leakRatio = computeLeakRatio(openCount, typeCounts, totalWeighted),
        openCount = openCount or 0,
        typeCounts = typeCounts or {},
        totalWeighted = totalWeighted or 0,
    }

    return leakInfo
end

local function computeHistoryStats(entries, fallback)
    local stats = { avg = fallback or 0, min = fallback or 0, max = fallback or 0 }
    if type(entries) ~= "table" or #entries == 0 then
        return stats
    end
    local sum = 0
    local minT, maxT = nil, nil
    local count = 0
    for _, rec in ipairs(entries) do
        if type(rec) == "table" then
            local t = rec.temp
            if t == nil then t = fallback or 0 end
            sum = sum + t
            if minT == nil or t < minT then minT = t end
            if maxT == nil or t > maxT then maxT = t end
            count = count + 1
        end
    end
    if count > 0 then
        stats.avg = sum / count
        stats.min = minT or stats.avg
        stats.max = maxT or stats.avg
    end
    if stats.min == nil then stats.min = stats.avg end
    if stats.max == nil then stats.max = stats.avg end
    return stats
end

local function computeRecentHistoryStats(entries, fallback, windowHours)
    local fallbackTemp = fallback or 0
    local stats = { avg = fallbackTemp, min = fallbackTemp, max = fallbackTemp }
    if type(entries) ~= "table" or #entries == 0 then
        return stats
    end

    local now = (_currentWorldHours and _currentWorldHours()) or 0
    local defaultWindow = RC_TempSim.OUTSIDE_HISTORY_MAX or 24
    local span = windowHours or defaultWindow
    if not span or span <= 0 then span = defaultWindow end
    if span <= 0 then span = 24 end
    local startT = now - span

    local haveAny = false
    local minT, maxT = nil, nil
    for _, rec in ipairs(entries) do
        if type(rec) == "table" then
            local hour = rec.hour or 0
            if hour >= startT then
                local temp = rec.temp
                if temp == nil then temp = fallbackTemp end
                if not haveAny or temp < minT then minT = temp end
                if not haveAny or temp > maxT then maxT = temp end
                haveAny = true
            end
        end
    end

    if not haveAny then
        minT, maxT = fallbackTemp, fallbackTemp
    end

    local avg = _avgOutsideOverWindow(entries, fallbackTemp, span)
    stats.avg = avg or fallbackTemp
    stats.min = minT or stats.avg
    stats.max = maxT or stats.avg
    return stats
end

local function computeOfflineTarget(outdoorTemp, stats, leakRatio, opts)
    local outT = outdoorTemp or 0
    local s = stats or { avg = outT, min = outT, max = outT }
    local ratio = leakRatio or 0
    if ratio < 0 then ratio = 0 elseif ratio > 1 then ratio = 1 end
    local baseline = s.avg or outT
    local offset = RC_TempSim.INDOOR_OFFSET_DEFAULT_C
    local retention = 1 - ratio
    if retention < 0 then retention = 0 end
    baseline = baseline + offset * retention
    local target = outT * ratio + baseline * (1 - ratio)
    local minClamp = s.min or target
    local maxClamp = s.max or target
    if minClamp > maxClamp then
        minClamp, maxClamp = maxClamp, minClamp
    end
    minClamp = minClamp - offset
    maxClamp = maxClamp + offset
    if target < minClamp then target = minClamp end
    if target > maxClamp then target = maxClamp end
    local skipSeedBias = opts and opts.skipSeedBias
    local allowPoweredSeed = false
    if type(opts) == "table" then
        if opts.allowPoweredSeed ~= nil then
            allowPoweredSeed = opts.allowPoweredSeed and true or false
        elseif opts.allowPoweredBaseline ~= nil then
            allowPoweredSeed = opts.allowPoweredBaseline and true or false
        elseif opts.allowPowered ~= nil then
            allowPoweredSeed = opts.allowPowered and true or false
        end
    end
    if not skipSeedBias and RC_TempSim and RC_TempSim.seedTempForRoom then
        local seed = RC_TempSim.seedTempForRoom(outdoorTemp, allowPoweredSeed)
        if seed then
            target = target + (seed - target) * (1 - ratio)
        end
    end
    return target
end

local function normalizeHeaterHours(hours)
    hours = tonumber(hours) or 0
    if hours <= 0 then return 0 end
    local cap = RC_TempSim.OFFLINE_HEATER_RUNTIME_CAP
    if cap and cap > 0 and hours > cap then
        hours = cap
    end
    return hours
end

local function logOfflineDebug(label, fmt, ...)
    if not fmt then return end
    local prefix = "[RC_TempSim][Offline]"
    if label then
        prefix = string.format("%s[%s]", prefix, tostring(label))
    end
    local message
    if select("#", ...) > 0 then
        local ok, formatted = pcall(string.format, fmt, ...)
        message = ok and formatted or fmt
    else
        message = fmt
    end
    -- print(string.format("%s %s", prefix, message))
end

local function summarizeHeaterRecords(heaterRecords)
    if type(heaterRecords) ~= "table" then return "none" end
    local parts = {}
    for idx, entry in ipairs(heaterRecords) do
        if type(entry) == "table" then
            local heat = tonumber(entry.heat) or 0
            local hours
            if entry.unlimited then
                hours = "∞"
            else
                local normalized = entry.hours
                hours = string.format("%.2f", normalized or 0)
            end
            local area = tonumber(entry.roomArea)
            local areaSuffix = ""
            if area and area > 0 then
                areaSuffix = string.format(" area=%.1f", area)
            end
            parts[#parts+1] = string.format("#%d heat=%.2f hours=%s%s", idx, heat, hours, areaSuffix)
        end
    end
    if #parts == 0 then return "none" end
    return table.concat(parts, "; ")
end

local function collectHeatersFromSquare(sq, seen, out, opts)
    if not sq then return out, seen end
    local helper = RCHeaters and RCHeaters.getRuntimeForObject or nil
    if not helper then return out, seen end
    seen = seen or {}
    out = out or {}
    local lists = {}
    if sq.getObjects then lists[#lists+1] = sq:getObjects() end
    if sq.getSpecialObjects then lists[#lists+1] = sq:getSpecialObjects() end
    if sq.getWorldObjects then lists[#lists+1] = sq:getWorldObjects() end
    for _, list in ipairs(lists) do
        if list then
            for idx = 0, list:size() - 1 do
                local obj = list:get(idx)
                if obj and not seen[obj] then
                    seen[obj] = true
                    local runtime = helper(obj)
                    if runtime and runtime.active and runtime.heatValue and runtime.heatValue ~= 0 then
                        local heat = tonumber(runtime.heatValue) or 0
                        if heat ~= 0 then
                            local rawHours = runtime.remainingHours
                            local unlimited = (rawHours == math.huge)
                            local hours = rawHours
                            if unlimited or hours > 0 then
                                local entry = {
                                    heat = heat,
                                    hours = hours,
                                }
                                if unlimited then
                                    entry.unlimited = true
                                end
                                if opts then
                                    local area = tonumber(opts.roomArea)
                                    if area and area > 0 then
                                        entry.roomArea = area
                                    end
                                    if opts.roomKey ~= nil then
                                        entry.roomKey = opts.roomKey
                                    end
                                end
                                out[#out+1] = entry
                            end
                        end
                    end
                end
            end
        end
    end
    return out, seen
end

local function collectActiveHeatersFromSquares(squares, seen, out, opts)
    if not squares then return out or {}, seen end
    seen = seen or {}
    out = out or {}
    for _, sq in ipairs(squares) do
        out, seen = collectHeatersFromSquare(sq, seen, out, opts)
    end
    return out, seen
end

local function computeOfflineHeatBoost(elapsedHours, heaterRecords, context)
    if not heaterRecords or type(heaterRecords) ~= "table" then return 0, 0, 0, 0 end
    if not elapsedHours or elapsedHours <= 0 then return 0, 0, 0, 0 end
    local totalHeatHours = 0
    local maxActive = 0
    local heatedArea = 0
    local countedRooms = {}
    for _, rec in ipairs(heaterRecords) do
        if type(rec) == "table" then
            local heat = tonumber(rec.heat) or 0
            if heat ~= 0 then
                local unlimited = rec.unlimited and true or false
                local active
                if unlimited then
                    active = elapsedHours
                else
                    local hours = rec.hours
                    if hours > elapsedHours then
                        hours = elapsedHours
                    end
                    active = hours
                end
                if active and active > 0 then
                    totalHeatHours = totalHeatHours + heat * active
                    if active > maxActive then
                        maxActive = active
                    end
                    local roomArea = tonumber(rec.roomArea)
                    if roomArea and roomArea > 0 then
                        local key = rec.roomKey
                        if key ~= nil then
                            if not countedRooms[key] then
                                countedRooms[key] = true
                                heatedArea = heatedArea + roomArea
                            end
                        else
                            heatedArea = heatedArea + roomArea
                        end
                    end
                end
            end
        end
    end
    local refArea = context and tonumber(context.refArea) or nil
    if not refArea or refArea <= 0 then
        refArea = RC_TempSim.OFFLINE_HEAT_REF_AREA or RC_TempSim.ROOM_SIZE_REF_SQ or 80
    end
    if not refArea or refArea <= 0 then
        refArea = 80
    end
    local effectiveArea = context and tonumber(context.capacity) or nil
    if heatedArea > 0 then
        if not effectiveArea or effectiveArea <= 0 then
            effectiveArea = heatedArea
        else
            effectiveArea = math.min(effectiveArea, heatedArea)
        end
    end
    if not effectiveArea or effectiveArea <= 0 then
        effectiveArea = refArea
    end
    if totalHeatHours == 0 then
        return 0, maxActive, heatedArea, effectiveArea
    end
    local updateMinutes = tonumber(RC_TempSim.UPDATE_EVERY_MINUTES) or 10
    if updateMinutes <= 0 then updateMinutes = 10 end
    local stepHours = updateMinutes / 60
    if stepHours <= 0 then stepHours = 1 / 60 end
    local baseBlend = RC_TempSim.HEAT_BASE_BLEND_PER_STEP or RC_TempSim.BASE_BLEND_PER_STEP or 0.25
    local stepsPerHour = 1 / stepHours
    local perHourMultiplier = baseBlend * stepsPerHour
    if perHourMultiplier <= 0 then
        return 0, maxActive, heatedArea, effectiveArea
    end
    local efficiency = tonumber(RC_TempSim.OFFLINE_HEAT_EFFICIENCY) or 0
    local base = totalHeatHours * perHourMultiplier * efficiency
    if base == 0 then
        return 0, maxActive, heatedArea, effectiveArea
    end

    local sizeFactor = 1.0
    if context then
        local capacity = effectiveArea
        if refArea and refArea > 0 and capacity and capacity > 0 then
            local ratio = capacity / refArea
            if ratio <= 0 then ratio = 1 end
            local exp = RC_TempSim.OFFLINE_HEAT_SIZE_EXP or 1.0
            local factor = (refArea / capacity) ^ exp
            local minF = RC_TempSim.OFFLINE_HEAT_MIN_FACTOR
            local maxF = RC_TempSim.OFFLINE_HEAT_MAX_FACTOR
            if minF and minF > 0 and factor < minF then factor = minF end
            if maxF and maxF > 0 and factor > maxF then factor = maxF end
            sizeFactor = factor
        end
    end

    local breachFactor = 1.0
    local leakRatio = context and context.leakRatio
    if leakRatio ~= nil then
        leakRatio = tonumber(leakRatio) or 0
        if leakRatio < 0 then leakRatio = 0 elseif leakRatio > 1 then leakRatio = 1 end
        local leakExp = RC_TempSim.OFFLINE_HEAT_BREACH_EXP or 1.0
        breachFactor = (1 - leakRatio) ^ leakExp
        local floor = RC_TempSim.OFFLINE_HEAT_BREACH_MIN
        if floor and floor > 0 and breachFactor < floor then
            breachFactor = floor
        end
    end

    if context and context.breachFactor ~= nil then
        local extra = tonumber(context.breachFactor) or 0
        if extra < 0 then extra = 0 elseif extra > 1 then extra = 1 end
        breachFactor = breachFactor * extra
    end

    return base * sizeFactor * breachFactor, maxActive, heatedArea, effectiveArea
end

local function applyOfflineTemperatureDrift(temps, elapsedHours, outdoorTemp, leakRatio, historyEntries, heaterRecords, opts)
    if not temps or type(temps) ~= "table" then return temps end
    if not elapsedHours or elapsedHours <= 0 then return temps end
    local ratio = tonumber(leakRatio) or 0
    if ratio < 0 then ratio = 0 elseif ratio > 1 then ratio = 1 end
    local stats = computeHistoryStats(historyEntries, outdoorTemp)
    local referenceOut = outdoorTemp
    if opts and opts.preferHistoryAvg and stats and stats.avg then
        referenceOut = stats.avg
    end
    local target = computeOfflineTarget(referenceOut, stats, ratio, opts)
    local baseTarget = target

    local areaMap = opts and opts.roomArea
    local fallbackArea = opts and opts.defaultArea or RC_TempSim.OFFLINE_HEAT_REF_AREA or RC_TempSim.ROOM_SIZE_REF_SQ or 80
    if not fallbackArea or fallbackArea <= 0 then
        fallbackArea = RC_TempSim.ROOM_SIZE_REF_SQ or 40
    end
    if not fallbackArea or fallbackArea <= 0 then
        fallbackArea = 40
    end

    local totalArea, hasNodes = 0, false
    for node,_ in pairs(temps) do
        hasNodes = true
        local area = nil
        if type(areaMap) == "table" then
            area = areaMap[node]
            if type(area) ~= "number" or area <= 0 then
                area = nil
            end
        end
        if not area then area = fallbackArea end
        totalArea = totalArea + area
    end
    if not hasNodes then
        totalArea = fallbackArea
    end

    local context = {
        capacity = totalArea,
        refArea = (opts and opts.refArea) or fallbackArea,
        leakRatio = ratio,
    }
    if opts and opts.breachFactor ~= nil then
        context.breachFactor = opts.breachFactor
    end

    local heatBoost, maxActive, heatedArea, effectiveArea = computeOfflineHeatBoost(elapsedHours, heaterRecords, context)
    target = target + heatBoost
    local maxHeated = RC_TempSim.HEAT_TARGET_MAX_C
    local minHeated = RC_TempSim.HEAT_TARGET_MIN_C
    if maxHeated and target > maxHeated then
        target = maxHeated
    end
    if minHeated and target < minHeated then
        target = minHeated
    end
    local leakLinear = RC_TempSim.OFFLINE_DECAY_LEAK_MULT * ratio
    local leakQuadratic = RC_TempSim.OFFLINE_DECAY_HIGH_LEAK_MULT * (ratio * ratio)
    local rawBase = RC_TempSim.OFFLINE_DECAY_BASE or 0
    local scaledBase = rawBase
    if scaledBase > 0 then
        local sealedExp = RC_TempSim.OFFLINE_DECAY_SEALED_EXP or 1.0
        if sealedExp < 0 then sealedExp = 0 end
        local sealedMin = RC_TempSim.OFFLINE_DECAY_SEALED_MIN or 0.0
        local ratioFactor = ratio ^ sealedExp
        if ratioFactor < 0 then
            ratioFactor = 0
        elseif ratioFactor > 1 then
            ratioFactor = 1
        end
        if sealedMin and sealedMin > 0 and ratioFactor < sealedMin then
            ratioFactor = sealedMin
        end
        scaledBase = scaledBase * ratioFactor
    end
    local rate = scaledBase + leakLinear + leakQuadratic
    local heatRate = 0
    if heatBoost and heatBoost > 0 then
        local perDeg = tonumber(RC_TempSim.OFFLINE_HEAT_RESPONSE_PER_DEG) or 0
        if perDeg > 0 then
            local leakFactor = 1 - ratio
            if leakFactor < 0 then
                leakFactor = 0
            elseif leakFactor > 1 then
                leakFactor = 1
            end
            heatRate = heatBoost * perDeg * leakFactor
            local cap = RC_TempSim.OFFLINE_HEAT_RESPONSE_CAP
            if cap and cap > 0 and heatRate > cap then
                heatRate = cap
            end
            rate = rate + heatRate
        end
    end
    if rate < 0 then rate = 0 end
    local blend = 1 - math.exp(-rate * elapsedHours)
    if blend < 0 then blend = 0 end
    if blend > 1 then blend = 1 end
    local sustained = false
    if maxActive and maxActive > 0 and elapsedHours and elapsedHours > 0 then
        local tolerance = math.min(1/60, elapsedHours * 0.1)
        if tolerance < 1e-3 then tolerance = 1e-3 end
        if maxActive >= (elapsedHours - tolerance) then
            sustained = true
        end
    end
    if opts and opts.debugLabel then
        local decayBase = scaledBase
        local heaterSummary = opts.debugHeaterSummary or summarizeHeaterRecords(heaterRecords)
        local capacityArea = context and context.capacity or effectiveArea
        logOfflineDebug(
            opts.debugLabel,
            "elapsed=%.2f h | outdoor=%.1f C (ref=%.1f) | leak=%.3f | stats(avg=%.2f,min=%.2f,max=%.2f) | base=%.2f + heat=%.2f => target=%.2f | heaters=%s | heatRate=%.3f",
            opts.debugElapsed or elapsedHours,
            outdoorTemp or 0,
            referenceOut or 0,
            ratio,
            stats and stats.avg or 0,
            stats and stats.min or 0,
            stats and stats.max or 0,
            baseTarget,
            heatBoost,
            target,
            heatedArea or 0,
            capacityArea or 0,
            rate,
            decayBase,
            leakLinear,
            leakQuadratic,
            blend,
            sustained and "true" or "false",
            maxActive or 0,
            heaterSummary,
            heatRate
        )
    end
    for node, current in pairs(temps) do
        if type(current) == "number" then
            local desired = target
            if sustained and desired < current then
                local sustainStrength = 1 - ratio
                if sustainStrength < 0 then
                    sustainStrength = 0
                elseif sustainStrength > 1 then
                    sustainStrength = 1
                end
                desired = target + (current - target) * sustainStrength
            end
            temps[node] = current + (desired - current) * blend
        else
            temps[node] = target
        end
    end
    return temps
end

local function applyInternalEqualizationToTemps(temps, graph, closedDoors, roomArea, outdoorTemp)
    if type(temps) ~= "table" then return temps end
    local baseRate = RC_TempSim.INTERNAL_EQ_RATE_PER_EDGE
    if baseRate <= 0 then return temps end
    local capMin = RC_TempSim.CAPACITY_MIN
    local capExp = RC_TempSim.CAPACITY_EXP
    local function tempFor(node)
        local t = temps[node]
        if type(t) ~= "number" then t = outdoorTemp end
        return t or outdoorTemp
    end
    local function capFor(node)
        local area = (roomArea and roomArea[node]) or 1
        local base = math.max(area or 0, capMin)
        return base ^ capExp
    end
    local accum, seen = {}, {}
    local function applyEdges(edgeMap, rate)
        if type(edgeMap) ~= "table" or rate <= 0 then return end
        for a, nbs in pairs(edgeMap) do
            local sa = tostring(a)
            for b,_ in pairs(nbs or {}) do
                if a ~= b then
                    local sb = tostring(b)
                    local key = (sa < sb) and (sa .. "|" .. sb) or (sb .. "|" .. sa)
                    if not seen[key] then
                        seen[key] = true
                        local tA, tB = tempFor(a), tempFor(b)
                        local CA, CB = capFor(a), capFor(b)
                        if CA > 0 and CB > 0 then
                            local flow = rate * (tB - tA)
                            accum[a] = (accum[a] or 0) + ( flow / CA)
                            accum[b] = (accum[b] or 0) + (-flow / CB)
                        end
                    end
                end
            end
        end
    end
    applyEdges(graph or {}, baseRate)
    local closedRate = baseRate * RC_TempSim.CLOSED_DOOR_EQ_RATE_MULT
    if closedRate > 0 then applyEdges(closedDoors or {}, closedRate) end
    for node, delta in pairs(accum) do
        local current = tempFor(node)
        temps[node] = current + delta
    end
    return temps
end

local function _samePlayerBuilding(a, b)
    if not a or not b then return false end
    if a == b then return true end
    local ida = a.id
    local idb = b.id
    return (ida ~= nil and idb ~= nil and ida == idb)
end

local function distancesFromOne(graph, src)
    local d, q, head = {}, { src }, 1
    d[src] = 0
    while head <= #q do
        local cur = q[head]; head = head + 1
        local cd  = d[cur] or 0
        for nb,_ in pairs(graph[cur] or {}) do
            if d[nb] == nil then
                d[nb] = cd + 1
                q[#q+1] = nb
            end
        end
    end
    return d
end

---------------------------------------------------------------------------
-- Heat source registry and helpers
---------------------------------------------------------------------------

RC_TempSim._heatSourceDefs  = RC_TempSim._heatSourceDefs  or {}
RC_TempSim._heatSourceOrder = RC_TempSim._heatSourceOrder or {}
RC_TempSim._nextHeatId      = RC_TempSim._nextHeatId      or 1

local function _normalizeStringSet(input)
    if not input then return nil, nil end
    local set, list = {}, {}
    local function add(value)
        if value and value ~= "" and not set[value] then
            set[value] = true
            list[#list+1] = value
        end
    end
    if type(input) == "table" then
        for k, v in pairs(input) do
            if type(k) == "string" and v then
                add(k)
            elseif type(v) == "string" then
                add(v)
            end
        end
    elseif type(input) == "string" then
        add(input)
    end
    if #list == 0 then return nil, nil end
    table.sort(list)
    return set, list
end

local function _removeHeatOrder(id)
    for idx = #RC_TempSim._heatSourceOrder, 1, -1 do
        if RC_TempSim._heatSourceOrder[idx] == id then
            table.remove(RC_TempSim._heatSourceOrder, idx)
        end
    end
end

local function _callHeatEntry(entry, kind, fn, ...)
    if type(fn) ~= "function" then return true, nil end
    local ok, res = pcall(fn, ...)
    if not ok then
        entry._errorFlags = entry._errorFlags or {}
        if not entry._errorFlags[kind] then
            entry._errorFlags[kind] = true
            -- print(string.format("[RC_TempSim] heat source '%s' %s error: %s", tostring(entry.id or "?"), kind, tostring(res)))
        end
        return false, nil
    end
    return true, res
end

local function _heatEntryMatches(entry, obj, spriteName, objectName)
    if entry.matchFn then
        local ok, result = _callHeatEntry(entry, "match", entry.matchFn, obj, entry)
        if not ok or not result then return false, nil end
        if type(result) == "number" then
            return true, result
        end
        return true, nil
    end

    if entry.className then
        if not (instanceof and instanceof(obj, entry.className)) then
            return false, nil
        end
    end

    local matched = true
    if entry.spriteSet or entry.spritePrefix or entry.objectNameSet then
        matched = false
        if entry.spriteSet and spriteName and entry.spriteSet[spriteName] then matched = true end
        if not matched and entry.spritePrefix and spriteName and spriteName:sub(1, #entry.spritePrefix) == entry.spritePrefix then
            matched = true
        end
        if not matched and entry.objectNameSet and objectName and entry.objectNameSet[objectName] then
            matched = true
        end
    end

    return matched, nil
end

local function _evaluateHeatEntry(entry, obj, spriteName, objectName)
    local matched, overrideHeat = _heatEntryMatches(entry, obj, spriteName, objectName)
    if not matched then return 0, false end

    if entry.activeFn then
        local okActive, isActive = _callHeatEntry(entry, "isActive", entry.activeFn, obj, entry)
        if not okActive or not isActive then return 0, false end
    end

    local heatValue = overrideHeat
    if heatValue == nil then
        if entry.heatFn then
            local okHeat, hv = _callHeatEntry(entry, "heat", entry.heatFn, obj, entry)
            if not okHeat then return 0, false end
            heatValue = hv
        else
            heatValue = entry.heat
        end
    end

    heatValue = tonumber(heatValue)
    if not heatValue or heatValue == 0 then return 0, false end

    return heatValue, true
end

local function _collectHeatFromObject(obj)
    if not obj then return 0, false end
    if not RC_TempSim._heatSourceOrder or #RC_TempSim._heatSourceOrder == 0 then return 0, false end

    local spriteName
    local objectName
    local total, matchedAny = 0, false

    for _, id in ipairs(RC_TempSim._heatSourceOrder) do
        local entry = RC_TempSim._heatSourceDefs[id]
        if entry then
            if (entry.spriteSet or entry.spritePrefix) and spriteName == nil then
                local sprite = obj.getSprite and obj:getSprite() or nil
                spriteName = sprite and sprite.getName and sprite:getName() or nil
            end
            if entry.objectNameSet and objectName == nil then
                objectName = obj.getObjectName and obj:getObjectName() or nil
                if not objectName and obj.getName then objectName = obj:getName() end
            end

            local amount, matched = _evaluateHeatEntry(entry, obj, spriteName, objectName)
            if matched then
                matchedAny = true
                total = total + amount
            end
        end
    end

    return total, matchedAny
end

function RC_TempSim.registerHeatSource(def)
    if type(def) ~= "table" then return nil end

    local heatValue = def.heat or def.heatValue or def.delta or def.deltaC
    local heatFn = def.heatFn
    if type(heatValue) == "function" then
        heatFn = heatValue
        heatValue = nil
    end
    if heatFn and type(heatFn) ~= "function" then heatFn = nil end
    if heatValue ~= nil then heatValue = tonumber(heatValue) end
    if heatValue == nil and not heatFn then return nil end

    local id = def.id or def.key
    if not id then
        id = string.format("heat_%03d", RC_TempSim._nextHeatId)
        RC_TempSim._nextHeatId = RC_TempSim._nextHeatId + 1
    end

    local spriteSet, spriteList = _normalizeStringSet(def.sprites or def.sprite)
    local objectNameSet, objectNameList = _normalizeStringSet(def.objectNames or def.objectName)

    local entry = {
        id = id,
        heat = heatValue,
        heatFn = heatFn,
        matchFn = (type(def.match) == "function" and def.match) or (type(def.matchFn) == "function" and def.matchFn) or nil,
        activeFn = (type(def.isActive) == "function" and def.isActive) or (type(def.activeFn) == "function" and def.activeFn) or nil,
        className = def.class or def.className,
        spriteSet = spriteSet,
        spritePrefix = def.spritePrefix,
        objectNameSet = objectNameSet,
        _spritesList = spriteList,
        _objectNamesList = objectNameList,
        description = def.description,
    }

    RC_TempSim._heatSourceDefs[id] = entry
    _removeHeatOrder(id)
    RC_TempSim._heatSourceOrder[#RC_TempSim._heatSourceOrder + 1] = id

    return id
end

function RC_TempSim.unregisterHeatSource(id)
    if not id then return end
    RC_TempSim._heatSourceDefs[id] = nil
    _removeHeatOrder(id)
end

function RC_TempSim.clearRegisteredHeatSources()
    RC_TempSim._heatSourceDefs = {}
    RC_TempSim._heatSourceOrder = {}
end

function RC_TempSim.getRegisteredHeatSources()
    local out = {}
    for _, id in ipairs(RC_TempSim._heatSourceOrder) do
        local entry = RC_TempSim._heatSourceDefs[id]
        if entry then
            out[#out+1] = {
                id = entry.id,
                class = entry.className,
                spritePrefix = entry.spritePrefix,
                sprites = entry._spritesList,
                objectNames = entry._objectNamesList,
                heat = entry.heat,
                hasHeatFn = entry.heatFn ~= nil,
                hasMatchFn = entry.matchFn ~= nil,
                description = entry.description,
            }
        end
    end
    return out
end

local function buildingKey(def)
    local okX = def and def.getX and def:getX()
    local okY = def and def.getY and def:getY()
    local okW = def and def.getW and def:getW()
    local okH = def and def.getH and def:getH()
    if okX and okY and okW and okH then
        return string.format("x%d:y%d:w%d:h%d", okX, okY, okW, okH)
    end
    return tostring(def)
end

function RC_TempSim.seedTempForRoom(outdoorC, opts)
    local outC = outdoorC or 0.0
    local allowPowered = false
    if type(opts) == "boolean" then
        allowPowered = opts
    elseif type(opts) == "table" then
        if opts.allowPowered ~= nil then
            allowPowered = opts.allowPowered and true or false
        elseif opts.allowPoweredBaseline ~= nil then
            allowPowered = opts.allowPoweredBaseline and true or false
        elseif opts.allowPoweredSeed ~= nil then
            allowPowered = opts.allowPoweredSeed and true or false
        end
    end

    local powered = false
    if allowPowered and type(_trackWorldPowerStatus) == "function" then
        local ok, pow = pcall(_trackWorldPowerStatus)
        if ok and type(pow) == "boolean" then
            powered = pow
        end
    end

    if powered then
        local warmOut = -15.0
        local coldOut = -30.0
        local maxC    = 22.0
        local minC    = 15.0

        if outC >= warmOut then
            return maxC
        end
        if outC <= coldOut then
            return minC
        end

        local span = warmOut - coldOut
        if span > 0 then
            local t = (outC - coldOut) / span
            return minC + (maxC - minC) * t
        end
        return maxC
    end

    return outC + RC_TempSim.SEED_BIAS_C
end

---------------------------------------------------------------------------
-- VANILLA back-end (RC_RoomLogic)
---------------------------------------------------------------------------

local function safeRoomName_vanilla(rd)
    if not rd then return "<nil>" end
    if RC_RoomLogic and RC_RoomLogic.roomName then
        local ok, n = pcall(RC_RoomLogic.roomName, rd)
        if ok and n then return n end
    end
    return tostring(rd)
end

local function roomKey_vanilla(rd, cache)
    if not rd then return nil end
    local z = (rd.getZ and rd:getZ()) or 0
    local name = safeRoomName_vanilla(rd)
    local squares = RC_RoomLogic.getRoomSquares(cache, rd) or {}
    local sx, sy, n = 0, 0, 0
    for _, sq in ipairs(squares) do
        if sq then sx = sx + sq:getX(); sy = sy + sq:getY(); n = n + 1 end
    end
    local cx = (n > 0) and math.floor(sx / n + 0.5) or 0
    local cy = (n > 0) and math.floor(sy / n + 0.5) or 0
    return string.format("%s|z%d|cx%d|cy%d", name, z, cx, cy)
end

-- RC_TempSim.lua
-- REPLACE this whole function (keep the same signature)
-- VANILLA loader: powered huge = powered curve; unpowered huge = 24h outside average
local function loadTemps_vanilla(def, roomDefs, seedZ, outsideC, leakInfo, graph, closed, roomArea)
    recordOutsideTemperature()
    local store = md_get()
    local bkey  = buildingKey(def)
    local rec   = store[bkey]
    local temps = {}

    -- Use canonical cache at z=0 so keys align with persist_vanilla
    local cache = RC_RoomLogic.ensureHugeCache(def, roomDefs, 0)

    local historyEntries = _cloneOutsideHistoryEntries()
    local nowHours = _currentWorldHours()
    local leakRatio = (leakInfo and leakInfo.leakRatio) or RC_TempSim.OFFLINE_LEAK_SEALED

    -- helpers (scoped)
    local function isHugeByCount()
        local n = (roomDefs and roomDefs.size and roomDefs:size()) or 0
        return n >= (RC_TempSim.HUGE_OFFLINE_MIN_ROOMS or 60)
    end

    local function avgOutside(windowHours)
        if type(_avgOutsideOverWindow) == "function" then
            return _avgOutsideOverWindow(historyEntries, outsideC, windowHours)
        end
        return outsideC
    end

    local function isGridPowered()
        if type(_trackWorldPowerStatus) == "function" then
            local ok, res = pcall(_trackWorldPowerStatus)
            if ok then
                if type(res) == "table" then return res[1] and true or false end
                return res and true or false
            end
        end
        local world = getWorld and getWorld() or nil
        if world and world.isHydroPowerOn then
            local ok, v = pcall(function() return world:isHydroPowerOn() end)
            return ok and (v == true) or false
        end
        return false
    end

    -- 1) Load from ModData (and advance offline if needed)
    if rec and rec.rooms then
        local huge = isHugeByCount()

        local function computeHugeAggregate(tempMap)
            if not huge then return nil, 0 end
            local sum, weight = 0, 0
            for i = 0, roomDefs:size() - 1 do
                local rd = roomDefs:get(i)
                if rd then
                    local current = tempMap and tempMap[rd] or nil
                    if type(current) ~= "number" then
                        current = RC_TempSim.seedTempForRoom(outsideC, true)
                    end
                    local area = roomArea and roomArea[rd] or 40
                    area = tonumber(area) or 40
                    if area < 1 then area = 1 end
                    sum = sum + current * area
                    weight = weight + area
                end
            end
            if weight > 0 then
                return sum / weight, weight
            end
            return nil, weight
        end

        local function assignHugeTempToRooms(value)
            if not huge or type(value) ~= "number" then return false end
            local changed = false
            for i = 0, roomDefs:size() - 1 do
                local rd = roomDefs:get(i)
                if rd then
                    temps[rd] = value
                    local rkey = roomKey_vanilla(rd, cache)
                    if rec and rec.rooms and rkey then
                        if rec.rooms[rkey] ~= value then
                            rec.rooms[rkey] = value
                            changed = true
                        end
                    end
                end
            end
            return changed
        end

        for i = 0, roomDefs:size() - 1 do
            local rd = roomDefs:get(i)
            if rd then
                local rkey = roomKey_vanilla(rd, cache)
                local t    = rkey and rec.rooms[rkey] or nil
                if t == nil then t = RC_TempSim.seedTempForRoom(outsideC, huge) end
                temps[rd] = t
            end
        end

        local buildingTemp, totalArea = computeHugeAggregate(temps)
        local needsTransmit = false
        local storedHours = rec.worldHours or nowHours
        local elapsed = nowHours - storedHours
        if elapsed and elapsed > 0 then
            if huge then
                local powered = isGridPowered()
                local outAvg  = avgOutside(math.min(elapsed, RC_TempSim.HUGE_OFFLINE_AVG_WINDOW_HOURS or 24)) -- last ≤24h
                local buildingTarget =
                    powered and computePoweredHugeIndoorTarget and computePoweredHugeIndoorTarget(outAvg)
                    or outAvg

                local baseTemp = buildingTemp or RC_TempSim.seedTempForRoom(outsideC, true)
                local basePerHr = RC_TempSim.HUGE_OFFLINE_MAX_DELTA_PER_HOUR or 0.25
                local cap = basePerHr * math.max(0, elapsed)
                if totalArea and totalArea > 0 then
                    local areaNorm = math.min(totalArea / 100.0, 1.0)
                    cap = cap * (0.6 + 0.4 * areaNorm)
                end
                local delta = (buildingTarget or baseTemp) - baseTemp
                if delta >  cap then delta =  cap end
                if delta < -cap then delta = -cap end
                buildingTemp = baseTemp + delta
            else
                local usedLeak = rec.leakRatio or leakRatio
                local driftOpts = {
                    roomArea = roomArea,
                    defaultArea = RC_TempSim.ROOM_SIZE_REF_SQ,
                }
                local debugLabel = string.format("vanilla:%s", tostring(bkey))
                local heaterSummary = summarizeHeaterRecords(rec.heaters)
                logOfflineDebug(debugLabel, "resuming cached building (elapsed %.2f h, leak %.3f) | heaters=%s", elapsed, usedLeak, heaterSummary)
                driftOpts.debugLabel = debugLabel
                driftOpts.debugElapsed = elapsed
                driftOpts.debugOutdoor = outsideC
                driftOpts.debugLeakRatio = usedLeak
                driftOpts.debugHeaterSummary = heaterSummary
                applyOfflineTemperatureDrift(temps, elapsed, outsideC, usedLeak, historyEntries, rec.heaters, driftOpts)
                rec.leakRatio = usedLeak
                rec.heaters   = nil
            end
            rec.worldHours = nowHours
            rec.version    = rec.version or 3
            needsTransmit = true
        end

        if huge then
            local normalized = buildingTemp or RC_TempSim.seedTempForRoom(outsideC, true)
            local updated = assignHugeTempToRooms(normalized)
            if updated then needsTransmit = true end
        end

        if needsTransmit then
            store[bkey] = rec
            md_transmit()
        end

        applyInternalEqualizationToTemps(temps, graph, closed, roomArea, outsideC)
        return temps, true
    end

    -- 2) First visit (no prior record)
    local target
    local huge = isHugeByCount()
    local powered = isGridPowered()

    if huge and powered and computePoweredHugeIndoorTarget then
        -- powered huge: seed to powered indoor target (22↘15 curve)
        target = computePoweredHugeIndoorTarget(outsideC)  -- unchanged behavior
    elseif huge and not powered then
        -- UNPOWERED huge: seed to the 24h average outdoor temperature
        target = avgOutside(RC_TempSim.HUGE_OFFLINE_AVG_WINDOW_HOURS or 24)
    else
        -- small/medium: keep original baseline
        local stats = computeHistoryStats(historyEntries, outsideC)
        target = computeOfflineTarget(outsideC, stats, leakRatio)
    end

    for i = 0, roomDefs:size() - 1 do
        local rd = roomDefs:get(i)
        if rd then temps[rd] = target end
    end

    applyInternalEqualizationToTemps(temps, graph, closed, roomArea, outsideC)
    return temps, false
end

local function collectActiveHeaters_vanilla(S)
    if not (S and S.currentBuildingDef and S.rooms) then return {} end
    if not (RCHeaters and RCHeaters.getRuntimeForObject) then return {} end
    local entries, seen = {}, {}
    local cacheByZ = {}
    for i = 0, S.rooms:size() - 1 do
        local rd = S.rooms:get(i)
        if rd then
            local z = rd:getZ() or 0
            local cache = cacheByZ[z]
            if not cache then
                cache = RC_RoomLogic.ensureHugeCache(S.currentBuildingDef, S.rooms, z)
                cacheByZ[z] = cache
            end
            local squares = RC_RoomLogic.getRoomSquares(cache, rd) or {}
            local area = (S.roomArea and S.roomArea[rd]) or #squares
            local opts = {
                roomKey = roomKey_vanilla(rd, cache) or tostring(rd),
                roomArea = area,
            }
            entries, seen = collectActiveHeatersFromSquares(squares, seen, entries, opts)
        end
    end
    return entries
end

local function breaches_vanilla(def, roomDefs)
    local zSet = {}
    for i = 0, roomDefs:size() - 1 do
        local rd = roomDefs:get(i)
        if rd and rd.getZ then zSet[rd:getZ()] = true end
    end
    local breaches = {}
    for z,_ in pairs(zSet) do
        local cache = RC_RoomLogic.ensureHugeCache(def, roomDefs, z)
        RC_RoomLogic.ensureHugePerimeterForAll(cache)
        for _, rec in ipairs(cache.outsideDoors or {})   do breaches[#breaches+1] = rec end
        for _, rec in ipairs(cache.outsideWindows or {}) do breaches[#breaches+1] = rec end
        for _, rec in ipairs(cache.outsideCurtains or {}) do breaches[#breaches+1] = rec end
        for _, rec in ipairs(cache.outsideGaps or {})    do breaches[#breaches+1] = rec end
    end
    return breaches
end

local function breaches_player(building)
    local out = {}
    for _, rec in ipairs(building.exteriorDoors or {})   do out[#out+1] = rec end
    for _, rec in ipairs(building.exteriorWindows or {}) do out[#out+1] = rec end
    for _, rec in ipairs(building.exteriorCurtains or {}) do out[#out+1] = rec end
    for _, rec in ipairs(building.exteriorGaps or {})    do out[#out+1] = rec end
    return out
end

local function breachIsOpen_player(rec)
    local recType = _breachType(rec)
    if not recType then return false end

    if recType == "gap" then
        local cell = getCell and getCell() or nil
        if not cell then return false end
        local sq = cell:getGridSquare(rec.x, rec.y, rec.z)
        if not sq then return false end

        local dir = rec.edge
        local nSq
        if     dir == "north" and sq.getN then nSq = sq:getN()
        elseif dir == "west"  and sq.getW then nSq = sq:getW()
        elseif dir == "south" and sq.getS then nSq = sq:getS()
        elseif dir == "east"  and sq.getE then nSq = sq:getE() end
        if not nSq or (nSq.getZ and nSq:getZ() ~= (rec.z or 0)) then return false end

        local ok, pass = pcall(RC_PlayerRoomLogic.edgeIsPassable, sq, dir, nSq)
        return ok and pass or false
    end

    if recType == "door" then
        return RC_PlayerRoomLogic.isDoorOpen(rec.door)
    elseif recType == "curtain" then
        return false
    elseif recType == "window" then
        local win = rec.window
        local open = false
        local smashed = false
        if win then
            open = RC_PlayerRoomLogic.isWindowOpen(win) or false
            smashed = RC_PlayerRoomLogic.isWindowSmashed(win) or false
            if open or smashed then return true end
        end
        if rec.open ~= nil then open = open or rec.open end
        if rec.smashed ~= nil then smashed = smashed or rec.smashed end
        return open or smashed
    end
    return false
end

local function _refreshLeakInfoForPersistence(S)
    if not S then return nil end

    local leakInfo
    if S.currentBuildingDef and S.rooms then
        S.breaches = breaches_vanilla(S.currentBuildingDef, S.rooms)
        leakInfo = leakInfoFromBreaches(S.breaches, breachIsOpen_vanilla)
    elseif S.playerBuilding then
        S.breaches = breaches_player(S.playerBuilding)
        leakInfo = leakInfoFromBreaches(S.breaches, breachIsOpen_player)
    end

    if leakInfo then
        S.lastLeakInfo = leakInfo
        return leakInfo
    end

    return S.lastLeakInfo
end

local function persist_vanilla(S)
    if not S or not S.currentBuildingDef or not S.rooms or not S.temps then return end
    local store = md_get()
    local bkey  = buildingKey(S.currentBuildingDef)
    local cache = RC_RoomLogic.ensureHugeCache(S.currentBuildingDef, S.rooms, 0)

    local roomsOut = {}
    for i = 0, S.rooms:size() - 1 do
        local rd = S.rooms:get(i)
        if rd then
            local rkey = roomKey_vanilla(rd, cache)
            if rkey then roomsOut[rkey] = S.temps[rd] end
        end
    end

    local rec = store[bkey] or {}
    rec.version    = 3
    rec.worldHours = _currentWorldHours()
    rec.rooms      = roomsOut
    local leakInfo = _refreshLeakInfoForPersistence(S)
    rec.leakRatio  = (leakInfo and leakInfo.leakRatio) or rec.leakRatio or RC_TempSim.OFFLINE_LEAK_SEALED
    local heaters = collectActiveHeaters_vanilla(S)
    if heaters and #heaters > 0 then
        rec.heaters = heaters
    else
        rec.heaters = nil
    end
    store[bkey] = rec
    md_transmit()
end

local function buildGraph_vanilla(def, roomDefs, seedZ)
    local graph, closed = {}, {}
    local function addClosed(a, b)
        if not a or not b then return end
        closed[a] = closed[a] or {}
        closed[b] = closed[b] or {}
        closed[a][b] = true; closed[b][a] = true
    end
    local function doorBetween(rsq, dir, otherSq)
        if not rsq then return nil end
        if dir == "north" then
            local d = rsq.getIsoDoor and rsq:getIsoDoor() or nil
            if d and RC_RoomLogic.isDoorNorth(d) == true then return d end
        elseif dir == "west" then
            local d = rsq.getIsoDoor and rsq:getIsoDoor() or nil
            if d and RC_RoomLogic.isDoorNorth(d) == false then return d end
        elseif dir == "south" then
            local sq = otherSq
            local d = sq and sq.getIsoDoor and sq:getIsoDoor() or nil
            if d and RC_RoomLogic.isDoorNorth(d) == true then return d end
        elseif dir == "east" then
            local sq = otherSq
            local d = sq and sq.getIsoDoor and sq:getIsoDoor() or nil
            if d and RC_RoomLogic.isDoorNorth(d) == false then return d end
        end
        return nil
    end
    local cache = RC_RoomLogic.ensureHugeCache(def, roomDefs, seedZ)
    for i = 0, roomDefs:size() - 1 do
        local rd = roomDefs:get(i)
        if rd then graph[rd] = graph[rd] or {}; closed[rd] = closed[rd] or {} end
    end
    for i = 0, roomDefs:size() - 1 do
        local rd = roomDefs:get(i)
        if rd then
            for _, nb in ipairs(RC_RoomLogic.getPassableNeighbors(cache, rd, rd:getZ())) do
                graph[rd][nb] = true; graph[nb] = graph[nb] or {}; graph[nb][rd] = true
            end
            for _, nb in ipairs(RC_RoomLogic.getStairsNeighbors(cache, rd)) do
                graph[rd][nb] = true; graph[nb] = graph[nb] or {}; graph[nb][rd] = true
            end
            local squares = RC_RoomLogic.getRoomSquares(cache, rd) or {}
            for _, rsq in ipairs(squares) do
                if rsq then
                    local function consider(dir, getNeighbor)
                        local nbSq = getNeighbor and getNeighbor(rsq) or nil
                        if not (nbSq and nbSq.getZ and nbSq:getZ() == rsq:getZ()) then return end
                        local nbRoom = nbSq:getRoom()
                        local nbDef = nbRoom and (cache.byIso[nbRoom] or (nbRoom.getRoomDef and nbRoom:getRoomDef())) or nil
                        if nbDef and nbDef ~= rd then
                            local pass
                            if dir == "north" then
                                pass = RC_RoomLogic.edgeIsPassable(rsq, "north", nbSq)
                            elseif dir == "west" then
                                pass = RC_RoomLogic.edgeIsPassable(rsq, "west", nbSq)
                            elseif dir == "south" then
                                pass = RC_RoomLogic.edgeIsPassable(rsq, "south", nbSq)
                            elseif dir == "east" then
                                pass = RC_RoomLogic.edgeIsPassable(rsq, "east", nbSq)
                            end
                            if not pass and doorBetween(rsq, dir, nbSq) then
                                addClosed(rd, nbDef)
                            end
                        end
                    end
                    if rsq.getN then consider("north", function(s) return s:getN() end) end
                    if rsq.getW then consider("west",  function(s) return s:getW() end) end
                    if rsq.getS then consider("south", function(s) return s:getS() end) end
                    if rsq.getE then consider("east",  function(s) return s:getE() end) end
                end
            end
        end
    end
    return graph, closed
end

local function computeAreas_vanilla(def, roomDefs, seedZ)
    local areas = {}
    local cache = RC_RoomLogic.ensureHugeCache(def, roomDefs, seedZ)
    for i = 0, roomDefs:size() - 1 do
        local rd = roomDefs:get(i)
        if rd then
            local squares = RC_RoomLogic.getRoomSquares(cache, rd) or {}
            areas[rd] = #squares
        end
    end
    return areas
end

local function invalidatePerimeters_vanilla(def, roomDefs)
    if RC_RoomLogic and RC_RoomLogic.markHugeScanDirty then
        RC_RoomLogic.markHugeScanDirty(def, roomDefs, { structural = false })
    end
end

---------------------------------------------------------------------------
-- PLAYER back-end (RC_PlayerRoomLogic)
---------------------------------------------------------------------------

local function regionFromSquare_player(sq)
    local ok, reg = pcall(function() return sq and sq.getIsoWorldRegion and sq:getIsoWorldRegion() end)
    if ok and reg and reg.isPlayerRoom then
        local ok2, isPR = pcall(function() return reg:isPlayerRoom() end)
        if ok2 and isPR then return reg end
    end
    return nil
end

local function safeRoomName_player(building, region)
    if RC_PlayerRoomLogic and RC_PlayerRoomLogic.GetRoomName then
        local n = RC_PlayerRoomLogic.GetRoomName(building, region)
        if n then return n end
    end
    return RC_PlayerRoomLogic and RC_PlayerRoomLogic.roomName and RC_PlayerRoomLogic.roomName(region) or tostring(region)
end

local function makeAnchorMap_player(building)
    local m = {}
    for _, r in ipairs(building.rooms or {}) do
        if r and r.region then
            m[r.region] = { z = r.z or 0, ax = r.anchorX or 0, ay = r.anchorY or 0 }
        end
    end
    return m
end

local function roomKey_player(region, anchorMap)
    local a = anchorMap[region]
    local z = (region and region.getZ and region:getZ()) or (a and a.z) or 0
    local ax, ay = (a and a.ax) or 0, (a and a.ay) or 0
    return string.format("PR|z%d|ax%d|ay%d", z, ax, ay)
end

local function loadTemps_player(building, outsideC, leakInfo, graph, closed, roomArea)
    recordOutsideTemperature()
    local store = md_get()
    local bkey  = building.id or tostring(building)
    local rec   = store[bkey]
    local temps = {}
    local anchorMap = makeAnchorMap_player(building)
    local historyEntries = _cloneOutsideHistoryEntries()
    local nowHours = _currentWorldHours()
    local leakRatio = (leakInfo and leakInfo.leakRatio) or RC_TempSim.OFFLINE_LEAK_SEALED

    if rec and rec.rooms then
        for _, reg in ipairs(building.regions or {}) do
            local rkey = roomKey_player(reg, anchorMap)
            local t    = rkey and rec.rooms[rkey] or nil
            if t == nil then t = RC_TempSim.seedTempForRoom(outsideC) end
            temps[reg] = t
        end

        local storedHours = rec.worldHours or nowHours
        local elapsed = nowHours - storedHours
        local usedLeak = rec.leakRatio or leakRatio
        if elapsed and elapsed > 0 then
            local driftOpts = {
                roomArea = roomArea,
                defaultArea = RC_TempSim.ROOM_SIZE_REF_SQ,
            }
            local debugLabel = string.format("player:%s", tostring(bkey))
            local heaterSummary = summarizeHeaterRecords(rec.heaters)
            logOfflineDebug(debugLabel, "resuming cached building (elapsed %.2f h, leak %.3f) | heaters=%s", elapsed, usedLeak, heaterSummary)
            driftOpts.debugLabel = debugLabel
            driftOpts.debugElapsed = elapsed
            driftOpts.debugOutdoor = outsideC
            driftOpts.debugLeakRatio = usedLeak
            driftOpts.debugHeaterSummary = heaterSummary
            applyOfflineTemperatureDrift(temps, elapsed, outsideC, usedLeak, historyEntries, rec.heaters, driftOpts)
            rec.worldHours = nowHours
            rec.leakRatio = usedLeak
            rec.heaters = nil
            rec.version = rec.version or 3
            store[bkey] = rec
            md_transmit()
        end
        applyInternalEqualizationToTemps(temps, graph, closed, roomArea, outsideC)
        return temps, true
    end

    local stats = computeRecentHistoryStats(historyEntries, outsideC, RC_TempSim.OUTSIDE_HISTORY_MAX or 24)
    local target = computeOfflineTarget(outsideC, stats, leakRatio)
    for _, reg in ipairs(building.regions or {}) do
        temps[reg] = target
    end
    applyInternalEqualizationToTemps(temps, graph, closed, roomArea, outsideC)
    return temps, false
end

local function _squaresForPlayerRegion(building, region)
    local z = (region and region.getZ and region:getZ()) or 0
    local zCache = building and building._perZCache and building._perZCache[z] or nil
    local seedSq = zCache and zCache.seedSqByRegion and zCache.seedSqByRegion[region] or nil
    if not seedSq then return {} end
    local q, head = { seedSq }, 1
    local visited = { [seedSq] = true }
    local out = {}
    local limit = RC_TempSim.HEAT_SCAN_LIMIT
    while head <= #q and #out < limit do
        local sq = q[head]; head = head + 1
        out[#out+1] = sq
        local function push(ns)
            if ns and not visited[ns] and ns:getZ() == z and regionFromSquare_player(ns) == region then
                visited[ns] = true
                q[#q+1] = ns
            end
        end
        if sq and sq.getN then push(sq:getN()) end
        if sq and sq.getS then push(sq:getS()) end
        if sq and sq.getW then push(sq:getW()) end
        if sq and sq.getE then push(sq:getE()) end
    end
    return out
end

local function collectActiveHeaters_player(S)
    if not (S and S.playerBuilding) then return {} end
    if not (RCHeaters and RCHeaters.getRuntimeForObject) then return {} end
    local entries, seen = {}, {}
    local anchorMap = makeAnchorMap_player(S.playerBuilding)
    for _, reg in ipairs(S.playerRegions or {}) do
        local squares = _squaresForPlayerRegion(S.playerBuilding, reg)
        local area = (S.roomArea and S.roomArea[reg]) or #squares
        local opts = {
            roomKey = roomKey_player(reg, anchorMap) or tostring(reg),
            roomArea = area,
        }
        entries, seen = collectActiveHeatersFromSquares(squares, seen, entries, opts)
    end
    return entries
end

local function persist_player(S)
    if not S or not S.playerBuilding or not S.playerRegions or not S.temps then return end
    local store = md_get()
    local bkey  = S.playerBuilding.id or tostring(S.playerBuilding)
    local anchorMap = makeAnchorMap_player(S.playerBuilding)

    local roomsOut = {}
    for _, reg in ipairs(S.playerRegions) do
        local rkey = roomKey_player(reg, anchorMap)
        if rkey then roomsOut[rkey] = S.temps[reg] end
    end

    local rec = store[bkey] or {}
    rec.version    = 3
    rec.worldHours = _currentWorldHours()
    rec.rooms      = roomsOut
    local leakInfo = _refreshLeakInfoForPersistence(S)
    rec.leakRatio  = (leakInfo and leakInfo.leakRatio) or rec.leakRatio or RC_TempSim.OFFLINE_LEAK_SEALED
    local heaters = collectActiveHeaters_player(S)
    if heaters and #heaters > 0 then
        rec.heaters = heaters
    else
        rec.heaters = nil
    end
    store[bkey] = rec
    md_transmit()
end

local function buildGraph_player(building)
    local g, closed = {}, {}
    local function add(a,b)
        if a and b then g[a]=g[a] or {}; g[b]=g[b] or {}; g[a][b]=true; g[b][a]=true end
    end
    local function addClosed(a,b)
        if a and b then closed[a]=closed[a] or {}; closed[b]=closed[b] or {}; closed[a][b]=true; closed[b][a]=true end
    end
    for _, r in ipairs(building.regions or {}) do
        g[r] = g[r] or {}; closed[r] = closed[r] or {}
        local z = (r.getZ and r:getZ()) or 0
        local zCache = building._perZCache and building._perZCache[z] or nil
        if zCache then
            for _, nb in ipairs(RC_PlayerRoomLogic.getPassableNeighbors(zCache, r, z) or {}) do add(r, nb) end
            for _, nb in ipairs(RC_PlayerRoomLogic.getStairsNeighbors(zCache, r) or {}) do add(r, nb) end
            for _, nb in ipairs(RC_PlayerRoomLogic.getDoorNeighbors(zCache, r, z) or {}) do
                if not (g[r] and g[r][nb]) then addClosed(r, nb) end
            end
        end
    end
    return g, closed
end

local function computeAreas_player(building)
    local areas = {}
    for _, reg in ipairs(building.regions or {}) do
        local z = (reg.getZ and reg:getZ()) or 0
        local zCache = building._perZCache and building._perZCache[z] or nil
        local seedSq = zCache and zCache.seedSqByRegion and zCache.seedSqByRegion[reg] or nil
        if not seedSq then
            areas[reg] = RC_TempSim.ROOM_SIZE_REF_SQ -- fallback
        else
            local q, head, visited, n = { seedSq }, 1, {}, 0
            visited[seedSq] = true
            while head <= #q do
                local s = q[head]; head = head + 1
                n = n + 1
                local function push(ns)
                    if ns and (not visited[ns]) and (ns:getZ()==z) and (regionFromSquare_player(ns) == reg) then
                        visited[ns] = true; q[#q+1] = ns
                    end
                end
                push(s.getN and s:getN() or nil)
                push(s.getS and s:getS() or nil)
                push(s.getW and s:getW() or nil)
                push(s.getE and s:getE() or nil)
                if n > 12000 then break end -- safety
            end
            areas[reg] = math.max(1, n)
        end
    end
    return areas
end

---------------------------------------------------------------------------
-- Heat detection and propagation
---------------------------------------------------------------------------

local function _collectHeatFromSquareObjects(square, seen)
    if not square then return 0, 0 end
    local total, matched = 0, 0
    local function scan(list)
        if not list then return end
        for i = 0, list:size() - 1 do
            local obj = list:get(i)
            if obj and not seen[obj] then
                seen[obj] = true
                local amount, hit = _collectHeatFromObject(obj)
                if hit then matched = matched + 1 end
                if amount and amount ~= 0 then total = total + amount end
            end
        end
    end
    if square.getObjects then scan(square:getObjects()) end
    if square.getSpecialObjects then scan(square:getSpecialObjects()) end
    if square.getStaticMovingObjects then scan(square:getStaticMovingObjects()) end
    return total, matched
end

local function _neighborSquare(square, dir)
    if not square then return nil end
    if dir == "N" then return square.getN and square:getN() or nil end
    if dir == "S" then return square.getS and square:getS() or nil end
    if dir == "E" then return square.getE and square:getE() or nil end
    if dir == "W" then return square.getW and square:getW() or nil end
    return nil
end

local function _nativeHeatForSquare(square, cell)
    if not square or RC_TempSim.INCLUDE_NATIVE_HEAT == false then return 0 end
    if square.isInARoom and square:isInARoom() then return 0 end
    cell = cell or (getCell and getCell() or nil)
    if not cell then return 0 end
    local getX = square.getX
    local getY = square.getY
    local getZ = square.getZ
    if not (getX and getY and getZ) then return 0 end
    local native = cell:getHeatSourceTemperature(getX(square), getY(square), getZ(square)) or 0
    if native ~= 0 then
        local mult = RC_TempSim.NATIVE_HEAT_MULT or 1
        if mult ~= 1 then
            native = native * mult
        end
    end
    return native
end

local function _collectNearbyHeat(square)
    if not square then return 0 end
    local seenSquares = {}
    local seenObjects = {}
    local total = 0
    local cell = (RC_TempSim.INCLUDE_NATIVE_HEAT ~= false) and (getCell and getCell() or nil) or nil

    local function addSquareHeat(sq, weight)
        if not sq or weight <= 0 or seenSquares[sq] then return end
        seenSquares[sq] = true
        local amount = _collectHeatFromSquareObjects(sq, seenObjects)
        if amount and amount > 0 then
            total = total + amount * weight
        end
        if cell and sq and sq.isInARoom and not sq:isInARoom() then
            local native = _nativeHeatForSquare(sq, cell)
            if native ~= 0 then
                total = total + native * weight
            end
        end
    end

    addSquareHeat(square, 1.0)

    local adjWeight = RC_TempSim.PROXIMITY_ADJACENT_WEIGHT
    if adjWeight > 0 then
        addSquareHeat(_neighborSquare(square, "N"), adjWeight)
        addSquareHeat(_neighborSquare(square, "S"), adjWeight)
        addSquareHeat(_neighborSquare(square, "E"), adjWeight)
        addSquareHeat(_neighborSquare(square, "W"), adjWeight)
    end

    local diagWeight = RC_TempSim.PROXIMITY_DIAGONAL_WEIGHT
    if diagWeight > 0 then
        local north = _neighborSquare(square, "N")
        local south = _neighborSquare(square, "S")
        addSquareHeat(_neighborSquare(north, "E"), diagWeight)
        addSquareHeat(_neighborSquare(north, "W"), diagWeight)
        addSquareHeat(_neighborSquare(south, "E"), diagWeight)
        addSquareHeat(_neighborSquare(south, "W"), diagWeight)
    end

    return total
end

local function computeProximityWarmRate(square)
    local heat = _collectNearbyHeat(square)
    if not heat or heat <= 0 then return 0 end
    local ratePerUnit = RC_TempSim.PROXIMITY_HEAT_RATE_PER_UNIT
    if ratePerUnit <= 0 then return 0 end
    local rate = heat * ratePerUnit
    local maxRate = RC_TempSim.PROXIMITY_HEAT_RATE_MAX
    if maxRate > 0 and rate > maxRate then
        rate = maxRate
    end
    return rate
end

local function _heatTotalsForSquares(squares)
    local includeNative = RC_TempSim.INCLUDE_NATIVE_HEAT ~= false
    local cell = includeNative and getCell and getCell() or nil
    local nativeMax = 0
    local customTotal = 0
    local matchedCount = 0
    local seenObjects = {}
    for _, sq in ipairs(squares or {}) do
        if includeNative and cell and sq and sq.getX then
            local native = cell:getHeatSourceTemperature(sq:getX(), sq:getY(), sq:getZ())
            if native and native > nativeMax then nativeMax = native end
        end
        local added, hits = _collectHeatFromSquareObjects(sq, seenObjects)
        if added and added ~= 0 then customTotal = customTotal + added end
        if hits and hits > 0 then matchedCount = matchedCount + hits end
    end
    if RC_TempSim.NATIVE_HEAT_MULT and RC_TempSim.NATIVE_HEAT_MULT ~= 1 then
        nativeMax = nativeMax * RC_TempSim.NATIVE_HEAT_MULT
    end
    return nativeMax, customTotal, matchedCount
end

local function _roomDefFromSquare(square)
    if not square or not square.getRoom then return nil end
    local room = square:getRoom()
    if not room or not room.getRoomDef then return nil end
    local ok, roomDef = pcall(function()
        return room:getRoomDef()
    end)
    if not ok then return nil end
    return roomDef
end

local function _focusSetForHugeCurrentRoom(S, square)
    if not (S and S.hugeInfo and S.hugeInfo.isHuge) then return nil end
    local roomDef = _roomDefFromSquare(square)
    if not roomDef then return nil end
    local set = {}
    set[roomDef] = true
    return set
end

local function gatherHeat_vanilla(S, focusSet)
    local out, stats = {}, { objects = 0, nativeRooms = 0, customRooms = 0, directRooms = 0, maxDirectAbs = 0 }
    if not (S and S.currentBuildingDef and S.rooms) then return out, stats end
    local def = S.currentBuildingDef
    local cacheByZ = {}
    for i = 0, S.rooms:size() - 1 do
        local rd = S.rooms:get(i)
        if rd and (not focusSet or focusSet[rd]) then
            local z = rd:getZ() or 0
            local cache = cacheByZ[z]
            if not cache then
                cache = RC_RoomLogic.ensureHugeCache(def, S.rooms, z)
                cacheByZ[z] = cache
            end
            local squares = RC_RoomLogic.getRoomSquares(cache, rd) or {}
            local nativeMax, customTotal, matchedCount = _heatTotalsForSquares(squares)
            if matchedCount > 0 then stats.objects = stats.objects + matchedCount end
            if nativeMax > 0 then stats.nativeRooms = (stats.nativeRooms or 0) + 1 end
            if customTotal ~= 0 then stats.customRooms = (stats.customRooms or 0) + 1 end
            local total = nativeMax + customTotal
            if total ~= 0 then
                out[rd] = total
                stats.directRooms = (stats.directRooms or 0) + 1
                local absTotal = math.abs(total)
                if absTotal > (stats.maxDirectAbs or 0) then stats.maxDirectAbs = absTotal end
                if total > 0 and (stats.maxDirectPositive == nil or total > stats.maxDirectPositive) then
                    stats.maxDirectPositive = total
                end
                if total < 0 and (stats.maxDirectNegative == nil or total < stats.maxDirectNegative) then
                    stats.maxDirectNegative = total
                end
            end
        end
    end
    return out, stats
end

local function peekHeatedSmallRooms_vanilla(S, focusSet)
    local out = {}
    local direct = nil
    do
        local d, _stats = gatherHeat_vanilla(S, focusSet)
        direct = d or {}
    end
    local limit = RC_TempSim.HUGE_SMALLROOM_MAX_SQ or 60
    for node, amount in pairs(direct) do
        if amount and amount > 0 then
            local area = (S.roomArea and S.roomArea[node]) or 999
            if area <= limit then
                out[node] = true
            end
        end
    end
    return out
end

local function _mergeSets(a, b)
    if not a and not b then return nil end
    local out = {}
    if type(a) == "table" then
        for k,v in pairs(a) do if v then out[k] = true end end
    end
    if type(b) == "table" then
        for k,v in pairs(b) do if v then out[k] = true end end
    end
    return out
end


local function gatherHeat_player(S)
    local out, stats = {}, { objects = 0, nativeRooms = 0, customRooms = 0, directRooms = 0, maxDirectAbs = 0 }
    local building = S and S.playerBuilding or nil
    if not building then return out, stats end
    for _, reg in ipairs(building.regions or {}) do
        local squares = _squaresForPlayerRegion(building, reg)
        local nativeMax, customTotal, matchedCount = _heatTotalsForSquares(squares)
        if matchedCount > 0 then stats.objects = stats.objects + matchedCount end
        if nativeMax > 0 then stats.nativeRooms = (stats.nativeRooms or 0) + 1 end
        if customTotal ~= 0 then stats.customRooms = (stats.customRooms or 0) + 1 end
        local total = nativeMax + customTotal
        if total ~= 0 then
            out[reg] = total
            stats.directRooms = (stats.directRooms or 0) + 1
            local absTotal = math.abs(total)
            if absTotal > (stats.maxDirectAbs or 0) then stats.maxDirectAbs = absTotal end
            if total > 0 and (stats.maxDirectPositive == nil or total > stats.maxDirectPositive) then
                stats.maxDirectPositive = total
            end
            if total < 0 and (stats.maxDirectNegative == nil or total < stats.maxDirectNegative) then
                stats.maxDirectNegative = total
            end
        end
    end
    return out, stats
end

local function propagateHeatAcrossGraph(graph, sources)
    if not sources then return nil end
    local out = {}
    local distExp = RC_TempSim.HEAT_DIST_EXP
    for node, amount in pairs(sources) do
        if amount and amount ~= 0 then
            local distances = distancesFromOne(graph or {}, node)
            for other, d in pairs(distances) do
                local attenuation = 1.0 / ((1 + d) ^ distExp)
                out[other] = (out[other] or 0) + amount * attenuation
            end
        end
    end
    return out
end

local function applyHeatToSimulation(S, influence, outC, stats)
    if not influence then return stats end
    stats = stats or {}
    stats.propagatedRooms = 0
    stats.propagatedMax = 0
    stats.propagatedMaxPositive = nil
    stats.propagatedMaxNegative = nil

    local base = RC_TempSim.HEAT_BASE_BLEND_PER_STEP
    local sizeExp = RC_TempSim.HEAT_SIZE_EXP
    local minFactor = RC_TempSim.HEAT_SIZE_FACTOR_MIN
    local maxFactor = RC_TempSim.HEAT_SIZE_FACTOR_MAX
    local influenceLimit = RC_TempSim.HEAT_INFLUENCE_MAX

    local hugeBoostEnabled = (S and S.hugeInfo) and true or false
    local gridOn = true
    if hugeBoostEnabled then
        local ok, on = pcall(function()
            local w = getWorld and getWorld() or nil
            return (w and w.isHydroPowerOn and w:isHydroPowerOn()) or false
        end)
        gridOn = ok and on or false
    end
    local smallSqLimit = RC_TempSim.HUGE_SMALLROOM_MAX_SQ or 60
    local smallRoomHeatMult = RC_TempSim.HUGE_SMALLROOM_HEAT_MULT or 1.5

    for node, amount in pairs(influence) do
        if amount and amount ~= 0 then
            stats.propagatedRooms = stats.propagatedRooms + 1
            local absAmount = math.abs(amount)
            if absAmount > (stats.propagatedMax or 0) then stats.propagatedMax = absAmount end
            if amount > 0 and (stats.propagatedMaxPositive == nil or amount > stats.propagatedMaxPositive) then
                stats.propagatedMaxPositive = amount
            end
            if amount < 0 and (stats.propagatedMaxNegative == nil or amount < stats.propagatedMaxNegative) then
                stats.propagatedMaxNegative = amount
            end

            local current = S.temps[node]
            if type(current) ~= "number" then
                current = outC
            end
            local area = (S.roomArea and S.roomArea[node]) or 1
            local ratio = RC_TempSim.ROOM_SIZE_REF_SQ / math.max(1, area)
            local sizeFactor = ratio ^ sizeExp
            if sizeFactor < minFactor then sizeFactor = minFactor end
            if sizeFactor > maxFactor then sizeFactor = maxFactor end

            local clamped = amount
            if influenceLimit and influenceLimit > 0 then
                if clamped >  influenceLimit then clamped =  influenceLimit end
                if clamped < -influenceLimit then clamped = -influenceLimit end
            end

            local targetBase = current
            if type(targetBase) ~= "number" then
                targetBase = outC
            end
            local target = targetBase + clamped
            if RC_TempSim.HEAT_TARGET_MIN_C and target < RC_TempSim.HEAT_TARGET_MIN_C then
                target = RC_TempSim.HEAT_TARGET_MIN_C
            end
            if RC_TempSim.HEAT_TARGET_MAX_C and target > RC_TempSim.HEAT_TARGET_MAX_C then
                target = RC_TempSim.HEAT_TARGET_MAX_C
            end

            local baseThis = base
            if hugeBoostEnabled and (not gridOn) and clamped > 0 and area <= smallSqLimit then
                baseThis = baseThis * smallRoomHeatMult
            end

            local nextTemp = current + baseThis * sizeFactor * (target - current)
            S.temps[node] = nextTemp
        end
    end

    return stats
end

---------------------------------------------------------------------------
-- Core build/rebuild
---------------------------------------------------------------------------

local function clearState(S)
    S.mode = nil
    S.ownerPlayer = nil
    S.currentBuildingDef = nil
    S.building = nil
    S.rooms = nil
    S.playerBuilding = nil
    S.playerRegions = nil
    S.graph = nil
    S.closedDoors = nil
    S.temps = nil
    S.breaches = nil
    S.byRoomName = nil
    S.roomArea = nil
    S.lastLeakInfo = nil
    S.hugeInfo = nil
    S.lastHugeFullUpdateHour = nil
    S.lastHugeScanRevision = nil
    S.lastHugeLightUpdateHour = nil
end

local function _isBlankString(value)
    if value == nil then return true end
    if type(value) ~= "string" then return false end
    if value == "" then return true end
    return value:find("%S") == nil
end

function RC_TempSim.rebuildForCurrentBuilding(player)
    local S = RC_TempSim._state
    local sq = player and player:getCurrentSquare() or nil
    local prevMode = S.mode
    local isOwner = (S.ownerPlayer == player)

    if not sq then
        if isOwner then
            if prevMode == "vanilla" then persist_vanilla(S)
            elseif prevMode == "player" then persist_player(S) end
            clearState(S)
        end
        return
    end

    local room = sq:getRoom()
    local roomDef = room and room.getRoomDef and room:getRoomDef() or nil
    local roomName = roomDef and roomDef.getName and roomDef:getName() or nil
    local roomNameIsBlank = _isBlankString(roomName)

    local building = room and room:getBuilding() or nil
    if building and building:getDef() and not roomNameIsBlank then
        local def = building:getDef()
        if S.mode == "vanilla" and S.currentBuildingDef == def then
            S.ownerPlayer = player
            return
        end
        if prevMode == "vanilla" and S.currentBuildingDef and S.currentBuildingDef ~= def then persist_vanilla(S)
        elseif prevMode == "player" and S.playerBuilding then persist_player(S) end

        local roomDefs = def:getRooms()
        local zLevel   = sq:getZ()
        local hugeScan = RC_RoomLogic.scanHugeBuilding and RC_RoomLogic.scanHugeBuilding(def, roomDefs, zLevel, sq) or nil
        local graph, closed, breaches, areas
        if hugeScan then
            graph = hugeScan.graph or {}
            closed = hugeScan.closed or {}
            breaches = hugeScan.breaches or {}
            areas = hugeScan.roomArea or {}
        else
            graph, closed = buildGraph_vanilla(def, roomDefs, zLevel)
            breaches = breaches_vanilla(def, roomDefs)
            areas    = computeAreas_vanilla(def, roomDefs, zLevel)
        end

        local out = getClimateManager():getTemperature()
        local leakInfo = leakInfoFromBreaches(breaches, breachIsOpen_vanilla)
        local temps, loaded = loadTemps_vanilla(def, roomDefs, zLevel, out, leakInfo, graph, closed, areas)

        local byRoomName = {}
        for i = 0, roomDefs:size() - 1 do
            local rd = roomDefs:get(i)
            if rd then byRoomName[rd] = safeRoomName_vanilla(rd) end
        end

        S.mode = "vanilla"
        S.currentBuildingDef = def
        S.building = building
        S.rooms = roomDefs
        S.playerBuilding = nil
        S.playerRegions = nil
        S.graph = graph
        S.closedDoors = closed
        S.temps = temps
        S.breaches = breaches
        S.byRoomName = byRoomName
        S.roomArea = areas
        S.lastLeakInfo = leakInfo
        S.hugeInfo = hugeScan and hugeScan.hugeInfo or nil
        if S.hugeInfo then S.hugeInfo.isHuge = true end
        if hugeScan and RC_RoomLogic and RC_RoomLogic.getHugeScanRevision then
            local ok, rev = pcall(RC_RoomLogic.getHugeScanRevision, def)
            if ok then
                S.lastHugeScanRevision = rev
            else
                S.lastHugeScanRevision = nil
            end
        else
            S.lastHugeScanRevision = nil
        end
        S.lastHugeFullUpdateHour = nil
        S.lastHugeLightUpdateHour = nil
        S.ownerPlayer = player

        -- print(string.format("TempSim: cached VANILLA building (%d rooms), breaches=%d, temps %s ModData",
        --     roomDefs and roomDefs:size() or -1, #breaches, loaded and "loaded from" or "seeded (no"))
        return
    end

    local preg = RC_PlayerRoomLogic and RC_PlayerRoomLogic.regionFromSquareOrNeighbors
            and RC_PlayerRoomLogic.regionFromSquareOrNeighbors(sq)
            or regionFromSquare_player(sq)

    if preg and roomNameIsBlank then
        local pb = RC_PlayerRoomLogic.BuildPlayerBuildingFromRegion(preg)
        if not pb then return end
        if S.mode == "player" and S.playerBuilding and _samePlayerBuilding(S.playerBuilding, pb) then
            S.ownerPlayer = player
            return
        end

        if S.mode == "vanilla" and S.currentBuildingDef then
            persist_vanilla(S)
        elseif S.mode == "player" and S.playerBuilding and (not _samePlayerBuilding(S.playerBuilding, pb)) then
            persist_player(S)
        end

        local graph, closed = buildGraph_player(pb)
        local breaches = breaches_player(pb)
        local areas    = computeAreas_player(pb)
        local out      = getClimateManager():getTemperature()
        local leakInfo = leakInfoFromBreaches(breaches, breachIsOpen_player)
        local temps, loaded = loadTemps_player(pb, out, leakInfo, graph, closed, areas)

        local byRoomName = {}
        local regions = {}
        for _, reg in ipairs(pb.regions or {}) do
            regions[#regions+1] = reg
            byRoomName[reg] = safeRoomName_player(pb, reg)
        end

        S.mode = "player"
        S.currentBuildingDef = nil
        S.building = nil
        S.rooms = nil
        S.playerBuilding = pb
        S.playerRegions = regions
        S.graph = graph
        S.closedDoors = closed
        S.temps = temps
        S.breaches = breaches
        S.byRoomName = byRoomName
        S.roomArea = areas
        S.lastLeakInfo = leakInfo
        S.lastHugeScanRevision = nil
        S.lastHugeFullUpdateHour = nil
        S.lastHugeLightUpdateHour = nil
        S.ownerPlayer = player

        -- print(string.format(
        --     "TempSim: cached PLAYER building %s (%d rooms), breaches=%d, temps %s ModData",
        --     tostring(pb.id or "?"), #regions, #breaches, (loaded and "loaded from" or "seeded (no)")
        -- ))
        return
    end

    if isOwner then
        if prevMode == "vanilla" then persist_vanilla(S)
        elseif prevMode == "player" then persist_player(S) end
        clearState(S)
    end
end

---------------------------------------------------------------------------
-- Update step
---------------------------------------------------------------------------

function RC_TempSim.updateStep(player)
    local S = RC_TempSim._state
    if not S.mode then return end
    recordOutsideTemperature()
    local nowHours = _currentWorldHours()
    local outC = getClimateManager():getTemperature()
    local heatDirect, heatStats

    local isVanillaHuge = (S.mode == "vanilla") and S.hugeInfo and S.hugeInfo.isHuge
    local minGapHours = math.max(0, (RC_TempSim.HUGE_MINUTES_BETWEEN_FULL_UPDATES or 60) / 60.0)
    local hugeRevision, revisionChanged = nil, false
    if S.mode == "vanilla" and S.currentBuildingDef and RC_RoomLogic and RC_RoomLogic.getHugeScanRevision then
        local ok, rev = pcall(RC_RoomLogic.getHugeScanRevision, S.currentBuildingDef)
        if ok then
            hugeRevision = rev
            if S.lastHugeScanRevision ~= rev then
                revisionChanged = true
            end
        end
    end

    local heavyDue = true
    if isVanillaHuge then
        heavyDue = false
        if not S.lastHugeFullUpdateHour then
            heavyDue = true
        else
            local elapsed = nowHours - (S.lastHugeFullUpdateHour or 0)
            if elapsed < 0 then elapsed = minGapHours end
            if elapsed >= minGapHours then heavyDue = true end
        end
        if not S.lastLeakInfo then heavyDue = true end
        if revisionChanged then heavyDue = true end
    end

    local skipHeavy = isVanillaHuge and not heavyDue

    if skipHeavy then
        local stepMinutes = tonumber(RC_TempSim.UPDATE_EVERY_MINUTES) or 10
        local stepHours = stepMinutes / 60
        if stepHours > 0 then
            local historyEntries = _cloneOutsideHistoryEntries()
            local leakRatio = (S.lastLeakInfo and S.lastLeakInfo.leakRatio)
                    or RC_TempSim.OFFLINE_LEAK_SEALED or 0
            local driftOpts = {
                roomArea = S.roomArea,
                defaultArea = RC_TempSim.ROOM_SIZE_REF_SQ,
            }
            applyOfflineTemperatureDrift(S.temps or {}, stepHours, outC, leakRatio, historyEntries, nil, driftOpts)
        end
        if isVanillaHuge then
            S.lastHugeLightUpdateHour = nowHours
        end
        -- return
    end

    local curZ = 0
    local psq = player and player.getCurrentSquare and player:getCurrentSquare() or nil
    if psq and psq.getZ then curZ = psq:getZ() or 0 end

    if S.mode == "vanilla" and not isVanillaHuge then
        if not (S.currentBuildingDef and S.rooms) then return end
        pcall(invalidatePerimeters_vanilla, S.currentBuildingDef, S.rooms)
        local hugeScan = RC_RoomLogic.scanHugeBuilding and RC_RoomLogic.scanHugeBuilding(S.currentBuildingDef, S.rooms, curZ, psq) or nil
        if hugeScan then
            S.graph = hugeScan.graph or {}
            S.closedDoors = hugeScan.closed or {}
            S.roomArea = hugeScan.roomArea or {}
            S.breaches = hugeScan.breaches or {}
            S.hugeInfo = hugeScan.hugeInfo or {}
            if S.hugeInfo then S.hugeInfo.isHuge = true end
        else
            local graph, closed = buildGraph_vanilla(S.currentBuildingDef, S.rooms, curZ)
            S.graph = graph
            S.closedDoors = closed
            S.roomArea = computeAreas_vanilla(S.currentBuildingDef, S.rooms, curZ)
            S.breaches = breaches_vanilla(S.currentBuildingDef, S.rooms)
            S.hugeInfo = nil
        end
    elseif S.mode == "player" then
        if not S.playerBuilding then return end
        local graph, closed = buildGraph_player(S.playerBuilding)
        S.graph = graph
        S.closedDoors = closed
        if not S.roomArea then S.roomArea = computeAreas_player(S.playerBuilding) end
        S.breaches = breaches_player(S.playerBuilding)
    end

    local ensureList = {}
    if S.mode == "vanilla" then
        local focusList = S.hugeInfo and S.hugeInfo.focusRooms or nil
        local focusSetExisting = S.hugeInfo and S.hugeInfo.focusRoomsSet or nil
        local seen = {}
        local function push(room)
            if room and not seen[room] then
                seen[room] = true
                ensureList[#ensureList+1] = room
            end
        end
        if focusList and #focusList > 0 then
            for _, rd in ipairs(focusList) do
                push(rd)
            end
        elseif focusSetExisting and _tableHasEntries(focusSetExisting) then
            for rd,_ in pairs(focusSetExisting) do
                push(rd)
            end
        end
        if S.rooms then
            for i = 0, S.rooms:size() - 1 do
                local rd = S.rooms:get(i)
                push(rd)
            end
        end
    else
        for _, reg in ipairs(S.playerRegions or {}) do ensureList[#ensureList+1] = reg end
    end

    local focusSet = nil
    if isVanillaHuge then
        focusSet = _focusSetForHugeCurrentRoom(S, psq)
    end
    local useHugeRework = false
    if not isVanillaHuge then
        if S.mode == "vanilla" then
            heatDirect, heatStats = gatherHeat_vanilla(S, nil)
        else
            heatDirect, heatStats = gatherHeat_player(S)
        end
        if type(heatDirect) ~= "table" then heatDirect = {} end
        if type(heatStats)  ~= "table" then heatStats  = {} end

        local graphForHeat = S.graph
        local closedForEq  = S.closedDoors

        if _tableHasEntries(heatDirect) then
            local heatInfluence = propagateHeatAcrossGraph(graphForHeat, heatDirect)
            heatStats = applyHeatToSimulation(S, heatInfluence, outC, heatStats)
            S.heatDirect    = heatDirect
            S.heatInfluence = heatInfluence or {}
            S.heatStats     = heatStats or {}
        else
            S.heatDirect, S.heatInfluence, S.heatStats = {}, {}, {}
        end

        applyInternalEqualizationToTemps(S.temps, graphForHeat, closedForEq, S.roomArea, outC)
    end


    local preciseRooms, preciseSet = nil, nil
    local baselineRooms = ensureList

    if isVanillaHuge then
        local powered, powerOffHour = _trackWorldPowerStatus()

        local stepMinutesLocal = tonumber(RC_TempSim.UPDATE_EVERY_MINUTES) or 10
        local stepHoursLocal   = stepMinutesLocal / 60

        local heatedSmallSet = nil
        if not powered then
            heatedSmallSet = peekHeatedSmallRooms_vanilla(S, focusSet)
        end

        local excludeSet = _mergeSets(preciseSet, heatedSmallSet)
        applyHugeBuildingBaseline(S, baselineRooms, outC, stepHoursLocal, powered, powerOffHour, excludeSet)

        if preciseRooms and #preciseRooms > 0 then
            applyPreciseHugeRoomSimulation(S, preciseRooms, outC)
        end

        clampPoweredHugeRooms(S, baselineRooms, powered, stepHoursLocal)

        local graphForHeat = S.graph
        local graphForEq, closedForEq = S.graph, S.closedDoors
        if focusSet and _tableHasEntries(focusSet) then
            graphForHeat = buildFilteredGraph(S.graph or {}, focusSet)
            graphForEq   = graphForHeat
            closedForEq  = buildFilteredGraph(S.closedDoors or {}, focusSet)
        end

        local direct, heatStats = gatherHeat_vanilla(S, focusSet)
        local influence = propagateHeatAcrossGraph(graphForHeat, direct)

        -- APPLY heat + store debug for UI/prints
        heatStats = applyHeatToSimulation(S, influence, outC, heatStats)
        S.heatDirect    = direct or {}
        S.heatInfluence = influence or {}
        S.heatStats     = heatStats or {}

        -- Equalize locally-filtered graph, then HVAC vents
        applyInternalEqualizationToTemps(S.temps, graphForEq, closedForEq, S.roomArea, outC)
        applyHVACVentCoupling(S, focusSet or {}, stepHoursLocal, direct)
        return
    end

    baselineRooms = baselineRooms or ensureList
    for _, node in ipairs(baselineRooms) do
        if S.temps[node] == nil then S.temps[node] = RC_TempSim.seedTempForRoom(outC) end
    end

    local totalNodes = #ensureList
    local stepMinutes = tonumber(RC_TempSim.UPDATE_EVERY_MINUTES) or 10
    local stepHours = stepMinutes / 60

    local leakRatio = 0
    local breachSources = 0
    local totalWeighted = 0
    local hadBreaches = false
    local exposure = {}
    local simplifyExposure = false
    local simplifyValue = nil
    local typeCountsForPrint = {}
    local openCountForPrint = 0

    if not isVanillaHuge then
        local isOpenFn = (S.mode == "vanilla") and breachIsOpen_vanilla or breachIsOpen_player
        local strengthsByRoom, openCount, typeCounts
        strengthsByRoom, openCount, typeCounts, totalWeighted = computeOpenBreachStrengths(S.breaches, isOpenFn)
        leakRatio = computeLeakRatio(openCount, typeCounts, totalWeighted)
        local leakInfo = {
            leakRatio = leakRatio,
            openCount = openCount or 0,
            typeCounts = typeCounts or {},
            totalWeighted = totalWeighted or 0,
        }
        S.lastLeakInfo = leakInfo
        hadBreaches = (openCount or 0) > 0
        openCountForPrint = openCount or 0
        typeCountsForPrint = typeCounts or {}

        if hadBreaches then
            for _, _ in pairs(strengthsByRoom) do
                breachSources = breachSources + 1
            end

            if totalNodes >= RC_TempSim.LARGE_BUILDING_ROOM_THRESHOLD
                    or breachSources >= RC_TempSim.LARGE_BUILDING_BREACH_THRESHOLD then
                simplifyExposure = true
                local average = 0
                if totalNodes > 0 and totalWeighted and totalWeighted > 0 then
                    average = totalWeighted / totalNodes
                elseif totalWeighted and totalWeighted > 0 then
                    average = totalWeighted
                end
                if average > 0 and average < RC_TempSim.LARGE_BUILDING_EXPOSURE_FLOOR then
                    average = RC_TempSim.LARGE_BUILDING_EXPOSURE_FLOOR
                end
                if average > RC_TempSim.EXPOSURE_MAX then
                    average = RC_TempSim.EXPOSURE_MAX
                end
                simplifyValue = average
            end

            if simplifyExposure and simplifyValue and simplifyValue > 0 then
                for _, node in ipairs(ensureList) do
                    exposure[node] = simplifyValue
                end
            else
                for src, strength in pairs(strengthsByRoom) do
                    local dmap = distancesFromOne(S.graph, src)
                    for node, d in pairs(dmap) do
                        local att = 1.0 / ((1 + d) ^ RC_TempSim.DIST_EXP)
                        exposure[node] = (exposure[node] or 0) + strength * att
                    end
                end
            end

            if S.hugeInfo and S.hugeInfo.distanceFromPerimeter then
                local distMap = S.hugeInfo.distanceFromPerimeter
                if type(distMap) == "table" and _tableHasEntries(distMap) then
                    local expPow = RC_TempSim.HUGE_DISTANCE_EXP or 1.0
                    local minFactor = RC_TempSim.HUGE_DISTANCE_MIN_FACTOR or 0.0
                    local maxFactor = RC_TempSim.HUGE_DISTANCE_MAX_FACTOR or nil
                    for node, value in pairs(exposure) do
                        local d = distMap[node]
                        local falloff
                        if d == nil then
                            falloff = minFactor
                        else
                            falloff = 1.0 / ((1 + d) ^ expPow)
                            if maxFactor and falloff > maxFactor then falloff = maxFactor end
                            if minFactor and falloff < minFactor then falloff = minFactor end
                        end
                        exposure[node] = (value or 0) * (falloff or 0)
                    end
                end
            end
        end
    end

    if not isVanillaHuge and hadBreaches then
        local base = RC_TempSim.BASE_BLEND_PER_STEP
        for node, curT in pairs(S.temps) do
            local exp = exposure[node]
            if exp and exp > 0 then
                local sz   = (S.roomArea and S.roomArea[node]) or 1
                local ref  = RC_TempSim.ROOM_SIZE_REF_SQ
                local ratio = ref / math.max(1, sz)
                local sizeFactor = ratio ^ RC_TempSim.SIZE_EXP
                if sizeFactor < RC_TempSim.SIZE_FACTOR_MIN then sizeFactor = RC_TempSim.SIZE_FACTOR_MIN end
                if sizeFactor > RC_TempSim.SIZE_FACTOR_MAX then sizeFactor = RC_TempSim.SIZE_FACTOR_MAX end
                if exp > RC_TempSim.EXPOSURE_MAX then exp = RC_TempSim.EXPOSURE_MAX end
                local alpha = base * sizeFactor * exp
                S.temps[node] = curT + alpha * (outC - curT)
            end
        end
    end


    if not isVanillaHuge then
        local leakRate = tonumber(RC_TempSim.INDOOR_LEAK_RATE_PER_HOUR) or 0
        if leakRate < 0 then leakRate = 0 end
        if leakRate > 0 then
            local stepMinutes = tonumber(RC_TempSim.UPDATE_EVERY_MINUTES) or 10
            if stepMinutes > 0 then
                local stepHours = stepMinutes / 60
                if stepHours > 0 then
                    local blend = 1 - math.exp(-leakRate * stepHours)
                    if blend > 0 then
                        if blend > 1 then blend = 1 end
                        for node, curT in pairs(S.temps) do
                            if type(curT) == "number" then
                                S.temps[node] = curT + (outC - curT) * blend
                            end
                        end
                    end
                end
            end
        end
    end


    if S.mode == "vanilla" and not isVanillaHuge then
        heatDirect, heatStats = gatherHeat_vanilla(S, (useHugeRework and focusSet) or nil)
    elseif S.mode == "player" then
        heatDirect, heatStats = gatherHeat_player(S)
    end

    if type(heatDirect) ~= "table" then heatDirect = {} end
    if type(heatStats) ~= "table" then heatStats = {} end
    local graphForHeat = S.graph
    local closedForEq = S.closedDoors
    if useHugeRework and focusSet then
        graphForHeat = buildFilteredGraph(S.graph or {}, focusSet)
        closedForEq = buildFilteredGraph(S.closedDoors or {}, focusSet)
    end

    local heatInfluence = nil
    if _tableHasEntries(heatDirect) then
        heatInfluence = propagateHeatAcrossGraph(graphForHeat, heatDirect)
        heatStats = applyHeatToSimulation(S, heatInfluence, outC, heatStats)
    end
    if not heatInfluence then heatInfluence = {} end
    S.heatDirect = heatDirect
    S.heatInfluence = heatInfluence
    S.heatStats = heatStats

    -- applyInternalEqualizationToTemps(S.temps, graphForHeat, closedForEq, S.roomArea, outC)

    if isVanillaHuge then
        S.lastHugeFullUpdateHour = nowHours
        S.lastHugeScanRevision = hugeRevision
        S.lastHugeLightUpdateHour = nil
    else
        S.lastHugeFullUpdateHour = nil
        S.lastHugeScanRevision = nil
        S.lastHugeLightUpdateHour = nil
    end

    -- print status
    -- print("--------------------------------------")
    if hadBreaches then
        local eff = totalWeighted; if eff > RC_TempSim.EXPOSURE_MAX then eff = RC_TempSim.EXPOSURE_MAX end
        -- print(string.format(
        --     "RC_TempSim (%s): Outdoor %.1f C | Breaches: %d  (W:%d D:%d G:%d C:%d) | Flow multiplier: %.2f (cap %.2f)",
        --     S.mode, outC, openCountForPrint, (typeCountsForPrint.window or 0), (typeCountsForPrint.door or 0), (typeCountsForPrint.gap or 0), (typeCountsForPrint.curtain or 0), eff, RC_TempSim.EXPOSURE_MAX
        -- ))
        if simplifyExposure then
            -- print(string.format(
            --     "  Using simplified exposure (rooms:%d, breach sources:%d, exposure:%.2f)",
            --     totalNodes,
            --     breachSources,
            --     simplifyValue or 0
            -- ))
        end
    else
        -- print(string.format("RC_TempSim (%s): Outdoor %.1f C | No open breaches (equalizing only)", S.mode, outC))
    end

    if heatStats and ((heatStats.directRooms or 0) > 0 or (heatStats.objects or 0) > 0 or (heatStats.nativeRooms or 0) > 0) then
        -- print(string.format(
        --     "Heat sources: rooms %d (native:%d custom:%d) | objects:%d | max direct: %.1f C | propagated rooms:%d (max %.1f C)",
        --     heatStats.directRooms or 0,
        --     heatStats.nativeRooms or 0,
        --     heatStats.customRooms or 0,
        --     heatStats.objects or 0,
        --     heatStats.maxDirectPositive or heatStats.maxDirectAbs or 0,
        --     heatStats.propagatedRooms or 0,
        --     heatStats.propagatedMax or 0
        -- ))
    end

    local byZ = {}
    if S.mode == "vanilla" then
        for i = 0, S.rooms:size() - 1 do
            local rd = S.rooms:get(i)
            if rd then
                local z = rd:getZ() or 0
                byZ[z] = byZ[z] or {}; table.insert(byZ[z], rd)
            end
        end
    else
        for _, reg in ipairs(S.playerRegions or {}) do
            local z = (reg.getZ and reg:getZ()) or 0
            byZ[z] = byZ[z] or {}; table.insert(byZ[z], reg)
        end
    end
    for z, list in pairs(byZ) do
        table.sort(list, function(a,b) return (S.byRoomName[a] or "") < (S.byRoomName[b] or "") end)
        for _, node in ipairs(list) do
            local t = S.temps[node] or outC
            local tags = {}
            if hadBreaches and exposure and exposure[node] and exposure[node] > 0 then
                tags[#tags+1] = "O"
            end
            local heatMark = heatInfluence and heatInfluence[node] or 0
            if heatMark > 0 then
                tags[#tags+1] = "H"
            elseif heatMark < 0 then
                tags[#tags+1] = "C"
            end
            local tag = "~"
            if #tags > 0 then tag = table.concat(tags, "+") end
            -- print(string.format("  %s : %.1f C  [%s]", S.byRoomName[node] or "<room>", t, tag))
        end
    end

    if S.mode == "vanilla" then persist_vanilla(S) else persist_player(S) end
    -- print("--------------------------------------")
end

local function _registerDefaultHeatSources()
    if RC_TempSim._defaultHeatSourcesRegistered then return end
    RC_TempSim._defaultHeatSourcesRegistered = true

    RC_TempSim.registerHeatSource({
        id = "rc_heat_isofire",
        class = "IsoFire",
        heat = 30,
        isActive = function(obj)
            if obj and obj.getLife then
                local life = obj:getLife()
                return (life or 0) > 0
            end
            return false
        end,
        description = "Campfires and placed fire objects",
    })

    RC_TempSim.registerHeatSource({
        id = "rc_heat_fireplace",
        class = "IsoFireplace",
        heat = 24,
        isActive = function(obj)
            if obj and obj.isLit then
                return obj:isLit()
            end
            return false
        end,
        description = "Built fireplaces",
    })

    RC_TempSim.registerHeatSource({
        id = "rc_heat_barbecue",
        class = "IsoBarbecue",
        heat = 20,
        isActive = function(obj)
            if obj and obj.isLit then
                return obj:isLit()
            end
            return false
        end,
        description = "Lit grills and barbecues",
    })

    RC_TempSim.registerHeatSource({
        id = "rc_heat_stove",
        class = "IsoStove",
        heat = 15,
        isActive = function(obj)
            if obj and obj.Activated then
                return obj:Activated()
            end
            return false
        end,
        description = "Powered ovens and stoves",
    })
end

_registerDefaultHeatSources()

local _wetnessCoveredParts = ArrayList and ArrayList.new() or nil

local function _snapshotBodyPartWetness(body)
    if not body or not body.getBodyParts or not BodyPartType or not BodyPartType.ToIndex then
        return nil
    end

    local parts = body:getBodyParts()
    if not parts or not parts.size or parts:size() <= 0 then
        return nil
    end

    local snapshot = {}
    for i = 0, parts:size() - 1 do
        local part = parts:get(i)
        if part and part.getType and part.getWetness then
            local partType = part:getType()
            if partType then
                local index = BodyPartType.ToIndex(partType)
                if index and index >= 0 then
                    snapshot[index] = part:getWetness() or 0.0
                end
            end
        end
    end

    return snapshot
end

local function _getClothingWetnessCoveredParts(item)
    if not item or not item.getModData then return nil end
    if not BloodBodyPartType or not BloodBodyPartType.MAX then return nil end

    local modData = item:getModData()
    if not modData then return nil end

    local cached = modData.rcWetCoveredParts
    if cached ~= nil then
        return cached
    end

    cached = false
    if item.getBloodClothingType and BloodClothingType and BloodClothingType.getCoveredParts and _wetnessCoveredParts then
        local bloodTypes = item:getBloodClothingType()
        if bloodTypes then
            local covered = BloodClothingType.getCoveredParts(bloodTypes, _wetnessCoveredParts)
            if covered and covered.size and covered:size() > 0 then
                local list = {}
                for i = 0, covered:size() - 1 do
                    local part = covered:get(i)
                    local index = part and part:index()
                    if index and index >= 0 then
                        list[#list + 1] = index
                    end
                end
                if #list > 0 then
                    cached = list
                end
            end
        end
    end

    modData.rcWetCoveredParts = cached
    return cached
end

local function _applyBodyWetnessDeltaToClothing(player, beforeMap, afterMap)
    if not player or not beforeMap or not afterMap then return end
    if not BloodBodyPartType or not BloodBodyPartType.MAX or not BloodBodyPartType.FromIndex then return end
    if not player.getWornItems then return end

    local updatesPerMin = RC_TempSim.COLD_UPDATES_PER_MIN or 120
    local minStepMinutes = updatesPerMin > 0 and (1.0 / updatesPerMin) or (1.0 / 60.0)

    local totalAfter, partCount = 0.0, 0
    for index, value in pairs(afterMap) do
        if type(index) == "number" and value then
            totalAfter = totalAfter + value
            partCount = partCount + 1
        end
    end
    local avgBodyAfter = partCount > 0 and (totalAfter / partCount) or 0.0

    local playerMD = player.getModData and player:getModData() or nil
    local elapsedMinutes = minStepMinutes
    local allowReturn = false

    if playerMD then
        local nowH = _currentWorldHours()
        local lastH = playerMD.rcWetLastUpdate or nowH
        local elapsedH = nowH - lastH
        if elapsedH < 0 then elapsedH = 0 end
        if elapsedH > 0.5 then elapsedH = 0.5 end
        playerMD.rcWetLastUpdate = nowH

        elapsedMinutes = elapsedH * 60.0
        if elapsedMinutes <= 0 then
            elapsedMinutes = minStepMinutes
        end

        local dryTimer = playerMD.rcWetBodyDryTimer or 0.0
        local dryThreshold = RC_TempSim.CLOTHING_WETNESS_BODY_DRY_THRESHOLD or 20.0
        if avgBodyAfter <= dryThreshold then
            dryTimer = dryTimer + elapsedMinutes
        else
            dryTimer = 0.0
        end
        playerMD.rcWetBodyDryTimer = dryTimer

        local delay = RC_TempSim.CLOTHING_WETNESS_RETURN_DELAY_MIN or 10.0
        if dryTimer >= delay then
            allowReturn = true
        end
    end

    local maxParts = BloodBodyPartType.MAX:index()
    if not maxParts or maxParts <= 0 then return end

    local worn = player:getWornItems()
    if not worn or not worn.size or worn:size() <= 0 then return end

    for i = 0, worn:size() - 1 do
        local clothing = worn:getItemByIndex(i)
        if clothing and clothing.getWetness and clothing.setWetness then
            local covered = _getClothingWetnessCoveredParts(clothing)
            if covered and covered ~= false and #covered > 0 then
                local beforeSum = 0.0
                local afterSum = 0.0
                local deltaSum = 0.0
                local count = 0

                for j = 1, #covered do
                    local partIndex = covered[j]
                    if partIndex and partIndex >= 0 and partIndex < maxParts then
                        local oldWet = beforeMap[partIndex] or 0.0
                        local newWet = afterMap[partIndex] or 0.0
                        beforeSum = beforeSum + oldWet
                        afterSum = afterSum + newWet
                        deltaSum = deltaSum + (newWet - oldWet)
                        count = count + 1
                    end
                end

                if count > 0 then
                    local avgDelta = deltaSum / count
                    local avgAfter = afterSum / count
                    local currentWet = clothing:getWetness() or 0.0

                    if avgAfter < 0 then avgAfter = 0 end
                    if avgAfter > 100.0 then avgAfter = 100.0 end

                    if currentWet < 0 then currentWet = 0 end
                    if currentWet > 100.0 then currentWet = 100.0 end

                    local resistance = clothing.getWaterResistance and clothing:getWaterResistance() or 0.0
                    if resistance < 0 then resistance = 0 end
                    if resistance > 1 then resistance = 1 end

                    if avgAfter > currentWet + 0.0001 then
                        local diff = avgAfter - currentWet
                        local bodyDrive = avgAfter / 100.0
                        if bodyDrive < 0 then bodyDrive = 0 end
                        local saturationFactor = 1.0 - (currentWet / 100.0)
                        if saturationFactor < 0.05 then saturationFactor = 0.05 end

                        local soakRate = (RC_TempSim.CLOTHING_WETNESS_SOAK_RATE_PER_MIN or 0.18) * bodyDrive
                        soakRate = soakRate * (1.0 - resistance)

                        local transfer = diff * soakRate * elapsedMinutes * saturationFactor
                        if avgDelta > 0 then
                            local deltaBonus = avgDelta * (RC_TempSim.CLOTHING_WETNESS_SOAK_DELTA_MULT or 0.0) * (1.0 - resistance) * elapsedMinutes
                            transfer = transfer + deltaBonus
                        end

                        local capacity = 100.0 - currentWet
                        if transfer > diff then transfer = diff end
                        if transfer > capacity then transfer = capacity end
                        if transfer > 0.0001 then
                            clothing:setWetness(currentWet + transfer)
                            currentWet = currentWet + transfer
                        end
                    elseif allowReturn and currentWet > avgAfter + 0.0001 then
                        local diff = currentWet - avgAfter
                        if diff > 0.0001 then
                            local dryRate = (RC_TempSim.CLOTHING_WETNESS_RETURN_RATE_PER_MIN or 0.12) * (1.0 - resistance)
                            local removal = diff * dryRate * elapsedMinutes
                            if removal > diff then removal = diff end
                            if removal > currentWet then removal = currentWet end
                            if removal > 0.0001 then
                                clothing:setWetness(currentWet - removal)
                                currentWet = currentWet - removal
                            end
                        end
                    end
                end
            end
        end
    end
end

---------------------------------------------------------------------------
-- Body temperature integration
---------------------------------------------------------------------------


RC_TempSim.BODY_TEMP_NORMAL_C = 37.0
RC_TempSim.BODY_TEMP_MIN_C    = 20.0
RC_TempSim.BODY_TEMP_MAX_C    = RC_TempSim.BODY_TEMP_MAX_C or 42.0
RC_TempSim.TICKS_PER_MIN        = 75

RC_TempSim.HYPOTHERMIA_STAGE_1_C = 36.5
RC_TempSim.HYPOTHERMIA_STAGE_2_C = 35.0
RC_TempSim.HYPOTHERMIA_STAGE_3_C = 30.0
RC_TempSim.HYPOTHERMIA_STAGE_4_C = 25.0

RC_TempSim.COLD_UPDATES_PER_MIN = 120

RC_TempSim.WARM_RATE_CPM        = 0.20
RC_TempSim.WARM_BASE_CPM        = 0.005
RC_TempSim.AMBIENT_RECOVERY_START_C = 10.0
RC_TempSim.AMBIENT_RECOVERY_FULL_C  = 25.0

RC_TempSim.HEAT_PUSH_PER_C_CPM  = 0.05

RC_TempSim.WARM_COMFORT_AIR_C        = RC_TempSim.WARM_COMFORT_AIR_C        or RC_TempSim.VANILLA_BODY_TEMP_AIR_THRESHOLD_C or 20.0
RC_TempSim.WARM_AIR_HEAT_CPM         = RC_TempSim.WARM_AIR_HEAT_CPM         or 0.0003
RC_TempSim.WARM_AIR_BASELINE_FRACTION = RC_TempSim.WARM_AIR_BASELINE_FRACTION or 0.005
RC_TempSim.WARM_AIR_CRITICAL_FRACTION = RC_TempSim.WARM_AIR_CRITICAL_FRACTION or 0.15
RC_TempSim.WARM_CRITICAL_AIR_C       = RC_TempSim.WARM_CRITICAL_AIR_C       or 36.0
RC_TempSim.WARM_CRITICAL_SPAN        = RC_TempSim.WARM_CRITICAL_SPAN        or 5.0
RC_TempSim.WARM_INSULATION_HEAT_CPM  = RC_TempSim.WARM_INSULATION_HEAT_CPM  or 0.0004
RC_TempSim.WARM_MOVEMENT_HEAT_CPM    = RC_TempSim.WARM_MOVEMENT_HEAT_CPM    or 0.008
RC_TempSim.WARM_STAGE_TEMP_BUFFER_C  = RC_TempSim.WARM_STAGE_TEMP_BUFFER_C  or 0.01
local _warmNormalTemp = RC_TempSim.BODY_TEMP_NORMAL_C or 37.0
local _warmTempBuffer = RC_TempSim.WARM_STAGE_TEMP_BUFFER_C or 0.0
RC_TempSim.WARM_STAGE_TEMP_CAPS = RC_TempSim.WARM_STAGE_TEMP_CAPS or {
    [0] = _warmNormalTemp - _warmTempBuffer,
    [1] = (_warmNormalTemp + 2.0) - _warmTempBuffer,
    [2] = (_warmNormalTemp + 3.0) - _warmTempBuffer,
    [3] = (_warmNormalTemp + 4.0) - _warmTempBuffer,
    [4] = RC_TempSim.BODY_TEMP_MAX_C or 42.0,
}
RC_TempSim.WARM_STAGE_LIGHT_INSULATION_THRESHOLD = RC_TempSim.WARM_STAGE_LIGHT_INSULATION_THRESHOLD or 1
RC_TempSim.WARM_STAGE_LIGHT_AIR_THRESHOLD        = RC_TempSim.WARM_STAGE_LIGHT_AIR_THRESHOLD or 36.0
RC_TempSim.WARM_STAGE_LIGHT_RUN_THRESHOLD        = RC_TempSim.WARM_STAGE_LIGHT_RUN_THRESHOLD or 0.45
RC_TempSim.WARM_STAGE_LIGHT_STILL_CAP            = RC_TempSim.WARM_STAGE_LIGHT_STILL_CAP or 3.0
RC_TempSim.WARM_LOW_INSULATION_WALK_THRESHOLD    = RC_TempSim.WARM_LOW_INSULATION_WALK_THRESHOLD or 0.25
RC_TempSim.WARM_HIGH_INSULATION_MIN_SCALE        = RC_TempSim.WARM_HIGH_INSULATION_MIN_SCALE or 0.15
RC_TempSim.WARM_HIGH_INSULATION_RATE_SPAN        = RC_TempSim.WARM_HIGH_INSULATION_RATE_SPAN or 0.75
RC_TempSim.WARM_HIGH_INSULATION_MIN_TEMP         = RC_TempSim.WARM_HIGH_INSULATION_MIN_TEMP or 20.0
RC_TempSim.WARM_HIGH_INSULATION_MAX_TEMP         = RC_TempSim.WARM_HIGH_INSULATION_MAX_TEMP or 30.0
RC_TempSim.WARM_STAGE_BASE_MIN_C           = RC_TempSim.WARM_STAGE_BASE_MIN_C           or 30.0
RC_TempSim.WARM_STAGE_BASE_STAGE2_C        = RC_TempSim.WARM_STAGE_BASE_STAGE2_C        or 36.0
RC_TempSim.WARM_STAGE_BASE_MAX_C           = RC_TempSim.WARM_STAGE_BASE_MAX_C           or 42.0
RC_TempSim.WARM_STAGE_BASE_MAX_STAGE       = RC_TempSim.WARM_STAGE_BASE_MAX_STAGE       or 3.0
RC_TempSim.WARM_STAGE_CLOTHING_MIN_INSULATION   = RC_TempSim.WARM_STAGE_CLOTHING_MIN_INSULATION   or 2.5
RC_TempSim.WARM_STAGE_CLOTHING_MAX_INSULATION   = RC_TempSim.WARM_STAGE_CLOTHING_MAX_INSULATION   or 3.5
RC_TempSim.WARM_STAGE_CLOTHING_BONUS            = RC_TempSim.WARM_STAGE_CLOTHING_BONUS            or 2.0
RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_THRESHOLD       = RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_THRESHOLD       or 3.0
RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_STAGE_BONUS     = RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_STAGE_BONUS     or 1.0
RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_CAP_BONUS       = RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_CAP_BONUS       or 0.4
RC_TempSim.WARM_STAGE_HEAVY_RUN_TEMP_THRESHOLD       = RC_TempSim.WARM_STAGE_HEAVY_RUN_TEMP_THRESHOLD       or 30.0
RC_TempSim.WARM_STAGE_HEAVY_RUN_CAP_BASE             = RC_TempSim.WARM_STAGE_HEAVY_RUN_CAP_BASE             or 1.2
RC_TempSim.WARM_STAGE_HEAVY_RUN_CAP_BONUS            = RC_TempSim.WARM_STAGE_HEAVY_RUN_CAP_BONUS            or 1.8
RC_TempSim.WARM_STAGE_MOVEMENT_BASE             = RC_TempSim.WARM_STAGE_MOVEMENT_BASE             or 3.0
RC_TempSim.WARM_STAGE_MOVEMENT_CLOTHING_SCALE   = RC_TempSim.WARM_STAGE_MOVEMENT_CLOTHING_SCALE   or 0.8
RC_TempSim.WARM_STAGE_ROUNDING_BIAS             = RC_TempSim.WARM_STAGE_ROUNDING_BIAS             or 0.2
RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_THRESHOLD  = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_THRESHOLD  or 0.7
RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_PENALTY_MIN_C = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_PENALTY_MIN_C or 20.0
RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_PENALTY_MAX_C = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_PENALTY_MAX_C or 36.0
RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_MAX_REDUCTION = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_MAX_REDUCTION or 1.0
RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_RUN_RELIEF      = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_RUN_RELIEF      or 1.0
RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_RUN_REQUIRED    = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_RUN_REQUIRED    or 3.5
RC_TempSim.WARM_STAGE_RUN_ACCUM_MIN_HEAT             = RC_TempSim.WARM_STAGE_RUN_ACCUM_MIN_HEAT             or 0.25
RC_TempSim.WARM_STAGE_RUN_ACCUM_GAIN                 = RC_TempSim.WARM_STAGE_RUN_ACCUM_GAIN                 or 6.0
RC_TempSim.WARM_STAGE_RUN_ACCUM_DECAY                = RC_TempSim.WARM_STAGE_RUN_ACCUM_DECAY                or 1.5
RC_TempSim.WARM_STAGE_RUN_ACCUM_MAX                  = RC_TempSim.WARM_STAGE_RUN_ACCUM_MAX                  or 10.0
RC_TempSim.WARM_LIGHT_CLOTHING_THRESHOLD = RC_TempSim.WARM_LIGHT_CLOTHING_THRESHOLD or 0.65
RC_TempSim.WARM_LIGHT_CLOTHING_RATE_MULT = RC_TempSim.WARM_LIGHT_CLOTHING_RATE_MULT or 0.25
RC_TempSim.WARM_RECOVERY_TARGET_C = RC_TempSim.WARM_RECOVERY_TARGET_C or 36.5
RC_TempSim.WARM_RECOVERY_AIR_THRESHOLD_C = RC_TempSim.WARM_RECOVERY_AIR_THRESHOLD_C or 24.0
RC_TempSim.WARM_RECOVERY_AIR_FULL_C = RC_TempSim.WARM_RECOVERY_AIR_FULL_C or 20.0
RC_TempSim.WARM_COOLING_START_C = RC_TempSim.WARM_COOLING_START_C or 22.0
RC_TempSim.WARM_COOLING_FULL_C = RC_TempSim.WARM_COOLING_FULL_C or 28.0
RC_TempSim.WARM_COOLING_MAX_RATE = RC_TempSim.WARM_COOLING_MAX_RATE or 0.001
RC_TempSim.WARM_STAGE_CAP_COOL_RATE                = RC_TempSim.WARM_STAGE_CAP_COOL_RATE                or 0.7
RC_TempSim.WARM_STAGE_CAP_COOL_MIN                 = RC_TempSim.WARM_STAGE_CAP_COOL_MIN                 or 0.05
RC_TempSim.WARM_STAGE_CAP_COOL_AIR_BONUS           = RC_TempSim.WARM_STAGE_CAP_COOL_AIR_BONUS           or 0.14
RC_TempSim.WARM_STAGE_CAP_COOL_HOT_PENALTY         = RC_TempSim.WARM_STAGE_CAP_COOL_HOT_PENALTY         or 0.014
RC_TempSim.WARM_STAGE_CAP_COOL_INSULATION_SCALE    = RC_TempSim.WARM_STAGE_CAP_COOL_INSULATION_SCALE    or 0.35
RC_TempSim.WARM_STAGE_CAP_COOL_MIN_MULT            = RC_TempSim.WARM_STAGE_CAP_COOL_MIN_MULT            or 0.1
RC_TempSim.WARM_STAGE_CAP_RELAX_RATE               = RC_TempSim.WARM_STAGE_CAP_RELAX_RATE               or 0.4
RC_TempSim.WARM_STAGE_SUNSTRUCK_RUN_TEMP           = RC_TempSim.WARM_STAGE_SUNSTRUCK_RUN_TEMP           or 30.0
RC_TempSim.WARM_STAGE_SUNSTRUCK_RUN_THRESHOLD      = RC_TempSim.WARM_STAGE_SUNSTRUCK_RUN_THRESHOLD      or 0.75

RC_TempSim.SWEAT_OVERHEAT_STAGE           = RC_TempSim.SWEAT_OVERHEAT_STAGE           or 1
RC_TempSim.SWEAT_BASE_RATE_PER_MIN        = RC_TempSim.SWEAT_BASE_RATE_PER_MIN        or 1
RC_TempSim.SWEAT_MAX_RATE_PER_MIN         = RC_TempSim.SWEAT_MAX_RATE_PER_MIN         or 6
RC_TempSim.SWEAT_RAMP_MINUTES             = RC_TempSim.SWEAT_RAMP_MINUTES             or 10.0
RC_TempSim.SWEAT_STAGE_RATE_MULT          = RC_TempSim.SWEAT_STAGE_RATE_MULT          or 1.0
RC_TempSim.SWEAT_WETNESS_START            = RC_TempSim.SWEAT_WETNESS_START            or 1.0
RC_TempSim.SWEAT_MAX_WETNESS              = RC_TempSim.SWEAT_MAX_WETNESS              or 95.0

RC_TempSim.BODY_HEATGEN_WETNESS_THRESHOLD         = RC_TempSim.BODY_HEATGEN_WETNESS_THRESHOLD         or 45.0
RC_TempSim.BODY_HEATGEN_WETNESS_MAX               = RC_TempSim.BODY_HEATGEN_WETNESS_MAX               or 85.0
RC_TempSim.BODY_HEATGEN_WETNESS_TARGET_REDUCTION  = RC_TempSim.BODY_HEATGEN_WETNESS_TARGET_REDUCTION  or 0.18

RC_TempSim.DRINK_COOL_MIN_C                = RC_TempSim.DRINK_COOL_MIN_C                or 0.2
RC_TempSim.DRINK_COOL_MAX_C                = RC_TempSim.DRINK_COOL_MAX_C                or 0.5
RC_TempSim.DRINK_COOL_THIRST_RANGE         = RC_TempSim.DRINK_COOL_THIRST_RANGE         or 0.35
RC_TempSim.DRINK_HEAT_PENALTY_ADD          = RC_TempSim.DRINK_HEAT_PENALTY_ADD          or 0.12
RC_TempSim.DRINK_HEAT_PENALTY_MAX          = RC_TempSim.DRINK_HEAT_PENALTY_MAX          or 0.28
RC_TempSim.DRINK_HEAT_PENALTY_DECAY_PER_MIN = RC_TempSim.DRINK_HEAT_PENALTY_DECAY_PER_MIN or 0.02

RC_TempSim.VEHICLE_HEATER_OUTDOOR_LIMIT_C = RC_TempSim.VEHICLE_HEATER_OUTDOOR_LIMIT_C or -45.0
RC_TempSim.VEHICLE_HEATER_RAMP_DEGREES_PER_MIN = RC_TempSim.VEHICLE_HEATER_RAMP_DEGREES_PER_MIN or 3.0
RC_TempSim.VEHICLE_HEATER_MAX_TIME_STEP_MIN = RC_TempSim.VEHICLE_HEATER_MAX_TIME_STEP_MIN or 5.0
RC_TempSim.VEHICLE_HEATER_COOLDOWN_DEGREES_PER_MIN = RC_TempSim.VEHICLE_HEATER_COOLDOWN_DEGREES_PER_MIN or 1.5
RC_TempSim.VEHICLE_HEATER_VANILLA_MAX_DELTA_C = RC_TempSim.VEHICLE_HEATER_VANILLA_MAX_DELTA_C or 25.0
if RC_TempSim.VEHICLE_HEATER_MAX_DELTA_C == nil then
    RC_TempSim.VEHICLE_HEATER_MAX_DELTA_C = RC_TempSim.VEHICLE_HEATER_VANILLA_MAX_DELTA_C
end

if RC_TempSim.DEBUG_LOG_COOLING == nil then RC_TempSim.DEBUG_LOG_COOLING = true end
RC_TempSim.DEBUG_LOG_MIN_DELTA        = RC_TempSim.DEBUG_LOG_MIN_DELTA        or 0.05
RC_TempSim.DEBUG_LOG_MIN_INTERVAL_MIN = RC_TempSim.DEBUG_LOG_MIN_INTERVAL_MIN or (1.0 / 6.0)

if RC_TempSim.DEBUG_LOG_WARMING == nil then RC_TempSim.DEBUG_LOG_WARMING = true end
RC_TempSim.DEBUG_LOG_WARMING_MIN_DELTA        = RC_TempSim.DEBUG_LOG_WARMING_MIN_DELTA        or 0.05
RC_TempSim.DEBUG_LOG_WARMING_MIN_INTERVAL_MIN = RC_TempSim.DEBUG_LOG_WARMING_MIN_INTERVAL_MIN or (1.0 / 6.0)

local BODY_TEMP_DATA_KEY = "RC_TempSimBodyTemp"

local COOLING_LOOKUP = {
    { air = -40, rate = 0.0160 },
    { air = -35, rate = 0.0130 },
    { air = -30, rate = 0.0100 },
    { air = -25, rate = 0.0090 },
    { air = -20, rate = 0.0080 },
    { air = -15, rate = 0.0050 },
    { air = -10, rate = 0.0020 },
    { air = -5,  rate = 0.0015 },
    { air = -4,  rate = 0.0014 },
    { air = -3,  rate = 0.0013 },
    { air = -2,  rate = 0.0012 },
    { air = -1,  rate = 0.0011 },
    { air = 0,   rate = 0.0010 },
    { air = 5,   rate = 0.0007 },
    { air = 10,  rate = 0.0004 },
    { air = 15,  rate = 0.0002 },
    { air = 20,  rate = 0.0 },
}

local function coldStageForCore(temp)
    if not temp then return 0 end
    local stage1 = RC_TempSim.HYPOTHERMIA_STAGE_1_C or 36.5
    local stage2 = RC_TempSim.HYPOTHERMIA_STAGE_2_C or 35.0
    local stage3 = RC_TempSim.HYPOTHERMIA_STAGE_3_C or 30.0
    local stage4 = RC_TempSim.HYPOTHERMIA_STAGE_4_C or 25.0
    if temp <= stage4 then return 4 end
    if temp <= stage3 then return 3 end
    if temp <= stage2 then return 2 end
    if temp <= stage1 then return 1 end
    return 0
end

local function getBodyTempData(player)
    if not player or not player.getModData then return nil end
    local md = player:getModData()
    md[BODY_TEMP_DATA_KEY] = md[BODY_TEMP_DATA_KEY] or {}
    local rec = md[BODY_TEMP_DATA_KEY]
    if rec.core == nil then
        rec.core = RC_TempSim.BODY_TEMP_NORMAL_C
    end
    if rec.bodyHeatGeneration == nil then
        rec.bodyHeatGeneration = RC_TempSim.BODY_HEATGEN_TARGET_IDLE or 0.0
    end
    return rec
end

local function lerp(a, b, t)
    return a + (b - a) * t
end

local function computeWarmRecoveryTarget(air)
    local normal = RC_TempSim.BODY_TEMP_NORMAL_C or 37.0
    local warmTarget = RC_TempSim.WARM_RECOVERY_TARGET_C or normal
    if not air or not warmTarget or warmTarget >= normal then
        return normal
    end

    local startC = RC_TempSim.WARM_RECOVERY_AIR_THRESHOLD_C or normal
    local fullC = RC_TempSim.WARM_RECOVERY_AIR_FULL_C or startC

    if air <= startC then
        return normal
    end

    if fullC <= startC then
        return math.min(normal, warmTarget)
    end

    local t = (air - startC) / (fullC - startC)
    if t < 0 then t = 0 end
    if t > 1 then t = 1 end

    return normal + (warmTarget - normal) * t
end

local function _fallbackItemInsulation(player)
    if not player or not player.getWornItems then return 0 end

    local worn = player:getWornItems()
    if not worn or not worn.size or worn:size() <= 0 then return 0 end

    local total = 0.0
    for i = 0, worn:size() - 1 do
        local item = worn:getItemByIndex(i)
        if item and item.getInsulation then
            local insulation = item:getInsulation()
            if insulation and insulation > 0 then
                total = total + insulation
            end
        end
    end

    return total
end

local function summarizeThermalNodes(player)
    local summary = {
        count = 0,
        totalInsulation = 0.0,
        avgInsulation = 0.0,
        totalWindresist = 0.0,
        avgWindresist = 0.0,
        totalClothingWetness = 0.0,
        avgClothingWetness = 0.0,
        totalBodyWetness = 0.0,
        avgBodyWetness = 0.0,
        minInsulation = nil,
        minInsulationName = nil,
        source = "fallback",
    }

    if not player or not player.getBodyDamage then
        summary.totalInsulation = _fallbackItemInsulation(player)
        summary.avgInsulation = summary.totalInsulation
        return summary
    end

    local bodyDamage = player:getBodyDamage()
    local thermoregulator = bodyDamage and bodyDamage.getThermoregulator and bodyDamage:getThermoregulator() or nil
    if not thermoregulator or not thermoregulator.getNodeSize then
        summary.totalInsulation = _fallbackItemInsulation(player)
        summary.avgInsulation = summary.totalInsulation
        return summary
    end

    local nodeCount = thermoregulator:getNodeSize() or 0
    if nodeCount <= 0 then
        summary.totalInsulation = _fallbackItemInsulation(player)
        summary.avgInsulation = summary.totalInsulation
        return summary
    end

    summary.source = "nodes"

    for i = 0, nodeCount - 1 do
        local node = thermoregulator:getNode(i)
        if node then
            local insulation = node.getInsulation and node:getInsulation() or 0.0
            local wind = node.getWindresist and node:getWindresist() or 0.0
            local clothingWet = node.getClothingWetness and node:getClothingWetness() or 0.0
            local bodyWet = node.getBodyWetness and node:getBodyWetness() or 0.0

            summary.totalInsulation = summary.totalInsulation + insulation
            summary.totalWindresist = summary.totalWindresist + wind
            summary.totalClothingWetness = summary.totalClothingWetness + clothingWet
            summary.totalBodyWetness = summary.totalBodyWetness + bodyWet
            summary.count = summary.count + 1

            if insulation then
                if summary.minInsulation == nil or insulation < summary.minInsulation then
                    summary.minInsulation = insulation
                    summary.minInsulationName = node.getName and node:getName() or tostring(i)
                end
            end
        end
    end

    if summary.count > 0 then
        summary.avgInsulation = summary.totalInsulation / summary.count
        summary.avgWindresist = summary.totalWindresist / summary.count
        summary.avgClothingWetness = summary.totalClothingWetness / summary.count
        summary.avgBodyWetness = summary.totalBodyWetness / summary.count
    else
        summary.totalInsulation = _fallbackItemInsulation(player)
        summary.avgInsulation = summary.totalInsulation
    end

    return summary
end

local function _getGameTimeHours()
    local gt = getGameTime and getGameTime() or nil
    return gt and gt.getWorldAgeHours and gt:getWorldAgeHours() or 0
end

local function _playerDebugName(player)
    if not player then return "player" end
    if player.getUsername then
        local username = player:getUsername()
        if username and username ~= "" then return username end
    end
    if player.getFullName then
        local fullName = player:getFullName()
        if fullName and fullName ~= "" then return fullName end
    end
    return tostring(player)
end

local function _heatStageClothingFactor(thermalSummary)
    if not thermalSummary then return 0.0 end

    local insulation = thermalSummary.avgInsulation or 0.0
    local minIns = RC_TempSim.WARM_STAGE_CLOTHING_MIN_INSULATION or 0.0
    local maxIns = RC_TempSim.WARM_STAGE_CLOTHING_MAX_INSULATION or (minIns + 1.0)
    if maxIns <= minIns then
        return insulation > minIns and 1.0 or 0.0
    end

    local normalized = (insulation - minIns) / (maxIns - minIns)
    if normalized < 0 then normalized = 0 end
    if normalized > 1 then normalized = 1 end
    return normalized
end

local function _heavyClothingFraction(thermalSummary)
    if not thermalSummary then return 0.0 end

    local insulation = thermalSummary.avgInsulation or 0.0
    local threshold = RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_THRESHOLD or 0.0
    if insulation <= threshold then
        return 0.0
    end

    local maxIns = RC_TempSim.WARM_STAGE_CLOTHING_MAX_INSULATION or (threshold + 1.0)
    local span = maxIns - threshold
    if span <= 0 then
        return 1.0
    end

    local normalized = (insulation - threshold) / span
    if normalized < 0 then normalized = 0 end
    if normalized > 1 then normalized = 1 end
    return normalized
end

local function _updateWarmRunAccumulator(rec, minutes, movementHeat)
    if not rec then return 0.0, 0.0 end
    if not minutes or minutes <= 0 then
        local current = rec._warmRunAccum or 0.0
        local required = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_RUN_REQUIRED or 0.0
        local fraction = 0.0
        if required > 0 then
            fraction = current / required
            if fraction > 1 then fraction = 1 end
        end
        return current, fraction
    end

    local accum = rec._warmRunAccum or 0.0
    local heat = math.max(0.0, movementHeat or 0.0)
    local minHeat = RC_TempSim.WARM_STAGE_RUN_ACCUM_MIN_HEAT or 0.0
    local gain = RC_TempSim.WARM_STAGE_RUN_ACCUM_GAIN or 0.0
    local decay = RC_TempSim.WARM_STAGE_RUN_ACCUM_DECAY or 0.0

    if heat > minHeat and gain > 0 then
        accum = accum + (heat - minHeat) * gain * minutes
    elseif decay > 0 then
        accum = accum - decay * minutes
    end

    if accum < 0 then accum = 0 end
    local maxAccum = RC_TempSim.WARM_STAGE_RUN_ACCUM_MAX
    if maxAccum and maxAccum > 0 and accum > maxAccum then
        accum = maxAccum
    end

    rec._warmRunAccum = accum

    local required = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_RUN_REQUIRED or 0.0
    local fraction = 0.0
    if required > 0 then
        fraction = accum / required
        if fraction > 1 then fraction = 1 end
    elseif accum > 0 then
        fraction = 1.0
    end

    return accum, fraction
end

local function _heatStageBasePotential(airC)
    if not airC then return 0.0 end

    local minC    = RC_TempSim.WARM_STAGE_BASE_MIN_C or 22.0
    local stage2C = RC_TempSim.WARM_STAGE_BASE_STAGE2_C or (minC + 4.0)
    local maxC    = RC_TempSim.WARM_STAGE_BASE_MAX_C or 40.0
    local maxStage = RC_TempSim.WARM_STAGE_BASE_MAX_STAGE or 3.0

    if airC <= minC then
        return 0.0
    end

    if airC <= stage2C then
        local span = stage2C - minC
        if span <= 0 then return 0.0 end
        local t = (airC - minC) / span
        return 2.0 * t
    end

    if airC >= maxC then
        return maxStage
    end

    local span = maxC - stage2C
    if span <= 0 then return 2.0 end
    local t = (airC - stage2C) / span
    if t < 0 then t = 0 end
    if t > 1 then t = 1 end
    return 2.0 + t * (maxStage - 2.0)
end

local function _heatStageAirCap(airC, basePotential, clothingFactor, runFraction, heavyFraction, movementNormalized, avgInsulation)
    if not airC then return 4.0 end

    local stage2C = RC_TempSim.WARM_STAGE_BASE_STAGE2_C or 26.0
    local maxC    = RC_TempSim.WARM_STAGE_BASE_MAX_C or 40.0
    local cap
    if airC < stage2C then
        cap = 2.0
    elseif airC < maxC then
        cap = 3.0
    else
        cap = 4.0
    end

    local penaltyMin = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_PENALTY_MIN_C or 25.0
    local penaltyMax = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_PENALTY_MAX_C or 32.0
    local threshold  = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_THRESHOLD or 0.35
    local maxReduction = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_MAX_REDUCTION or 1.0

    if airC >= penaltyMin and airC < penaltyMax and clothingFactor < threshold then
        local span = penaltyMax - penaltyMin
        if span < 0.0001 then span = 0.0001 end
        local t = (airC - penaltyMin) / span
        if t < 0 then t = 0 end
        if t > 1 then t = 1 end

        local deficit = threshold - clothingFactor
        if deficit < 0 then deficit = 0 end
        local normalized = deficit / threshold
        if normalized > 1 then normalized = 1 end

        local reduction = maxReduction * normalized * (1.0 - t)
        if runFraction and runFraction > 0 then
            local reliefMax = RC_TempSim.WARM_STAGE_LIGHT_CLOTHING_RUN_RELIEF or 0.0
            if reliefMax > 0 then
                local relief = runFraction * reliefMax
                if relief > 1 then relief = 1 end
                reduction = reduction * (1.0 - relief)
            end
        end
        cap = cap - reduction
    end

    local minCap = math.ceil((basePotential or 0.0) - 1e-4)
    local heavyBoost = 0.0
    if heavyFraction and heavyFraction > 0 then
        local baseBonus = RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_CAP_BONUS or 0.0
        if baseBonus > 0 then
            heavyBoost = heavyBoost + heavyFraction * baseBonus
        end

        local runBase = RC_TempSim.WARM_STAGE_HEAVY_RUN_CAP_BASE or 0.0
        local runBonus = RC_TempSim.WARM_STAGE_HEAVY_RUN_CAP_BONUS or 0.0
        if runFraction and runFraction > 0 then
            local addition = 0.0
            if runBase > 0 then
                addition = addition + runBase
            end
            local runTempThreshold = RC_TempSim.WARM_STAGE_HEAVY_RUN_TEMP_THRESHOLD or stage2C
            if runBonus > 0 and airC >= runTempThreshold then
                addition = addition + heavyFraction * runBonus
            end
            if addition > 0 then
                heavyBoost = heavyBoost + runFraction * addition
            end
        end
    end

    if heavyBoost ~= 0 then
        cap = cap + heavyBoost
    end

    local lightInsulationThreshold = RC_TempSim.WARM_STAGE_LIGHT_INSULATION_THRESHOLD or 0.65
    local lightAirThreshold = RC_TempSim.WARM_STAGE_LIGHT_AIR_THRESHOLD or 36.0
    local lightRunThreshold = RC_TempSim.WARM_STAGE_LIGHT_RUN_THRESHOLD or 0.45
    local lightStillCap = RC_TempSim.WARM_STAGE_LIGHT_STILL_CAP or 3.0

    local insulation = avgInsulation or 0.0
    local movement = movementNormalized or 0.0

    if insulation >= 0 and insulation < lightInsulationThreshold and airC < lightAirThreshold then
        if movement >= lightRunThreshold then
            if cap < 4 then
                cap = 4.0
            end
        else
            if cap > lightStillCap then
                cap = lightStillCap
            end
        end
    end

    if cap < minCap then cap = minCap end
    if cap < 0 then cap = 0 end
    if cap > 4 then cap = 4 end
    return cap
end

local function applyStageCapCooling(current, target, minutes, clothingMult, thermalSummary, airC)
    if not target or not current then return current, 0.0 end
    if current <= target then return current, 0.0 end
    if not minutes or minutes <= 0 then return current, 0.0 end

    local delta = current - target
    local baseRate = RC_TempSim.WARM_STAGE_CAP_COOL_RATE or 0.0
    local rate = baseRate

    local airBonus = RC_TempSim.WARM_STAGE_CAP_COOL_AIR_BONUS or 0.0
    local hotPenalty = RC_TempSim.WARM_STAGE_CAP_COOL_HOT_PENALTY or 0.0
    if airC then
        if airC < target and airBonus ~= 0 then
            rate = rate + (target - airC) * airBonus
        elseif airC > target and hotPenalty ~= 0 then
            rate = rate - (airC - target) * hotPenalty
        end
    end

    local insulation = thermalSummary and thermalSummary.avgInsulation or 0.0
    insulation = math.max(0.0, insulation or 0.0)
    local insulationScale = RC_TempSim.WARM_STAGE_CAP_COOL_INSULATION_SCALE or 0.0
    if insulationScale > 0 and insulation > 0 then
        rate = rate / (1.0 + insulation * insulationScale)
    end

    if clothingMult and clothingMult > 0 then
        local mult = clothingMult
        local minMult = RC_TempSim.WARM_STAGE_CAP_COOL_MIN_MULT or 0.0
        if minMult > 0 and mult < minMult then
            mult = minMult
        end
        rate = rate * mult
    end

    local minRate = RC_TempSim.WARM_STAGE_CAP_COOL_MIN or 0.0
    if rate < minRate then rate = minRate end
    if rate < 0 then rate = 0 end

    local applied = rate * minutes
    if applied > delta then applied = delta end

    local cooled = current - applied
    if cooled > target then
        cooled = target
        applied = current - cooled
    end

    return cooled, applied
end

local function computeHeatStageCap(airC, thermalSummary, movementHeat, runFraction)
    if not airC then return nil, nil, nil end

    local clothingFactor = _heatStageClothingFactor(thermalSummary)
    local basePotential = _heatStageBasePotential(airC)
    local clothingContribution = clothingFactor * (RC_TempSim.WARM_STAGE_CLOTHING_BONUS or 2.0)
    local heavyFraction = _heavyClothingFraction(thermalSummary)
    if heavyFraction > 0 then
        local heavyBonus = RC_TempSim.WARM_STAGE_HEAVY_CLOTHING_STAGE_BONUS or 0.0
        if heavyBonus > 0 then
            clothingContribution = clothingContribution + heavyFraction * heavyBonus
        end
    end

    local movementContribution = 0.0
    local normalized = _normalizedMovementHeat(movementHeat)
    local sunRunTemp = RC_TempSim.WARM_STAGE_SUNSTRUCK_RUN_TEMP or 28.0
    local sunRunThreshold = RC_TempSim.WARM_STAGE_SUNSTRUCK_RUN_THRESHOLD or 0.75
    local runActive = false
    if normalized and normalized >= sunRunThreshold then
        runActive = true
    end
    if not runActive and runFraction and runFraction >= sunRunThreshold then
        runActive = true
    end
    local sunstruckEligible = airC and airC >= sunRunTemp and runActive
    if normalized > 0 then
        local moveBase = RC_TempSim.WARM_STAGE_MOVEMENT_BASE or 2.0
        local moveScale = RC_TempSim.WARM_STAGE_MOVEMENT_CLOTHING_SCALE or 0.8
        movementContribution = normalized * (moveBase + moveScale * clothingFactor)
    end

    local rawStage = basePotential + clothingContribution + movementContribution
    local rounding = RC_TempSim.WARM_STAGE_ROUNDING_BIAS or 0.5
    local stageInt = math.floor(rawStage + rounding)
    if stageInt < 0 then stageInt = 0 end
    if stageInt > 4 then stageInt = 4 end

    local stage2C = RC_TempSim.WARM_STAGE_BASE_STAGE2_C or 26.0
    if airC and airC <= stage2C then
        if not (movementContribution > 0 and clothingContribution > 0) then
            if stageInt > 2 then stageInt = 2 end
        end
    end

    local avgInsulation = thermalSummary and thermalSummary.avgInsulation or nil
    local maxStageFloat = _heatStageAirCap(airC, basePotential, clothingFactor, runFraction, heavyFraction, normalized, avgInsulation)
    if sunstruckEligible and (not maxStageFloat or maxStageFloat < 4.0) then
        maxStageFloat = 4.0
    end
    if maxStageFloat then
        local maxStageInt = math.floor(maxStageFloat + 0.0001)
        if maxStageInt < 0 then maxStageInt = 0 end
        if maxStageInt > 4 then maxStageInt = 4 end
        if stageInt > maxStageInt then
            stageInt = maxStageInt
        end
    end

    if sunstruckEligible and stageInt < 4 then
        stageInt = 4
    end

    local tempCap = nil
    local caps = RC_TempSim.WARM_STAGE_TEMP_CAPS
    if caps then
        tempCap = caps[stageInt]
        if not tempCap then
            if stageInt >= 4 then
                tempCap = RC_TempSim.BODY_TEMP_MAX_C or 42.0
            else
                tempCap = caps[math.max(0, stageInt - 1)] or (RC_TempSim.BODY_TEMP_NORMAL_C + 0.5)
            end
        end
    end

    return stageInt, tempCap, {
        stage = stageInt,
        basePotential = basePotential,
        clothingFactor = clothingFactor,
        clothingContribution = clothingContribution,
        heavyFraction = heavyFraction,
        movementHeat = movementHeat or 0.0,
        movementContribution = movementContribution,
        rawStage = rawStage,
        maxStage = maxStageFloat,
        runFraction = runFraction,
        sunstruckEligible = sunstruckEligible,
    }
end

local function logHeatGain(player, airC, minutes, startCore, endCore, clothingMult, movementHeat, thermalSummary, movementRecovery, warmBreakdown, proxGain, heatStressBreakdown, stageFloorGain, stageCapClamp, stageCapInfo, bodyHeatInfo, directDetails)
    if not RC_TempSim.DEBUG_LOG_WARMING then return end

    local totalGain = (endCore or 0) - (startCore or 0)
    if totalGain <= 0 then return end

    local key = _playerDebugName(player)
    local debugState = RC_TempSim._debugWarming or {}
    RC_TempSim._debugWarming = debugState

    local prev = debugState[key]
    local nowHours = _getGameTimeHours()
    local shouldLog = prev == nil

    local function deltaExceeds(a, b)
        return math.abs((a or 0) - (b or 0)) >= RC_TempSim.DEBUG_LOG_WARMING_MIN_DELTA
    end

    if not shouldLog then
        if deltaExceeds(totalGain, prev.totalGain) then shouldLog = true end
        if deltaExceeds(airC, prev.airC) then shouldLog = true end
        if deltaExceeds(clothingMult, prev.clothingMult) then shouldLog = true end
        if deltaExceeds(movementHeat, prev.movementHeat) then shouldLog = true end
        local minutesSince = (nowHours - (prev.lastHours or 0)) * 60.0
        if minutesSince >= (RC_TempSim.DEBUG_LOG_WARMING_MIN_INTERVAL_MIN or 0) then
            shouldLog = shouldLog or deltaExceeds(totalGain, prev.totalGain)
        end
    end

    if not shouldLog then return end

    debugState[key] = {
        totalGain = totalGain,
        airC = airC,
        clothingMult = clothingMult,
        movementHeat = movementHeat,
        bodyHeat = bodyHeatInfo and bodyHeatInfo.after or movementHeat,
        lastHours = nowHours,
    }

    local parts = {}
    if warmBreakdown and (warmBreakdown.applied or 0) > 0 then
        table.insert(parts, string.format(
            "recovery=%.3f (base=%.3f ambient=%.3f hot=%.3f push=%.3f)",
            warmBreakdown.applied or 0,
            warmBreakdown.baseApplied or 0,
            warmBreakdown.ambientApplied or 0,
            warmBreakdown.hotApplied or 0,
            warmBreakdown.pushApplied or 0
        ))
    end

    if movementRecovery and movementRecovery > 0 then
        table.insert(parts, string.format("movementRecov=%.3f", movementRecovery))
    end

    if directDetails and directDetails.applied and directDetails.applied > 0 then
        table.insert(parts, string.format(
            "movementDirect=%.3f (rate=%.3f warm=%.2f intensity=%.2f)",
            directDetails.applied or 0,
            directDetails.rate or 0,
            directDetails.warmMultiplier or 0,
            directDetails.intensity or 0
        ))
    end

    if proxGain and proxGain > 0 then
        table.insert(parts, string.format("proximity=%.3f", proxGain))
    end

    if heatStressBreakdown and (heatStressBreakdown.applied or 0) > 0 then
        table.insert(parts, string.format(
            "heatStress=%.3f (env=%.3f ins=%.3f move=%.3f warmExcess=%.1f)",
            heatStressBreakdown.applied or 0,
            heatStressBreakdown.envApplied or 0,
            heatStressBreakdown.insulationApplied or 0,
            heatStressBreakdown.movementApplied or 0,
            heatStressBreakdown.warmExcess or 0
        ))
    end

    if stageFloorGain and stageFloorGain > 0 then
        table.insert(parts, string.format("stageFloor=%.3f", stageFloorGain))
    end

    if stageCapClamp and stageCapClamp > 0 then
        local stageDesc = ""
        if stageCapInfo then
            local maxStage = stageCapInfo.maxStage or 0
            stageDesc = string.format(
                " stage=%d raw=%.2f max=%.2f cloth=%.2f move=%.2f",
                stageCapInfo.stage or -1,
                stageCapInfo.rawStage or 0,
                maxStage,
                stageCapInfo.clothingContribution or 0,
                stageCapInfo.movementContribution or 0
            )
        end
        table.insert(parts, string.format("stageCap=%.3f%s", stageCapClamp, stageDesc))
    end

    table.insert(parts, string.format("minutes=%.2f", minutes or 0))
    table.insert(parts, string.format("clothingMult=%.2f", clothingMult or 0))
    if thermalSummary then
        table.insert(parts, string.format("avgIns=%.3f", thermalSummary.avgInsulation or 0))
    end
    table.insert(parts, string.format("movementHeat=%.2f", movementHeat or 0))
    if bodyHeatInfo then
        table.insert(parts, string.format(
            "bodyHeat=%.3f->%.3f target=%.3f norm=%.2f inc=%.3f×%.2f dec=%.3f×%.2f load=%.2f weight=%.2f passive=%.2f active=%.2f",
            bodyHeatInfo.before or 0,
            bodyHeatInfo.after or 0,
            bodyHeatInfo.target or 0,
            bodyHeatInfo.normalized or movementHeat or 0,
            bodyHeatInfo.increaseRate or 0,
            bodyHeatInfo.increaseMult or 0,
            bodyHeatInfo.decreaseRate or 0,
            bodyHeatInfo.decreaseMult or 0,
            bodyHeatInfo.heavyBonus or 0,
            bodyHeatInfo.weightBonus or 0,
            bodyHeatInfo.passiveTarget or 0,
            bodyHeatInfo.activityTarget or 0
        ))
    end

    local message = string.format(
        "[RC_TempSim][Warm] %s air=%.1f core=%.2f->%.2f (+%.3f)",
        key,
        airC or 0,
        startCore or 0,
        endCore or 0,
        totalGain
    )

    if #parts > 0 then
        message = string.format("%s %s", message, table.concat(parts, ", "))
    end

    -- print(message)
end

local function logCoolingSummary(player, summary, mult, context)
    if not RC_TempSim.DEBUG_LOG_COOLING then return end

    local debugState = RC_TempSim._debugCooling or {}
    RC_TempSim._debugCooling = debugState

    local key = "player"
    if player then
        if player.getUsername then
            key = player:getUsername() or key
        elseif player.getDescriptor and player:getDescriptor() and player:getDescriptor().getForename then
            key = tostring(player:getDescriptor():getForename())
        else
            key = tostring(player)
        end
    end

    local prev = debugState[key]
    local nowHours = _getGameTimeHours()
    local shouldLog = prev == nil

    local function deltaExceeds(a, b)
        return math.abs((a or 0) - (b or 0)) >= RC_TempSim.DEBUG_LOG_MIN_DELTA
    end

    if not shouldLog then
        if deltaExceeds(summary.avgInsulation, prev.avgInsulation) then shouldLog = true end
        if deltaExceeds(summary.avgClothingWetness, prev.avgClothingWetness) then shouldLog = true end
        if deltaExceeds(summary.avgBodyWetness, prev.avgBodyWetness) then shouldLog = true end
        if deltaExceeds(mult, prev.mult) then shouldLog = true end
        if deltaExceeds(summary.avgWindresist, prev.avgWindresist) then shouldLog = true end
        if (summary.source or "") ~= (prev.source or "") then shouldLog = true end
        local minutesSince = (nowHours - (prev.lastHours or 0)) * 60.0
        if minutesSince >= RC_TempSim.DEBUG_LOG_MIN_INTERVAL_MIN then
            shouldLog = shouldLog or deltaExceeds(summary.totalInsulation, prev.totalInsulation)
        end
    end

    if not shouldLog then return end

    debugState[key] = {
        avgInsulation = summary.avgInsulation,
        totalInsulation = summary.totalInsulation,
        avgWindresist = summary.avgWindresist,
        avgClothingWetness = summary.avgClothingWetness,
        avgBodyWetness = summary.avgBodyWetness,
        minInsulation = summary.minInsulation,
        minInsulationName = summary.minInsulationName,
        mult = mult,
        lastHours = nowHours,
        source = summary.source,
    }

    local label = string.format("[RC_TempSim] %s coolingMult=%.3f", key, mult)
    local parts = {
        string.format("source=%s", summary.source or "unknown"),
        string.format("avgClothIns=%.3f", summary.avgInsulation or 0),
        string.format("totalIns=%.3f", summary.totalInsulation or 0),
        string.format("avgWind=%.3f", summary.avgWindresist or 0),
        string.format("clothWet=%.3f", summary.avgClothingWetness or 0),
        string.format("bodyWet=%.3f", summary.avgBodyWetness or 0),
    }

    if summary.minInsulation ~= nil then
        table.insert(parts, string.format("minIns=%.3f@%s", summary.minInsulation, summary.minInsulationName or "?"))
    end

    if context then
        if context.airC ~= nil then
            table.insert(parts, string.format("air=%.1f", context.airC))
        end
        if context.severity ~= nil then
            table.insert(parts, string.format("severity=%.2f", context.severity))
        end
        if context.scaledInsulation ~= nil then
            table.insert(parts, string.format("scaledIns=%.2f", context.scaledInsulation))
        end
        if context.scaleStrength ~= nil then
            table.insert(parts, string.format("scaleStr=%.2f", context.scaleStrength))
        end
        if context.boost and context.boost ~= 1 then
            table.insert(parts, string.format("boost=%.2f", context.boost))
        end
        if context.effectiveMin ~= nil then
            table.insert(parts, string.format("effMin=%.3f", context.effectiveMin))
        end
        if context.windIntensity ~= nil then
            table.insert(parts, string.format("wind=%.2f", context.windIntensity))
        end
        if context.windExposure ~= nil then
            table.insert(parts, string.format("windExp=%.2f", context.windExposure))
        end
        if context.windChillMult and context.windChillMult ~= 1.0 then
            table.insert(parts, string.format("windMult=%.3f", context.windChillMult))
        end
    end

    local message = label
    if #parts > 0 then
        message = string.format("%s %s", label, table.concat(parts, ", "))
    end
    -- print(message)
end

local function _severeColdFactor(airC)
    if airC == nil then
        return 0
    end

    local startC = RC_TempSim.CLOTHING_SEVERE_COLD_START_C or -5.0
    local fullC  = RC_TempSim.CLOTHING_SEVERE_COLD_FULL_C or -35.0
    local warmThreshold = math.max(startC, fullC)
    local coldThreshold = math.min(startC, fullC)

    if airC >= warmThreshold then
        return 0
    end

    if airC <= coldThreshold then
        return 1
    end

    local span = warmThreshold - coldThreshold
    if span <= 0 then
        return 0
    end

    return _clamp01((warmThreshold - airC) / span)
end

local function _highInsulationDelayMultiplier(insulation)
    if not insulation then return 1.0 end

    local start = RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_START or 0
    local strength = RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_STRENGTH or 0
    if strength <= 0 then return 1.0 end

    if insulation <= start then
        return 1.0
    end

    local span = RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_SPAN or 1.0
    if span <= 0 then span = 1.0 end

    local normalized = (insulation - start) / span
    if normalized < 0 then normalized = 0 end

    local exp = RC_TempSim.CLOTHING_HIGH_INSULATION_DELAY_EXP or 1.0
    if exp < 1.0 then exp = 1.0 end

    local factor = 1.0 / (1.0 + strength * math.pow(normalized, exp))
    if factor < 0 then factor = 0 end
    if factor > 1 then factor = 1 end

    return factor
end

local function computeInsulationStageFloor(thermalSummary, airC)
    if not thermalSummary then return nil end

    local insulation = thermalSummary.avgInsulation or 0
    if insulation <= 0 then return nil end

    if not airC then return nil end
    local minOutdoor = RC_TempSim.CLOTHING_STAGE_CAP_MIN_OUTDOOR_C or -math.huge
    if airC <= minOutdoor then
        return nil
    end

    local freezeCap = RC_TempSim.CLOTHING_STAGE_FREEZE_CAP_INSULATION or 0
    local coldCap = RC_TempSim.CLOTHING_STAGE_COLD_CAP_INSULATION or 0

    local stageFloor = nil
    if insulation >= coldCap and coldCap > 0 then
        stageFloor = RC_TempSim.HYPOTHERMIA_STAGE_3_C
    elseif insulation >= freezeCap and freezeCap > 0 then
        stageFloor = RC_TempSim.HYPOTHERMIA_STAGE_4_C
    end

    if not stageFloor then
        return nil
    end

    local minC = RC_TempSim.BODY_TEMP_MIN_C or stageFloor
    if minC < stageFloor then
        minC = stageFloor
    end

    return minC
end

local function clothingCoolingMultiplier(player, airC)
    local summary = summarizeThermalNodes(player)
    local insulation = summary.avgInsulation or 0.0

    local scale = RC_TempSim.CLOTHING_INSULATION_SCALE or 1.0
    if scale < 0 then scale = 0 end

    local scaled = math.max(0.0, insulation * scale)
    local scaleStrength = 1.0 + scale
    local effective = scaled * scaleStrength

    -- Increase the impact of both insulation and the sandbox scale.  The
    -- sandbox scale now amplifies the effective insulation directly as well
    -- as providing an additional divisor, putting the sandbox option in
    -- control of how quickly clothing slows cooling.
    local curveExp = RC_TempSim.CLOTHING_INSULATION_CURVE_EXP
    local curveCoeff = nil
    local mult
    if effective > 0 and curveExp and curveExp > 1.0 then
        local anchor = RC_TempSim.CLOTHING_INSULATION_CURVE_ANCHOR
        if anchor and anchor > 0 then
            local anchorScaled = math.max(0.0, anchor * scale)
            local anchorEffective = anchorScaled * scaleStrength
            if anchorEffective > 0 then
                curveCoeff = math.pow(anchorEffective, 1.0 - curveExp)
            end
        end
        if not curveCoeff then
            curveCoeff = 1.0
        end
        mult = 1.0 / (1.0 + curveCoeff * math.pow(effective, curveExp))
    else
        mult = 1.0 / (1.0 + effective)
    end

    local delayMult = _highInsulationDelayMultiplier(insulation)
    if delayMult < 1.0 then
        mult = mult * delayMult
    end

    local severity = _severeColdFactor(airC)
    local boostApplied = 1.0
    local effectiveMin = nil
    local stageFloor = nil
    local windChillMult = 1.0
    local windIntensity = 0.0
    local windExposure = 0.0

    if severity > 0 then
        local protectionScale = RC_TempSim.CLOTHING_LOW_PROTECTION_SCALE or 0
        local protection = 0.0
        if protectionScale > 0 then
            -- Use a saturating curve so very high insulation never reaches a
            -- perfect 1.0 protection value; this keeps severe cold penalties
            -- and boosts responsive at extreme insulation levels.
            protection = scaled / (scaled + protectionScale)
            if protection < 0 then protection = 0 end
            if protection > 1 then protection = 1 end
        end

        local boostMax = RC_TempSim.CLOTHING_SEVERE_COLD_EXPOSURE_BOOST or 0
        if boostMax > 0 then
            boostApplied = 1.0 + boostMax * severity * (1.0 - protection)
            mult = mult * boostApplied
        end

        local minMult = RC_TempSim.CLOTHING_COOLING_MULT_MIN
        if minMult then
            local fraction = RC_TempSim.CLOTHING_DYNAMIC_MIN_FRACTION or 0
            if fraction < 0 then fraction = 0 end
            if fraction > 1 then fraction = 1 end
            local scaledMin = minMult / scaleStrength
            if scaledMin < 0 then scaledMin = 0 end
            local floor = scaledMin * fraction
            effectiveMin = scaledMin - (scaledMin - floor) * severity * protection
            if effectiveMin < 0 then effectiveMin = 0 end
            if mult < effectiveMin then
                mult = effectiveMin
            end
        end
    elseif RC_TempSim.CLOTHING_COOLING_MULT_MIN then
        local scaledMin = RC_TempSim.CLOTHING_COOLING_MULT_MIN / scaleStrength
        if scaledMin < 0 then scaledMin = 0 end
        if mult < scaledMin then mult = scaledMin end
    end

    if RC_TempSim.CLOTHING_COOLING_MULT_MAX then
        mult = math.min(RC_TempSim.CLOTHING_COOLING_MULT_MAX, mult)
    end

    local windEffect = RC_TempSim.WIND_CHILL_EFFECT_MULT or 0.0
    if windEffect ~= 0 then
        local climate = getClimateManager and getClimateManager() or nil
        if climate then
            local ok, value = pcall(function() return climate:getWindIntensity() end)
            if ok and type(value) == "number" then
                windIntensity = math.max(0.0, value)
            elseif climate.getWindPower then
                local okPower, power = pcall(function() return climate:getWindPower() end)
                if okPower and type(power) == "number" then
                    windIntensity = math.max(0.0, power)
                end
            end
        end

        if windIntensity > 0 then
            local resist = math.max(0.0, summary.avgWindresist or 0.0)
            local resistFraction = 0.0
            if resist > 0 then
                resistFraction = resist / (resist + 1.0)
            end
            local exposed = 1.0 - resistFraction
            if exposed < 0 then exposed = 0 end
            windExposure = windIntensity * exposed
            if windExposure ~= 0 then
                windChillMult = 1.0 + windExposure * windEffect
                if windChillMult < 0 then windChillMult = 0 end
                mult = mult * windChillMult
            end
        end
    end

    stageFloor = computeInsulationStageFloor(summary, airC)

    logCoolingSummary(player, summary, mult, {
        airC = airC,
        severity = severity,
        scaledInsulation = scaled,
        scaleStrength = scaleStrength,
        curveExp = curveExp,
        curveCoeff = curveCoeff,
        effectiveInsulation = effective,
        boost = boostApplied,
        effectiveMin = effectiveMin,
        delayMult = delayMult,
        stageFloor = stageFloor,
        windIntensity = windIntensity,
        windExposure = windExposure,
        windChillMult = windChillMult,
    })

    return mult, summary
end

local function _vehicleHeaterStateTable()
    local states = RC_TempSim._vehicleHeaterState
    if not states then
        states = setmetatable({}, { __mode = "k" })
        RC_TempSim._vehicleHeaterState = states
    end
    return states
end

local function _resetVehicleHeaterRamp(heater)
    if not heater then return end
    local states = RC_TempSim._vehicleHeaterState
    if states then
        states[heater] = nil
    end
end

local function _applyVehicleHeaterRamp(heater, targetDelta, rateOverride)
    targetDelta = tonumber(targetDelta) or 0
    if not heater then return targetDelta end

    local states = _vehicleHeaterStateTable()
    local now = _currentWorldHours()

    local rec = states[heater]
    if not rec then
        rec = { applied = 0, lastHour = now }
        states[heater] = rec
    end

    local minutes = (now - (rec.lastHour or now)) * 60
    if minutes < 0 then minutes = 0 end
    local maxMinutes = RC_TempSim.VEHICLE_HEATER_MAX_TIME_STEP_MIN or 0
    if maxMinutes > 0 and minutes > maxMinutes then minutes = maxMinutes end
    rec.lastHour = now

    local applied = rec.applied or 0
    local rate = tonumber(rateOverride) or RC_TempSim.VEHICLE_HEATER_RAMP_DEGREES_PER_MIN or 0
    local maxStep = rate * minutes

    if maxStep > 0 then
        local delta = targetDelta - applied
        if delta > maxStep then
            delta = maxStep
        elseif delta < -maxStep then
            delta = -maxStep
        end
        applied = applied + delta
    end

    rec.applied = applied
    rec.target = targetDelta

    return applied
end

local function _vehicleHeaterCooldown(heater)
    if not heater then return 0 end

    local states = RC_TempSim._vehicleHeaterState
    if not states then return 0 end

    local rec = states[heater]
    if not rec then return 0 end

    local cooldownRate = RC_TempSim.VEHICLE_HEATER_COOLDOWN_DEGREES_PER_MIN
    if not cooldownRate or cooldownRate <= 0 then
        cooldownRate = RC_TempSim.VEHICLE_HEATER_RAMP_DEGREES_PER_MIN
    end
    local applied = _applyVehicleHeaterRamp(heater, 0, cooldownRate)
    if math.abs(applied) < 0.001 then
        states[heater] = nil
        return 0
    end

    return applied
end

local function _vehicleHasCompromisedWindows(vehicle)
    if not vehicle or not vehicle.getPartCount or not vehicle.getPartByIndex then return false end

    if vehicle.windowsOpen then
        local ok, openCount = pcall(function() return vehicle:windowsOpen() end)
        if ok and openCount and openCount > 0 then
            return true
        end
    end

    local partCount = vehicle:getPartCount()
    if not partCount or partCount <= 0 then return false end

    for idx = 0, partCount - 1 do
        local part = vehicle:getPartByIndex(idx)
        if part and part.getWindow then
            local window = part:getWindow()
            if window then
                if window.isDestroyed and window:isDestroyed() then
                    return true
                end
                if window.isOpen and window:isOpen() then
                    return true
                end
                if window.getOpenDelta then
                    local ok, delta = pcall(function() return window:getOpenDelta() end)
                    if ok and delta and delta > 0 then
                        return true
                    end
                end
            end
            if part.getCondition then
                local condition = part:getCondition()
                if condition and condition == 0 then
                    return true
                end
            end
        end
    end

    return false
end

local function _dropVehicleHeaterTowardsAmbient(heater)
    local cooled = _vehicleHeaterCooldown(heater)
    if cooled ~= 0 then
        return cooled
    end
    _resetVehicleHeaterRamp(heater)
    return 0
end

local function computeVehicleHeaterAirAdjustment(player, square)
    if not player or not player.getVehicle then return 0 end

    local vehicle = player:getVehicle()
    if not vehicle then return 0 end

    local insideDelta = vehicle.getInsideTemperature and vehicle:getInsideTemperature() or nil
    insideDelta = tonumber(insideDelta) or 0

    local heater = vehicle.getHeater and vehicle:getHeater() or nil
    local heaterData = heater and heater:getModData() or nil
    local heaterActive = heaterData and heaterData.active or false
    local heaterTemp = heaterData and tonumber(heaterData.temperature) or 0
    local engineRunning = vehicle.isEngineRunning and vehicle:isEngineRunning() or false
    local hasCompromisedWindows = _vehicleHasCompromisedWindows(vehicle)

    local baseMaxHeat = RC_TempSim.VEHICLE_HEATER_VANILLA_MAX_DELTA_C or 25.0
    if not baseMaxHeat or baseMaxHeat <= 0 then baseMaxHeat = 25.0 end
    local sandboxMaxHeat = RC_TempSim.VEHICLE_HEATER_MAX_DELTA_C
    if sandboxMaxHeat == nil then
        sandboxMaxHeat = baseMaxHeat
    elseif sandboxMaxHeat < 0 then
        sandboxMaxHeat = 0
    end
    local heatScale = 1.0
    if sandboxMaxHeat == 0 then
        heatScale = 0
    elseif sandboxMaxHeat ~= baseMaxHeat then
        heatScale = sandboxMaxHeat / baseMaxHeat
    end

    local function scaleHeatDelta(delta)
        if not delta or delta <= 0 then return delta end
        if heatScale == 0 then
            return 0
        end
        local scaled = delta
        if heatScale ~= 1 then
            scaled = delta * heatScale
        end
        if sandboxMaxHeat > 0 and scaled > sandboxMaxHeat then
            scaled = sandboxMaxHeat
        end
        return scaled
    end

    insideDelta = scaleHeatDelta(insideDelta)
    heaterTemp = scaleHeatDelta(heaterTemp)

    if math.abs(insideDelta) < 0.001 and heaterActive and heaterTemp and heaterTemp ~= 0 then
        local pcPart = vehicle.getPartById and vehicle:getPartById("PassengerCompartment") or nil
        if pcPart then
            local pcData = pcPart:getModData()
            local pcTemp = pcData and tonumber(pcData.temperature) or nil
            pcTemp = scaleHeatDelta(pcTemp)
            if pcTemp and math.abs(pcTemp) > math.abs(insideDelta) then
                insideDelta = pcTemp
            end
        end

        if math.abs(insideDelta) < 0.001 and engineRunning then
            insideDelta = heaterTemp
        end
    end

    local heaterEffective = heaterActive and heaterTemp and heaterTemp ~= 0 and engineRunning and not hasCompromisedWindows

    if not heaterEffective then
        return _dropVehicleHeaterTowardsAmbient(heater)
    end

    if math.abs(insideDelta) < 0.001 then
        return _dropVehicleHeaterTowardsAmbient(heater)
    end

    if insideDelta > 0 then
        if not (heaterTemp and heaterTemp > 0) then
            return _dropVehicleHeaterTowardsAmbient(heater)
        end

        local limit = RC_TempSim.VEHICLE_HEATER_OUTDOOR_LIMIT_C or -20.0
        local outsideC = nil
        local climate = getClimateManager and getClimateManager() or nil
        if climate then
            if square and climate.getAirTemperatureForSquare then
                outsideC = climate:getAirTemperatureForSquare(square)
            elseif climate.getTemperature then
                outsideC = climate:getTemperature()
            end
        end

        if outsideC and outsideC <= limit then
            local threshold = RC_TempSim.VEHICLE_HEATER_INSULATION_THRESHOLD
            if threshold == nil then threshold = RC_TempSim.CHILLY_INSULATION_THRESHOLD end
            if threshold == nil then threshold = 0.6 end

            local avgIns = nil
            if summarizeThermalNodes then
                local summary = summarizeThermalNodes(player)
                avgIns = summary and summary.avgInsulation or nil
            end
            if avgIns == nil then avgIns = 0 end

            if avgIns <= threshold then
                _resetVehicleHeaterRamp(heater)
                return 0
            end
        end
        insideDelta = _applyVehicleHeaterRamp(heater, insideDelta)
    elseif insideDelta < 0 then
        if not (heaterTemp and heaterTemp < 0) then
            return _dropVehicleHeaterTowardsAmbient(heater)
        end
        insideDelta = _applyVehicleHeaterRamp(heater, insideDelta)
    end

    if insideDelta > 0 then
        insideDelta = scaleHeatDelta(insideDelta)
    end

    return insideDelta
end

local function coolingRateForAir(airC)
    local first = COOLING_LOOKUP[1]
    local last  = COOLING_LOOKUP[#COOLING_LOOKUP]

    local rate = last.rate

    if airC <= first.air then
        rate = first.rate
    elseif airC < last.air then
        for i = 1, #COOLING_LOOKUP - 1 do
            local lower = COOLING_LOOKUP[i]
            local upper = COOLING_LOOKUP[i + 1]
            if airC >= lower.air and airC <= upper.air then
                local span = upper.air - lower.air
                if span == 0 then
                    rate = upper.rate
                else
                    local t = (airC - lower.air) / span
                    rate = lerp(lower.rate, upper.rate, t)
                end
                break
            end
        end
    end

    local warmRate = RC_TempSim.WARM_COOLING_MAX_RATE or 0
    if warmRate and warmRate > rate then
        local warmStart = RC_TempSim.WARM_COOLING_START_C or last.air or 0
        local warmFull = RC_TempSim.WARM_COOLING_FULL_C or warmStart
        if airC >= warmStart then
            local t = 1.0
            if warmFull > warmStart then
                t = (airC - warmStart) / (warmFull - warmStart)
                if t < 0 then t = 0 end
                if t > 1 then t = 1 end
            end
            if t > 0 then
                rate = rate + (warmRate - rate) * t
            end
        end
    end

    if rate < 0 then rate = 0 end

    return rate
end

local function _minutesSinceLast(rec)
    local gt = getGameTime and getGameTime() or nil
    local nowH = gt and gt.getWorldAgeHours and gt:getWorldAgeHours() or 0
    local lastH = rec._lastH or nowH
    rec._lastH = nowH
    local dtm = (nowH - lastH) * 60
    if dtm < 0 then dtm = 0 end
    if dtm > 5 then dtm = 5 end
    return dtm
end

local function coolBodyTowardsAir(current, air, minutes, coolingMult)
    if air >= current then return current end
    local ratePerTick = coolingRateForAir(air)
    if ratePerTick <= 0 then return current end
    local mult = coolingMult or 1.0
    local dT = ratePerTick * RC_TempSim.TICKS_PER_MIN * minutes * mult
    local nextTemp = current - dT
    if nextTemp < air then nextTemp = air end
    if nextTemp < RC_TempSim.BODY_TEMP_MIN_C then nextTemp = RC_TempSim.BODY_TEMP_MIN_C end
    return nextTemp
end

local function _resolveBodyHeatActivityTarget(player)
    local target = RC_TempSim.BODY_HEATGEN_TARGET_IDLE or 0.0
    if not player then return target end

    local moving = player.isPlayerMoving and player:isPlayerMoving() or false
    if moving then
        if player.isSprinting and player:isSprinting() then
            target = RC_TempSim.BODY_HEATGEN_TARGET_SPRINT or target
        elseif player.isRunning and player:isRunning() then
            target = RC_TempSim.BODY_HEATGEN_TARGET_RUN or target
        elseif player.isSneaking and player:isSneaking() then
            target = RC_TempSim.BODY_HEATGEN_TARGET_SNEAK or target
        else
            target = RC_TempSim.BODY_HEATGEN_TARGET_WALK or target
        end
    end

    if player.isAttacking and player:isAttacking() then
        local combat = RC_TempSim.BODY_HEATGEN_TARGET_COMBAT
        if combat and combat > target then
            target = combat
        end
    end

    return target
end

local function _resolveBodyHeatPassiveTarget(airC)
    if not airC then
        return RC_TempSim.BODY_HEATGEN_TARGET_IDLE or 0.0
    end

    local threshold = RC_TempSim.BODY_HEATGEN_WARM_IDLE_THRESHOLD_C or 18.0
    local full = RC_TempSim.BODY_HEATGEN_WARM_IDLE_FULL_C or (threshold + 10.0)
    local idle = RC_TempSim.BODY_HEATGEN_TARGET_IDLE or 0.0
    local warmTarget = RC_TempSim.BODY_HEATGEN_WARM_IDLE_TARGET or idle

    if full <= threshold then
        full = threshold + 1.0
    end

    if airC <= threshold then
        return idle
    end

    local t = (airC - threshold) / (full - threshold)
    if t < 0 then t = 0 end
    if t > 1 then t = 1 end

    return lerp(idle, warmTarget, t)
end

local function _resolveBodyHeatLoadBonus(player)
    local bonusMax = RC_TempSim.BODY_HEATGEN_HEAVY_LOAD_BONUS_MAX or 0.0
    if bonusMax <= 0 or not player or not player.getInventory then return 0.0 end

    local inv = player:getInventory()
    if not inv or not inv.getCapacityWeight or not player.getMaxWeight then
        return 0.0
    end

    local capacity = inv:getCapacityWeight() or 0.0
    local maxWeight = player:getMaxWeight() or 0.0
    if maxWeight <= 0 then return 0.0 end

    local fraction = capacity / maxWeight
    if fraction < 0 then fraction = 0 end
    if fraction > 1 then fraction = 1 end

    return bonusMax * (fraction * fraction)
end

local function _resolveBodyHeatWeightBonus(player)
    local bonusMax = RC_TempSim.BODY_HEATGEN_WEIGHT_BONUS_MAX or 0.0
    if bonusMax <= 0 or not player or not player.getNutrition then return 0.0 end

    local nutrition = player:getNutrition()
    if not nutrition or not nutrition.getWeight then return 0.0 end

    local weight = nutrition:getWeight() or 0.0
    local ref = RC_TempSim.BODY_HEATGEN_WEIGHT_REF or 80.0
    local max = RC_TempSim.BODY_HEATGEN_WEIGHT_MAX or (ref + 1.0)
    if max <= ref then max = ref + 1.0 end

    if weight <= ref then return 0.0 end

    local t = (weight - ref) / (max - ref)
    if t < 0 then t = 0 end
    if t > 1 then t = 1 end

    return bonusMax * t
end

local function _resolveBodyHeatIncreaseRate(player, airC)
    local base = RC_TempSim.BODY_HEATGEN_INCREASE_RATE_BASE or 0.45
    local neutralMult = RC_TempSim.BODY_HEATGEN_INCREASE_RATE_NEUTRAL_MULT or 1.0
    local coldMult = RC_TempSim.BODY_HEATGEN_INCREASE_RATE_COLD_MULT or 0.4
    local warmMult = RC_TempSim.BODY_HEATGEN_INCREASE_RATE_WARM_MULT or 1.75
    local cold = RC_TempSim.BODY_HEATGEN_COLD_AIR_THRESHOLD_C or 0.0
    local warm = RC_TempSim.BODY_HEATGEN_WARM_AIR_THRESHOLD_C or 25.0

    if warm <= cold then
        warm = cold + 1.0
    end

    local mult = neutralMult
    if airC ~= nil then
        if airC <= cold then
            mult = coldMult
        elseif airC >= warm then
            mult = warmMult
        else
            local t = (airC - cold) / (warm - cold)
            mult = coldMult + (warmMult - coldMult) * t
        end
    end

    local rate = base * mult

    if player and player.getStats then
        local stats = player:getStats()
        if stats and stats.getEndurance then
            local endurance = stats:getEndurance() or 1.0
            local staminaMult = RC_TempSim.BODY_HEATGEN_STAMINA_RATE_MULT or 0.0
            if staminaMult ~= 0 then
                rate = rate * (1.0 + (1.0 - endurance) * staminaMult)
            end
        end
    end

    if rate < 0 then rate = 0 end

    return rate, mult
end

local function _resolveBodyHeatDecreaseRate(player, airC)
    local base = RC_TempSim.BODY_HEATGEN_DECREASE_RATE_BASE or 0.5
    local coldMult = RC_TempSim.BODY_HEATGEN_DECREASE_RATE_COLD_MULT or 1.2
    local warmMult = RC_TempSim.BODY_HEATGEN_DECREASE_RATE_WARM_MULT or 0.6
    local cold = RC_TempSim.BODY_HEATGEN_COLD_AIR_THRESHOLD_C or 0.0
    local warm = RC_TempSim.BODY_HEATGEN_WARM_AIR_THRESHOLD_C or 25.0

    if warm <= cold then
        warm = cold + 1.0
    end

    local mult = 1.0
    if airC ~= nil then
        if airC <= cold then
            mult = coldMult
        elseif airC >= warm then
            mult = warmMult
        else
            local t = (airC - cold) / (warm - cold)
            mult = coldMult + (warmMult - coldMult) * t
        end
    end

    local rate = base * mult
    if rate < 0 then rate = 0 end

    return rate, mult
end

local function _resolveBodyWetness(player)
    if not player or not player.getBodyDamage then return nil end
    local body = player:getBodyDamage()
    if not (body and body.getWetness) then return nil end
    return body:getWetness()
end

local function _computeWetnessHeatPenalty(wetness)
    if not wetness then return 0.0 end
    local threshold = RC_TempSim.BODY_HEATGEN_WETNESS_THRESHOLD or 0.0
    local maxWet = RC_TempSim.BODY_HEATGEN_WETNESS_MAX or (threshold + 1.0)
    local maxReduction = RC_TempSim.BODY_HEATGEN_WETNESS_TARGET_REDUCTION or 0.0
    if maxReduction <= 0 then return 0.0 end
    if maxWet <= threshold then maxWet = threshold + 1.0 end
    if wetness <= threshold then return 0.0 end
    local normalized = (wetness - threshold) / (maxWet - threshold)
    if normalized > 1 then normalized = 1 end
    if normalized < 0 then normalized = 0 end
    return maxReduction * normalized
end

local function _resolveColdActivityAllowance(airC, thermalSummary)
    local severity = 0.0
    if _severeColdFactor and airC ~= nil then
        severity = _severeColdFactor(airC) or 0.0
    end

    if severity <= 0 then
        return 1.0, 0.0, nil
    end

    local insulation = 0.0
    if thermalSummary and thermalSummary.avgInsulation then
        insulation = math.max(0.0, thermalSummary.avgInsulation)
    end

    local baseReq = RC_TempSim.BODY_HEATGEN_COLD_REQUIRED_INSULATION_BASE or 0.0
    local perSeverity = RC_TempSim.BODY_HEATGEN_COLD_REQUIRED_INSULATION_PER_SEVERITY or 0.0
    local required = baseReq + perSeverity * severity
    if required < 0 then required = 0 end

    if required <= 0 then
        return 1.0, severity, required
    end

    local fraction = 0.0
    if insulation > 0 then
        fraction = insulation / required
    end

    if fraction >= 1 then
        return 1.0, severity, required
    end

    local minFraction = RC_TempSim.BODY_HEATGEN_COLD_MIN_ACTIVITY_FRACTION or 0.0
    if minFraction < 0 then minFraction = 0 end
    if minFraction > 1 then minFraction = 1 end

    local allowance = minFraction + (1.0 - minFraction) * fraction
    if allowance < minFraction then allowance = minFraction end
    if allowance > 1 then allowance = 1 end

    return allowance, severity, required
end

local function _applyDrinkPenaltyDecay(rec, minutes)
    if not rec then return 0.0 end
    local current = rec._drinkHeatPenalty or 0.0
    if current <= 0 then return 0.0 end
    local decay = RC_TempSim.DRINK_HEAT_PENALTY_DECAY_PER_MIN or 0.0
    if not minutes or minutes <= 0 or decay <= 0 then
        return current
    end
    current = current - decay * minutes
    if current < 0 then current = 0 end
    rec._drinkHeatPenalty = current
    return current
end

local function _applyDrinkCooling(player, thirstDelta)
    if not player or not thirstDelta or thirstDelta <= 0 then return end
    local rec = getBodyTempData(player)
    if not rec then return end

    local minCool = RC_TempSim.DRINK_COOL_MIN_C or 0.0
    local maxCool = RC_TempSim.DRINK_COOL_MAX_C or minCool
    if maxCool < minCool then maxCool = minCool end
    if minCool <= 0 and maxCool <= 0 then return end

    local range = RC_TempSim.DRINK_COOL_THIRST_RANGE or 0.35
    if range <= 0 then range = 0.35 end
    local normalized = thirstDelta / range
    if normalized > 1 then normalized = 1 end
    if normalized < 0 then normalized = 0 end

    local cooling = minCool + (maxCool - minCool) * normalized
    if cooling < 0 then cooling = 0 end

    rec.core = (rec.core or RC_TempSim.BODY_TEMP_NORMAL_C or 37.0) - cooling
    local minC = RC_TempSim.BODY_TEMP_MIN_C or 20.0
    if rec.core < minC then rec.core = minC end
    rec._lastDrinkCooling = cooling
    rec._lastDrinkThirstDelta = thirstDelta

    local penaltyAdd = (RC_TempSim.DRINK_HEAT_PENALTY_ADD or 0.0) * normalized
    if penaltyAdd > 0 then
        local current = rec._drinkHeatPenalty or 0.0
        local maxPenalty = RC_TempSim.DRINK_HEAT_PENALTY_MAX or current
        current = current + penaltyAdd
        if maxPenalty > 0 and current > maxPenalty then current = maxPenalty end
        rec._drinkHeatPenalty = current
    end

    if player.setTemperature then
        player:setTemperature(rec.core)
    end
end

local function updateBodyHeatGeneration(rec, player, minutes, airC, thermalSummary)
    if not rec then return 0.0, nil end

    local current = rec.bodyHeatGeneration
    if current == nil then
        current = RC_TempSim.BODY_HEATGEN_TARGET_IDLE or 0.0
    end

    if not minutes or minutes <= 0 then
        local normalized = _normalizedMovementHeat(current)
        rec.bodyHeatGeneration = current
        rec.bodyHeatNormalized = normalized
        return current, {
            before = current,
            after = current,
            target = current,
            normalized = normalized,
            delta = 0,
            increaseRate = 0,
            decreaseRate = 0,
        }
    end

    local activityTargetRaw = _resolveBodyHeatActivityTarget(player)
    local passiveTargetRaw = _resolveBodyHeatPassiveTarget(airC)
    local heavyBonusRaw = _resolveBodyHeatLoadBonus(player)
    local weightBonusRaw = _resolveBodyHeatWeightBonus(player)
    local wetness = _resolveBodyWetness(player)
    local wetPenalty = _computeWetnessHeatPenalty(wetness)
    local drinkPenalty = _applyDrinkPenaltyDecay(rec, minutes)

    local coldAllowance, coldSeverity, coldRequired = _resolveColdActivityAllowance(airC, thermalSummary)
    if not coldAllowance then coldAllowance = 1.0 end
    if coldAllowance < 0 then coldAllowance = 0 end
    if coldAllowance > 1 then coldAllowance = 1 end

    local idleHeat = RC_TempSim.BODY_HEATGEN_TARGET_IDLE or 0.0
    local passiveBaseline = math.max(passiveTargetRaw, idleHeat)
    local activityBaseline = math.max(activityTargetRaw, idleHeat)
    local movementExtra = 0.0
    if activityBaseline > passiveBaseline then
        movementExtra = activityBaseline - passiveBaseline
    end

    local effectiveActivityTarget = activityBaseline
    if coldAllowance < 1.0 then
        effectiveActivityTarget = passiveBaseline + movementExtra * coldAllowance
    end

    local heavyBonus = heavyBonusRaw
    local weightBonus = weightBonusRaw
    if coldAllowance < 1.0 then
        heavyBonus = heavyBonus * coldAllowance
        weightBonus = weightBonus * coldAllowance
    end

    local target = effectiveActivityTarget
    if passiveBaseline > target then
        target = passiveBaseline
    end
    target = target + heavyBonus + weightBonus

    local coldPenalty = 0.0
    local penaltyMax = RC_TempSim.BODY_HEATGEN_SEVERE_COLD_TARGET_PENALTY or 0.0
    if penaltyMax > 0 and coldAllowance < 1.0 and coldSeverity and coldSeverity > 0 then
        coldPenalty = penaltyMax * coldSeverity * (1.0 - coldAllowance)
        if coldPenalty > 0 then
            target = target - coldPenalty
        else
            coldPenalty = 0.0
        end
    end

    if wetPenalty > 0 then
        target = target - wetPenalty
    end
    if drinkPenalty and drinkPenalty > 0 then
        target = target - drinkPenalty
    end

    local maxTarget = RC_TempSim.BODY_HEATGEN_TARGET_MAX or 1.25
    if target > maxTarget then target = maxTarget end
    if target < 0 then target = 0 end

    local incRate, incMult = _resolveBodyHeatIncreaseRate(player, airC)
    local decRate, decMult = _resolveBodyHeatDecreaseRate(player, airC)

    local before = current
    local delta = 0.0

    if current < target then
        local gain = math.min(target - current, incRate * minutes)
        current = current + gain
        delta = gain
    elseif current > target then
        local loss = math.min(current - target, decRate * minutes)
        current = current - loss
        delta = -loss
    end

    if current < 0 then current = 0 end
    if current > maxTarget then current = maxTarget end

    rec.bodyHeatGeneration = current
    rec.bodyHeatTarget = target
    rec.bodyHeatIncreaseRate = incRate
    rec.bodyHeatDecreaseRate = decRate
    rec.bodyHeatLastDelta = delta
    rec.bodyHeatHeavyLoadBonus = heavyBonus
    rec.bodyHeatWeightBonus = weightBonus
    rec.bodyHeatPassiveTarget = passiveBaseline
    rec.bodyHeatActivityTarget = effectiveActivityTarget
    rec.bodyHeatColdAllowance = coldAllowance
    rec.bodyHeatColdSeverity = coldSeverity
    rec.bodyHeatColdRequiredInsulation = coldRequired
    rec.bodyHeatColdPenalty = coldPenalty
    rec.bodyHeatActivityRaw = activityTargetRaw
    rec.bodyHeatPassiveRaw = passiveTargetRaw
    rec.bodyHeatHeavyRaw = heavyBonusRaw
    rec.bodyHeatWeightRaw = weightBonusRaw

    local normalized = _normalizedMovementHeat(current)
    rec.bodyHeatNormalized = normalized

    return current, {
        before = before,
        after = current,
        target = target,
        delta = delta,
        increaseRate = incRate,
        increaseMult = incMult,
        decreaseRate = decRate,
        decreaseMult = decMult,
        normalized = normalized,
        heavyBonus = heavyBonus,
        heavyBonusRaw = heavyBonusRaw,
        weightBonus = weightBonus,
        weightBonusRaw = weightBonusRaw,
        passiveTarget = passiveBaseline,
        passiveTargetRaw = passiveTargetRaw,
        activityTarget = effectiveActivityTarget,
        activityTargetRaw = activityTargetRaw,
        wetnessPenalty = wetPenalty,
        drinkPenalty = drinkPenalty,
        wetness = wetness,
        coldAllowance = coldAllowance,
        coldSeverity = coldSeverity,
        coldRequired = coldRequired,
        coldPenalty = coldPenalty,
    }
end

local function warmBodyTowardsNormal(current, air, minutes, targetOverride)
    local target = targetOverride or computeWarmRecoveryTarget(air)
    if not target then
        target = RC_TempSim.BODY_TEMP_NORMAL_C
    end
    if current >= target then return current, nil end

    local maxRate = RC_TempSim.WARM_RATE_CPM
    local baseRate = RC_TempSim.WARM_BASE_CPM
    if baseRate > maxRate then baseRate = maxRate end
    if baseRate < 0 then baseRate = 0 end

    local startC = RC_TempSim.AMBIENT_RECOVERY_START_C
    local fullC  = RC_TempSim.AMBIENT_RECOVERY_FULL_C
    if fullC <= startC then
        fullC = startC + 0.1
    end

    local ambientRate = 0.0
    if air > startC then
        local t = (air - startC) / (fullC - startC)
        if t < 0 then t = 0 end
        if t > 1 then t = 1 end
        ambientRate = (maxRate - baseRate) * t
    end

    local hotBonus = 0.0
    if air > fullC then
        hotBonus = maxRate * math.min(0.5, (air - fullC) * 0.025)
    end

    local baseDelta = baseRate * minutes
    local ambientDelta = ambientRate * minutes
    local hotDelta = hotBonus * minutes
    local pushDelta = 0

    if air > current then
        pushDelta = RC_TempSim.HEAT_PUSH_PER_C_CPM * (air - current) * minutes
    end

    local planned = baseDelta + ambientDelta + hotDelta + pushDelta
    local nextTemp = current + planned
    local clamp = 0
    if nextTemp > target then
        clamp = nextTemp - target
        nextTemp = target
    end

    local applied = nextTemp - current
    local breakdown = nil
    if planned > 0 or applied > 0 then
        breakdown = {
            base = baseDelta,
            ambient = ambientDelta,
            hot = hotDelta,
            push = pushDelta,
            planned = planned,
            applied = applied,
            clamp = clamp,
            target = target,
        }

        if planned > 0 and applied < planned then
            local ratio = applied / planned
            breakdown.baseApplied = baseDelta * ratio
            breakdown.ambientApplied = ambientDelta * ratio
            breakdown.hotApplied = hotDelta * ratio
            breakdown.pushApplied = pushDelta * ratio
        else
            breakdown.baseApplied = baseDelta
            breakdown.ambientApplied = ambientDelta
            breakdown.hotApplied = hotDelta
            breakdown.pushApplied = pushDelta
        end
    end

    return nextTemp, breakdown
end

local function applyHeatStress(core, air, minutes, clothingMult, movementHeat, thermalSummary)
    if not air or not minutes or minutes <= 0 then return core, nil end

    local comfort = RC_TempSim.WARM_COMFORT_AIR_C or 20.0
    local warmExcess = air - comfort
    local insulationFactor = 0.0
    if clothingMult then
        insulationFactor = 1.0 - math.max(0.0, math.min(1.0, clothingMult))
    end
    if insulationFactor < 0 then insulationFactor = 0 end

    local heatDelta = 0.0
    local envContribution = 0.0
    local insulationContribution = 0.0
    local movementContribution = 0.0
    local rateMult = 1.0
    local lightClothingInfo = nil

    local avgIns = 0.0
    if thermalSummary then
        avgIns = math.max(0.0, thermalSummary.avgInsulation or 0.0)
    end
    local movementFraction = _normalizedMovementHeat(movementHeat)
    local insulationThreshold = RC_TempSim.WARM_STAGE_LIGHT_INSULATION_THRESHOLD or 0.65
    local ambientScale = 1.0
    local sunRunTemp = RC_TempSim.WARM_STAGE_SUNSTRUCK_RUN_TEMP or 28.0
    local highInsMinTemp = RC_TempSim.WARM_HIGH_INSULATION_MIN_TEMP or 20.0
    local highInsMaxTemp = RC_TempSim.WARM_HIGH_INSULATION_MAX_TEMP or sunRunTemp

    if air and air < sunRunTemp and avgIns < insulationThreshold then
        local walkThreshold = RC_TempSim.WARM_LOW_INSULATION_WALK_THRESHOLD or 0.25
        if movementFraction <= walkThreshold then
            ambientScale = 0.0
        else
            local span = 1.0 - walkThreshold
            if span < 0.0001 then span = 0.0001 end
            ambientScale = (movementFraction - walkThreshold) / span
            if ambientScale < 0 then ambientScale = 0 end
            if ambientScale > 1 then ambientScale = 1 end
        end
    elseif air and air >= highInsMinTemp and air <= highInsMaxTemp and avgIns > insulationThreshold then
        local span = RC_TempSim.WARM_HIGH_INSULATION_RATE_SPAN or 0.75
        if span < 0.0001 then span = 0.0001 end
        local excess = avgIns - insulationThreshold
        if excess < 0 then excess = 0 end
        local ratio = excess / span
        if ratio > 1 then ratio = 1 end
        local minScale = RC_TempSim.WARM_HIGH_INSULATION_MIN_SCALE or 0.15
        if minScale < 0 then minScale = 0 end
        if minScale > 1 then minScale = 1 end
        ambientScale = minScale + (1.0 - minScale) * ratio
    end

    if thermalSummary then
        local lightThreshold = RC_TempSim.WARM_LIGHT_CLOTHING_THRESHOLD or 0.0
        if lightThreshold > 0 and avgIns < lightThreshold then
            local normalized = (lightThreshold - avgIns) / lightThreshold
            if normalized < 0 then normalized = 0 end
            if normalized > 1 then normalized = 1 end
            local minMult = RC_TempSim.WARM_LIGHT_CLOTHING_RATE_MULT or 1.0
            if minMult < 0 then minMult = 0 end
            if minMult > 1 then minMult = 1 end
            rateMult = 1.0 - normalized * (1.0 - minMult)
            lightClothingInfo = {
                avgInsulation = avgIns,
                threshold = lightThreshold,
                normalized = normalized,
                rateMult = rateMult,
            }
        end
    end

    if warmExcess > 0 then
        local envRate = RC_TempSim.WARM_AIR_HEAT_CPM or 0.0
        if envRate < 0 then envRate = 0 end

        local baseline = RC_TempSim.WARM_AIR_BASELINE_FRACTION or 0.0
        if baseline < 0 then baseline = 0 end
        if baseline > 1 then baseline = 1 end

        local criticalFraction = RC_TempSim.WARM_AIR_CRITICAL_FRACTION or 0.0
        local criticalStart = RC_TempSim.WARM_CRITICAL_AIR_C or 0.0
        local criticalSpan = RC_TempSim.WARM_CRITICAL_SPAN or 0.0
        local criticalBonus = 0.0
        if criticalFraction > 0 and criticalSpan > 0 and air > criticalStart then
            local t = (air - criticalStart) / criticalSpan
            if t > 1 then t = 1 end
            if t > 0 then
                criticalBonus = criticalFraction * t
            end
        end

        local ampBase = baseline + criticalBonus
        if ampBase < 0 then ampBase = 0 end
        if ampBase > 1 then ampBase = 1 end

        local amp = ampBase + (1.0 - ampBase) * insulationFactor
        if amp < 0 then amp = 0 end

        envContribution = warmExcess * envRate * amp * minutes
        envContribution = envContribution * ambientScale

        local insulationRate = RC_TempSim.WARM_INSULATION_HEAT_CPM or 0.0
        if insulationRate > 0 and insulationFactor > 0 then
            insulationContribution = warmExcess * insulationFactor * insulationRate * minutes
            insulationContribution = insulationContribution * ambientScale
        end
    end

    if warmExcess > 0 then
        local moveRate = RC_TempSim.WARM_MOVEMENT_HEAT_CPM or 0.0
        if moveRate > 0 then
            local moveFraction = _normalizedMovementHeat(movementHeat)
            local retention = 0.25 + 0.75 * insulationFactor
            if retention < 0 then retention = 0 end
            movementContribution = warmExcess * moveFraction * moveRate * retention * minutes
        end
    end

    if rateMult < 1.0 then
        envContribution = envContribution * rateMult
        insulationContribution = insulationContribution * rateMult
        movementContribution = movementContribution * rateMult
    end

    heatDelta = envContribution + insulationContribution + movementContribution

    if heatDelta <= 0 then
        return core, {
            applied = 0,
            planned = heatDelta,
            env = envContribution,
            insulation = insulationContribution,
            movement = movementContribution,
            envApplied = 0,
            insulationApplied = 0,
            movementApplied = 0,
            warmExcess = warmExcess,
            insulationFactor = insulationFactor,
            movementFraction = _normalizedMovementHeat(movementHeat),
            lightClothing = lightClothingInfo,
            rateMult = rateMult,
        }
    end

    local maxC = RC_TempSim.BODY_TEMP_MAX_C or (RC_TempSim.BODY_TEMP_NORMAL_C + 4.0)
    local nextTemp = core + heatDelta
    local clamp = 0
    if maxC and nextTemp > maxC then
        clamp = nextTemp - maxC
        nextTemp = maxC
    end

    local applied = nextTemp - core
    local breakdown = {
        applied = applied,
        planned = heatDelta,
        clamp = clamp,
        env = envContribution,
        insulation = insulationContribution,
        movement = movementContribution,
        warmExcess = warmExcess,
        insulationFactor = insulationFactor,
        movementFraction = _normalizedMovementHeat(movementHeat),
        lightClothing = lightClothingInfo,
        rateMult = rateMult,
    }

    if heatDelta > 0 and applied < heatDelta then
        local ratio = applied / heatDelta
        breakdown.envApplied = envContribution * ratio
        breakdown.insulationApplied = insulationContribution * ratio
        breakdown.movementApplied = movementContribution * ratio
    else
        breakdown.envApplied = envContribution
        breakdown.insulationApplied = insulationContribution
        breakdown.movementApplied = movementContribution
    end

    return nextTemp, breakdown
end

local function _resolveBodyTemperature(player, rec)
    if rec and rec.core then
        return rec.core
    end
    if player and player.getTemperature then
        local temp = player:getTemperature()
        if temp then return temp end
    end
    return RC_TempSim.BODY_TEMP_NORMAL_C or 37.0
end

local function _heatStageFromBodyTemperature(temp)
    local normal = RC_TempSim.BODY_TEMP_NORMAL_C or 37.0
    local caps = RC_TempSim.WARM_STAGE_TEMP_CAPS
    if caps then
        for stage = 4, 1, -1 do
            local cap = caps[stage]
            if cap and temp >= cap then
                return stage
            end
        end
        if temp >= (caps[1] or (normal + 2.0)) then
            return 1
        end
        return 0
    end

    if temp >= normal + 4.0 then return 4 end
    if temp >= normal + 3.0 then return 3 end
    if temp >= normal + 2.0 then return 2 end
    if temp >= normal + 1.0 then return 1 end
    return 0
end

local function applyOverheatSweating(player, rec, minutes, heatStage)
    if not player or not rec or not minutes or minutes <= 0 then return end

    rec._lastSweatSuppressed = nil

    local stageThreshold = RC_TempSim.SWEAT_OVERHEAT_STAGE or 2
    stageThreshold = math.floor(stageThreshold)
    if stageThreshold < 0 then stageThreshold = 0 end
    if stageThreshold > 4 then stageThreshold = 4 end

    local bodyTemp = _resolveBodyTemperature(player, rec)
    local bodyStage = _heatStageFromBodyTemperature(bodyTemp)

    local effectiveStage = bodyStage
    if heatStage ~= nil then
        local providedStage = math.floor(heatStage)
        if providedStage < 0 then providedStage = 0 end
        if providedStage > 4 then providedStage = 4 end
        if effectiveStage == nil or providedStage > effectiveStage then
            effectiveStage = providedStage
        end
    end

    if not effectiveStage then effectiveStage = 0 end

    if effectiveStage < stageThreshold then
        local reduce = (rec._overheatAccum or 0.0) - minutes
        if reduce < 0 then reduce = 0 end
        rec._overheatAccum = reduce
        rec._lastSweatApplied = 0
        rec._lastOverheatStage = effectiveStage
        return
    end

    local coldAllowance = rec.bodyHeatColdAllowance
    if coldAllowance == nil then coldAllowance = 1.0 end
    local coldSeverity = rec.bodyHeatColdSeverity or 0.0
    if coldSeverity < 0 then coldSeverity = 0 end
    local allowanceThreshold = RC_TempSim.SWEAT_COLD_ALLOWANCE_THRESHOLD or 0.6
    local severityMult = RC_TempSim.SWEAT_COLD_ALLOWANCE_SEVERITY_MULT or 1.0
    if coldSeverity > 0 and coldAllowance < 1.0 then
        local severityFactor = coldSeverity * severityMult
        if severityFactor < 0 then severityFactor = 0 end
        if severityFactor > 1 then severityFactor = 1 end
        local effectiveThreshold = allowanceThreshold + (1.0 - allowanceThreshold) * severityFactor
        if effectiveThreshold > 1 then effectiveThreshold = 1 end
        if coldAllowance < effectiveThreshold then
            local reduce = (rec._overheatAccum or 0.0) - minutes
            if reduce < 0 then reduce = 0 end
            rec._overheatAccum = reduce
            rec._lastSweatApplied = 0
            rec._lastOverheatStage = effectiveStage
            rec._lastSweatSuppressed = {
                allowance = coldAllowance,
                threshold = effectiveThreshold,
                severity = coldSeverity,
            }
            return
        end
    end

    local accum = (rec._overheatAccum or 0.0) + minutes
    rec._overheatAccum = accum
    rec._lastOverheatStage = effectiveStage

    local body = player.getBodyDamage and player:getBodyDamage() or nil
    if not (body and body.increaseBodyWetness and body.getWetness) then return end

    local wetness = body:getWetness() or 0.0
    local wetStart = RC_TempSim.SWEAT_WETNESS_START or 55.0
    local wetMax = RC_TempSim.SWEAT_MAX_WETNESS or 95.0
    if wetMax <= wetStart then wetMax = wetStart + 1.0 end
    local wetMult = 1.0
    if wetness >= wetStart then
        local span = wetMax - wetStart
        if span < 0.0001 then span = 0.0001 end
        local normalized = (wetness - wetStart) / span
        if normalized > 1 then normalized = 1 end
        if normalized < 0 then normalized = 0 end
        wetMult = 1.0 - normalized
        if wetMult <= 0 then
            rec._lastSweatApplied = 0
            return
        end
    end

    local baseRate = RC_TempSim.SWEAT_BASE_RATE_PER_MIN or 0.0
    local maxRate = RC_TempSim.SWEAT_MAX_RATE_PER_MIN or baseRate
    if maxRate < baseRate then maxRate = baseRate end
    if baseRate <= 0 and maxRate <= 0 then return end

    local rampMinutes = RC_TempSim.SWEAT_RAMP_MINUTES or 1.0
    if rampMinutes <= 0 then rampMinutes = 1.0 end
    local ramp = accum / rampMinutes
    if ramp > 1 then ramp = 1 end
    if ramp < 0 then ramp = 0 end

    local normalTemp = RC_TempSim.BODY_TEMP_NORMAL_C or 37.0
    local caps = RC_TempSim.WARM_STAGE_TEMP_CAPS
    local stageStartTemp = normalTemp
    if stageThreshold > 0 then
        if caps then
            stageStartTemp = (caps[math.max(0, stageThreshold - 1)] or normalTemp) + 0.0001
        else
            stageStartTemp = normalTemp + stageThreshold
        end
    end

    local maxTemp = nil
    if caps then
        maxTemp = caps[4] or caps[stageThreshold] or (RC_TempSim.BODY_TEMP_MAX_C or (normalTemp + 5.0))
    else
        maxTemp = RC_TempSim.BODY_TEMP_MAX_C or (normalTemp + 5.0)
    end
    if not maxTemp or maxTemp <= stageStartTemp then
        maxTemp = stageStartTemp + 0.0001
    end

    local tempExcess = bodyTemp - stageStartTemp
    local tempMult = 0.0
    if tempExcess > 0 then
        tempMult = tempExcess / (maxTemp - stageStartTemp)
        if tempMult > 1 then tempMult = 1 end
    end
    if tempMult <= 0 then
        rec._lastSweatApplied = 0
        return
    end

    local stageMult = 1.0
    local extraStage = effectiveStage - stageThreshold
    if extraStage > 0 then
        stageMult = stageMult + extraStage * (RC_TempSim.SWEAT_STAGE_RATE_MULT or 0.0)
    end

    local rate = (baseRate + (maxRate - baseRate) * ramp) * stageMult * wetMult * tempMult
    if rate <= 0 then
        rec._lastSweatApplied = 0
        return
    end

    local amount = rate * minutes
    if amount <= 0 then
        rec._lastSweatApplied = 0
        return
    end

    local beforeWet = _snapshotBodyPartWetness(body)
    body:increaseBodyWetness(amount)
    local afterWet = _snapshotBodyPartWetness(body)
    if beforeWet and afterWet then
        _applyBodyWetnessDeltaToClothing(player, beforeWet, afterWet)
    end
    rec._lastSweatApplied = amount
end

local function shouldForceImmediateChilly(player, airC, coolingMult, thermalSummary, movementHeat)
    if not airC then return false end
    local square = nil
    if player and player.getCurrentSquare then
        square = player:getCurrentSquare()
    elseif getSpecificPlayer then
        local specific = getSpecificPlayer(0)
        if specific and specific.getCurrentSquare then
            square = specific:getCurrentSquare()
        end
    end
    local warmthProx = computeProximityWarmRate(square)
    if warmthProx > 0.2 then return false end
    local chillTemp = RC_TempSim.CHILLY_AIR_TEMP_THRESHOLD_C
    if airC > chillTemp then return false end

    local heatFraction = _normalizedMovementHeat(movementHeat)

    local insufficient = false
    local multThreshold = RC_TempSim.CHILLY_COOLING_MULT_THRESHOLD
    local effectiveCooling = coolingMult
    if effectiveCooling and heatFraction > 0 then
        local coolingBonus = heatFraction * (RC_TempSim.MOVEMENT_CHILLY_COOLING_BONUS_PER_HEAT or 0)
        effectiveCooling = effectiveCooling - coolingBonus
        if effectiveCooling < 0 then effectiveCooling = 0 end
    end
    if effectiveCooling and multThreshold and effectiveCooling >= multThreshold then
        insufficient = true
    end

    local effectiveInsulation = thermalSummary and thermalSummary.avgInsulation or nil
    if effectiveInsulation and heatFraction > 0 then
        local insulationBonus = heatFraction * (RC_TempSim.MOVEMENT_CHILLY_INSULATION_BONUS_PER_HEAT or 0)
        effectiveInsulation = effectiveInsulation + insulationBonus
    end

    if not insufficient and effectiveInsulation then
        local insThreshold = RC_TempSim.CHILLY_INSULATION_THRESHOLD
        if insThreshold and effectiveInsulation <= insThreshold then
            insufficient = true
        end
    end

    if insufficient and heatFraction > 0 and chillTemp then
        local chillDelta = chillTemp - airC
        if chillDelta > 0 then
            local buffer = heatFraction * (RC_TempSim.MOVEMENT_CHILLY_AIR_BUFFER_PER_HEAT or 0)
            if buffer >= chillDelta then
                insufficient = false
            end
        end
    end

    return insufficient
end

local function applyHypothermiaHealthPenalty(player, rec, minutes)
    if not player or not rec or not minutes or minutes <= 0 then return end
    if not MoodleType or not player.getMoodles then return end

    local moodles = player:getMoodles()
    local stage = moodles and moodles.getMoodleLevel and moodles:getMoodleLevel(MoodleType.Hypothermia) or 0
    if stage < RC_TempSim.HYPOTHERMIA_DAMAGE_STAGE_THRESHOLD then
        rec._hypoDamageAccum = 0
        return
    end

    local interval = RC_TempSim.HYPOTHERMIA_DAMAGE_INTERVAL_MIN
    if interval <= 0 then interval = 1.0 end

    rec._hypoDamageAccum = (rec._hypoDamageAccum or 0) + minutes

    local damagePerTick = RC_TempSim.HYPOTHERMIA_DAMAGE_PER_TICK
    if damagePerTick <= 0 then return end

    local bd = player.getBodyDamage and player:getBodyDamage() or nil
    if not (bd and bd.ReduceGeneralHealth) then return end

    while rec._hypoDamageAccum >= interval do
        rec._hypoDamageAccum = rec._hypoDamageAccum - interval
        bd:ReduceGeneralHealth(damagePerTick)
    end
end

local function applyColdSicknessFromExposure(player, rec, minutes, airC, core, thermalSummary)
    if not player or not rec or not minutes or minutes <= 0 then return end

    local stats = player.getStats and player:getStats() or nil
    local body  = player.getBodyDamage and player:getBodyDamage() or nil
    if not stats or not body or not stats.getSickness or not stats.setSickness or not body.getColdStrength then return end

    local currentSickness = stats:getSickness() or 0.0
    local previousCold = rec._lastColdSickness or 0.0
    local otherSickness = currentSickness - previousCold
    if otherSickness < 0 then otherSickness = 0 end

    local updatesPerMinute = RC_TempSim.COLD_UPDATES_PER_MIN
    if updatesPerMinute <= 0 then updatesPerMinute = 120 end
    local totalUpdates = math.floor(minutes * updatesPerMinute + 0.5)
    if totalUpdates < 1 then totalUpdates = 1 end

    local catchIncreaseRate = (ZomboidGlobals and ZomboidGlobals.CatchAColdIncreaseRate) or 0.003
    catchIncreaseRate = catchIncreaseRate * RC_TempSim.CATCH_COLD_RATE_MULT
    local catchDecreaseRate = (ZomboidGlobals and ZomboidGlobals.CatchAColdDecreaseRate) or 0.175
    local catchStageRequirement = RC_TempSim.CATCH_COLD_THRESHOLD or 2
    catchStageRequirement = math.floor(catchStageRequirement)
    if catchStageRequirement < 1 then catchStageRequirement = 1 end
    if catchStageRequirement > 4 then catchStageRequirement = 4 end

    local worsenRatePerHour = RC_TempSim.COLD_WORSEN_RATE_PER_HOUR
    local recoverRatePerHour = RC_TempSim.COLD_RECOVER_RATE_PER_HOUR
    local worsenRatePerMin   = worsenRatePerHour / 60.0
    local recoverRatePerMin  = recoverRatePerHour / 60.0
    local worsenChangePerUpdate  = (worsenRatePerMin / updatesPerMinute)
    local recoverChangePerUpdate = (recoverRatePerMin / updatesPerMinute)

    local medicineDecayPerMin = RC_TempSim.COLD_MEDICINE_DECAY_PER_HOUR / 60.0
    local medicineDecayPerUpdate = (medicineDecayPerMin / updatesPerMinute)

    local catchStageMet = coldStageForCore(core) >= catchStageRequirement

    local thermo = body.getThermoregulator and body:getThermoregulator() or nil
    local catchDelta = thermo and thermo.getCatchAColdDelta and thermo:getCatchAColdDelta() or 0.0
    if catchDelta < 0 then catchDelta = 0 end

    local hasCold = body.isHasACold and body:isHasACold() or false
    local coldStrength = body:getColdStrength() or 0.0
    local catchMeter = body.getCatchACold and body:getCatchACold() or 0.0
    local coldReduction = body.getColdReduction and body:getColdReduction() or 0.0

    local hasProne = player.HasTrait and player:HasTrait("ProneToIllness") or false
    local hasResilient = player.HasTrait and player:HasTrait("Resilient") or false
    local hasOutdoorsman = player.HasTrait and player:HasTrait("Outdoorsman") or false

    local catchTraitMult = 1.0
    if hasProne then catchTraitMult = catchTraitMult * 1.7 end
    if hasResilient then catchTraitMult = catchTraitMult * 0.45 end
    if hasOutdoorsman then catchTraitMult = catchTraitMult * 0.25 end

    local recoverTraitMult = 1.0
    if hasProne then recoverTraitMult = 0.5 end
    if hasResilient then recoverTraitMult = 1.5 end

    local worsenTraitMult = 1.0
    if hasProne then worsenTraitMult = worsenTraitMult * 1.2 end
    if hasResilient then worsenTraitMult = worsenTraitMult * 0.8 end

    local moodles = player.getMoodles and player:getMoodles() or nil
    local wetLevel = moodles and moodles.getMoodleLevel and moodles:getMoodleLevel(MoodleType.Wet) or 0
    local hypoLevel = moodles and moodles.getMoodleLevel and moodles:getMoodleLevel(MoodleType.Hypothermia) or 0
    local square = player.getCurrentSquare and player:getCurrentSquare() or nil
    local fatigue = stats.getFatigue and stats:getFatigue() or stats.fatigue or 0.0
    local hunger = stats.getHunger and stats:getHunger() or stats.hunger or 0.0
    local thirst = stats.getThirst and stats:getThirst() or stats.thirst or 0.0

    local restingBase = false
    if hasCold then
        if square and square.isInARoom and square:isInARoom()
                and wetLevel <= 0
                and hypoLevel < 1
                and fatigue <= 0.5
                and hunger <= 0.25
                and thirst <= 0.25 then
            restingBase = true
        end
    end

    for _ = 1, totalUpdates do
        if not hasCold then
            if catchStageMet and catchDelta > 0 then
                catchMeter = catchMeter + catchIncreaseRate * catchDelta * catchTraitMult
                if catchMeter >= 100.0 then
                    catchMeter = 0.0
                    hasCold = true
                    coldStrength = math.max(coldStrength, 20.0)
                    if body.setHasACold then body:setHasACold(true) end
                    if body.setColdStrength then body:setColdStrength(coldStrength) end
                    if body.setTimeToSneezeOrCough then body:setTimeToSneezeOrCough(0.0) end
                    if square and square.isInARoom and square:isInARoom()
                            and wetLevel <= 0
                            and hypoLevel < 1
                            and fatigue <= 0.5
                            and hunger <= 0.25
                            and thirst <= 0.25 then
                        restingBase = true
                    else
                        restingBase = false
                    end
                end
            else
                catchMeter = catchMeter - catchDecreaseRate
                if catchMeter < 0 then catchMeter = 0 end
            end
        else
            local recovering = restingBase
            if coldReduction > 0 then
                recovering = true
                coldReduction = coldReduction - medicineDecayPerUpdate
                if coldReduction < 0 then coldReduction = 0 end
            end

            if recovering then
                local delta = recoverChangePerUpdate * recoverTraitMult
                coldStrength = coldStrength - delta
                if coldReduction > 0 then
                    coldStrength = coldStrength - delta
                end
                if coldStrength <= 0 then
                    coldStrength = 0
                    hasCold = false
                    catchMeter = 0
                    break
                end
            else
                coldStrength = coldStrength + worsenChangePerUpdate * worsenTraitMult
                if coldStrength > 100.0 then coldStrength = 100.0 end
            end
        end
    end

    if not hasCold then
        coldStrength = 0
        if body.setHasACold then body:setHasACold(false) end
    elseif body.setHasACold then
        body:setHasACold(true)
    end

    if body.setCatchACold then body:setCatchACold(catchMeter) end
    if body.setColdReduction then body:setColdReduction(coldReduction) end
    if body.setColdStrength then body:setColdStrength(coldStrength) end

    rec._coldStrength = coldStrength

    local coldSickness = math.min(1.0, (coldStrength / 100.0) * RC_TempSim.COLD_SICKNESS_MULT)
    local newSickness = otherSickness + coldSickness
    if newSickness > 1.0 then newSickness = 1.0 end
    stats:setSickness(newSickness)
    rec._lastColdSickness = coldSickness
end


function RC_TempSim.getSimulatedTemperatureForSquare(square)
    local climate = getClimateManager and getClimateManager() or nil
    local fallback = climate and climate:getTemperature() or 0
    if not square then return fallback end

    local S = RC_TempSim._state
    if not S or not S.mode or not S.temps then
        local native = _nativeHeatForSquare(square)
        if native ~= 0 then
            local air = fallback + native
            if RC_TempSim.HEAT_TARGET_MIN_C and air < RC_TempSim.HEAT_TARGET_MIN_C then air = RC_TempSim.HEAT_TARGET_MIN_C end
            if RC_TempSim.HEAT_TARGET_MAX_C and air > RC_TempSim.HEAT_TARGET_MAX_C then air = RC_TempSim.HEAT_TARGET_MAX_C end
            return air
        end
        return fallback
    end

    if S.mode == "vanilla" then
        local rd = RC_RoomLogic and RC_RoomLogic.roomDefFromSquare and RC_RoomLogic.roomDefFromSquare(square) or nil
        if rd and S.temps[rd] ~= nil then
            return S.temps[rd]
        end
    elseif S.mode == "player" then
        local reg = RC_PlayerRoomLogic and RC_PlayerRoomLogic.regionFromSquareOrNeighbors
                and RC_PlayerRoomLogic.regionFromSquareOrNeighbors(square)
                or nil
        if reg and S.temps[reg] ~= nil then
            return S.temps[reg]
        end
    end

    local native = _nativeHeatForSquare(square)
    if native ~= 0 then
        local air = fallback + native
        if RC_TempSim.HEAT_TARGET_MIN_C and air < RC_TempSim.HEAT_TARGET_MIN_C then air = RC_TempSim.HEAT_TARGET_MIN_C end
        if RC_TempSim.HEAT_TARGET_MAX_C and air > RC_TempSim.HEAT_TARGET_MAX_C then air = RC_TempSim.HEAT_TARGET_MAX_C end
        return air
    end

    return fallback
end

function RC_TempSim.getSimulatedTemperatureForPlayer(player)
    local square = player.getCurrentSquare and player:getCurrentSquare() or nil
    local air = RC_TempSim.getSimulatedTemperatureForSquare(square)

    local vehicleDelta = computeVehicleHeaterAirAdjustment(player, square)
    if vehicleDelta ~= 0 then
        air = air + vehicleDelta
    end

    return air
end

function RC_TempSim.applyDrinkCooling(player, thirstDelta)
    _applyDrinkCooling(player, thirstDelta)
end

function RC_TempSim.updatePlayerBodyTemperature(player)
    if not player then return end

    local rec = getBodyTempData(player)
    if not rec then return end

    local minutes = _minutesSinceLast(rec)
    if minutes <= 0 then return end

    local square = player.getCurrentSquare and player:getCurrentSquare() or nil

    local airC   = RC_TempSim.getSimulatedTemperatureForPlayer(player)
    local core   = rec.core or RC_TempSim.BODY_TEMP_NORMAL_C
    local initialCore = core

    local warmRecoveryTarget = computeWarmRecoveryTarget(airC)

    local clothingMult, thermalSummary = clothingCoolingMultiplier(player, airC)
    if not clothingMult then
        clothingMult = 1.0
    end

    rec._lastAirC = airC
    rec._lastAvgInsulation = thermalSummary and thermalSummary.avgInsulation or nil

    local bodyHeatValue, bodyHeatInfo = updateBodyHeatGeneration(rec, player, minutes, airC, thermalSummary)
    local movementHeat = 0.0
    if bodyHeatInfo then
        local idleHeat = RC_TempSim.BODY_HEATGEN_TARGET_IDLE or 0.0
        local activityTarget = math.max(idleHeat, bodyHeatInfo.activityTarget or 0.0)
        local passiveTarget = math.max(idleHeat, bodyHeatInfo.passiveTarget or 0.0)
        local baseline = activityTarget
        if passiveTarget > baseline then
            baseline = passiveTarget
        end
        baseline = baseline + math.max(bodyHeatInfo.heavyBonus or 0.0, 0.0)
        baseline = baseline + math.max(bodyHeatInfo.weightBonus or 0.0, 0.0)

        local movementComponent = (bodyHeatInfo.after or bodyHeatValue or 0.0) - baseline
        if movementComponent < 0 then
            movementComponent = 0
        end

        local activityComponent = math.max(0.0, activityTarget - idleHeat)
        if movementComponent < activityComponent then
            movementComponent = activityComponent
        end

        movementHeat = _normalizedMovementHeat(movementComponent)
        bodyHeatInfo.movementComponent = movementComponent
    else
        movementHeat = _normalizedMovementHeat(bodyHeatValue)
    end
    local runAccum, runFraction = _updateWarmRunAccumulator(rec, minutes, movementHeat)

    local coldMovementRetention = 1.0
    local coldPenaltyStrength = RC_TempSim.BODY_HEATGEN_COLD_MOVEMENT_REFUND_SUPPRESSION or 0.0
    if coldPenaltyStrength < 0 then coldPenaltyStrength = 0 end
    if (coldPenaltyStrength > 0) and airC then
        local severity = _severeColdFactor(airC)
        if severity > 0 then
            local insulation = 0.0
            if thermalSummary and thermalSummary.avgInsulation then
                insulation = math.max(0.0, thermalSummary.avgInsulation)
            end
            local scale = RC_TempSim.CLOTHING_INSULATION_SCALE or 1.0
            if scale < 0 then scale = 0 end
            local scaledIns = insulation * scale
            local protectionScale = RC_TempSim.CLOTHING_LOW_PROTECTION_SCALE or 0.0
            local protection = 0.0
            if protectionScale > 0 then
                protection = scaledIns / (scaledIns + protectionScale)
                if protection < 0 then protection = 0 end
                if protection > 1 then protection = 1 end
            end
            local coldPenalty = severity * (1.0 - protection)
            if coldPenalty > 0 then
                local suppression = coldPenalty * coldPenaltyStrength
                if suppression > 1 then suppression = 1 end
                coldMovementRetention = 1.0 - suppression
            end
        end
    end
    local minRetention = RC_TempSim.BODY_HEATGEN_COLD_MOVEMENT_MIN_RETENTION
    if minRetention then
        if minRetention < 0 then minRetention = 0 end
        if minRetention > 1 then minRetention = 1 end
        if coldMovementRetention < minRetention then
            coldMovementRetention = minRetention
        end
    end

    local coolingReduction = RC_TempSim.BODY_HEATGEN_COOLING_MULT or 1.0
    if coolingReduction < 0 then coolingReduction = 0 end
    local movementCoolingEquivalent = 1.0 - movementHeat * coolingReduction * coldMovementRetention
    if movementCoolingEquivalent < 0 then movementCoolingEquivalent = 0 end
    local minCooling = RC_TempSim.BODY_HEATGEN_COOLING_MIN or 0.0
    if movementCoolingEquivalent < minCooling then movementCoolingEquivalent = minCooling end
    if movementCoolingEquivalent > 1 then movementCoolingEquivalent = 1 end
    local coolingMult = clothingMult * movementCoolingEquivalent

    local beforeCooling = core
    core = coolBodyTowardsAir(core, airC, minutes, coolingMult)

    local movementRecovery = 0.0
    local recoveryMult = RC_TempSim.BODY_HEATGEN_RECOVERY_MULT or 1.0
    if recoveryMult < 0 then recoveryMult = 0 end
    if movementHeat > 0 and beforeCooling > core and recoveryMult > 0 then
        local coolingLoss = beforeCooling - core
        if coolingLoss > 0 then
            local heatGain = coolingLoss * movementHeat * recoveryMult * coldMovementRetention
            local targetTemp = warmRecoveryTarget or RC_TempSim.BODY_TEMP_NORMAL_C
            local maxGain = targetTemp - core
            if maxGain < 0 then maxGain = 0 end
            if heatGain > maxGain then
                heatGain = maxGain
            end
            if heatGain > 0 then
                core = core + heatGain
                movementRecovery = heatGain
            end
        end
    end

    local directDetails = nil
    local directThreshold = RC_TempSim.BODY_HEATGEN_DIRECT_GAIN_THRESHOLD or 0.65
    local directMaxRate = RC_TempSim.BODY_HEATGEN_DIRECT_GAIN_MAX_PER_MIN or 0.0
    if directMaxRate > 0 then
        local above = movementHeat - directThreshold
        if above > 0 then
            local span = 1.0 - directThreshold
            if span <= 0 then span = 1.0 end
            local intensity = above / span
            if intensity < 0 then intensity = 0 end
            if intensity > 1 then intensity = 1 end

            local warmMult = 1.0
            local warmThreshold = RC_TempSim.BODY_HEATGEN_WARM_AIR_THRESHOLD_C or 25.0
            local warmFull = RC_TempSim.BODY_HEATGEN_DIRECT_WARM_FULL_C or (warmThreshold + 5.0)
            if warmFull <= warmThreshold then warmFull = warmThreshold + 1.0 end
            local warmBonus = RC_TempSim.BODY_HEATGEN_DIRECT_WARM_MULT or 1.0
            if airC and warmBonus and warmBonus ~= 1.0 and airC >= warmThreshold then
                local t = (airC - warmThreshold) / (warmFull - warmThreshold)
                if t < 0 then t = 0 end
                if t > 1 then t = 1 end
                warmMult = 1.0 + (warmBonus - 1.0) * t
            end

            local rate = directMaxRate * warmMult
            if rate > 0 then
                local gain = rate * intensity * minutes
                gain = gain * coldMovementRetention
                if gain > 0 then
                    local targetTemp = warmRecoveryTarget or RC_TempSim.BODY_TEMP_NORMAL_C
                    local maxGain = targetTemp - core
                    if maxGain < 0 then maxGain = 0 end
                    if gain > maxGain then gain = maxGain end
                    if gain > 0 then
                        core = core + gain
                        directDetails = {
                            intensity = intensity,
                            warmMultiplier = warmMult,
                            applied = gain,
                            rate = rate,
                        }
                    end
                end
            end
        end
    end

    local warmBreakdown = nil
    local proximityGain = 0.0
    if core < RC_TempSim.BODY_TEMP_NORMAL_C then
        local warmedCore, breakdown = warmBodyTowardsNormal(core, airC, minutes, warmRecoveryTarget)
        core = warmedCore
        warmBreakdown = breakdown

        local proxRate = computeProximityWarmRate(square)
        if proxRate > 0 then
            local delta = proxRate * minutes
            local maxGain = RC_TempSim.BODY_TEMP_NORMAL_C - core
            if delta > maxGain then delta = maxGain end
            core = core + delta
            proximityGain = delta
        end
    end

    local chillyStageTemp = RC_TempSim.HYPOTHERMIA_STAGE_1_C
    if core > chillyStageTemp and shouldForceImmediateChilly(player, airC, coolingMult, thermalSummary, movementHeat) then
        local forced = chillyStageTemp - (RC_TempSim.CHILLY_FORCE_EPSILON or 0.01)
        if forced < RC_TempSim.BODY_TEMP_MIN_C then
            forced = RC_TempSim.BODY_TEMP_MIN_C
        end
        core = forced
        rec._lastForcedChilly = (_currentWorldHours and _currentWorldHours()) or 0
    end

    local heatStressBreakdown = nil
    core, heatStressBreakdown = applyHeatStress(core, airC, minutes, clothingMult, movementHeat, thermalSummary)

    local maxC = RC_TempSim.BODY_TEMP_MAX_C
    if maxC and core > maxC then
        core = maxC
    end

    local stageFloor = computeInsulationStageFloor(thermalSummary, airC)
    local stageFloorGain = 0.0
    if stageFloor and core < stageFloor then
        stageFloorGain = stageFloor - core
        core = stageFloor
    end

    local stageCapLevel, stageCapTemp, stageCapInfo = computeHeatStageCap(airC, thermalSummary, movementHeat, runFraction)
    local stageCapTarget = stageCapTemp
    if stageCapTemp then
        local prevTarget = rec._lastStageCapTarget
        local relaxRate = RC_TempSim.WARM_STAGE_CAP_RELAX_RATE or 0.0
        if stageCapInfo then
            stageCapInfo.tempCap = stageCapTemp
        end
        if prevTarget and prevTarget > stageCapTemp then
            local allowedDrop = 0.0
            if relaxRate > 0 then
                allowedDrop = relaxRate * minutes
                if allowedDrop < 0 then allowedDrop = 0 end
            end
            if allowedDrop <= 0 then
                stageCapTarget = prevTarget
            else
                local relaxed = prevTarget - allowedDrop
                if relaxed < stageCapTemp then relaxed = stageCapTemp end
                stageCapTarget = relaxed
            end
        end
        rec._lastStageCapTarget = stageCapTarget
        if stageCapInfo then
            stageCapInfo.relaxedTempCap = stageCapTarget
        end
    else
        rec._lastStageCapTarget = nil
        if stageCapInfo then
            stageCapInfo.tempCap = nil
            stageCapInfo.relaxedTempCap = nil
        end
    end
    if stageCapInfo then
        stageCapInfo.stage = stageCapLevel or stageCapInfo.stage
        stageCapInfo.runFraction = stageCapInfo.runFraction or runFraction
        stageCapInfo.runAccum = stageCapInfo.runAccum or runAccum
    end
    local coldRunHeatGain = 0.0
    if stageCapTarget and core < stageCapTarget then
        local normalTemp = RC_TempSim.BODY_TEMP_NORMAL_C or 37.0
        if stageCapTarget > normalTemp then
            local movementThreshold = RC_TempSim.BODY_HEATGEN_COLD_PUSH_MOVEMENT_THRESHOLD or 0.75
            local clothingThreshold = RC_TempSim.BODY_HEATGEN_COLD_PUSH_CLOTHING_THRESHOLD or 0.55
            local coldAirMax = RC_TempSim.BODY_HEATGEN_COLD_PUSH_AIR_MAX_C or (RC_TempSim.WARM_COMFORT_AIR_C or 20.0)
            local minStage = RC_TempSim.BODY_HEATGEN_COLD_PUSH_MIN_STAGE or 0
            local clothingFactor = stageCapInfo and stageCapInfo.clothingFactor or 0.0
            local movementLevel = movementHeat or 0.0
            local coldEnough = (airC == nil) or airC <= coldAirMax
            local runningEnough = movementLevel >= movementThreshold
            local clothedEnough = clothingFactor >= clothingThreshold
            local stageEnough = (stageCapLevel or 0) >= minStage
            if coldEnough and runningEnough and clothedEnough and stageEnough then
                local rate = RC_TempSim.BODY_HEATGEN_COLD_PUSH_RATE_PER_MIN or 0.0
                if rate > 0 and minutes > 0 then
                    local potential = stageCapTarget - core
                    if potential > 0 then
                        local gain = rate * minutes
                        if gain > potential then gain = potential end
                        if gain > 0 then
                            local avgIns = thermalSummary and thermalSummary.avgInsulation or nil
                            local lowInsThreshold = RC_TempSim.BODY_HEATGEN_COLD_PUSH_LOW_INS_THRESHOLD or 3.0
                            local lowInsFactor = nil
                            if avgIns and avgIns < lowInsThreshold then
                                local baseline = beforeCooling or core
                                local lost = 0.0
                                if baseline and baseline > core then
                                    lost = baseline - core
                                end
                                if lost < 0 then lost = 0 end
                                local normalized = avgIns / lowInsThreshold
                                if normalized < 0 then normalized = 0 end
                                lowInsFactor = normalized
                                local maxRecovery = lost * normalized
                                if maxRecovery < gain then
                                    gain = maxRecovery
                                end
                            end
                            if gain > 0 then
                                core = core + gain
                                coldRunHeatGain = gain
                                if stageCapInfo and lowInsFactor then
                                    stageCapInfo.lowInsulationFactor = lowInsFactor
                                end
                            end
                        end
                    end
                end
            end
        end
    end
    local stageCapClamp = 0.0
    if stageCapTarget and core > stageCapTarget then
        local cooledCore, clampApplied = applyStageCapCooling(core, stageCapTarget, minutes, clothingMult, thermalSummary, airC)
        stageCapClamp = clampApplied
        core = cooledCore
    end
    if stageCapInfo then
        stageCapInfo.coldRunHeatGain = (stageCapInfo.coldRunHeatGain or 0.0) + coldRunHeatGain
    end

    local heatStage = stageCapLevel or (stageCapInfo and stageCapInfo.stage) or nil
    applyOverheatSweating(player, rec, minutes, heatStage)

    logHeatGain(
        player,
        airC,
        minutes,
        initialCore,
        core,
        clothingMult,
        movementHeat,
        thermalSummary,
        movementRecovery,
        warmBreakdown,
        proximityGain,
        heatStressBreakdown,
        stageFloorGain,
        stageCapClamp,
        stageCapInfo,
        bodyHeatInfo,
        directDetails
    )

    rec.core = core
    player:setTemperature(core)
    local name = nil
    if player.getUsername then
        name = player:getUsername()
    end
    if not name or name == "" then
        if player.getFullName then
            name = player:getFullName()
        end
    end
    if not name or name == "" then
        name = tostring(player)
    end
    -- print(string.format("[RealisticCold] Body temp for %s: %.2f°C (air %.2f°C, movementHeat %.2f)", name or "player", core or 0, airC or 0, movementHeat or 0))
    applyColdSicknessFromExposure(player, rec, minutes, airC, core, thermalSummary)
    applyHypothermiaHealthPenalty(player, rec, minutes)
end

local function _enforceBodyTemperatureForPlayer(player)
    if not player then return end
    local rec = getBodyTempData(player)
    if not rec or not rec.core then return end
    player:setTemperature(rec.core)
end

local function _forEachActivePlayer(callback)
    if not callback then return end
    if not getSpecificPlayer then return end
    local hasIterator = false
    if getNumActivePlayers and getSpecificPlayer then
        local count = getNumActivePlayers()
        if count and count > 0 then
            hasIterator = true
            for idx = 0, count - 1 do
                local ply = getSpecificPlayer(idx)
                if ply then callback(ply) end
            end
        end
    end

    if hasIterator then return end

    local player = getSpecificPlayer and getSpecificPlayer(0) or nil
    if player then callback(player) end
end

function RC_TempSim.enforceBodyTemperatures()
    _forEachActivePlayer(_enforceBodyTemperatureForPlayer)
end

---------------------------------------------------------------------------
-- Events
---------------------------------------------------------------------------

function RC_TempSim.OnPlayerUpdate(player)
    if not player then return end
    RC_TempSim.rebuildForCurrentBuilding(player)
    RC_TempSim.updatePlayerBodyTemperature(player)
end

local function _performScheduledUpdate()
    local player = getSpecificPlayer(0)
    if not player then return end
    RC_TempSim.rebuildForCurrentBuilding(player)
    RC_TempSim.updateStep(player)
end

function RC_TempSim.OnEveryTenMinutes()
    _performScheduledUpdate()
end

function RC_TempSim.OnEveryMinuteUpdate(minutes)
    _performScheduledUpdate()
end

function RC_TempSim.OnEveryHours()
    recordOutsideTemperature()
end

local function _triggerRealisticColdMinuteUpdate()
    triggerEvent(MINUTE_EVENT_NAME, 1)
end

local function _ensureMinuteEventTrigger()
    if RC_TempSim._minuteEventTriggerRegistered then return end
    if Events and Events.EveryOneMinute and Events.EveryOneMinute.Add then
        Events.EveryOneMinute.Add(_triggerRealisticColdMinuteUpdate)
        RC_TempSim._minuteEventTriggerRegistered = true
    end
end

function RC_TempSim.applyUpdateIntervalSettings()
    local useMinute = _shouldUseOneMinuteUpdates()
    local defaultMinutes = RC_TempSim.DEFAULT_UPDATE_EVERY_MINUTES or 10
    local desiredMinutes = useMinute and 1 or defaultMinutes

    RC_TempSim.UPDATE_EVERY_MINUTES = desiredMinutes

    _ensureMinuteEventTrigger()

    if Events then
        if Events.EveryTenMinutes and Events.EveryTenMinutes.Remove then
            Events.EveryTenMinutes.Remove(RC_TempSim.OnEveryTenMinutes)
        end
        if Events.OnRealisticColdMinuteUpdate and Events.OnRealisticColdMinuteUpdate.Remove then
            Events.OnRealisticColdMinuteUpdate.Remove(RC_TempSim.OnEveryMinuteUpdate)
        end

        if useMinute then
            if Events.OnRealisticColdMinuteUpdate and Events.OnRealisticColdMinuteUpdate.Add then
                Events.OnRealisticColdMinuteUpdate.Add(RC_TempSim.OnEveryMinuteUpdate)
            end
        else
            if Events.EveryTenMinutes and Events.EveryTenMinutes.Add then
                Events.EveryTenMinutes.Add(RC_TempSim.OnEveryTenMinutes)
            end
        end
    end

    RC_TempSim._usingOneMinuteUpdates = useMinute
end

local originalPauseHandler = MainScreen.onMenuItemMouseDownMainMenu

MainScreen.onMenuItemMouseDownMainMenu = function(item, x, y)
    if item and item.internal == "EXIT" and MainScreen.instance and MainScreen.instance.inGame then
        if Thermoregulator and Thermoregulator.setSimulationMultiplier then
            Thermoregulator.setSimulationMultiplier(1)
        end
        if Thermoregulator_tryouts and Thermoregulator_tryouts.setSimulationMultiplier then
            Thermoregulator_tryouts.setSimulationMultiplier(1)
        end
    end
    return originalPauseHandler(item, x, y)
end

local function _refreshSandboxAndSchedule()
    RC_TempSim.refreshSandboxVars()
    RC_TempSim.applyUpdateIntervalSettings()
end

_refreshSandboxAndSchedule()

Events.OnCreatePlayer.Add(RC_TempSim.initializeThermoregulatorOverrides)

Events.OnInitWorld.Add(_refreshSandboxAndSchedule)
Events.OnGameStart.Add(_refreshSandboxAndSchedule)
Events.OnLoad.Add(_refreshSandboxAndSchedule)

Events.OnPlayerUpdate.Add(RC_TempSim.OnPlayerUpdate)
Events.EveryHours.Add(RC_TempSim.OnEveryHours)
Events.OnTick.Add(RC_TempSim.enforceBodyTemperatures)

require("RC_DrinkHooks")
