RC_RoomLogic = RC_RoomLogic or {}

-- tweak if needed
RC_RoomLogic.HUGE_ROOM_THRESHOLD = 30

-- cache for huge buildings, keyed by buildingDef + z
RC_RoomLogic._hugeCache = RC_RoomLogic._hugeCache or {}

function RC_RoomLogic.isHugeBuilding(roomDefs)
    if not roomDefs or not roomDefs.size then return false end
    local ok, size = pcall(function() return roomDefs:size() end)
    if not ok or not size then return false end
    return size >= RC_RoomLogic.HUGE_ROOM_THRESHOLD
end

---------------------------------------------------------------------------
-- SHARED HELPERS (moved out of CustomKey to avoid too many locals)
---------------------------------------------------------------------------

function RC_RoomLogic.roomName(roomLike)
    if not roomLike then return "<nil>" end
    if roomLike.getName then
        local ok, name = pcall(function() return roomLike:getName() end)
        if ok and name and name ~= "" then return name end
    end
    if roomLike.getIsoRoom then
        local okIso, iso = pcall(function() return roomLike:getIsoRoom() end)
        if okIso and iso and iso.getName then
            local okName, n = pcall(function() return iso:getName() end)
            if okName and n and n ~= "" then return n end
        end
    end
    return tostring(roomLike)
end

function RC_RoomLogic.isDoorNorth(door)
    if not door then return nil end
    if door.isNorth then local ok,v=pcall(function() return door:isNorth() end); if ok then return v end end
    if door.getNorth then local ok,v=pcall(function() return door:getNorth() end); if ok then return v end end
    return nil
end

local function _appendList(dst, zlist)
    if not zlist then return end
    for i = 0, zlist:size() - 1 do
        dst[#dst+1] = zlist:get(i)
    end
end

function RC_RoomLogic.getAllSquareObjects(sq)
    local out = {}
    if not sq then return out end
    if sq.getObjects        then _appendList(out, sq:getObjects()) end
    if sq.getSpecialObjects then _appendList(out, sq:getSpecialObjects()) end
    if sq.getIsoDoor then
        local ok, d = pcall(function() return sq:getIsoDoor() end)
        if ok and d then out[#out+1] = d end
    end
    return out
end

local function doorOrientationFromSprite(obj)
    if not obj or not obj.getSprite then return nil end
    local spr = obj:getSprite()
    if not spr then return nil end
    local props = spr.getProperties and spr:getProperties() or nil

    if props and props.Val then
        local t = props:Val("Type") or props:Val("type") or props:Val("SpriteType") or props:Val("sprite.type")
        if t == "doorN" or t == "DoorN" then return "north" end
        if t == "doorW" or t == "DoorW" then return "west" end
    end

    if obj.isNorth then
        local ok, n = pcall(function() return obj:isNorth() end)
        if ok then return n and "north" or "west" end
    end

    local sprName = spr.getName and spr:getName() or ""
    sprName = string.lower(tostring(sprName))
    if sprName:find("door[_-]?n") then return "north" end
    if sprName:find("door[_-]?w") then return "west" end

    return nil
end

local function windowOrientationFromSprite(obj)
    if not obj or not obj.getSprite then return nil end
    local spr = obj:getSprite()
    if not spr then return nil end
    local props = spr.getProperties and spr:getProperties() or nil

    if props and props.Val then
        local t = props:Val("Type") or props:Val("type") or props:Val("SpriteType") or props:Val("sprite.type")
        if t == "windowN" or t == "WindowN" then return "north" end
        if t == "windowW" or t == "WindowW" then return "west" end
    end

    -- if props and props.Is then
    --     if props:Is(IsoFlagType.WindowN) or props:Is(IsoFlagType.windowN) then return "north" end
    --     if props:Is(IsoFlagType.WindowW) or props:Is(IsoFlagType.windowW) then return "west" end
    -- end

    local sprName = spr.getName and spr:getName() or ""
    sprName = string.lower(tostring(sprName))
    if sprName:find("window[_-]?n") then return "north" end
    if sprName:find("window[_-]?w") then return "west" end

    return nil
end

local function _northFlagFromOrientation(orient)
    if orient == nil then return nil end
    if type(orient) == "boolean" then return orient end
    if type(orient) == "string" then
        if orient == "north" then return true end
        if orient == "west" then return false end
    end
    return nil
end

local function _doorNorthFlag(obj)
    local direct = _northFlagFromOrientation(RC_RoomLogic.isDoorNorth(obj))
    if direct ~= nil then return direct end
    return _northFlagFromOrientation(doorOrientationFromSprite(obj))
end

local function _doorOrientation(obj)
    if not obj then return nil end
    local northFlag = _doorNorthFlag(obj)
    if northFlag ~= nil then
        return northFlag and "north" or "west"
    end
    return doorOrientationFromSprite(obj)
end

local function _windowNorthFlag(obj)
    local direct = _northFlagFromOrientation(RC_RoomLogic.isWindowNorth(obj))
    if direct ~= nil then return direct end
    return _northFlagFromOrientation(windowOrientationFromSprite(obj))
end

local function _squaresForEdgeObject(obj, orientFn)
    if not obj or not orientFn then return nil, nil, nil end
    local sq = obj.getSquare and obj:getSquare() or nil
    if not sq then return nil, nil, nil end
    local northFlag = orientFn(obj)
    if northFlag == nil then return nil, nil, nil end
    local dir = northFlag and "north" or "west"
    local neighbor = nil
    if dir == "north" then
        neighbor = sq.getN and sq:getN() or nil
    else
        neighbor = sq.getW and sq:getW() or nil
    end
    if neighbor and neighbor.getZ and sq.getZ and neighbor:getZ() ~= sq:getZ() then
        neighbor = nil
    end
    return sq, neighbor, dir
end

local function _squaresAreUnblocked(a, b)
    if not (a and b) then return nil end
    if a.isBlockedTo then
        local ok, blocked = pcall(function() return a:isBlockedTo(b) end)
        if ok and blocked ~= nil then
            return not blocked
        end
    end
    return nil
end

local function _booleanFromField(obj, field)
    if not obj or not field then return nil end
    local ok, value = pcall(function() return obj[field] end)
    if not ok or value == nil then return nil end
    if type(value) == "function" then return nil end
    return value and true or false
end

local function _hasOpenAPI(obj)
    if not obj then return false end
    if obj.isOpen or obj.IsOpen or obj.isDoorOpen then return true end
    if _booleanFromField(obj, "open") ~= nil then return true end
    if _booleanFromField(obj, "Open") ~= nil then return true end
    return false
end

local function isDoorFrameObject(obj)
    if not obj then return false end
    if instanceof and instanceof(obj, "IsoThumpable") and obj.isDoorFrame then
        local ok, res = pcall(function() return obj:isDoorFrame() end)
        if ok and res then return true end
    end
    if obj and obj.isDoorFrame == true then
        return true
    end
    return false
end

local function isWindowObject(obj)
    if not obj then return false end
    if instanceof and instanceof(obj, "IsoWindow") then
        return true
    end
    if instanceof and instanceof(obj, "IsoThumpable") and obj.isWindow then
        local ok, res = pcall(function() return obj:isWindow() end)
        if ok and res then return true end
    end
    if obj.isWindow == true then
        return true
    end
    return false
end

local function isCurtainObject(obj)
    if not obj then return false end
    if instanceof and instanceof(obj, "IsoCurtain") then return true end
    if obj.getObjectName then
        local ok, name = pcall(function() return obj:getObjectName() end)
        if ok and name == "Curtain" then return true end
    end
    return false
end

local function isDoorObject(obj)
    -- print("obj: ", obj)
    if not obj then return false end
    if instanceof and instanceof(obj, "IsoDoor") then return true end
    if instanceof and instanceof(obj, "IsoThumpable") and obj.isDoor then
        local ok, res = pcall(function() return obj:isDoor() end)
        if ok and res then return true end
    end
    -- When instanceof checks fail (e.g. player-placed IsoThumpable clones),
    -- fall back to sprite metadata so that we still recognise the door leaf.
    if not isDoorFrameObject(obj) then
        if isWindowObject(obj) then
            return false
        end
        if isCurtainObject(obj) then
            return false
        end
        local orient = _doorOrientation(obj)
        if orient == "north" or orient == "west" then
            return true
        end
    end
    return false
end

function RC_RoomLogic.isDoorOpen(door)
    if not door then return false end
    local methods = { "isOpen", "IsOpen", "isDoorOpen" }
    for _, m in ipairs(methods) do
        local fn = door[m]
        if fn then
            local ok, v = pcall(fn, door)
            if ok and v ~= nil then
                return v and true or false
            end
        end
    end

    local fieldState = _booleanFromField(door, "open") or _booleanFromField(door, "Open")
    if fieldState ~= nil then
        return fieldState
    end

    local sq, neighbor = _squaresForEdgeObject(door, _doorNorthFlag)
    local passable = _squaresAreUnblocked(sq, neighbor)
    if passable ~= nil then
        return passable and true or false
    end

    return false
end

function RC_RoomLogic.isWindowNorth(win)
    if not win then return nil end
    if win.isNorth then local ok,v=pcall(function() return win:isNorth() end); if ok then return v end end
    if win.getNorth then local ok,v=pcall(function() return win:getNorth() end); if ok then return v end end
    return nil
end

function RC_RoomLogic.isWindowOpen(win)
    if not win then return false end
    if win.isOpen then local ok,v=pcall(function() return win:isOpen() end); if ok then return v and true or false end end
    if win.IsOpen then local ok,v=pcall(function() return win:IsOpen() end); if ok then return v and true or false end end

    local sq, neighbor = _squaresForEdgeObject(win, _windowNorthFlag)
    local passable = _squaresAreUnblocked(sq, neighbor)
    if passable ~= nil then
        return passable and true or false
    end

    return false
end

function RC_RoomLogic.isWindowSmashed(win)
    if not win then return false end
    if win.isSmashed then local ok,v=pcall(function() return win:isSmashed() end); if ok then return v and true or false end end
    return false
end

function RC_RoomLogic.isWindowBarricaded(win)
    if not win then return false end
    if win.isBarricaded then
        local ok, v = pcall(function() return win:isBarricaded() end)
        if ok and v ~= nil then
            return v and true or false
        end
    end

    local function hasBarricade(method)
        if not method then return false end
        if type(method) ~= "function" then return false end
        local ok, barricade = pcall(method, win)
        return ok and barricade ~= nil
    end

    if hasBarricade(win.getBarricadeOnSameSquare) then return true end
    if hasBarricade(win.getBarricadeOnOppositeSquare) then return true end

    local field = _booleanFromField(win, "Barricaded")
    if field ~= nil then
        return field
    end

    return false
end

local function _barricadePlankCount(barricade)
    if not barricade or not barricade.getNumPlanks then return 0 end
    local ok, num = pcall(function() return barricade:getNumPlanks() end)
    if ok and type(num) == "number" then
        return num
    end
    return 0
end

function RC_RoomLogic.windowBarricadePlankCount(win)
    if not win then return 0 end

    local total = 0
    local function addFrom(method)
        if not method or type(method) ~= "function" then return end
        local ok, barricade = pcall(method, win)
        if ok and barricade then
            total = total + _barricadePlankCount(barricade)
        end
    end

    addFrom(win.getBarricadeOnSameSquare)
    addFrom(win.getBarricadeOnOppositeSquare)

    if total < 0 then return 0 end
    if total > 4 then return 4 end
    return total
end

function RC_RoomLogic.isCurtainOpen(curtain)
    if not isCurtainObject(curtain) then return false end
    if curtain.IsOpen then
        local ok, open = pcall(function() return curtain:IsOpen() end)
        if ok and open ~= nil then
            return open and true or false
        end
    end
    local state = _booleanFromField(curtain, "open")
    if state ~= nil then
        return state and true or false
    end
    return false
end

function RC_RoomLogic.windowHasClosedCurtains(win)
    if not win then return false end

    local squares = {}
    if win.getSquare then
        local ok, sq = pcall(function() return win:getSquare() end)
        if ok and sq then squares[#squares+1] = sq end
    end
    if win.getOppositeSquare then
        local ok, sq = pcall(function() return win:getOppositeSquare() end)
        if ok and sq then squares[#squares+1] = sq end
    end

    for _, sq in ipairs(squares) do
        local objs = RC_RoomLogic.getAllSquareObjects(sq)
        for _, obj in ipairs(objs) do
            if isCurtainObject(obj) and (not RC_RoomLogic.isCurtainOpen(obj)) then
                return true
            end
        end
    end

    if win.HasCurtains then
        local ok, curtain = pcall(function() return win:HasCurtains() end)
        if ok and (not RC_RoomLogic.isCurtainOpen(curtain)) then
            return true
        end
    end

    return false
end

function RC_RoomLogic.getCurtainOnEdge(sq0, dir) -- "north" / "west"
    if not sq0 then return nil end
    local objs = RC_RoomLogic.getAllSquareObjects(sq0)
    for _, obj in ipairs(objs) do
        if isCurtainObject(obj) then
            local north = _doorNorthFlag(obj)
            if (dir == "north" and north == true) or (dir == "west" and north == false) then
                if _hasOpenAPI(obj) then
                    return obj
                end
            end
        end
    end
    return nil
end

function RC_RoomLogic.getWindowOnEdge(sq0, dir) -- "north" / "west"
    if not sq0 then return nil end
    local objs = RC_RoomLogic.getAllSquareObjects(sq0)
    for _, obj in ipairs(objs) do
        local isWindow = instanceof and instanceof(obj, "IsoWindow")
        if not isWindow and instanceof and instanceof(obj, "IsoThumpable") and obj.isWindow then
            local ok, res = pcall(function() return obj:isWindow() end)
            isWindow = ok and res and true or false
        end
        if isWindow then
            local north = _windowNorthFlag(obj)
            if (dir == "north" and north == true) or (dir == "west" and north == false) then
                return obj
            end
        end
    end
    return nil
end

local function _doorMatchesEdge(obj, edge)
    return _doorOrientation(obj) == edge
end

function RC_RoomLogic.getDoorOnEdge(sq0, edge)
    if not sq0 then return nil end
    local objs = RC_RoomLogic.getAllSquareObjects(sq0)
    for _, obj in ipairs(objs) do
        if isDoorObject(obj) and _doorMatchesEdge(obj, edge) then
            if _hasOpenAPI(obj) then
                return obj
            end
            -- door frames without open/closed API are ignored; treat as blocked later
        end
    end
    return nil
end

function RC_RoomLogic.hasDoorFrameOnEdge(sq0, edge)
    if not sq0 then return false end
    local objs = RC_RoomLogic.getAllSquareObjects(sq0)
    for _, obj in ipairs(objs) do
        if isDoorFrameObject(obj) and _doorMatchesEdge(obj, edge) then
            return true
        end
    end
    return false
end

function RC_RoomLogic.isTableEmpty(t)
    if t == nil then return true end
    for _ in pairs(t) do return false end
    return true
end

function RC_RoomLogic.edgeHasBlockingOnSquare(sq0, dir)
    -- print("edgeHasBlockingOnSquare: ", sq0)
    if not sq0 then return false end
    local objs = RC_RoomLogic.getAllSquareObjects(sq0)
    for _, obj in ipairs(objs) do
        if obj then
            if isDoorFrameObject(obj) then
                -- Empty door frames should be treated as gaps, never as blocking walls.
                -- Skip to next object.
            elseif not (instanceof and instanceof(obj, "IsoDoor")) then
                local spr = obj and obj:getSprite() or nil
                local props = spr and spr:getProperties() or nil
                -- if props and props.Is then
                --     if dir == "north" then
                --         if props:Is(IsoFlagType.WallN) or props:Is(IsoFlagType.collideN) then return true end
                --     elseif dir == "west" then
                --         if props:Is(IsoFlagType.WallW) or props:Is(IsoFlagType.collideW) then return true end
                --     elseif dir == "south" then
                --         if props:Is(IsoFlagType.WallS) or props:Is(IsoFlagType.collideS) then return true end
                --     elseif dir == "east" then
                --         if props:Is(IsoFlagType.WallE) or props:Is(IsoFlagType.collideE) then return true end
                --     end
                -- end
            end
        end
    end
    return false
end

-- Passable between two *rooms* across an edge?
function RC_RoomLogic.edgeIsPassable(sqA, dir, sqB)
    -- print("edgeIsPassable: ", sqA, " | ", sqB)
    if not sqA or not sqB then return false end

    local function openDoorOn(square, edge)
        local door = RC_RoomLogic.getDoorOnEdge(square, edge)
        if door then
            return RC_RoomLogic.isDoorOpen(door) and true or false
        end
        return nil
    end

    -- 1) Door check (note: S/E doors live on the neighbor square)
    if dir == "north" then
        local od = openDoorOn(sqA, "north")
        if od ~= nil then return od end
    elseif dir == "west" then
        local od = openDoorOn(sqA, "west")
        if od ~= nil then return od end
    elseif dir == "south" then
        local od = openDoorOn(sqB, "north") -- door on neighbor's north edge
        if od ~= nil then return od end
    elseif dir == "east" then
        local od = openDoorOn(sqB, "west") -- door on neighbor's west edge
        if od ~= nil then return od end
    end

    -- 2) Explicit wall objects (same rule: S/E walls live on neighbor’s N/W getter)
    if dir == "north" and sqA.getWallN then
        local wallObj = sqA:getWallN()
        if wallObj == nil then
            if not RC_RoomLogic.edgeHasBlockingOnSquare(sqB, "south") then return true end
        end
        return false
    elseif dir == "west" and sqA.getWallW then
        local wallObj = sqA:getWallW()
        if wallObj == nil then
            if not RC_RoomLogic.edgeHasBlockingOnSquare(sqB, "east") then return true end
        end
        return false
    elseif dir == "south" and sqB.getWallN then
        local wallObj = sqB:getWallN()
        if wallObj == nil then
            if not RC_RoomLogic.edgeHasBlockingOnSquare(sqA, "south") then return true end
        end
        return false
    elseif dir == "east" and sqB.getWallW then
        local wallObj = sqB:getWallW()
        if wallObj == nil then
            if not RC_RoomLogic.edgeHasBlockingOnSquare(sqA, "east") then return true end
        end
        return false
    end

    -- 3) Fallback via sprite flags on both sides
    if dir == "north" then
        if RC_RoomLogic.edgeHasBlockingOnSquare(sqA,"north") then return false end
        if RC_RoomLogic.edgeHasBlockingOnSquare(sqB,"south") then return false end
        return true
    elseif dir == "west" then
        if RC_RoomLogic.edgeHasBlockingOnSquare(sqA,"west") then return false end
        if RC_RoomLogic.edgeHasBlockingOnSquare(sqB,"east") then return false end
        return true
    elseif dir == "south" then
        if RC_RoomLogic.edgeHasBlockingOnSquare(sqA,"south") then return false end
        if RC_RoomLogic.edgeHasBlockingOnSquare(sqB,"north") then return false end
        return true
    elseif dir == "east" then
        if RC_RoomLogic.edgeHasBlockingOnSquare(sqA,"east") then return false end
        if RC_RoomLogic.edgeHasBlockingOnSquare(sqB,"west") then return false end
        return true
    end

    return false
end

-- Stairs detection using obj:isStairsNorth/West
function RC_RoomLogic.hasStairsNorth(sq0)
    if not sq0 or not sq0.getObjects then return false end
    local objs = sq0:getObjects()
    if not objs then return false end
    for i = 0, objs:size() - 1 do
        local obj = objs:get(i)
        if obj and obj.isStairsNorth then
            local ok, res = pcall(function() return obj:isStairsNorth() end)
            if ok and res then return true end
        end
    end
    return false
end
function RC_RoomLogic.hasStairsWest(sq0)
    if not sq0 or not sq0.getObjects then return false end
    local objs = sq0:getObjects()
    if not objs then return false end
    for i = 0, objs:size() - 1 do
        local obj = objs:get(i)
        if obj and obj.isStairsWest then
            local ok, res = pcall(function() return obj:isStairsWest() end)
            if ok and res then return true end
        end
    end
    return false
end

function RC_RoomLogic.stairsOrientation(sq0) -- "N" or "W"
    if RC_RoomLogic.hasStairsNorth(sq0) then return "N" end
    if RC_RoomLogic.hasStairsWest(sq0)  then return "W" end
    return nil
end

function RC_RoomLogic.findBottomStairSquare(sq0, orient)
    local s = sq0
    local steps = 0
    while s and steps < 4 do
        local nextS = (orient == "N" and s.getS and s:getS())
                   or (orient == "W" and s.getE and s:getE()) or nil
        if nextS and ((orient == "N" and RC_RoomLogic.hasStairsNorth(nextS))
                   or (orient == "W" and RC_RoomLogic.hasStairsWest(nextS))) then
            s = nextS; steps = steps + 1
        else
            break
        end
    end
    return s
end

-- From ANY stair tile, compute landings
function RC_RoomLogic.stairsEndpointsFromAnyStair(sq0)
    local orient = RC_RoomLogic.stairsOrientation(sq0)
    if not orient or not sq0 then return nil, nil end
    local bottomBase = RC_RoomLogic.findBottomStairSquare(sq0, orient)
    if not bottomBase then return nil, nil end
    local x, y, z = bottomBase:getX(), bottomBase:getY(), bottomBase:getZ()
    local cell = getCell and getCell() or nil
    if not cell then return nil, nil end
    if orient == "N" then
        local bottomLanding = bottomBase.getS and bottomBase:getS() or cell:getGridSquare(x, y+1, z)
        local topLanding    = cell:getGridSquare(x, y-3, z+1)
        return bottomLanding, topLanding
    else -- "W"
        local bottomLanding = bottomBase.getE and bottomBase:getE() or cell:getGridSquare(x+1, y, z)
        local topLanding    = cell:getGridSquare(x-3, y, z+1)
        return bottomLanding, topLanding
    end
end

function RC_RoomLogic.roomDefFromSquare(sq0)
    if not sq0 then return nil end
    local iso = sq0:getRoom()
    if not iso then return nil end
    if iso.getRoomDef then
        local ok, rd = pcall(function() return iso:getRoomDef() end)
        if ok and rd then return rd end
    end
    return nil
end

function RC_RoomLogic.roomDefFromSquareOrNeighbors(sq0)
    local rd = RC_RoomLogic.roomDefFromSquare(sq0)
    if rd or (not sq0) then return rd end
    local z = sq0:getZ()
    local function try(s)
        if s and s:getZ() == z then
            local r = RC_RoomLogic.roomDefFromSquare(s)
            if r then return r end
        end
    end
    return try(sq0.getN and sq0:getN())
        or  try(sq0.getS and sq0:getS())
        or  try(sq0.getW and sq0:getW())
        or  try(sq0.getE and sq0:getE())
        or  nil
end

function RC_RoomLogic.buildingKey(def, z)
    return tostring(def) .. ":" .. tostring(z)
end

-- get squares for a RoomDef (with a small cache object you supply)
function RC_RoomLogic.getRoomSquares(cache, rd)
    local sc = cache.squaresCache[rd]
    if sc then return sc end
    local squares = {}
    local iso = rd.getIsoRoom and rd:getIsoRoom() or nil
    if iso and iso.getSquares then
        local sqList = iso:getSquares()
        for j = 0, sqList:size() - 1 do
            squares[#squares+1] = sqList:get(j)
        end
    end
    cache.squaresCache[rd] = squares
    return squares
end

-- Lateral (same-Z) neighbors for HUGE-mode BFS (uses rd:getZ())
function RC_RoomLogic.getPassableNeighbors(cache, rd, zFallback)
    local neighbors, seen = {}, {}
    local rdZ = (rd and rd.getZ) and rd:getZ() or zFallback
    local squares = RC_RoomLogic.getRoomSquares(cache, rd)
    for _, rsq in ipairs(squares) do
        if rsq then
            -- north
            if rsq.getN then
                local nSq = rsq:getN()
                if nSq and nSq:getZ() == rdZ then
                    local nIso = nSq:getRoom()
                    local nDef = nIso and (cache.byIso[nIso] or (nIso.getRoomDef and nIso:getRoomDef())) or nil
                    if nDef and nDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "north", nSq) then
                        if not seen[nDef] then seen[nDef]=true; neighbors[#neighbors+1] = nDef end
                    end
                end
            end
            -- west
            if rsq.getW then
                local wSq = rsq:getW()
                if wSq and wSq:getZ() == rdZ then
                    local wIso = wSq:getRoom()
                    local wDef = wIso and (cache.byIso[wIso] or (wIso.getRoomDef and wIso:getRoomDef())) or nil
                    if wDef and wDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "west", wSq) then
                        if not seen[wDef] then seen[wDef]=true; neighbors[#neighbors+1] = wDef end
                    end
                end
            end
        end
    end
    return neighbors
end

-- Cross-floor neighbors via stairs (HUGE-mode), cached per room
function RC_RoomLogic.getStairsNeighbors(cache, rd)
    local cached = cache.stairsNeighborsCache[rd]
    if cached then return cached end
    local nbs, seen = {}, {}
    local squares = RC_RoomLogic.getRoomSquares(cache, rd)
    for _, rsq in ipairs(squares) do
        local bottomSq, topSq = RC_RoomLogic.stairsEndpointsFromAnyStair(rsq)
        if bottomSq or topSq then
            local bottomDef = RC_RoomLogic.roomDefFromSquareOrNeighbors(bottomSq) or rd
            local topDef    = RC_RoomLogic.roomDefFromSquareOrNeighbors(topSq)
            if bottomDef and topDef then
                if bottomDef == rd and not seen[topDef] then
                    seen[topDef] = true; nbs[#nbs+1] = topDef
                elseif topDef == rd and not seen[bottomDef] then
                    seen[bottomDef] = true; nbs[#nbs+1] = bottomDef
                end
            end
        end
    end
    cache.stairsNeighborsCache[rd] = nbs
    return nbs
end

-- Build (or return) the perimeter cache for a (buildingDef, zLevel)
function RC_RoomLogic.ensureHugeCache(buildingDef, roomDefs, zLevel, opts)
    opts = opts or {}
    local yieldFn = opts.yieldFn
    local function step(units)
        if not yieldFn then return end
        yieldFn(units or 1)
    end
    local key = RC_RoomLogic.buildingKey(buildingDef, zLevel)
    local cache = RC_RoomLogic._hugeCache[key]
    if cache then
        cache.outsideDoors = cache.outsideDoors or {}
        cache.outsideWindows = cache.outsideWindows or {}
        cache.outsideCurtains = cache.outsideCurtains or {}
        cache.outsideGaps = cache.outsideGaps or {}
        cache.outsideByRoom = cache.outsideByRoom or {}
        cache.perimeterRooms = cache.perimeterRooms or {}
        cache.perimeterRoomSet = cache.perimeterRoomSet or {}
        cache.squaresCache = cache.squaresCache or {}
        cache.stairsNeighborsCache = cache.stairsNeighborsCache or {}
        cache.exteriorByRoom = cache.exteriorByRoom or {}
        cache.exteriorComputed = cache.exteriorComputed or {}
        cache.seenExteriorDoors = cache.seenExteriorDoors or {}
        cache.seenExteriorWindows = cache.seenExteriorWindows or {}
        cache.seenExteriorCurtains = cache.seenExteriorCurtains or {}
        return cache
    end

    -- map IsoRoom -> RoomDef (same Z)
    local byIso, roomDefsZ = {}, {}
    for i = 0, roomDefs:size() - 1 do
        local rd = roomDefs:get(i)
        if rd and zLevel == rd:getZ() then
            roomDefsZ[#roomDefsZ+1] = rd
            local iso = rd.getIsoRoom and rd:getIsoRoom() or nil
            if iso then byIso[iso] = rd end
            step()
        end
    end

    cache = {
        buildingDef = buildingDef,
        z = zLevel,
        byIso = byIso,
        roomDefsZ = roomDefsZ,
        perimeterRooms = {},
        perimeterRoomSet = {},
        outsideDoors = {},
        outsideWindows = {},
        outsideCurtains = {},
        outsideGaps = {},
        outsideByRoom = {},
        squaresCache = {},          -- roomDef -> {squares}
        stairsNeighborsCache = {},  -- roomDef -> {neighbors via stairs}
        exteriorByRoom = {},
        exteriorComputed = {},
        seenExteriorDoors = {},
        seenExteriorWindows = {},
        seenExteriorCurtains = {},
    }
    RC_RoomLogic._hugeCache[key] = cache
    return cache
end

local function _hugeCacheAddExterior(cache, rd, rec)
    if not cache or not rd or not rec then return end
    cache.outsideByRoom[rd] = cache.outsideByRoom[rd] or {}
    table.insert(cache.outsideByRoom[rd], rec)
    if rec.type == "door" then
        cache.outsideDoors[#cache.outsideDoors+1] = rec
    elseif rec.type == "window" then
        cache.outsideWindows[#cache.outsideWindows+1] = rec
    elseif rec.type == "curtain" then
        cache.outsideCurtains[#cache.outsideCurtains+1] = rec
    elseif rec.type == "gap" then
        cache.outsideGaps[#cache.outsideGaps+1] = rec
    end
end

local function _recordExteriorDoor(cache, rd, door, sqOnDoor, edgeLabel, records)
    if not cache or not rd or not door or not sqOnDoor then return end
    cache.seenExteriorDoors = cache.seenExteriorDoors or {}
    local keyObj = tostring(door)
    if cache.seenExteriorDoors[keyObj] then return end
    cache.seenExteriorDoors[keyObj] = true
    local rec = {
        type = "door",
        room = rd,
        x = sqOnDoor:getX(),
        y = sqOnDoor:getY(),
        z = sqOnDoor:getZ(),
        edge = edgeLabel,
        door = door,
    }
    _hugeCacheAddExterior(cache, rd, rec)
    if records then records[#records+1] = rec end
end

local function _recordExteriorWindow(cache, rd, win, sqWithWin, edgeLabel, records)
    if not cache or not rd or not win or not sqWithWin then return end
    cache.seenExteriorWindows = cache.seenExteriorWindows or {}
    local keyObj = tostring(win)
    if cache.seenExteriorWindows[keyObj] then return end
    cache.seenExteriorWindows[keyObj] = true
    local rec = {
        type = "window",
        room = rd,
        x = sqWithWin:getX(),
        y = sqWithWin:getY(),
        z = sqWithWin:getZ(),
        edge = edgeLabel,
        window = win,
        open = RC_RoomLogic.isWindowOpen(win),
        smashed = RC_RoomLogic.isWindowSmashed(win),
        barricaded = RC_RoomLogic.isWindowBarricaded(win),
        barricadePlanks = RC_RoomLogic.windowBarricadePlankCount(win),
        curtainClosed = RC_RoomLogic.windowHasClosedCurtains(win),
    }
    _hugeCacheAddExterior(cache, rd, rec)
    if records then records[#records+1] = rec end
end

local function _recordExteriorCurtain(cache, rd, curtain, sqWithCurtain, edgeLabel, records)
    if not cache or not rd or not curtain or not sqWithCurtain then return end
    cache.seenExteriorCurtains = cache.seenExteriorCurtains or {}
    local keyObj = tostring(curtain)
    if cache.seenExteriorCurtains[keyObj] then return end
    cache.seenExteriorCurtains[keyObj] = true
    local rec = {
        type = "curtain",
        room = rd,
        x = sqWithCurtain:getX(),
        y = sqWithCurtain:getY(),
        z = sqWithCurtain:getZ(),
        edge = edgeLabel,
        curtain = curtain,
        open = RC_RoomLogic.isCurtainOpen(curtain),
    }
    _hugeCacheAddExterior(cache, rd, rec)
    if records then records[#records+1] = rec end
end

local function _recordExteriorGap(cache, rd, sqOnEdge, edgeLabel, records)
    if not cache or not rd or not sqOnEdge then return end
    local rec = {
        type = "gap",
        room = rd,
        x = sqOnEdge:getX(),
        y = sqOnEdge:getY(),
        z = sqOnEdge:getZ(),
        edge = edgeLabel,
    }
    _hugeCacheAddExterior(cache, rd, rec)
    if records then records[#records+1] = rec end
end

function RC_RoomLogic.ensureHugePerimeterForRoom(cache, rd, opts)
    if not cache or not rd then return {} end
    cache.exteriorByRoom = cache.exteriorByRoom or {}
    cache.exteriorComputed = cache.exteriorComputed or {}
    if cache.exteriorComputed[rd] then
        return cache.exteriorByRoom[rd] or {}
    end

    opts = opts or {}
    local yieldFn = opts.yieldFn
    local function step(units)
        if not yieldFn then return end
        yieldFn(units or 1)
    end

    local records = {}
    local squares = RC_RoomLogic.getRoomSquares(cache, rd)
    local zLevel = (rd.getZ and rd:getZ()) or cache.z or 0
    local isPerimeter = false

    for _, rsq in ipairs(squares or {}) do
        step()
        if rsq then
            local nSq = rsq.getN and rsq:getN() or nil
            local sSq = rsq.getS and rsq:getS() or nil
            local wSq = rsq.getW and rsq:getW() or nil
            local eSq = rsq.getE and rsq:getE() or nil

            local nIso = nSq and (nSq:getZ()==zLevel) and nSq:getRoom() or nil
            local sIso = sSq and (sSq:getZ()==zLevel) and sSq:getRoom() or nil
            local wIso = wSq and (wSq:getZ()==zLevel) and wSq:getRoom() or nil
            local eIso = eSq and (eSq:getZ()==zLevel) and eSq:getRoom() or nil

            if nSq and not nIso then
                isPerimeter = true
                local dN = RC_RoomLogic.getDoorOnEdge(rsq, "north")
                if dN then
                    _recordExteriorDoor(cache, rd, dN, rsq, "north", records)
                else
                    local cN = RC_RoomLogic.getCurtainOnEdge(rsq, "north")
                    if cN then
                        _recordExteriorCurtain(cache, rd, cN, rsq, "north", records)
                    end
                    local wN = RC_RoomLogic.getWindowOnEdge(rsq, "north")
                    if wN then
                        _recordExteriorWindow(cache, rd, wN, rsq, "north", records)
                    else
                        local frameN = RC_RoomLogic.hasDoorFrameOnEdge(rsq, "north")
                        if frameN then
                            _recordExteriorGap(cache, rd, rsq, "north", records)
                        elseif not RC_RoomLogic.edgeHasBlockingOnSquare(rsq,"north")
                           and not RC_RoomLogic.edgeHasBlockingOnSquare(nSq,"south") then
                            _recordExteriorGap(cache, rd, rsq, "north", records)
                        end
                    end
                end
            end

            if wSq and not wIso then
                isPerimeter = true
                local dW = RC_RoomLogic.getDoorOnEdge(rsq, "west")
                if dW then
                    _recordExteriorDoor(cache, rd, dW, rsq, "west", records)
                else
                    local cW = RC_RoomLogic.getCurtainOnEdge(rsq, "west")
                    if cW then
                        _recordExteriorCurtain(cache, rd, cW, rsq, "west", records)
                    end
                    local wW = RC_RoomLogic.getWindowOnEdge(rsq, "west")
                    if wW then
                        _recordExteriorWindow(cache, rd, wW, rsq, "west", records)
                    else
                        local frameW = RC_RoomLogic.hasDoorFrameOnEdge(rsq, "west")
                        if frameW then
                            _recordExteriorGap(cache, rd, rsq, "west", records)
                        elseif not RC_RoomLogic.edgeHasBlockingOnSquare(rsq,"west")
                           and not RC_RoomLogic.edgeHasBlockingOnSquare(wSq,"east") then
                            _recordExteriorGap(cache, rd, rsq, "west", records)
                        end
                    end
                end
            end

            if sSq and not sIso then
                isPerimeter = true
                local dS = RC_RoomLogic.getDoorOnEdge(sSq, "north")
                if dS then
                    _recordExteriorDoor(cache, rd, dS, sSq, "south", records)
                else
                    local cS = RC_RoomLogic.getCurtainOnEdge(sSq, "north")
                    if cS then
                        _recordExteriorCurtain(cache, rd, cS, sSq, "south", records)
                    end
                    local wS = RC_RoomLogic.getWindowOnEdge(sSq, "north")
                    if wS then
                        _recordExteriorWindow(cache, rd, wS, sSq, "south", records)
                    else
                        local frameS = RC_RoomLogic.hasDoorFrameOnEdge(sSq, "north")
                        if frameS then
                            _recordExteriorGap(cache, rd, rsq, "south", records)
                        elseif not RC_RoomLogic.edgeHasBlockingOnSquare(rsq,"south")
                           and not RC_RoomLogic.edgeHasBlockingOnSquare(sSq,"north") then
                            _recordExteriorGap(cache, rd, rsq, "south", records)
                        end
                    end
                end
            end

            if eSq and not eIso then
                isPerimeter = true
                local dE = RC_RoomLogic.getDoorOnEdge(eSq, "west")
                if dE then
                    _recordExteriorDoor(cache, rd, dE, eSq, "east", records)
                else
                    local cE = RC_RoomLogic.getCurtainOnEdge(eSq, "west")
                    if cE then
                        _recordExteriorCurtain(cache, rd, cE, eSq, "east", records)
                    end
                    local wE = RC_RoomLogic.getWindowOnEdge(eSq, "west")
                    if wE then
                        _recordExteriorWindow(cache, rd, wE, eSq, "east", records)
                    else
                        local frameE = RC_RoomLogic.hasDoorFrameOnEdge(eSq, "west")
                        if frameE then
                            _recordExteriorGap(cache, rd, rsq, "east", records)
                        elseif not RC_RoomLogic.edgeHasBlockingOnSquare(rsq,"east")
                           and not RC_RoomLogic.edgeHasBlockingOnSquare(eSq,"west") then
                            _recordExteriorGap(cache, rd, rsq, "east", records)
                        end
                    end
                end
            end
        end
    end

    if isPerimeter then
        cache.perimeterRoomSet = cache.perimeterRoomSet or {}
        if not cache.perimeterRoomSet[rd] then
            cache.perimeterRoomSet[rd] = true
            cache.perimeterRooms[#cache.perimeterRooms+1] = rd
        end
    end

    cache.exteriorByRoom[rd] = records
    cache.exteriorComputed[rd] = true
    return records
end

function RC_RoomLogic.ensureHugePerimeterForAll(cache, opts)
    if not cache then return end
    for _, rd in ipairs(cache.roomDefsZ or {}) do
        RC_RoomLogic.ensureHugePerimeterForRoom(cache, rd, opts)
    end
end

RC_RoomLogic.HUGE_SCAN_BUDGET_PER_CALL = RC_RoomLogic.HUGE_SCAN_BUDGET_PER_CALL or 4000
RC_RoomLogic._hugeScanStates = RC_RoomLogic._hugeScanStates or {}

RC_RoomLogic.HUGE_LIGHT_RESCAN_MINUTES = RC_RoomLogic.HUGE_LIGHT_RESCAN_MINUTES or 5
RC_RoomLogic.HUGE_IDLE_RESCAN_MINUTES = RC_RoomLogic.HUGE_IDLE_RESCAN_MINUTES or 30

local function _currentWorldHours()
    local gt = getGameTime and getGameTime() or nil
    if gt and gt.getWorldAgeHours then
        local ok, hours = pcall(function() return gt:getWorldAgeHours() end)
        if ok and hours then
            return hours
        end
    end
    return 0
end

local function _collectZLevels(roomDefs)
    local list, seen = {}, {}
    if not roomDefs then return list end
    for i = 0, roomDefs:size() - 1 do
        local rd = roomDefs:get(i)
        if rd and rd.getZ then
            local z = rd:getZ()
            if z and not seen[z] then
                seen[z] = true
                list[#list+1] = z
            end
        end
    end
    table.sort(list)
    return list
end

function RC_RoomLogic.markHugeScanDirty(buildingDef, roomDefs, opts)
    if not buildingDef then return end
    local state = RC_RoomLogic._hugeScanStates[buildingDef]
    if not state then
        state = { buildingDef = buildingDef }
        RC_RoomLogic._hugeScanStates[buildingDef] = state
    end
    state.buildingDef = buildingDef
    if roomDefs then state.roomDefs = roomDefs end
    opts = opts or {}

    local structural = opts.structural
    if structural == nil then structural = true end

    -- If we have no cached result yet, force an immediate structural rebuild.
    if not state.lastResult then
        structural = true
    end

    if structural then
        state.dirty = true
        state.pendingZ = _collectZLevels(roomDefs or state.roomDefs)
        state.pendingZSet = {}
        for _, z in ipairs(state.pendingZ or {}) do
            state.pendingZSet[z] = true
        end
        -- Ensure any cached perimeter data is discarded immediately so that
        -- follow-up scans (even for smaller buildings that don't use the HUGE
        -- coroutine path) see newly created gaps or filled breaches.
        if RC_RoomLogic._hugeCache and RC_RoomLogic.buildingKey then
            local zList = state.pendingZ
            if zList and #zList > 0 then
                for _, z in ipairs(zList) do
                    local key = RC_RoomLogic.buildingKey(buildingDef, z)
                    RC_RoomLogic._hugeCache[key] = nil
                end
            else
                -- Fallback: clear any cache entries that belong to this building
                -- when we do not yet know the active Z levels.
                local prefix = tostring(buildingDef) .. ":"
                for key,_ in pairs(RC_RoomLogic._hugeCache) do
                    if type(key) == "string" and key:sub(1, #prefix) == prefix then
                        RC_RoomLogic._hugeCache[key] = nil
                    end
                end
            end
        end
        if state.co then
            state.co = nil
        end
        state.revision = (state.revision or 0) + 1
        state.nextFullScanHour = nil
    else
        local nowHours = _currentWorldHours()
        state.lastLightRefreshHour = nowHours
        local intervalMin = RC_RoomLogic.HUGE_LIGHT_RESCAN_MINUTES or 0
        if intervalMin and intervalMin > 0 then
            local nextHour = nowHours + (intervalMin / 60.0)
            if not state.nextFullScanHour or nextHour < state.nextFullScanHour then
                state.nextFullScanHour = nextHour
            end
        end
    end
end

function RC_RoomLogic.getHugeScanRevision(buildingDef)
    if not buildingDef then return 0 end
    local state = RC_RoomLogic._hugeScanStates and RC_RoomLogic._hugeScanStates[buildingDef] or nil
    if state and state.revision then return state.revision end
    return 0
end

local function _createHugeScanCoroutine(state, seedZ, playerSquare)
    local buildingDef = state.buildingDef
    local roomDefs = state.roomDefs
    local zList = state.pendingZ or _collectZLevels(roomDefs)
    local defaultBudget = RC_RoomLogic.HUGE_SCAN_BUDGET_PER_CALL or 4000

    return coroutine.create(function(initialBudget)
        local budget = initialBudget or defaultBudget
        if not budget or budget <= 0 then budget = defaultBudget end

        local function consume(units)
            units = units or 1
            budget = budget - units
            if budget <= 0 then
                budget = coroutine.yield()
                if not budget or budget <= 0 then budget = defaultBudget end
            end
        end

        local result = {
            mode = "huge",
            graph = {},
            closed = {},
            roomArea = {},
            breaches = {},
            hugeInfo = {},
            cacheByZ = {},
        }
        local cacheByZ = result.cacheByZ
        local perimeterSet = {}

        for _, z in ipairs(zList) do
            consume(1)
            local key = RC_RoomLogic.buildingKey(buildingDef, z)
            RC_RoomLogic._hugeCache[key] = nil
            cacheByZ[z] = RC_RoomLogic.ensureHugeCache(buildingDef, roomDefs, z, { yieldFn = consume })
        end

        local function ensureCacheLocal(z)
            local cache = cacheByZ[z]
            if cache then return cache end
            consume(1)
            cache = RC_RoomLogic.ensureHugeCache(buildingDef, roomDefs, z, { yieldFn = consume })
            cacheByZ[z] = cache
            return cache
        end

        local function ensureNode(rd)
            if not rd then return end
            if not result.graph[rd] then result.graph[rd] = {} end
            if not result.closed[rd] then result.closed[rd] = {} end
        end

        local function addEdge(a, b)
            if not a or not b or a == b then return end
            ensureNode(a); ensureNode(b)
            result.graph[a][b] = true
            result.graph[b][a] = true
        end

        local function addClosed(a, b)
            if not a or not b or a == b then return end
            ensureNode(a); ensureNode(b)
            result.closed[a][b] = true
            result.closed[b][a] = true
        end

        local function doorBetween(rsq, dir, otherSq)
            if not rsq then return nil end
            if dir == "north" then
                local d = rsq.getIsoDoor and rsq:getIsoDoor() or nil
                if d and RC_RoomLogic.isDoorNorth(d) == true then return d end
            elseif dir == "west" then
                local d = rsq.getIsoDoor and rsq:getIsoDoor() or nil
                if d and RC_RoomLogic.isDoorNorth(d) == false then return d end
            elseif dir == "south" then
                local sq = otherSq
                local d = sq and sq.getIsoDoor and sq:getIsoDoor() or nil
                if d and RC_RoomLogic.isDoorNorth(d) == true then return d end
            elseif dir == "east" then
                local sq = otherSq
                local d = sq and sq.getIsoDoor and sq:getIsoDoor() or nil
                if d and RC_RoomLogic.isDoorNorth(d) == false then return d end
            end
            return nil
        end

        local function traverseFrom(startRoom)
            if not startRoom then return nil end
            local queue = { startRoom }
            local head = 1
            local depthMap = { [startRoom] = 0 }
            local visitedOrder = {}
            local visitedSet = { [startRoom] = true }
            local localPerimeter = {}
            local depthLimit = nil

            while head <= #queue do
                local cur = queue[head]; head = head + 1
                local depth = depthMap[cur] or 0
                if depthLimit and depth > depthLimit then break end
                visitedOrder[#visitedOrder+1] = cur

                local z = (cur.getZ and cur:getZ()) or seedZ or 0
                local cache = ensureCacheLocal(z)
                local exposures = RC_RoomLogic.ensureHugePerimeterForRoom(cache, cur, { yieldFn = consume }) or {}
                if exposures and #exposures > 0 then
                    localPerimeter[cur] = true
                    if not depthLimit then depthLimit = depth end
                end

                if not depthLimit or depth < depthLimit then
                    local passNeighbors = RC_RoomLogic.getPassableNeighbors(cache, cur, z) or {}
                    for _, nb in ipairs(passNeighbors) do
                        if not visitedSet[nb] then
                            visitedSet[nb] = true
                            depthMap[nb] = depth + 1
                            queue[#queue+1] = nb
                        end
                        consume(1)
                    end
                    local stairsNeighbors = RC_RoomLogic.getStairsNeighbors(cache, cur) or {}
                    for _, nb in ipairs(stairsNeighbors) do
                        if not visitedSet[nb] then
                            visitedSet[nb] = true
                            depthMap[nb] = depth + 1
                            queue[#queue+1] = nb
                        end
                        consume(1)
                    end
                end
                consume(1)
            end

            if #visitedOrder == 0 then return nil end

            local focusSet = {}
            for _, rd in ipairs(visitedOrder) do focusSet[rd] = true end

            return {
                list = visitedOrder,
                set = focusSet,
                perimeter = localPerimeter,
                depthLimit = depthLimit,
            }
        end

        local startRoom = nil
        if playerSquare then
            startRoom = RC_RoomLogic.roomDefFromSquareOrNeighbors(playerSquare)
        end

        local focusInfo = traverseFrom(startRoom)
        local focusRoomsList, focusRoomsSet
        if focusInfo and focusInfo.list and #focusInfo.list > 0 then
            focusRoomsList = focusInfo.list
            focusRoomsSet = focusInfo.set
            perimeterSet = focusInfo.perimeter or {}
        else
            focusRoomsList = {}
            focusRoomsSet = {}
            local roomCount = roomDefs and roomDefs:size() or 0
            for i = 0, roomCount - 1 do
                local rd = roomDefs:get(i)
                if rd and not focusRoomsSet[rd] then
                    focusRoomsSet[rd] = true
                    focusRoomsList[#focusRoomsList+1] = rd
                    consume(1)
                end
            end
        end

        result.hugeInfo.focusRooms = focusRoomsList
        result.hugeInfo.focusRoomsSet = focusRoomsSet
        result.hugeInfo.focusStartRoom = startRoom
        result.hugeInfo.focusDepthLimit = focusInfo and focusInfo.depthLimit or nil

        for _, rd in ipairs(focusRoomsList) do
            consume(1)
            local z = (rd.getZ and rd:getZ()) or seedZ or 0
            local cache = ensureCacheLocal(z)
            ensureNode(rd)

            local squares = RC_RoomLogic.getRoomSquares(cache, rd) or {}
            result.roomArea[rd] = #squares

            local exposures = RC_RoomLogic.ensureHugePerimeterForRoom(cache, rd, { yieldFn = consume }) or {}
            for _, rec in ipairs(exposures) do
                result.breaches[#result.breaches+1] = rec
                perimeterSet[rd] = true
                consume(1)
            end

            local passNeighbors = RC_RoomLogic.getPassableNeighbors(cache, rd, z) or {}
            for _, nb in ipairs(passNeighbors) do
                if focusRoomsSet[nb] then
                    addEdge(rd, nb)
                end
                consume(1)
            end
            local stairsNeighbors = RC_RoomLogic.getStairsNeighbors(cache, rd) or {}
            for _, nb in ipairs(stairsNeighbors) do
                if focusRoomsSet[nb] then
                    addEdge(rd, nb)
                end
                consume(1)
            end

            for _, rsq in ipairs(squares) do
                consume(1)
                if rsq then
                    local function consider(dir, getNeighbor)
                        local nbSq = getNeighbor and getNeighbor(rsq) or nil
                        if not nbSq or not nbSq.getZ or nbSq:getZ() ~= rsq:getZ() then return end
                        local nbIso = nbSq:getRoom()
                        if not nbIso then return end
                        local nbCache = ensureCacheLocal(nbSq:getZ())
                        local nbDef = (nbCache.byIso and nbCache.byIso[nbIso]) or (nbIso.getRoomDef and nbIso:getRoomDef()) or nil
                        if nbDef and nbDef ~= rd and focusRoomsSet[nbDef] then
                            if not RC_RoomLogic.edgeIsPassable(rsq, dir, nbSq) then
                                local door = doorBetween(rsq, dir, nbSq)
                                if door then addClosed(rd, nbDef) end
                            end
                        end
                    end
                    if rsq.getN then consider("north", function(s) return s:getN() end) end
                    if rsq.getW then consider("west",  function(s) return s:getW() end) end
                    if rsq.getS then consider("south", function(s) return s:getS() end) end
                    if rsq.getE then consider("east",  function(s) return s:getE() end) end
                end
            end
        end

        local perimeterList = {}
        for room,_ in pairs(perimeterSet) do
            if focusRoomsSet[room] then
                perimeterList[#perimeterList+1] = room
            end
            consume(1)
        end
        table.sort(perimeterList, function(a, b)
            local ar = RC_RoomLogic.roomName(a) or ""
            local br = RC_RoomLogic.roomName(b) or ""
            if ar ~= br then return ar < br end
            return tostring(a) < tostring(b)
        end)
        result.hugeInfo.perimeterRooms = perimeterList

        local distFromPerimeter, q, head = {}, {}, 1
        for room,_ in pairs(perimeterSet) do
            if focusRoomsSet[room] then
                distFromPerimeter[room] = 0
                q[#q+1] = room
                consume(1)
            end
        end
        while head <= #q do
            local cur = q[head]; head = head + 1
            local cd = distFromPerimeter[cur] or 0
            for nb,_ in pairs(result.graph[cur] or {}) do
                if focusRoomsSet[nb] and distFromPerimeter[nb] == nil then
                    distFromPerimeter[nb] = cd + 1
                    q[#q+1] = nb
                end
                consume(1)
            end
            consume(1)
        end
        result.hugeInfo.distanceFromPerimeter = distFromPerimeter

        return result
    end)
end
local function _entryIsOpen(entry)
    if not entry then return false end
    if entry.type == "door" then
        return RC_RoomLogic.isDoorOpen(entry.door)
    elseif entry.type == "window" then
        local win = entry.window
        local open = false
        local smashed = false
        if win then
            open = RC_RoomLogic.isWindowOpen(win) or false
            smashed = RC_RoomLogic.isWindowSmashed(win) or false
            if open or smashed then return true end
        end
        if entry.open ~= nil then open = open or entry.open end
        if entry.smashed ~= nil then smashed = smashed or entry.smashed end
        return open or smashed
    elseif entry.type == "gap" then
        return true
    end
    return false
end

local function _updateHugeResult(result, buildingDef, roomDefs, seedZ, playerSquare)
    if not result then return end
    result.hugeInfo = result.hugeInfo or {}
    local cacheByZ = result.cacheByZ or {}
    local function ensureCache(z)
        local cache = cacheByZ[z]
        if cache then return cache end
        cache = RC_RoomLogic.ensureHugeCache(buildingDef, roomDefs, z)
        cacheByZ[z] = cache
        result.cacheByZ = cacheByZ
        return cache
    end

    local function resolveRoomFromSquare(sq)
        if not sq then return nil end
        local iso = sq:getRoom()
        if not iso then return nil end
        local z = sq:getZ() or seedZ or 0
        local cache = ensureCache(z)
        if cache and cache.byIso and cache.byIso[iso] then
            return cache.byIso[iso]
        end
        if iso.getRoomDef then
            local ok, rd = pcall(function() return iso:getRoomDef() end)
            if ok and rd then return rd end
        end
        return nil
    end

    local startRoom = resolveRoomFromSquare(playerSquare)
    result.hugeInfo.startRoom = startRoom

    local targetRoomBest = {}
    local px, py, pz = nil, nil, nil
    if playerSquare then
        px, py = playerSquare:getX(), playerSquare:getY()
        pz = playerSquare:getZ()
    end

    if px and py then
        local function considerEntry(entry)
            if not entry or not entry.room then return end
            if pz ~= nil and entry.z ~= pz then return end
            if not _entryIsOpen(entry) then return end
            local dx, dy = (entry.x or px) - px, (entry.y or py) - py
            local d2 = dx * dx + dy * dy
            local cur = targetRoomBest[entry.room]
            if (not cur) or (d2 < cur.dist2) then
                targetRoomBest[entry.room] = { dist2 = d2, entry = entry }
            end
        end

        for _, entry in ipairs(result.breaches or {}) do
            considerEntry(entry)
        end
    end

    result.hugeInfo.openTargets = targetRoomBest

    local nearestPath, nearestEntry = nil, nil
    if startRoom then
        local visited, prev = { [startRoom] = true }, {}
        local queue, idx = { startRoom }, 1
        local found = nil
        while idx <= #queue and not found do
            local cur = queue[idx]; idx = idx + 1
            if targetRoomBest[cur] then
                found = cur
                break
            end
            for nb,_ in pairs(result.graph[cur] or {}) do
                if not visited[nb] then
                    visited[nb] = true
                    prev[nb] = cur
                    queue[#queue+1] = nb
                end
            end
        end
        if found then
            nearestEntry = targetRoomBest[found] and targetRoomBest[found].entry or nil
            nearestPath = {}
            local cur = found
            while cur do
                table.insert(nearestPath, 1, cur)
                cur = prev[cur]
            end
        end
    end

    result.hugeInfo.nearestExitPath = nearestPath
    result.hugeInfo.nearestExitEntry = nearestEntry
end

---------------------------------------------------------------------------
-- MAIN KEY HANDLER (now lean — very few locals)
---------------------------------------------------------------------------
---
function RC_RoomLogic.RunNonHugeScan(buildingDef, roomDefs, zLevel, playerSquare)
    -- print("Running scan!")
    -- maps & containers
    local byIso, byRoom, rooms = {}, {}, {}

    local function ensureEntry(rd)
        if not rd or byRoom[rd] then return byRoom[rd] end
        local iso = rd.getIsoRoom and rd:getIsoRoom() or nil
        local squares = {}
        if iso and iso.getSquares then
            local sqList = iso:getSquares()
            for j = 0, sqList:size() - 1 do squares[#squares+1] = sqList:get(j) end
        end
        local entry = { room = rd, iso = iso, squares = squares }
        rooms[#rooms+1] = entry
        byRoom[rd] = entry
        if iso then byIso[iso] = rd end
        return entry
    end

    -- seed with current floor rooms
    for i = 0, roomDefs:size() - 1 do
        local rd = roomDefs:get(i)
        if rd and zLevel == rd:getZ() then ensureEntry(rd) end
    end

    local graph = {}
    local function addEdge(a,b)
        if not a or not b then return end
        graph[a] = graph[a] or {}
        graph[b] = graph[b] or {}
        graph[a][b] = true; graph[b][a] = true
    end

    local outsideDoors, seenDoor = {}, {}
    local outsideWindows, seenWindow = {}, {}
    local outsideGaps = {}

    local function recordExteriorDoor(roomDef, door, doorSq, edgeLabel)
        if not door or not doorSq then return end
        local keyObj = tostring(door)
        if seenDoor[keyObj] then return end
        seenDoor[keyObj] = true
        outsideDoors[#outsideDoors+1] = {
            room = roomDef, x = doorSq:getX(), y = doorSq:getY(), z = doorSq:getZ(),
            edge = edgeLabel, open = RC_RoomLogic.isDoorOpen(door), door = door
        }
    end
    local function recordExteriorWindow(roomDef, win, winSq, edgeLabel)
        if not win or not winSq then return end
        local keyObj = tostring(win)
        if seenWindow[keyObj] then return end
        seenWindow[keyObj] = true
        outsideWindows[#outsideWindows+1] = {
            room = roomDef, x = winSq:getX(), y = winSq:getY(), z = winSq:getZ(),
            edge = edgeLabel, open = RC_RoomLogic.isWindowOpen(win),
            smashed = RC_RoomLogic.isWindowSmashed(win), window = win
        }
    end
    local function recordExteriorGap(roomDef, sqOnEdge, edgeLabel)
        if not sqOnEdge then return end
        outsideGaps[#outsideGaps+1] = {
            room = roomDef, x = sqOnEdge:getX(), y = sqOnEdge:getY(), z = sqOnEdge:getZ(),
            edge = edgeLabel
        }
    end

    -- **** REUSED SCRATCH VARS (avoid >200 locals) ****
    local rsq, nSq, wSq, sSq, eSq
    local nIso, wIso, sIso, eIso
    local nDef, wDef, sDef, eDef
    local dN, dW, dS, dE, wN, wW, wS, wE

    ------------------------------------------------------------------------
    -- TOP-FLOOR BRIDGE: look straight down from the player, find a room,
    -- then find stairs on that lower floor and link bottom/top rooms.
    ------------------------------------------------------------------------
    local function tryBridgeFromBelow()
        if not playerSquare then return end
        local topZ = playerSquare:getZ()
        if not topZ or topZ <= 0 then return end
        local cell = getCell and getCell() or nil
        if not cell then return end
        local px, py = playerSquare:getX(), playerSquare:getY()

        local bz = topZ - 1
        local belowSq, belowIso, belowDef
        while bz >= 0 do
            belowSq = cell:getGridSquare(px, py, bz)
            if not belowSq then break end
            belowIso = belowSq:getRoom()
            if belowIso then
                belowDef = byIso[belowIso] or (belowIso.getRoomDef and belowIso:getRoomDef()) or nil
                if not belowDef then return end
                local entryB = ensureEntry(belowDef)

                -- First, try to find stairs *inside that room*.
                local found = false
                for _, s in ipairs(entryB.squares) do
                    local bL, tL = RC_RoomLogic.stairsEndpointsFromAnyStair(s)
                    if bL or tL then
                        local bd = RC_RoomLogic.roomDefFromSquareOrNeighbors(bL) or belowDef
                        local td = RC_RoomLogic.roomDefFromSquareOrNeighbors(tL)
                        if td then
                            if not byRoom[bd] then ensureEntry(bd) end
                            if not byRoom[td] then ensureEntry(td) end
                            addEdge(bd, td)
                            found = true
                            break
                        end
                    end
                end
                if found then return end

                -- Fallback: small radius search on the same lower Z for any stair tile.
                local r = 6
                local x, y, z = px, py, bz
                for dy = -r, r do
                    if found then break end
                    for dx = -r, r do
                        rsq = cell:getGridSquare(x+dx, y+dy, z)
                        if rsq then
                            local bL, tL = RC_RoomLogic.stairsEndpointsFromAnyStair(rsq)
                            if bL or tL then
                                local bd = RC_RoomLogic.roomDefFromSquareOrNeighbors(bL)
                                local td = RC_RoomLogic.roomDefFromSquareOrNeighbors(tL)
                                if bd and td then
                                    if not byRoom[bd] then ensureEntry(bd) end
                                    if not byRoom[td] then ensureEntry(td) end
                                    addEdge(bd, td)
                                    found = true
                                    break
                                end
                            end
                        end
                    end
                end
                return
            end
            bz = bz - 1
        end
    end

    -- Lateral connectivity + exterior detection (N/W/S/E)
    for _, entry in ipairs(rooms) do
        local rd = entry.room
        graph[rd] = graph[rd] or {}
        for _, rsq in ipairs(entry.squares) do
            if rsq then
                -- NORTH neighbor
                if rsq.getN then
                    nSq = rsq:getN()
                    if nSq and nSq:getZ() == zLevel then
                        nIso = nSq:getRoom()
                        nDef = nIso and (byIso[nIso] or (nIso.getRoomDef and nIso:getRoomDef())) or nil
                        if nDef and nDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "north", nSq) then
                            addEdge(rd, nDef)
                        end
                        if not nIso then
                            dN = rsq.getIsoDoor and rsq:getIsoDoor() or nil
                            if dN and RC_RoomLogic.isDoorNorth(dN) == true then
                                recordExteriorDoor(rd, dN, rsq, "north")
                            end
                            wN = RC_RoomLogic.getWindowOnEdge(rsq, "north")
                            if wN then
                                recordExteriorWindow(rd, wN, rsq, "north")
                            elseif (not dN)
                               and (not RC_RoomLogic.edgeHasBlockingOnSquare(rsq,"north"))
                               and (not RC_RoomLogic.edgeHasBlockingOnSquare(nSq,"south")) then
                                recordExteriorGap(rd, rsq, "north")
                            end
                        end
                    end
                end

                -- WEST neighbor
                if rsq.getW then
                    wSq = rsq:getW()
                    if wSq and wSq:getZ() == zLevel then
                        wIso = wSq:getRoom()
                        wDef = wIso and (byIso[wIso] or (wIso.getRoomDef and wIso:getRoomDef())) or nil
                        if wDef and wDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "west", wSq) then
                            addEdge(rd, wDef)
                        end
                        if not wIso then
                            dW = rsq.getIsoDoor and rsq:getIsoDoor() or nil
                            if dW and RC_RoomLogic.isDoorNorth(dW) == false then
                                recordExteriorDoor(rd, dW, rsq, "west")
                            end
                            wW = RC_RoomLogic.getWindowOnEdge(rsq, "west")
                            if wW then
                                recordExteriorWindow(rd, wW, rsq, "west")
                            elseif (not dW)
                               and (not RC_RoomLogic.edgeHasBlockingOnSquare(rsq,"west"))
                               and (not RC_RoomLogic.edgeHasBlockingOnSquare(wSq,"east")) then
                                recordExteriorGap(rd, rsq, "west")
                            end
                        end
                    end
                end

                -- SOUTH neighbor (objects live on SOUTH square)
                if rsq.getS then
                    sSq = rsq:getS()
                    if sSq and sSq:getZ() == zLevel then
                        sIso = sSq:getRoom()
                        sDef = sIso and (byIso[sIso] or (sIso.getRoomDef and sIso:getRoomDef())) or nil
                        if sDef and sDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "south", sSq) then
                            addEdge(rd, sDef)
                        end
                        if not sIso then
                            dS = sSq.getIsoDoor and sSq:getIsoDoor() or nil
                            if dS and RC_RoomLogic.isDoorNorth(dS) == true then
                                recordExteriorDoor(rd, dS, sSq, "south")
                            end
                            wS = RC_RoomLogic.getWindowOnEdge(sSq, "north")
                            if wS then
                                recordExteriorWindow(rd, wS, sSq, "south")
                            elseif (not dS)
                               and (not RC_RoomLogic.edgeHasBlockingOnSquare(rsq,"south"))
                               and (not RC_RoomLogic.edgeHasBlockingOnSquare(sSq,"north")) then
                                recordExteriorGap(rd, rsq, "south")
                            end
                        end
                    end
                end

                -- EAST neighbor (objects live on EAST square)
                if rsq.getE then
                    eSq = rsq:getE()
                    if eSq and eSq:getZ() == zLevel then
                        eIso = eSq:getRoom()
                        eDef = eIso and (byIso[eIso] or (eIso.getRoomDef and eIso:getRoomDef())) or nil
                        if eDef and eDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "east", eSq) then
                            addEdge(rd, eDef)
                        end
                        if not eIso then
                            dE = eSq.getIsoDoor and eSq:getIsoDoor() or nil
                            if dE and RC_RoomLogic.isDoorNorth(dE) == false then
                                recordExteriorDoor(rd, dE, eSq, "east")
                            end
                            wE = RC_RoomLogic.getWindowOnEdge(eSq, "west")
                            if wE then
                                recordExteriorWindow(rd, wE, eSq, "east")
                            elseif (not dE)
                               and (not RC_RoomLogic.edgeHasBlockingOnSquare(rsq,"east"))
                               and (not RC_RoomLogic.edgeHasBlockingOnSquare(eSq,"west")) then
                                recordExteriorGap(rd, rsq, "east")
                            end
                        end
                    end
                end
            end
        end
    end

    -- Bridge from below BEFORE we do the normal stairs sweep (covers top floor)
    tryBridgeFromBelow()

    -- STAIRS connectivity (cross-floor)
    local initialCount = #rooms
    local addedSomething = true
    local bottomSq, topSq, bottomDef, topDef
    while addedSomething do
        addedSomething = false
        for _, entry in ipairs(rooms) do
            local rd = entry.room
            for _, rsq in ipairs(entry.squares) do
                bottomSq, topSq = RC_RoomLogic.stairsEndpointsFromAnyStair(rsq)
                if bottomSq or topSq then
                    bottomDef = RC_RoomLogic.roomDefFromSquareOrNeighbors(bottomSq) or rd
                    topDef    = RC_RoomLogic.roomDefFromSquareOrNeighbors(topSq)
                    if topDef then
                        if not byRoom[topDef]    then ensureEntry(topDef);    addedSomething = true end
                        if not byRoom[bottomDef] then ensureEntry(bottomDef); addedSomething = true end
                        addEdge(bottomDef, topDef)
                    end
                end
            end
        end
    end

    -- Lateral edges for any new rooms (other floors)
    for i = initialCount + 1, #rooms do
        local entry = rooms[i]
        local rd = entry.room
        local rdZ = (rd and rd.getZ) and rd:getZ() or zLevel

        for _, rsq in ipairs(entry.squares) do
            if rsq then
                if rsq.getN then
                    nSq = rsq:getN()
                    if nSq and nSq:getZ() == rdZ then
                        nIso = nSq:getRoom()
                        nDef = nIso and (byIso[nIso] or (nIso.getRoomDef and nIso:getRoomDef())) or nil
                        if nDef and nDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "north", nSq) then
                            addEdge(rd, nDef)
                        end
                    end
                end
                if rsq.getW then
                    wSq = rsq:getW()
                    if wSq and wSq:getZ() == rdZ then
                        wIso = wSq:getRoom()
                        wDef = wIso and (byIso[wIso] or (wIso.getRoomDef and wIso:getRoomDef())) or nil
                        if wDef and wDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "west", wSq) then
                            addEdge(rd, wDef)
                        end
                    end
                end
                if rsq.getS then
                    sSq = rsq:getS()
                    if sSq and sSq:getZ() == rdZ then
                        sIso = sSq:getRoom()
                        sDef = sIso and (byIso[sIso] or (sIso.getRoomDef and sIso:getRoomDef())) or nil
                        if sDef and sDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "south", sSq) then
                            addEdge(rd, sDef)
                        end
                    end
                end
                if rsq.getE then
                    eSq = rsq:getE()
                    if eSq and eSq:getZ() == rdZ then
                        eIso = eSq:getRoom()
                        eDef = eIso and (byIso[eIso] or (eIso.getRoomDef and eIso:getRoomDef())) or nil
                        if eDef and eDef ~= rd and RC_RoomLogic.edgeIsPassable(rsq, "east", eSq) then
                            addEdge(rd, eDef)
                        end
                    end
                end
            end
        end
    end

    -- Connected components (may span floors)
    local visited, groups = {}, {}
    local function roomLess(a, b) return RC_RoomLogic.roomName(a) < RC_RoomLogic.roomName(b) end
    for _, entry in ipairs(rooms) do
        local r = entry.room
        if not visited[r] then
            local stack, comp = { r }, {}
            visited[r] = true
            while #stack > 0 do
                local cur = table.remove(stack)
                comp[#comp+1] = cur
                for neighbor,_ in pairs(graph[cur] or {}) do
                    if not visited[neighbor] then
                        visited[neighbor] = true; stack[#stack+1] = neighbor
                    end
                end
            end
            table.sort(comp, roomLess)
            groups[#groups+1] = comp
        end
    end

    -- print("--------------------------------------")
    for gi, comp in ipairs(groups) do
        local names = {}
        for _, r in ipairs(comp) do names[#names+1] = RC_RoomLogic.roomName(r) end
        -- print(string.format("Connected rooms %d : %s", gi, table.concat(names, ", ")))
    end

    RC_RoomLogic.OutsideDoors = outsideDoors
    table.sort(RC_RoomLogic.OutsideDoors, function(a,b)
        local ar, br = RC_RoomLogic.roomName(a.room) or "", RC_RoomLogic.roomName(b.room) or ""
        if ar ~= br then return ar < br end
        if a.y ~= b.y then return a.y < b.y end
        if a.x ~= b.x then return a.x < b.x end
        if a.z ~= b.z then return a.z < b.z end
        return (a.edge or "") < (b.edge or "")
    end)
    -- print(string.format("Exterior doors: %d", #RC_RoomLogic.OutsideDoors))
    for _, d in ipairs(RC_RoomLogic.OutsideDoors) do
        -- if d.open then print("Room: " .. RC_RoomLogic.roomName(d.room) .. ", door open") end
    end

    RC_RoomLogic.OutsideWindows = outsideWindows
    table.sort(RC_RoomLogic.OutsideWindows, function(a,b)
        local ar, br = RC_RoomLogic.roomName(a.room) or "", RC_RoomLogic.roomName(b.room) or ""
        if ar ~= br then return ar < br end
        if a.y ~= b.y then return a.y < b.y end
        if a.x ~= b.x then return a.x < b.x end
        if a.z ~= b.z then return a.z < b.z end
        return (a.edge or "") < (b.edge or "")
    end)
    -- print(string.format("Exterior windows: %d", #RC_RoomLogic.OutsideWindows))
    -- for _, w in ipairs(RC_RoomLogic.OutsideWindows) do
    --     if w.open or w.smashed then print("Room: " .. RC_RoomLogic.roomName(w.room) .. ", window open or smashed") end
    -- end

    RC_RoomLogic.OutsideCurtains = cache and cache.outsideCurtains or {}
    table.sort(RC_RoomLogic.OutsideCurtains, function(a,b)
        local ar, br = RC_RoomLogic.roomName(a.room) or "", RC_RoomLogic.roomName(b.room) or ""
        if ar ~= br then return ar < br end
        if a.y ~= b.y then return a.y < b.y end
        if a.x ~= b.x then return a.x < b.x end
        if a.z ~= b.z then return a.z < b.z end
        return (a.edge or "") < (b.edge or "")
    end)
    -- print(string.format("Exterior curtains: %d", #RC_RoomLogic.OutsideCurtains))
    -- for _, c in ipairs(RC_RoomLogic.OutsideCurtains) do
    --     if c.open then print("Room: " .. RC_RoomLogic.roomName(c.room) .. ", curtain open") end
    -- end

    RC_RoomLogic.OutsideGaps = outsideGaps
    table.sort(RC_RoomLogic.OutsideGaps, function(a,b)
        local ar, br = RC_RoomLogic.roomName(a.room) or "", RC_RoomLogic.roomName(b.room) or ""
        if ar ~= br then return ar < br end
        if a.y ~= b.y then return a.y < b.y end
        if a.x ~= b.x then return a.x < b.x end
        if a.z ~= b.z then return a.z < b.z end
        return (a.edge or "") < (b.edge or "")
    end)
    -- print(string.format("Missing walls (gaps): %d", #RC_RoomLogic.OutsideGaps))
    for _, g in ipairs(RC_RoomLogic.OutsideGaps) do
        -- print(string.format("Room: %s, missing wall at %d,%d,%d (%s)",
        --     RC_RoomLogic.roomName(g.room), g.x, g.y, g.z, g.edge))
    end
    -- print("--------------------------------------")
end

function RC_RoomLogic.scanHugeBuilding(buildingDef, roomDefs, seedZ, playerSquare)
    if not buildingDef or not roomDefs then return nil end
    if not RC_RoomLogic.isHugeBuilding(roomDefs) then return nil end

    local state = RC_RoomLogic._hugeScanStates[buildingDef]
    if state and state.scanComplete then
        state.buildingDef = buildingDef
        state.roomDefs = roomDefs
        if playerSquare then
            state.focusPlayerSquare = playerSquare
        end
        local completedResult = state.lastResult
        if completedResult then
            _updateHugeResult(completedResult, buildingDef, roomDefs, seedZ, playerSquare)
        end
        return completedResult
    end

    if not state then
        state = { buildingDef = buildingDef }
        RC_RoomLogic._hugeScanStates[buildingDef] = state
    end
    state.buildingDef = buildingDef
    state.roomDefs = roomDefs

    if playerSquare then
        state.focusPlayerSquare = playerSquare
    end

    local focusRoom = RC_RoomLogic.roomDefFromSquareOrNeighbors and RC_RoomLogic.roomDefFromSquareOrNeighbors(playerSquare) or nil

    local nowHours = _currentWorldHours()
    local idleInterval = RC_RoomLogic.HUGE_IDLE_RESCAN_MINUTES or 0
    if idleInterval and idleInterval > 0 and state.lastResult and not state.dirty then
        if state.nextFullScanHour and nowHours >= state.nextFullScanHour then
            state.dirty = true
            state.pendingZ = _collectZLevels(roomDefs)
            state.pendingZSet = {}
            for _, z in ipairs(state.pendingZ or {}) do
                state.pendingZSet[z] = true
            end
            state.nextFullScanHour = nil
        elseif not state.nextFullScanHour and state.lastFullScanHour then
            state.nextFullScanHour = state.lastFullScanHour + (idleInterval / 60.0)
        end
    end

    if state.lastResult and not state.dirty and focusRoom then
        local focusSet = state.lastResult.hugeInfo and state.lastResult.hugeInfo.focusRoomsSet or nil
        local inGraph = state.lastResult.graph and state.lastResult.graph[focusRoom] or nil
        local covered = (focusSet and focusSet[focusRoom]) or inGraph
        if not covered then
            state.dirty = true
            state.pendingZ = _collectZLevels(roomDefs)
            state.pendingZSet = {}
            for _, z in ipairs(state.pendingZ or {}) do
                state.pendingZSet[z] = true
            end
            state.co = nil
            state.lastResult = nil
        end
    end

    if not state.lastResult then
        state.dirty = true
    end

    if state.dirty and not state.pendingZ then
        state.pendingZ = _collectZLevels(roomDefs)
        state.pendingZSet = {}
        for _, z in ipairs(state.pendingZ or {}) do
            state.pendingZSet[z] = true
        end
    end

    local defaultBudget = RC_RoomLogic.HUGE_SCAN_BUDGET_PER_CALL or 4000
    if not defaultBudget or defaultBudget <= 0 then defaultBudget = 4000 end

    if state.dirty and not state.co then
        local focusSq = state.focusPlayerSquare or playerSquare
        state.co = _createHugeScanCoroutine(state, seedZ, focusSq)
    end

    local function resumeWithBudget(budget)
        if not state.co then return end
        local ok, resultOrNeed = coroutine.resume(state.co, budget)
        if not ok then
            if getDebug and getDebug() then
                -- print(string.format("RC_RoomLogic.scanHugeBuilding coroutine error: %s", tostring(resultOrNeed)))
            end
            state.co = nil
            state.dirty = false
            return
        end
        if coroutine.status(state.co) == "dead" then
            state.lastResult = resultOrNeed
            state.co = nil
            state.dirty = false
            state.pendingZ = nil
            state.pendingZSet = nil
            state.lastFullScanHour = nowHours
            state.scanComplete = true
            local interval = RC_RoomLogic.HUGE_IDLE_RESCAN_MINUTES or 0
            if interval and interval > 0 then
                state.nextFullScanHour = nowHours + (interval / 60.0)
            else
                state.nextFullScanHour = nil
            end
        end
    end

    if state.co then
        if not state.lastResult then
            local bigBudget = defaultBudget * 100
            if bigBudget <= 0 then bigBudget = 400000 end
            local attempts = 0
            while state.co and coroutine.status(state.co) ~= "dead" and attempts < 64 do
                resumeWithBudget(bigBudget)
                attempts = attempts + 1
            end
        else
            resumeWithBudget(defaultBudget)
        end
    end

    local result = state.lastResult
    if result then
        _updateHugeResult(result, buildingDef, roomDefs, seedZ, playerSquare)
    end
    return result
end
