-- RC_PlayerRoomLogic.lua
-- Room/Perimeter logic for *player-made* structures (no BuildingDef).
-- Mirrors RC_RoomLogic API style but operates on IsoWorldRegion with isPlayerRoom().

RC_PlayerRoomLogic = RC_PlayerRoomLogic or {}

-- small cache of region-clusters; key = tostring(seedRegion) .. ":" .. z
RC_PlayerRoomLogic._cluster = RC_PlayerRoomLogic._cluster or {}

RC_PlayerRoomLogic.DEBUG_DOORS = RC_PlayerRoomLogic.DEBUG_DOORS or false

local function _dbgDoor(fmt, ...)
    if RC_PlayerRoomLogic.DEBUG_DOORS then
        -- print("[PRL][DOOR] " .. string.format(fmt, ...))
    end
end

local function _pairKey(a, b)
    if not a or not b then return nil end
    local sa, sb = tostring(a), tostring(b)
    if sa < sb then return sa .. "|" .. sb else return sb .. "|" .. sa end
end

function RC_PlayerRoomLogic.ClearCache()
    RC_PlayerRoomLogic._cluster = {}
    RC_PlayerRoomLogic._buildingByRegion = {}
    RC_PlayerRoomLogic._allPlayerBuildings = {}
end


-- ========== Shared helpers (mostly copied / adapted from RC_RoomLogic) ==========

local function regionFromSquare(sq)
    if not sq or not sq.getIsoWorldRegion then return nil end
    local ok, reg = pcall(function() return sq:getIsoWorldRegion() end)
    if not ok or not reg then return nil end
    if reg.isPlayerRoom then
        local ok2, isPR = pcall(function() return reg:isPlayerRoom() end)
        if ok2 and isPR then return reg end
    end
    return nil
end

function RC_PlayerRoomLogic.roomName(roomLike)
    if not roomLike then return "<nil>" end
    -- Regions do not have names; synthesize something stable-ish.
    -- Try to show Z and a short id hash
    local z = 0
    if roomLike.getZ then
        local okZ, vz = pcall(function() return roomLike:getZ() end)
        if okZ and type(vz)=="number" then z = vz end
    end
    return string.format("P-Region(z=%d)@%s", z, tostring(roomLike))
end

-- Doors/windows helpers (identical style to RC_RoomLogic)
function RC_PlayerRoomLogic.isDoorNorth(door)
    if not door then return nil end
    if door.isNorth then local ok,v=pcall(function() return door:isNorth() end); if ok then return v end end
    if door.getNorth then local ok,v=pcall(function() return door:getNorth() end); if ok then return v end end
    return nil
end

local function doorOrientationFromSprite(obj)
    if not obj or not obj.getSprite then return nil end
    local spr = obj:getSprite()
    if not spr then return nil end
    local props = spr.getProperties and spr:getProperties() or nil

    if props and props.Val then
        local t = props:Val("Type") or props:Val("type") or props:Val("SpriteType") or props:Val("sprite.type")
        if t == "doorN" or t == "DoorN" then return "north" end
        if t == "doorW" or t == "DoorW" then return "west" end
    end

    if props and props.Is then
        if props:Is(IsoFlagType.DoorWallN) or props:Is(IsoFlagType.doorN) or props:Is(IsoFlagType.HingeN) then
            return "north"
        end
        if props:Is(IsoFlagType.DoorWallW) or props:Is(IsoFlagType.doorW) or props:Is(IsoFlagType.HingeW) then
            return "west"
        end
    end

    if obj.isNorth then
        local ok, n = pcall(function() return obj:isNorth() end)
        if ok then return n and "north" or "west" end
    end

    local sprName = spr.getName and spr:getName() or ""
    sprName = string.lower(tostring(sprName))
    if sprName:find("door[_-]?n") then return "north" end
    if sprName:find("door[_-]?w") then return "west" end

    return nil
end

local function _northFlagFromOrientation(orient)
    if orient == nil then return nil end
    if type(orient) == "boolean" then return orient end
    if type(orient) == "string" then
        if orient == "north" then return true end
        if orient == "west" then return false end
    end
    return nil
end

local function _doorNorthFlag(obj)
    local direct = _northFlagFromOrientation(RC_PlayerRoomLogic.isDoorNorth(obj))
    if direct ~= nil then return direct end
    return _northFlagFromOrientation(doorOrientationFromSprite(obj))
end

local function _squaresForEdgeObject(obj, orientFn)
    if not obj or not orientFn then return nil, nil, nil end
    local sq = obj.getSquare and obj:getSquare() or nil
    if not sq then return nil, nil, nil end
    local northFlag = orientFn(obj)
    if northFlag == nil then return nil, nil, nil end
    local dir = northFlag and "north" or "west"
    local neighbor = nil
    if dir == "north" then
        neighbor = sq.getN and sq:getN() or nil
    else
        neighbor = sq.getW and sq:getW() or nil
    end
    if neighbor and neighbor.getZ and sq.getZ and neighbor:getZ() ~= sq:getZ() then
        neighbor = nil
    end
    return sq, neighbor, dir
end

local function _squaresAreUnblocked(a, b)
    if not (a and b) then return nil end
    if a.isBlockedTo then
        local ok, blocked = pcall(function() return a:isBlockedTo(b) end)
        if ok and blocked ~= nil then
            return not blocked
        end
    end
    return nil
end

function RC_PlayerRoomLogic.isDoorOpen(door)
    if not door then return false end

    local methods = { "isOpen", "IsOpen", "isDoorOpen" }
    for _, m in ipairs(methods) do
        local fn = door[m]
        if fn then
            local ok, v = pcall(fn, door)
            if ok and v ~= nil then
                return v and true or false
            end
        end
    end

    local sq, neighbor = _squaresForEdgeObject(door, _doorNorthFlag)
    local passable = _squaresAreUnblocked(sq, neighbor)
    if passable ~= nil then
        return passable and true or false
    end

    return false
end

function RC_PlayerRoomLogic.isWindowNorth(win)
    if not win then return nil end
    if win.isNorth then local ok,v=pcall(function() return win:isNorth() end); if ok then return v end end
    if win.getNorth then local ok,v=pcall(function() return win:getNorth() end); if ok then return v end end
    return nil
end

local function windowOrientationFromSprite(obj)
    if not obj or not obj.getSprite then return nil end
    local spr = obj:getSprite()
    if not spr then return nil end
    local props = spr.getProperties and spr:getProperties() or nil

    if props and props.Val then
        local t = props:Val("Type") or props:Val("type") or props:Val("SpriteType") or props:Val("sprite.type")
        if t == "windowN" or t == "WindowN" then return "north" end
        if t == "windowW" or t == "WindowW" then return "west" end
    end

    if props and props.Is then
        if props:Is(IsoFlagType.WindowN) or props:Is(IsoFlagType.windowN) then return "north" end
        if props:Is(IsoFlagType.WindowW) or props:Is(IsoFlagType.windowW) then return "west" end
    end

    local sprName = spr.getName and spr:getName() or ""
    sprName = string.lower(tostring(sprName))
    if sprName:find("window[_-]?n") then return "north" end
    if sprName:find("window[_-]?w") then return "west" end

    return nil
end

local function _windowNorthFlag(obj)
    local direct = _northFlagFromOrientation(RC_PlayerRoomLogic.isWindowNorth(obj))
    if direct ~= nil then return direct end
    return _northFlagFromOrientation(windowOrientationFromSprite(obj))
end

function RC_PlayerRoomLogic.isWindowOpen(win)
    if not win then return false end
    if win.isOpen then
        local ok, v = pcall(function() return win:isOpen() end)
        if ok and v ~= nil then return v and true or false end
    end
    if win.IsOpen then
        local ok, v = pcall(function() return win:IsOpen() end)
        if ok and v ~= nil then return v and true or false end
    end

    local sq, neighbor = _squaresForEdgeObject(win, _windowNorthFlag)
    local passable = _squaresAreUnblocked(sq, neighbor)
    if passable ~= nil then
        return passable and true or false
    end

    return false
end

function RC_PlayerRoomLogic.isWindowSmashed(win)
    if not win then return false end
    if win.isSmashed then local ok,v=pcall(function() return win:isSmashed() end); if ok then return v and true or false end end
    return false
end

local function _booleanFromField(obj, field)
    if not obj or not field then return nil end
    local ok, value = pcall(function() return obj[field] end)
    if not ok or value == nil or type(value) == "function" then return nil end
    return value and true or false
end

local function isCurtainObject(obj)
    if not obj then return false end
    if instanceof and instanceof(obj, "IsoCurtain") then return true end
    if obj.getObjectName then
        local ok, name = pcall(function() return obj:getObjectName() end)
        if ok and name == "Curtain" then return true end
    end
    return false
end

function RC_PlayerRoomLogic.isCurtainOpen(curtain)
    if not isCurtainObject(curtain) then return false end
    if curtain.IsOpen then
        local ok, open = pcall(function() return curtain:IsOpen() end)
        if ok and open ~= nil then
            return open and true or false
        end
    end
    local state = _booleanFromField(curtain, "open")
    if state ~= nil then
        return state and true or false
    end
    return false
end

local function _appendList(dst, zlist)
    if not zlist then return end
    for i = 0, zlist:size() - 1 do
        dst[#dst+1] = zlist:get(i)
    end
end

-- Collect *everything* that might be a door on this square:
-- - regular objects
-- - special objects (IsoDoor / IsoWindow frequently live here)
-- - direct getters (getIsoDoor / getDoorFrame)
local function getAllSquareObjects(sq)
    local out = {}
    if not sq then return out end
    if sq.getObjects         then _appendList(out, sq:getObjects()) end
    if sq.getSpecialObjects  then _appendList(out, sq:getSpecialObjects()) end
    if sq.getIsoDoor         then local ok,d = pcall(function() return sq:getIsoDoor() end); if ok and d then out[#out+1]=d end end
    -- DO NOT add getDoorFrame() here: it is usually a WoodDoorFrame and has no open API.
    return out
end

function RC_PlayerRoomLogic.getWindowOnEdge(sq0, dir) -- "north" / "west"
    local objs = getAllSquareObjects(sq0)
    for _, obj in ipairs(objs) do
        local isWindow = instanceof and instanceof(obj, "IsoWindow")
        if not isWindow and instanceof and instanceof(obj, "IsoThumpable") and obj.isWindow then
            local ok, res = pcall(function() return obj:isWindow() end)
            isWindow = ok and res and true or false
        end
        if isWindow then
            local north = _windowNorthFlag(obj)
            if (dir == "north" and north == true) or (dir == "west" and north == false) then
                return obj
            end
        end
    end
    return nil
end

local function _hasOpenAPI(obj)
    return obj and (obj.isOpen or obj.IsOpen or obj.isDoorOpen) and true or false
end

function RC_PlayerRoomLogic.getCurtainOnEdge(sq0, dir) -- "north" / "west"
    if not sq0 or (dir ~= "north" and dir ~= "west") then
        return nil
    end
    local objs = getAllSquareObjects(sq0)
    for _, obj in ipairs(objs) do
        if isCurtainObject(obj) then
            local north = _doorNorthFlag(obj)
            if (dir == "north" and north == true) or (dir == "west" and north == false) then
                if _hasOpenAPI(obj) then
                    return obj
                end
            end
        end
    end
    return nil
end

function RC_PlayerRoomLogic.edgeHasBlockingOnSquare(sq0, dir)
    local objs = getAllSquareObjects(sq0)
    for _, obj in ipairs(objs) do
        if obj and not (instanceof and instanceof(obj, "IsoDoor")) then
            local spr   = obj.getSprite and obj:getSprite() or nil
            local props = spr and spr:getProperties() or nil
            if props and props.Is then
                if     dir == "north" and (props:Is(IsoFlagType.WallN) or props:Is(IsoFlagType.collideN)) then return true
                elseif dir == "west"  and (props:Is(IsoFlagType.WallW) or props:Is(IsoFlagType.collideW)) then return true
                elseif dir == "south" and (props:Is(IsoFlagType.WallS) or props:Is(IsoFlagType.collideS)) then return true
                elseif dir == "east"  and (props:Is(IsoFlagType.WallE) or props:Is(IsoFlagType.collideE)) then return true
                end
            end
        end
    end
    return false
end

local function isDoorObject(obj)
    if not obj then return false end
    -- Real world-placed doors:
    if instanceof and instanceof(obj, "IsoDoor") then return true end
    -- Player-built doors:
    if instanceof and instanceof(obj, "IsoThumpable") and obj.isDoor then
        if obj:isDoor() then return true end
    end
end

-- Return a *door object* that sits on a specific edge ("north" or "west") of this square.
function RC_PlayerRoomLogic.getDoorOnEdge(sq0, edge)
    if not sq0 then return nil end
    local objs = getAllSquareObjects(sq0)
    local fallback = nil

    for _, obj in ipairs(objs) do
        if isDoorObject(obj) then
            -- Try to get orientation from the object first (works for IsoThumpable/IsoDoor),
            -- then fallback to sprite flags.
            local northFlag = _doorNorthFlag(obj)
            local orient = nil
            if northFlag ~= nil then
                orient = northFlag and "north" or "west"
            else
                orient = doorOrientationFromSprite(obj)
            end

            if orient == edge then
                if _hasOpenAPI(obj) then
                    -- Real door leaf (IsoDoor or thumpable door) -> use it
                    return obj
                end
                -- door-ish but no state API (likely a frame); remember but keep looking
                fallback = fallback or obj
            end
        end
    end

    -- If we only saw a frame, don't return it; it can't tell us open/closed properly.
    -- That forces callers to treat the edge as blocked unless another check passes.
    return nil
end

-- Passable between *regions* across an edge?
function RC_PlayerRoomLogic.edgeIsPassable(sqA, dir, sqB)
    if not sqA or not sqB then return false end

    -- 1) Door check (S/E doors live on neighbor square)
    if dir == "north" then
        local d = RC_PlayerRoomLogic.getDoorOnEdge(sqA, "north")
        if d then
            local open = RC_PlayerRoomLogic.isDoorOpen(d)
            _dbgDoor("edgeIsPassable NORTH: door found (open=%s) at (%d,%d,%d)",
                tostring(open), sqA:getX(), sqA:getY(), sqA:getZ())
            return open
        end
    elseif dir == "west" then
        local d = RC_PlayerRoomLogic.getDoorOnEdge(sqA, "west")
        if d then
            local open = RC_PlayerRoomLogic.isDoorOpen(d)
            _dbgDoor("edgeIsPassable WEST:  door found (open=%s) at (%d,%d,%d)",
                tostring(open), sqA:getX(), sqA:getY(), sqA:getZ())
            return open
        end
    elseif dir == "south" then
        local d = RC_PlayerRoomLogic.getDoorOnEdge(sqB, "north")
        if d then
            local open = RC_PlayerRoomLogic.isDoorOpen(d)
            _dbgDoor("edgeIsPassable SOUTH: door found (open=%s) at neighbor (%d,%d,%d)",
                tostring(open), sqB:getX(), sqB:getY(), sqB:getZ())
            return open
        end
    elseif dir == "east" then
        local d = RC_PlayerRoomLogic.getDoorOnEdge(sqB, "west")
        if d then
            local open = RC_PlayerRoomLogic.isDoorOpen(d)
            _dbgDoor("edgeIsPassable EAST:  door found (open=%s) at neighbor (%d,%d,%d)",
                tostring(open), sqB:getX(), sqB:getY(), sqB:getZ())
            return open
        end
    end

    -- 2) Explicit wall objects (S/E walls live on neighbor’s N/W getter)
    if dir == "north" and sqA.getWallN then
        local wallObj = sqA:getWallN()
        if wallObj == nil then
            if not RC_PlayerRoomLogic.edgeHasBlockingOnSquare(sqB, "south") then
                _dbgDoor("edgeIsPassable NORTH: no wall, no neighbor block -> PASS")
                return true
            end
        end
        _dbgDoor("edgeIsPassable NORTH: wall present or neighbor blocks -> BLOCK")
        return false
    elseif dir == "west" and sqA.getWallW then
        local wallObj = sqA:getWallW()
        if wallObj == nil then
            if not RC_PlayerRoomLogic.edgeHasBlockingOnSquare(sqB, "east") then
                _dbgDoor("edgeIsPassable WEST: no wall, no neighbor block -> PASS")
                return true
            end
        end
        _dbgDoor("edgeIsPassable WEST: wall present or neighbor blocks -> BLOCK")
        return false
    elseif dir == "south" and sqB.getWallN then
        local wallObj = sqB:getWallN()
        if wallObj == nil then
            if not RC_PlayerRoomLogic.edgeHasBlockingOnSquare(sqA, "south") then
                _dbgDoor("edgeIsPassable SOUTH: no wall, no neighbor block -> PASS")
                return true
            end
        end
        _dbgDoor("edgeIsPassable SOUTH: wall present or neighbor blocks -> BLOCK")
        return false
    elseif dir == "east" and sqB.getWallW then
        local wallObj = sqB:getWallW()
        if wallObj == nil then
            if not RC_PlayerRoomLogic.edgeHasBlockingOnSquare(sqA, "east") then
                _dbgDoor("edgeIsPassable EAST: no wall, no neighbor block -> PASS")
                return true
            end
        end
        _dbgDoor("edgeIsPassable EAST: wall present or neighbor blocks -> BLOCK")
        return false
    end

    -- 3) Fallback via sprite flags on both sides
    local function blocked(sq, e)
        return RC_PlayerRoomLogic.edgeHasBlockingOnSquare(sq, e)
    end
    if dir == "north" then
        local pass = (not blocked(sqA,"north")) and (not blocked(sqB,"south"))
        _dbgDoor("edgeIsPassable NORTH (fallback): %s", pass and "PASS" or "BLOCK")
        return pass
    elseif dir == "west" then
        local pass = (not blocked(sqA,"west")) and (not blocked(sqB,"east"))
        _dbgDoor("edgeIsPassable WEST (fallback): %s", pass and "PASS" or "BLOCK")
        return pass
    elseif dir == "south" then
        local pass = (not blocked(sqA,"south")) and (not blocked(sqB,"north"))
        _dbgDoor("edgeIsPassable SOUTH (fallback): %s", pass and "PASS" or "BLOCK")
        return pass
    elseif dir == "east" then
        local pass = (not blocked(sqA,"east")) and (not blocked(sqB,"west"))
        _dbgDoor("edgeIsPassable EAST (fallback): %s", pass and "PASS" or "BLOCK")
        return pass
    end

    return false
end

-- Stairs detection (same as your building logic)
local function hasStairsNorth(sq0)
    if not sq0 or not sq0.getObjects then return false end
    local objs = sq0:getObjects()
    if not objs then return false end
    for i = 0, objs:size() - 1 do
        local obj = objs:get(i)
        if obj and obj.isStairsNorth then
            local ok, res = pcall(function() return obj:isStairsNorth() end)
            if ok and res then return true end
        end
    end
    return false
end
local function hasStairsWest(sq0)
    if not sq0 or not sq0.getObjects then return false end
    local objs = sq0:getObjects()
    if not objs then return false end
    for i = 0, objs:size() - 1 do
        local obj = objs:get(i)
        if obj and obj.isStairsWest then
            local ok, res = pcall(function() return obj:isStairsWest() end)
            if ok and res then return true end
        end
    end
    return false
end

local function stairsOrientation(sq0) -- "N" or "W" or nil
    if hasStairsNorth(sq0) then return "N" end
    if hasStairsWest(sq0)  then return "W" end
    return nil
end

local function findBottomStairSquare(sq0, orient)
    local s = sq0
    local steps = 0
    while s and steps < 4 do
        local nextS = (orient == "N" and s.getS and s:getS())
                   or (orient == "W" and s.getE and s:getE()) or nil
        if nextS and ((orient == "N" and hasStairsNorth(nextS))
                   or (orient == "W" and hasStairsWest(nextS))) then
            s = nextS; steps = steps + 1
        else
            break
        end
    end
    return s
end

function RC_PlayerRoomLogic.stairsEndpointsFromAnyStair(sq0)
    local orient = stairsOrientation(sq0)
    if not orient or not sq0 then return nil, nil end
    local bottomBase = findBottomStairSquare(sq0, orient)
    if not bottomBase then return nil, nil end
    local x, y, z = bottomBase:getX(), bottomBase:getY(), bottomBase:getZ()
    local cell = getCell and getCell() or nil
    if not cell then return nil, nil end
    if orient == "N" then
        local bottomLanding = bottomBase.getS and bottomBase:getS() or cell:getGridSquare(x, y+1, z)
        local topLanding    = cell:getGridSquare(x, y-3, z+1)
        return bottomLanding, topLanding
    else -- "W"
        local bottomLanding = bottomBase.getE and bottomBase:getE() or cell:getGridSquare(x+1, y, z)
        local topLanding    = cell:getGridSquare(x-3, y, z+1)
        return bottomLanding, topLanding
    end
end

-- Region resolvers
function RC_PlayerRoomLogic.regionFromSquareOrNeighbors(sq0)
    local reg = regionFromSquare(sq0)
    if reg or (not sq0) then return reg end
    local z = sq0:getZ()
    local function try(s)
        if s and s:getZ() == z then
            local r = regionFromSquare(s)
            if r then return r end
        end
    end
    return try(sq0.getN and sq0:getN())
        or  try(sq0.getS and sq0:getS())
        or  try(sq0.getW and sq0:getW())
        or  try(sq0.getE and sq0:getE())
        or  nil
end

-- ========== Squares collection ==========

RC_PlayerRoomLogic.MAX_FIND_BOUNDARY_STEPS = RC_PlayerRoomLogic.MAX_FIND_BOUNDARY_STEPS or 8000

RC_PlayerRoomLogic.SEED_SEARCH_RADIUS = RC_PlayerRoomLogic.SEED_SEARCH_RADIUS or 12

local function sameRegion(sq, region)
    return sq and regionFromSquare(sq) == region
end

local function ensureSeedSq(cache, region, zLevel)
    cache.seedSqByRegion = cache.seedSqByRegion or {}

    -- cached seed still valid?
    local s = cache.seedSqByRegion[region]
    if s and sameRegion(s, region) and (not zLevel or s:getZ() == zLevel) then
        return s
    end

    -- try player's current square
    local player = getSpecificPlayer and getSpecificPlayer(0) or nil
    local psq = player and (
        (player.getCurrentSquare and player:getCurrentSquare()) or
        (player.getSquare and player:getSquare())
    ) or nil
    local z = zLevel or (psq and psq:getZ()) or 0
    if psq and psq:getZ() == z and sameRegion(psq, region) then
        cache.seedSqByRegion[region] = psq
        return psq
    end

    -- spiral search around player for any square in this region
    local cell = getCell and getCell() or nil
    if not cell or not psq then return nil end
    local x0, y0 = psq:getX(), psq:getY()
    local R = RC_PlayerRoomLogic.SEED_SEARCH_RADIUS

    for r = 0, R do
        -- top & bottom rows of the ring
        for dx = -r, r do
            local s1 = cell:getGridSquare(x0 + dx, y0 - r, z)
            if sameRegion(s1, region) then cache.seedSqByRegion[region] = s1; return s1 end
            local s2 = cell:getGridSquare(x0 + dx, y0 + r, z)
            if sameRegion(s2, region) then cache.seedSqByRegion[region] = s2; return s2 end
        end
        -- left & right columns (skip corners already checked)
        for dy = -r + 1, r - 1 do
            local s3 = cell:getGridSquare(x0 - r, y0 + dy, z)
            if sameRegion(s3, region) then cache.seedSqByRegion[region] = s3; return s3 end
            local s4 = cell:getGridSquare(x0 + r, y0 + dy, z)
            if sameRegion(s4, region) then cache.seedSqByRegion[region] = s4; return s4 end
        end
    end

    return nil
end

local function isBoundarySquare(sq, region)
    if not sq then return false end
    local z = sq:getZ()
    local n = sq.getN and sq:getN() or nil
    local s = sq.getS and sq:getS() or nil
    local w = sq.getW and sq:getW() or nil
    local e = sq.getE and sq:getE() or nil
    if (not n or n:getZ() ~= z or not sameRegion(n, region)) then return true end
    if (not s or s:getZ() ~= z or not sameRegion(s, region)) then return true end
    if (not w or w:getZ() ~= z or not sameRegion(w, region)) then return true end
    if (not e or e:getZ() ~= z or not sameRegion(e, region)) then return true end
    return false
end

-- NEW: perimeter-only finder
function RC_PlayerRoomLogic.getRegionBoundarySquares(cache, region, zLevel)
    cache.boundaryCache = cache.boundaryCache or {}
    local got = cache.boundaryCache[region]
    if got then return got end

    local seed = ensureSeedSq(cache, region, zLevel)
    if not seed then
        cache.boundaryCache[region] = {}
        if RC_PlayerRoomLogic.DEBUG_DOORS then
            -- print("[PRL] boundary: NO SEED for region", tostring(region))
        end
        return cache.boundaryCache[region]
    end

    local z = zLevel or seed:getZ()
    local visited = {}
    local q, head = { seed }, 1
    visited[seed] = true

    -- 1) find any boundary tile with short BFS
    local boundaryStart = nil
    local steps = 0
    local function tryPush(s)
        if s and s:getZ() == z and not visited[s] and sameRegion(s, region) then
            visited[s] = true
            q[#q+1] = s
        end
    end

    while head <= #q do
        local cur = q[head]; head = head + 1
        if isBoundarySquare(cur, region) then
            boundaryStart = cur
            break
        end
        if cur.getN then tryPush(cur:getN()) end
        if cur.getS then tryPush(cur:getS()) end
        if cur.getW then tryPush(cur:getW()) end
        if cur.getE then tryPush(cur:getE()) end
        steps = steps + 1
        if steps > RC_PlayerRoomLogic.MAX_FIND_BOUNDARY_STEPS then break end
    end

    if not boundaryStart then
        -- fallback: at least return the seed so door scans still run
        cache.boundaryCache[region] = { seed }
        if RC_PlayerRoomLogic.DEBUG_DOORS then
            -- print("[PRL] boundary: no boundary found; using seed only")
        end
        return cache.boundaryCache[region]
    end

    -- 2) walk only boundary tiles
    local boundaryVisited, boundaryList = {}, {}
    local qb, hb = { boundaryStart }, 1
    boundaryVisited[boundaryStart] = true

    local function tryPushBoundary(s)
        if s and s:getZ() == z and not boundaryVisited[s]
           and sameRegion(s, region) and isBoundarySquare(s, region) then
            boundaryVisited[s] = true
            qb[#qb+1] = s
        end
    end

    while hb <= #qb do
        local cur = qb[hb]; hb = hb + 1
        boundaryList[#boundaryList+1] = cur
        if cur.getN then tryPushBoundary(cur:getN()) end
        if cur.getS then tryPushBoundary(cur:getS()) end
        if cur.getW then tryPushBoundary(cur:getW()) end
        if cur.getE then tryPushBoundary(cur:getE()) end
    end

    cache.boundaryCache[region] = boundaryList
    if RC_PlayerRoomLogic.DEBUG_DOORS then
        -- print(("[PRL] boundary squares for %s: %d"):format(tostring(region), #boundaryList))
    end
    return boundaryList
end

-- ========== Graph building on player regions ==========

-- Same-Z neighbors: look at edges of all squares in 'region' and see if the neighbor square
-- belongs to a *different* isPlayerRoom() region, and the edge is passable
function RC_PlayerRoomLogic.getPassableNeighbors(cache, region, zFallback)
    local neighbors, seen = {}, {}
    local regZ = (region and region.getZ) and region:getZ() or zFallback
    local squares = RC_PlayerRoomLogic.getRegionBoundarySquares(cache, region, regZ)
    for _, rsq in ipairs(squares) do
        if rsq then
            -- For each direction, determine neighbor region and passability.
            local function check(dir, getNeighbor)
                local nSq = getNeighbor(rsq)
                if not nSq or nSq:getZ() ~= regZ then return end
                local nReg = regionFromSquare(nSq)
                if nReg and nReg ~= region then
                    local pass =
                        (dir == "north" and RC_PlayerRoomLogic.edgeIsPassable(rsq, "north", nSq)) or
                        (dir == "west"  and RC_PlayerRoomLogic.edgeIsPassable(rsq, "west",  nSq)) or
                        (dir == "south" and RC_PlayerRoomLogic.edgeIsPassable(rsq, "south", nSq)) or
                        (dir == "east"  and RC_PlayerRoomLogic.edgeIsPassable(rsq, "east",  nSq)) or
                        false
                    if pass and not seen[nReg] then
                        seen[nReg] = true
                        neighbors[#neighbors+1] = nReg
                    end
                end
            end
            if rsq.getN then check("north", function(s) return s:getN() end) end
            if rsq.getW then check("west",  function(s) return s:getW() end) end
            if rsq.getS then check("south", function(s) return s:getS() end) end
            if rsq.getE then check("east",  function(s) return s:getE() end) end
        end
    end
    return neighbors
end

-- Cross-floor neighbors via stairs: from any stair tile in this region, get bottom/top landings
-- and resolve to regions on z or z+1.
function RC_PlayerRoomLogic.getStairsNeighbors(cache, region)
    local cached = cache.stairsNeighborsCache[region]
    if cached then return cached end
    local nbs, seen = {}, {}
    local regZ = (region and region.getZ and region:getZ()) or cache.z or 0
    local squares = RC_PlayerRoomLogic.getRegionBoundarySquares(cache, region, regZ)
    for _, rsq in ipairs(squares) do
        local bottomSq, topSq = RC_PlayerRoomLogic.stairsEndpointsFromAnyStair(rsq)
        if bottomSq or topSq then
            local bottomReg = RC_PlayerRoomLogic.regionFromSquareOrNeighbors(bottomSq) or region
            local topReg    = RC_PlayerRoomLogic.regionFromSquareOrNeighbors(topSq)
            if bottomReg and topReg then
                if bottomReg == region and not seen[topReg] then
                    seen[topReg] = true; nbs[#nbs+1] = topReg
                elseif topReg == region and not seen[bottomReg] then
                    seen[bottomReg] = true; nbs[#nbs+1] = bottomReg
                end
            end
        end
    end
    cache.stairsNeighborsCache[region] = nbs
    return nbs
end

function RC_PlayerRoomLogic.getDoorNeighbors(cache, region, zFallback)
    _dbgDoor("ENTER getDoorNeighbors for region %s", tostring(region))
    local neighbors, seen = {}, {}
    local regZ = zFallback

    -- ensure we can iterate our *own* squares
    local squares = RC_PlayerRoomLogic.getRegionBoundarySquares(cache, region, regZ)

    local function add(nb)
        if nb and nb ~= region and not seen[nb] then
            seen[nb] = true
            neighbors[#neighbors+1] = nb
        end
    end

    local function markSeeds(curSq, nbSq, nbReg)
        cache.seedSqByRegion = cache.seedSqByRegion or {}
        if curSq and region then
            cache.seedSqByRegion[region] = cache.seedSqByRegion[region] or curSq
        end
        if nbReg and nbSq then
            cache.seedSqByRegion[nbReg] = cache.seedSqByRegion[nbReg] or nbSq
        end
    end

    for _, rsq in ipairs(squares) do
        local z = rsq:getZ()
        -- NORTH
        if rsq.getN then
            local nSq  = rsq:getN()
            local nReg = (nSq and nSq:getZ() == z) and regionFromSquare(nSq) or nil
            local d    = RC_PlayerRoomLogic.getDoorOnEdge(rsq, "north")
            if d and nReg then
                _dbgDoor("STRUCTURAL door between regions via NORTH at (%d,%d,%d)", rsq:getX(), rsq:getY(), z)
                markSeeds(rsq, nSq, nReg); add(nReg)
            end
        end
        -- WEST
        if rsq.getW then
            local wSq  = rsq:getW()
            local wReg = (wSq and wSq:getZ() == z) and regionFromSquare(wSq) or nil
            local d    = RC_PlayerRoomLogic.getDoorOnEdge(rsq, "west")
            if d and wReg then
                _dbgDoor("STRUCTURAL door between regions via WEST at (%d,%d,%d)", rsq:getX(), rsq:getY(), z)
                markSeeds(rsq, wSq, wReg); add(wReg)
            end
        end
        -- SOUTH (door lives on neighbor's NORTH)
        if rsq.getS then
            local sSq  = rsq:getS()
            local sReg = (sSq and sSq:getZ() == z) and regionFromSquare(sSq) or nil
            local d    = sSq and RC_PlayerRoomLogic.getDoorOnEdge(sSq, "north") or nil
            if d and sReg then
                _dbgDoor("STRUCTURAL door between regions via SOUTH@neighborN at (%d,%d,%d)", rsq:getX(), rsq:getY(), z)
                markSeeds(rsq, sSq, sReg); add(sReg)
            end
        end
        -- EAST (door lives on neighbor's WEST)
        if rsq.getE then
            local eSq  = rsq:getE()
            local eReg = (eSq and eSq:getZ() == z) and regionFromSquare(eSq) or nil
            local d    = eSq and RC_PlayerRoomLogic.getDoorOnEdge(eSq, "west") or nil
            if d and eReg then
                _dbgDoor("STRUCTURAL door between regions via EAST@neighborW at (%d,%d,%d)", rsq:getX(), rsq:getY(), z)
                markSeeds(rsq, eSq, eReg); add(eReg)
            end
        end
    end
    return neighbors
end

-- Perimeter scan (doors/windows/gaps) for a *single* region on a z-level.
-- Also records INTERIOR doors (between two player regions).
local function scanPerimeterForRegion(cache, region, zLevel,
    outsideDoors, outsideWindows, outsideCurtains, outsideGaps, outsideByRegion,
    interiorDoors, interiorByPair, interiorSeen)

    local function addByRegion(reg, entry)
        outsideByRegion[reg] = outsideByRegion[reg] or {}
        table.insert(outsideByRegion[reg], entry)
    end
    local function recordDoor(reg, door, sqOnDoor, edgeLabel)
        if not door or not sqOnDoor then return end
        local rec = { type="door", room=reg, x=sqOnDoor:getX(), y=sqOnDoor:getY(), z=sqOnDoor:getZ(),
                      edge=edgeLabel, door=door }
        outsideDoors[#outsideDoors+1] = rec; addByRegion(reg, rec)
    end
    local function recordWindow(reg, win, sqWithWin, edgeLabel)
        if not win or not sqWithWin then return end
        local rec = { type="window", room=reg, x=sqWithWin:getX(), y=sqWithWin:getY(), z=sqWithWin:getZ(),
                      edge=edgeLabel, window=win,
                      open=RC_PlayerRoomLogic.isWindowOpen(win),
                      smashed=RC_PlayerRoomLogic.isWindowSmashed(win),
                      barricadePlanks = RC_RoomLogic and RC_RoomLogic.windowBarricadePlankCount and RC_RoomLogic.windowBarricadePlankCount(win) or nil,
                      curtainClosed = RC_RoomLogic and RC_RoomLogic.windowHasClosedCurtains and RC_RoomLogic.windowHasClosedCurtains(win) or nil }
        outsideWindows[#outsideWindows+1] = rec; addByRegion(reg, rec)
    end
    local function recordCurtain(reg, curtain, sqWithCurtain, edgeLabel)
        if not curtain or not sqWithCurtain then return end
        local rec = { type="curtain", room=reg, x=sqWithCurtain:getX(), y=sqWithCurtain:getY(), z=sqWithCurtain:getZ(),
                      edge=edgeLabel, curtain=curtain, open=RC_PlayerRoomLogic.isCurtainOpen(curtain) }
        outsideCurtains[#outsideCurtains+1] = rec; addByRegion(reg, rec)
    end
    local function recordGap(reg, sqOnEdge, edgeLabel)
        if not sqOnEdge then return end
        local rec = { type="gap", room=reg, x=sqOnEdge:getX(), y=sqOnEdge:getY(), z=sqOnEdge:getZ(),
                      edge=edgeLabel }
        outsideGaps[#outsideGaps+1] = rec; addByRegion(reg, rec)
    end

    local function recordInterior(regA, regB, door, sqHost, edgeLabel)
        if not (regA and regB and door and sqHost) then return end
        interiorSeen = interiorSeen or {}
        if interiorSeen[door] then return end   -- dedupe by actual object
        interiorSeen[door] = true
        local rec = {
            type="door", a=regA, b=regB, door=door, edge=edgeLabel,
            x=sqHost:getX(), y=sqHost:getY(), z=sqHost:getZ()
        }
        interiorDoors[#interiorDoors+1] = rec
        local k = _pairKey(regA, regB) or "?"
        interiorByPair[k] = interiorByPair[k] or {}
        interiorByPair[k][#interiorByPair[k]+1] = rec
    end

    local squares = RC_PlayerRoomLogic.getRegionBoundarySquares(cache, region, zLevel)
    for _, rsq in ipairs(squares) do
        if rsq then
            -- convenience
            local z = rsq:getZ()

            -- NORTH edge: door/window live on THIS square
            do
                local nSq  = rsq.getN and rsq:getN() or nil
                local nReg = (nSq and (nSq:getZ()==z)) and regionFromSquare(nSq) or nil
                if nReg and nReg ~= region then
                    -- INTERIOR: door between region and nReg?
                    local dN = RC_PlayerRoomLogic.getDoorOnEdge(rsq, "north")
                    if dN then recordInterior(region, nReg, dN, rsq, "north") end
                else
                    -- EXTERIOR: nothing (or different z)
                    local dN = RC_PlayerRoomLogic.getDoorOnEdge(rsq, "north")
                    if dN then
                        recordDoor(region, dN, rsq, "north")
                    else
                        local cN = RC_PlayerRoomLogic.getCurtainOnEdge(rsq, "north")
                        if cN then recordCurtain(region, cN, rsq, "north") end
                        local wN = RC_PlayerRoomLogic.getWindowOnEdge(rsq, "north")
                        if wN then
                            recordWindow(region, wN, rsq, "north")
                        elseif not nReg then
                            local pass = nSq and RC_PlayerRoomLogic.edgeIsPassable(rsq, "north", nSq) or false
                            if pass then recordGap(region, rsq, "north") end
                        end
                    end
                end
            end

            -- WEST edge: door/window live on THIS square
            do
                local wSq  = rsq.getW and rsq:getW() or nil
                local wReg = (wSq and (wSq:getZ()==z)) and regionFromSquare(wSq) or nil
                if wReg and wReg ~= region then
                    local dW = RC_PlayerRoomLogic.getDoorOnEdge(rsq, "west")
                    if dW then recordInterior(region, wReg, dW, rsq, "west") end
                else
                    local dW = RC_PlayerRoomLogic.getDoorOnEdge(rsq, "west")
                    if dW then
                        recordDoor(region, dW, rsq, "west")
                    else
                        local cW = RC_PlayerRoomLogic.getCurtainOnEdge(rsq, "west")
                        if cW then recordCurtain(region, cW, rsq, "west") end
                        local wW = RC_PlayerRoomLogic.getWindowOnEdge(rsq, "west")
                        if wW then
                            recordWindow(region, wW, rsq, "west")
                        elseif not wReg then
                            local pass = wSq and RC_PlayerRoomLogic.edgeIsPassable(rsq, "west", wSq) or false
                            if pass then recordGap(region, rsq, "west") end
                        end
                    end
                end
            end

            -- SOUTH edge: door/window live on NEIGHBOR'S NORTH edge
            do
                local sSq  = rsq.getS and rsq:getS() or nil
                local sReg = (sSq and (sSq:getZ()==z)) and regionFromSquare(sSq) or nil
                if sReg and sReg ~= region then
                    local dS = sSq and RC_PlayerRoomLogic.getDoorOnEdge(sSq, "north") or nil
                    if dS then recordInterior(region, sReg, dS, sSq or rsq, "south") end
                else
                    local dS = sSq and RC_PlayerRoomLogic.getDoorOnEdge(sSq, "north") or nil
                    if dS then
                        recordDoor(region, dS, sSq or rsq, "south")
                    else
                        local cS = sSq and RC_PlayerRoomLogic.getCurtainOnEdge(sSq, "north") or nil
                        if cS then recordCurtain(region, cS, sSq or rsq, "south") end
                        local wS = sSq and RC_PlayerRoomLogic.getWindowOnEdge(sSq, "north") or nil
                        if wS then
                            recordWindow(region, wS, sSq or rsq, "south")
                        elseif not sReg then
                            local pass = sSq and RC_PlayerRoomLogic.edgeIsPassable(rsq, "south", sSq) or false
                            if pass then recordGap(region, rsq, "south") end
                        end
                    end
                end
            end

            -- EAST edge: door/window live on NEIGHBOR'S WEST edge
            do
                local eSq  = rsq.getE and rsq:getE() or nil
                local eReg = (eSq and (eSq:getZ()==z)) and regionFromSquare(eSq) or nil
                if eReg and eReg ~= region then
                    local dE = eSq and RC_PlayerRoomLogic.getDoorOnEdge(eSq, "west") or nil
                    if dE then recordInterior(region, eReg, dE, eSq or rsq, "east") end
                else
                    local dE = eSq and RC_PlayerRoomLogic.getDoorOnEdge(eSq, "west") or nil
                    if dE then
                        recordDoor(region, dE, eSq or rsq, "east")
                    else
                        local cE = eSq and RC_PlayerRoomLogic.getCurtainOnEdge(eSq, "west") or nil
                        if cE then recordCurtain(region, cE, eSq or rsq, "east") end
                        local wE = eSq and RC_PlayerRoomLogic.getWindowOnEdge(eSq, "west") or nil
                        if wE then
                            recordWindow(region, wE, eSq or rsq, "east")
                        elseif not eReg then
                            local pass = eSq and RC_PlayerRoomLogic.edgeIsPassable(rsq, "east", eSq) or false
                            if pass then recordGap(region, rsq, "east") end
                        end
                    end
                end
            end
        end
    end
end



-- Build or return the cluster cache for player regions starting at 'seedRegion' on 'zLevel'
function RC_PlayerRoomLogic.ensureClusterCache(seedRegion, zLevel)
    if not seedRegion then return nil end
    local key = tostring(seedRegion) .. ":" .. tostring(zLevel or 0)

    if RC_PlayerRoomLogic.DEBUG_DOORS then
        RC_PlayerRoomLogic._cluster[key] = nil
    end

    local cache = RC_PlayerRoomLogic._cluster[key]
    if cache then return cache end

    local tempCache = {
        squaresCache = {},
        stairsNeighborsCache = {},
        seedSqByRegion = {},        -- NEW: seed square per region
    }

    -- Seed the starting region from the player's square (same z)
    local psq = getSpecificPlayer and getSpecificPlayer(0) and getSpecificPlayer(0):getCurrentSquare() or nil
    if psq and psq:getZ() == (zLevel or 0) and regionFromSquare(psq) == seedRegion then
        tempCache.seedSqByRegion[seedRegion] = psq
        if RC_PlayerRoomLogic.DEBUG_DOORS then
            -- print(("[PRL] Seed square for start region set to (%d,%d,%d)"):format(psq:getX(), psq:getY(), psq:getZ()))
        end
    end

    -- BFS over regions to collect the cluster on this z
    local regionsZ, byRegion = {}, {}
    local q, head = { seedRegion }, 1
    byRegion[seedRegion] = true

    while head <= #q do
        local cur = q[head]; head = head + 1
        regionsZ[#regionsZ+1] = cur
        -- STRUCTURAL neighbors via shared door (ignore open/closed)
        for _, nb in ipairs(RC_PlayerRoomLogic.getDoorNeighbors(tempCache, cur, zLevel)) do
            if not byRegion[nb] then byRegion[nb] = true; q[#q+1] = nb end
        end
    end

    -- Perimeter scan for all collected regions
    local outsideDoors, outsideWindows, outsideCurtains, outsideGaps = {}, {}, {}, {}
    local outsideByRegion = {}

    local interiorDoors, interiorByPair = {}, {}
    local interiorSeen = {}

    for _, reg in ipairs(regionsZ) do
        scanPerimeterForRegion(
            tempCache, reg, zLevel,
            outsideDoors, outsideWindows, outsideCurtains, outsideGaps, outsideByRegion,
            interiorDoors, interiorByPair, interiorSeen
        )
    end

    cache = {
        z = zLevel or 0,
        regionsZ = regionsZ,
        outsideDoors = outsideDoors,
        outsideWindows = outsideWindows,
        outsideCurtains = outsideCurtains,
        outsideGaps = outsideGaps,
        outsideByRegion = outsideByRegion,
        interiorDoors = interiorDoors,
        interiorByPair = interiorByPair,
        squaresCache = tempCache.squaresCache,
        stairsNeighborsCache = {},
        seedSqByRegion = tempCache.seedSqByRegion,
    }

    RC_PlayerRoomLogic._cluster[key] = cache
    return cache
end

-- ========== Quick scanner, bound to G (like RC_RoomLogic) ==========

function RC_PlayerRoomLogic.RunScanForPlayerRegions(seedRegion, zLevel)
    -- print(("[PRL] === RUN SCAN (z=%s) DEBUG=%s ==="):format(tostring(zLevel), tostring(RC_PlayerRoomLogic.DEBUG_DOORS)))
    local cache = RC_PlayerRoomLogic.ensureClusterCache(seedRegion, zLevel)
    if not cache then return end

    -- Build adjacency graph (same-z + stairs + open interior doors)
    local graph = {}
    local function addEdge(a,b)
        if not a or not b then return end
        graph[a] = graph[a] or {}
        graph[b] = graph[b] or {}
        graph[a][b] = true; graph[b][a] = true
    end

    for _, reg in ipairs(cache.regionsZ) do
        graph[reg] = graph[reg] or {}
        -- gaps/archways/etc
        for _, nb in ipairs(RC_PlayerRoomLogic.getPassableNeighbors(cache, reg, cache.z)) do
            addEdge(reg, nb)
        end
        -- stairs
        for _, nb in ipairs(RC_PlayerRoomLogic.getStairsNeighbors(cache, reg)) do
            addEdge(reg, nb)
        end
    end

    -- edges from OPEN interior doors (authoritative)
    for _, drec in ipairs(cache.interiorDoors or {}) do
        if RC_PlayerRoomLogic.isDoorOpen(drec.door) then
            addEdge(drec.a, drec.b)
        end
    end

    -- Connected components
    local visited, groups = {}, {}
    local function rname(a) return RC_PlayerRoomLogic.roomName(a) end
    for _, r in ipairs(cache.regionsZ) do
        if not visited[r] then
            local stack, comp = { r }, {}
            visited[r] = true
            while #stack > 0 do
                local cur = table.remove(stack)
                comp[#comp+1] = cur
                for nb,_ in pairs(graph[cur] or {}) do
                    if not visited[nb] then
                        visited[nb] = true; stack[#stack+1] = nb
                    end
                end
            end
            table.sort(comp, function(a,b) return rname(a) < rname(b) end)
            groups[#groups+1] = comp
        end
    end

    -- RC_PlayerRoomLogic.DebugPrintPlayerBuilding(seedRegion)
end

RC_PlayerRoomLogic._buildingByRegion = RC_PlayerRoomLogic._buildingByRegion or {}
RC_PlayerRoomLogic._allPlayerBuildings = RC_PlayerRoomLogic._allPlayerBuildings or {}

local function _ensureSeed(cache, region)
    local z = (region and region.getZ and region:getZ()) or cache.z or 0
    return (cache.seedSqByRegion and cache.seedSqByRegion[region])
        or ensureSeedSq(cache, region, z)
end

local function _roomAnchorXY(cache, region)
    local s = _ensureSeed(cache, region)
    if s then return s:getX() or 0, s:getY() or 0 end
    return 0, 0
end

local function _sortRegionsReadable(regs, cache)
    table.sort(regs, function(a,b)
        local za = (a.getZ and a:getZ()) or 0
        local zb = (b.getZ and b:getZ()) or 0
        if za ~= zb then return za < zb end
        local ax, ay = _roomAnchorXY(cache, a)
        local bx, by = _roomAnchorXY(cache, b)
        if ax + ay ~= bx + by then return (ax + ay) < (bx + by) end
        if ax ~= bx then return ax < bx end
        return ay < by
    end)
end

local function _mkBuildingId(seedReg, seedSq)
    local z = (seedReg and seedReg.getZ and seedReg:getZ()) or (seedSq and seedSq:getZ()) or 0
    local x = seedSq and seedSq:getX() or 0
    local y = seedSq and seedSq:getY() or 0
    return string.format("PB@%d,%d,%d", x, y, z)
end

function RC_PlayerRoomLogic.BuildPlayerBuildingFromRegion(seedRegion)
    if not seedRegion then return nil end

    -- If we already mapped this region, return cached building
    local cached = RC_PlayerRoomLogic._buildingByRegion[seedRegion]
    if cached then return cached end

    -- We’ll BFS across regions, spanning floors with stairs.
    local byRegion = {}                 -- visited set of regions
    local regions  = {}                 -- all regions in building
    local queues   = { seedRegion }     -- regions to process
    byRegion[seedRegion] = true

    -- We also aggregate z-level caches (your cluster caches are per-z)
    local perZCache = {}                -- [z] -> cluster cache seeded from any region on that z
    local function ensureZCache(reg)
        local z = (reg and reg.getZ and reg:getZ()) or 0
        if not perZCache[z] then
            perZCache[z] = RC_PlayerRoomLogic.ensureClusterCache(reg, z)
        end
        return perZCache[z]
    end

    -- Door/Window/Gaps aggregation
    local exteriorDoors, exteriorWindows, exteriorCurtains, exteriorGaps = {}, {}, {}, {}
    local interiorDoors = {}
    local seenInterior = {}

    -- For structural adjacency (doors regardless of open/closed)
    local graph = {}

    local function addEdge(a,b)
        if not a or not b then return end
        graph[a] = graph[a] or {}
        graph[b] = graph[b] or {}
        graph[a][b] = true; graph[b][a] = true
    end

    -- BFS
    local head = 1
    while head <= #queues do
        local reg = queues[head]; head = head + 1
        regions[#regions+1] = reg

        local z = (reg and reg.getZ and reg:getZ()) or 0
        local zCache = ensureZCache(reg)

        -- Aggregate perimeters on this z (already computed by ensureClusterCache)
        -- Only copy entries that belong to 'reg' to avoid double work.
        for _, r in ipairs(zCache.outsideByRegion[reg] or {}) do
            if r.type == "door"   then exteriorDoors[#exteriorDoors+1] = r
            elseif r.type=="window" then exteriorWindows[#exteriorWindows+1] = r
            elseif r.type=="curtain" then exteriorCurtains[#exteriorCurtains+1] = r
            elseif r.type=="gap"    then exteriorGaps[#exteriorGaps+1] = r
            end
        end

        -- Interior doors that touch this region
        for _, d in ipairs(zCache.interiorDoors or {}) do
            if d.a == reg or d.b == reg then
                if not seenInterior[d.door] then
                    interiorDoors[#interiorDoors+1] = d
                    seenInterior[d.door] = true
                end
                addEdge(d.a, d.b)
            end
        end

        -- Structural door neighbors (same-z)
        for _, nb in ipairs(RC_PlayerRoomLogic.getDoorNeighbors(zCache, reg, z)) do
            addEdge(reg, nb)
            if not byRegion[nb] then
                byRegion[nb] = true
                queues[#queues+1] = nb
            end
        end

        -- Stairs neighbors (cross-floor)
        for _, nb in ipairs(RC_PlayerRoomLogic.getStairsNeighbors(zCache, reg)) do
            -- stairs neighbor may be on another z; ensure its z-cache
            ensureZCache(nb)
            addEdge(reg, nb)
            if not byRegion[nb] then
                byRegion[nb] = true
                queues[#queues+1] = nb
            end
        end
    end

    -- Compose building table
    -- Pick a stable seed square for id
    local anyZ = (seedRegion.getZ and seedRegion:getZ()) or 0
    local seedCache = perZCache[anyZ] or RC_PlayerRoomLogic.ensureClusterCache(seedRegion, anyZ)
    local seedSq = _ensureSeed(seedCache, seedRegion)
    local bldg = {
        id = _mkBuildingId(seedRegion, seedSq),
        regions = regions,
        graph = graph,
        exteriorDoors = exteriorDoors,
        exteriorWindows = exteriorWindows,
        exteriorCurtains = exteriorCurtains,
        exteriorGaps = exteriorGaps,
        interiorDoors = interiorDoors,
        -- filled below:
        roomNames = {},    -- [region] = "Readable Name"
        rooms = {},        -- array of { region, name, z, anchorX, anchorY }
        _perZCache = perZCache,  -- keep for internal lookups (anchors, etc.)
    }

    -- Assign readable room names, sorted by floor then anchor
    -- (Room 01, Room 02 … per building; include floor in name for clarity)
    _sortRegionsReadable(bldg.regions, seedCache)
    for i, reg in ipairs(bldg.regions) do
        local z = (reg.getZ and reg:getZ()) or 0
        local cx, cy = _roomAnchorXY(bldg._perZCache[z] or seedCache, reg)
        local name = string.format("Room %02d (z=%d)", i, z)
        bldg.roomNames[reg] = name
        bldg.rooms[#bldg.rooms+1] = {
            region = reg, name = name, z = z, anchorX = cx, anchorY = cy
        }
    end

    -- Cache it by every region it contains for O(1) lookup next time
    RC_PlayerRoomLogic._allPlayerBuildings[#RC_PlayerRoomLogic._allPlayerBuildings+1] = bldg
    for _, reg in ipairs(bldg.regions) do
        RC_PlayerRoomLogic._buildingByRegion[reg] = bldg
    end
    return bldg
end

function RC_PlayerRoomLogic.GetPlayerBuildingByRegion(region)
    return region and RC_PlayerRoomLogic._buildingByRegion[region] or nil
end

function RC_PlayerRoomLogic.GetRooms(building)
    return building and building.rooms or {}
end

function RC_PlayerRoomLogic.GetRoomName(building, region)
    if not (building and region) then return nil end
    return building.roomNames[region]
end

function RC_PlayerRoomLogic.GetExteriorDoors(building)
    return building and building.exteriorDoors or {}
end

function RC_PlayerRoomLogic.GetInteriorDoors(building)
    return building and building.interiorDoors or {}
end

function RC_PlayerRoomLogic.GetExteriorWindows(building)
    return building and building.exteriorWindows or {}
end

function RC_PlayerRoomLogic.GetExteriorCurtains(building)
    return building and building.exteriorCurtains or {}
end

function RC_PlayerRoomLogic.GetExteriorGaps(building)
    return building and building.exteriorGaps or {}
end

function RC_PlayerRoomLogic.GetGraph(building)
    return building and building.graph or {}
end

function RC_PlayerRoomLogic.LogDetectedDoors(buildingOrRegion)
    local building = nil
    if buildingOrRegion and buildingOrRegion.exteriorDoors and buildingOrRegion.regions then
        building = buildingOrRegion
    else
        local region = buildingOrRegion
        if not region then
            local player = getSpecificPlayer and getSpecificPlayer(0) or nil
            local sq = player and (player.getCurrentSquare and player:getCurrentSquare()) or nil
            if sq and RC_PlayerRoomLogic.regionFromSquareOrNeighbors then
                region = RC_PlayerRoomLogic.regionFromSquareOrNeighbors(sq)
            end
        end
        if region then
            building = RC_PlayerRoomLogic.BuildPlayerBuildingFromRegion(region)
        end
    end

    if not building then
        -- print("[PRL] LogDetectedDoors: no player building found")
        return
    end

    local function roomName(reg)
        return RC_PlayerRoomLogic.GetRoomName(building, reg) or RC_PlayerRoomLogic.roomName(reg)
    end

    -- print(string.format("[PRL] Detected doors for building %s", tostring(building.id or "<unknown>")))

    local exterior = RC_PlayerRoomLogic.GetExteriorDoors(building)
    -- print(string.format("  Exterior doors: %d", #exterior))
    for _, rec in ipairs(exterior) do
        local status = RC_PlayerRoomLogic.isDoorOpen(rec.door) and "open" or "closed"
        local rn = roomName(rec.room)
        -- print(string.format("    [%d,%d,%d] edge=%s room=%s status=%s",
        --     rec.x or -1, rec.y or -1, rec.z or -1, tostring(rec.edge), tostring(rn), status))
    end

    local interior = RC_PlayerRoomLogic.GetInteriorDoors(building)
    -- print(string.format("  Interior doors: %d", #interior))
    for _, rec in ipairs(interior) do
        local status = RC_PlayerRoomLogic.isDoorOpen(rec.door) and "open" or "closed"
        -- print(string.format("    [%d,%d,%d] edge=%s %s <-> %s status=%s",
        --     rec.x or -1, rec.y or -1, rec.z or -1,
        --     tostring(rec.edge), tostring(roomName(rec.a)), tostring(roomName(rec.b)), status))
    end
end

function RC_PlayerRoomLogic.GetConnectedRoomGroups(building, opts)
    if not building then return {} end
    opts = opts or {}
    local onlyPassable = opts.onlyPassable and true or false

    local function addEdge(g, a, b)
        if not a or not b then return end
        g[a] = g[a] or {}; g[b] = g[b] or {}
        g[a][b] = true;    g[b][a] = true
    end

    -- Build the graph we’ll traverse
    local graph = {}
    if not onlyPassable then
        -- Structural graph is already cached on the building
        graph = building.graph or {}
    else
        -- Passable-now graph:
        --  - open interior doors
        --  - passable neighbors (gaps/archways/open doors)
        --  - stairs neighbors
        for _, rec in ipairs(building.interiorDoors or {}) do
            if RC_PlayerRoomLogic.isDoorOpen(rec.door) then
                addEdge(graph, rec.a, rec.b)
            end
        end
        for _, r in ipairs(building.regions or {}) do
            local z = (r.getZ and r:getZ()) or 0
            local zCache = building._perZCache and building._perZCache[z] or nil
            if zCache then
                for _, nb in ipairs(RC_PlayerRoomLogic.getPassableNeighbors(zCache, r, z) or {}) do
                    addEdge(graph, r, nb)
                end
                for _, nb in ipairs(RC_PlayerRoomLogic.getStairsNeighbors(zCache, r) or {}) do
                    addEdge(graph, r, nb)
                end
            end
        end
    end

    -- Connected components over 'graph'
    local visited, groups = {}, {}
    local regs = building.regions or {}

    local function anchorXY(reg)
        local z = (reg.getZ and reg:getZ()) or 0
        local zCache = (building._perZCache and building._perZCache[z]) or {}
        local s = (zCache.seedSqByRegion and zCache.seedSqByRegion[reg]) or nil
        if not s and RC_PlayerRoomLogic then
            -- fallback using local helpers
            local _ensureSeed = function(cache, region)
                local zf = (region and region.getZ and region:getZ()) or cache.z or 0
                return (cache.seedSqByRegion and cache.seedSqByRegion[region])
                    or (ensureSeedSq and ensureSeedSq(cache, region, zf)) or nil
            end
            s = _ensureSeed(zCache, reg)
        end
        if s then return s:getX() or 0, s:getY() or 0 end
        return 0, 0
    end

    local function sortReadable(comp)
        table.sort(comp, function(a,b)
            local za = (a.getZ and a:getZ()) or 0
            local zb = (b.getZ and b:getZ()) or 0
            if za ~= zb then return za < zb end
            local ax, ay = anchorXY(a); local bx, by = anchorXY(b)
            if ax + ay ~= bx + by then return (ax + ay) < (bx + by) end
            if ax ~= bx then return ax < bx end
            return ay < by
        end)
    end

    for _, r in ipairs(regs) do
        if not visited[r] then
            local stack, comp = { r }, {}
            visited[r] = true
            while #stack > 0 do
                local cur = table.remove(stack)
                comp[#comp+1] = cur
                for nb,_ in pairs(graph[cur] or {}) do
                    if not visited[nb] then
                        visited[nb] = true
                        stack[#stack+1] = nb
                    end
                end
            end
            sortReadable(comp)
            groups[#groups+1] = comp
        end
    end
    return groups
end

function RC_PlayerRoomLogic.DebugPrintPlayerBuilding(seedRegion)
    local b = RC_PlayerRoomLogic.BuildPlayerBuildingFromRegion(seedRegion)
    if not b then return end

    -- print(("----- Player Building %s -----"):format(b.id))

    -- Passable-now groups (open doors/gaps + stairs)
    local liveGroups = RC_PlayerRoomLogic.GetConnectedRoomGroups(b, { onlyPassable = true })
    -- print(("Currently connected rooms:"):format(#liveGroups))
    for gi, comp in ipairs(liveGroups) do
        local names = {}
        for _, reg in ipairs(comp) do
            names[#names+1] = RC_PlayerRoomLogic.GetRoomName(b, reg)
                or RC_PlayerRoomLogic.roomName(reg)
        end
        -- print(("  Group %d: %s"):format(gi, table.concat(names, ", ")))
    end

    -- Flat room list (kept for reference)
    -- print(("Rooms: %d"):format(#b.rooms))
    for _, r in ipairs(b.rooms) do
        -- print(("  %s @ (%d,%d) z=%d"):format(r.name, r.anchorX or -1, r.anchorY or -1, r.z or 0))
    end

    -- print(("Exterior doors: %d"):format(#b.exteriorDoors))
    for _, rec in ipairs(b.exteriorDoors) do
        local status = RC_PlayerRoomLogic.isDoorOpen(rec.door) and "open" or "closed"
        local rn = RC_PlayerRoomLogic.GetRoomName(b, rec.room) or RC_PlayerRoomLogic.roomName(rec.room)
        -- print(("  [%d,%d,%d] %s  room=%s  status=%s"):format(rec.x, rec.y, rec.z, tostring(rec.edge), rn, status))
    end

    -- print(("Interior doors: %d"):format(#b.interiorDoors))
    for _, rec in ipairs(b.interiorDoors) do
        local status = RC_PlayerRoomLogic.isDoorOpen(rec.door) and "open" or "closed"
        local ra = RC_PlayerRoomLogic.GetRoomName(b, rec.a) or RC_PlayerRoomLogic.roomName(rec.a)
        local rb = RC_PlayerRoomLogic.GetRoomName(b, rec.b) or RC_PlayerRoomLogic.roomName(rec.b)
        -- print(("  [%d,%d,%d] %s  %s CONNECTING %s  %s")
        --     :format(rec.x, rec.y, rec.z, tostring(rec.edge), ra, rb, status))
    end
    -- print("--------------------------------")
end