-- RealisticCold propane tank handling for heaters

local RCHeaters = {}

RCHeaters.Heaters = {
    ["RC_TempSimMod.Mov_RCElectricHeater1"] = {
        powerType = "electric",
        baseModel = "RC_TempSimMod.ESpaceHeater1_OFF",
        baseModelOff = "RC_TempSimMod.ESpaceHeater1_OFF",
        baseModelOn = "RC_TempSimMod.ESpaceHeater1_ON",
        sprite = "appliances_misc_01_24",
        textureOff = "common/media/textures/heaters/ESpaceHeater1_OFF.png",
        textureOn = "common/media/textures/heaters/ESpaceHeater1_ON.png",
        heatValue = 16,
        heatLevels = {
            low = 8,
            medium = 12,
            high = 16,
        },
    },
    ["RC_TempSimMod.Mov_RCElectricHeater2"] = {
        powerType = "electric",
        baseModel = "RC_TempSimMod.ESpaceHeater2_OFF",
        baseModelOff = "RC_TempSimMod.ESpaceHeater2_OFF",
        baseModelOn = "RC_TempSimMod.ESpaceHeater2_ON",
        sprite = "appliances_misc_01_25",
        textureOff = "common/media/textures/heaters/ESpaceHeater2_OFF.png",
        textureOn = "common/media/textures/heaters/ESpaceHeater2_ON.png",
        heatValue = 12,
        heatLevels = {
            low = 6,
            medium = 9,
            high = 12,
        },
    },
    ["RC_TempSimMod.Mov_RCGasHeater"] = {
        powerType = "propane",
        baseModel = "RC_TempSimMod.GSpaceHeater1_OFF",
        baseModelOff = "RC_TempSimMod.GSpaceHeater1_OFF",
        baseModelOn = "RC_TempSimMod.GSpaceHeater1_ON",
        tankModel = "RC_TempSimMod.GSpaceHeater1_Tank_OFF",
        tankModelOff = "RC_TempSimMod.GSpaceHeater1_Tank_OFF",
        tankModelOn = "RC_TempSimMod.GSpaceHeater1_Tank_ON",
        sprite = "appliances_misc_01_26",
        textureOff = "common/media/textures/heaters/GSpaceHeater1_OFF.png",
        textureOn = "common/media/textures/heaters/GSpaceHeater1_ON.png",
        heatValue = 14,
        heatLevels = {
            low = 7,
            medium = 11,
            high = 14,
        },
        drainPerHour = 0.04,
    },
    ["RC_TempSimMod.Mov_RCIndustrialHeater"] = {
        powerType = "propane",
        baseModel = "RC_TempSimMod.IndustrialSpaceHeater_OFF",
        baseModelOff = "RC_TempSimMod.IndustrialSpaceHeater_OFF",
        baseModelOn = "RC_TempSimMod.IndustrialSpaceHeater_ON",
        tankModel = "RC_TempSimMod.IndustrialSpaceHeater_Tank_OFF",
        tankModelOff = "RC_TempSimMod.IndustrialSpaceHeater_Tank_OFF",
        tankModelOn = "RC_TempSimMod.IndustrialSpaceHeater_Tank_ON",
        sprite = "appliances_misc_01_27",
        textureOff = "common/media/textures/heaters/IndustrialSpaceHeater_OFF.png",
        textureOn = "common/media/textures/heaters/IndustrialSpaceHeater_ON.png",
        heatValue = 22,
        heatLevels = {
            low = 11,
            medium = 17,
            high = 22,
        },
        drainPerHour = 0.12,
    },
}

RCHeaters.HeatersByModel = {}
RCHeaters.HeatersBySprite = {}

local function registerHeaterModel(fullType, modelName)
    if modelName and not RCHeaters.HeatersByModel[modelName] then
        RCHeaters.HeatersByModel[modelName] = fullType
    end
end

for fullType, info in pairs(RCHeaters.Heaters) do
    registerHeaterModel(fullType, info.baseModel)
    registerHeaterModel(fullType, info.baseModelOff)
    registerHeaterModel(fullType, info.baseModelOn)
    registerHeaterModel(fullType, info.tankModel)
    registerHeaterModel(fullType, info.tankModelOff)
    registerHeaterModel(fullType, info.tankModelOn)
    if info.sprite then
        RCHeaters.HeatersBySprite[info.sprite] = fullType
    end
end

local refreshHeaterAppearance
local setHeaterActive
local setElectricHeaterActive
local deactivateElectricHeaterDueToPowerLoss

local function getTextOrDefault(key, defaultValue)
    if not getText then return defaultValue end
    local text = getText(key)
    if text and text ~= key then
        return text
    end
    return defaultValue
end

local function isPropaneHeater(info)
    return info and info.powerType == "propane"
end

local function isElectricHeater(info)
    return info and info.powerType == "electric"
end

local function getHeaterInfoByItem(item)
    if not item then return nil, nil end
    local fullType = item:getFullType()
    local info = RCHeaters.Heaters[fullType]
    return info, fullType
end

local function getHeaterInfoByObject(object)
    if not instanceof(object, "IsoWorldInventoryObject") then return nil, nil end
    local item = object:getItem()
    local fullType = item:getFullType()
    if fullType then
        return RCHeaters.Heaters[fullType], fullType
    end
    return nil, nil
end

local function resolveModDataHolder(target)
    if not target then return nil end

    if target.getItem then
        local ok, item = pcall(target.getItem, target)
        if ok and item and item.getModData then
            return item
        end
    end

    if target.getModData then
        return target
    end

    return nil
end

local function refreshModelByRotation(target)
    if not target then return end

    local item = target
    local ok, result = pcall(target.getItem, target)
    if ok then
        item = result
    end

    local modDataHolder = nil
    if item.getModData then
        modDataHolder = item
    elseif target.getModData then
        modDataHolder = target
    end

    local modData = modDataHolder and modDataHolder:getModData() or nil
    local lastDirection = modData and modData.RCLastRotationDir or -1
    local direction = -lastDirection

    local zRot = item:getWorldZRotation()
    item:setWorldZRotation(zRot + direction)

    if modData then
        modData.RCLastRotationDir = direction
        if modDataHolder and modDataHolder.transmitModData then
            modDataHolder:transmitModData()
        end
    end
end

local function clamp01(value)
    value = tonumber(value) or 0
    if value < 0 then return 0 end
    if value > 1 then return 1 end
    return value
end

local HEAT_SETTING_INFO = {
    [1] = {
        name = "low",
        textKey = "IGUI_RCHeatSettingLow",
        fallback = "Low",
        multiplier = 0.4,
        knobAngle = 45,
        markerAngle = 45,
    },
    [2] = {
        name = "medium",
        textKey = "IGUI_RCHeatSettingMedium",
        fallback = "Medium",
        multiplier = 0.7,
        knobAngle = 150,
        markerAngle = 150,
    },
    [3] = {
        name = "high",
        textKey = "IGUI_RCHeatSettingHigh",
        fallback = "High",
        multiplier = 1.0,
        knobAngle = 255,
        markerAngle = 255,
    },
}

local HEAT_SETTING_NAME_TO_ID = {}
local HEAT_SETTING_ORDER = { 1, 2, 3 }
for id, def in pairs(HEAT_SETTING_INFO) do
    if def.name then
        HEAT_SETTING_NAME_TO_ID[def.name] = id
    end
end

local DEFAULT_HEAT_SETTING = 3
local HEATER_MAX_TIMER_SECONDS = 120 * 60

local function _normalizeHeatSetting(value)
    if value == nil then
        return DEFAULT_HEAT_SETTING
    end

    if type(value) == "string" then
        local lowered = string.lower(value)
        if HEAT_SETTING_NAME_TO_ID[lowered] then
            return HEAT_SETTING_NAME_TO_ID[lowered]
        end
    end

    local numeric = tonumber(value)
    if not numeric then
        return DEFAULT_HEAT_SETTING
    end

    numeric = math.floor(numeric + 0.5)

    if HEAT_SETTING_INFO[numeric] then
        return numeric
    end

    if numeric <= 0 then
        return HEAT_SETTING_NAME_TO_ID.low or DEFAULT_HEAT_SETTING
    elseif numeric < 100 then
        return HEAT_SETTING_NAME_TO_ID.low or DEFAULT_HEAT_SETTING
    elseif numeric < 200 then
        return HEAT_SETTING_NAME_TO_ID.medium or DEFAULT_HEAT_SETTING
    else
        return HEAT_SETTING_NAME_TO_ID.high or DEFAULT_HEAT_SETTING
    end
end

local function _normalizeTimerSeconds(value)
    value = tonumber(value)
    if not value or value < 0 then return 0 end
    value = math.floor(value + 0.5)
    if value > HEATER_MAX_TIMER_SECONDS then
        value = HEATER_MAX_TIMER_SECONDS
    end
    return value
end

local function _getHeatSettingFromModData(md)
    if not md then return DEFAULT_HEAT_SETTING end
    local value = md.RCHeatSetting
    if value == nil then return DEFAULT_HEAT_SETTING end
    return _normalizeHeatSetting(value)
end

local function _getTimerSettingFromModData(md)
    if not md then return 0 end
    local value = md.RCHeatTimerSetting
    if value == nil then return 0 end
    return _normalizeTimerSeconds(value)
end

local function getHeatSettingDisplayText(settingId)
    local def = HEAT_SETTING_INFO[settingId]
    if not def then
        return tostring(settingId)
    end
    return getTextOrDefault(def.textKey, def.fallback)
end

local function getHeatOutputForSetting(info, settingId)
    local def = HEAT_SETTING_INFO[settingId] or HEAT_SETTING_INFO[DEFAULT_HEAT_SETTING]
    if not def then
        return 0
    end

    info = info or {}

    if info.heatLevels and def.name and info.heatLevels[def.name] then
        local levelValue = tonumber(info.heatLevels[def.name])
        if levelValue then
            return levelValue
        end
    end

    local multiplier = def.multiplier or 1
    if info.heatMultipliers and def.name and info.heatMultipliers[def.name] then
        local override = tonumber(info.heatMultipliers[def.name])
        if override then
            multiplier = override
        end
    end

    local base = tonumber(info.heatValue) or 0
    return base * multiplier
end

local function _getTimerEndFromModData(md)
    if not md then return nil end
    local value = md.RCHeatTimerEnd
    if value == nil then return nil end
    return tonumber(value)
end

local function _getTimerRunningForSecondsFromModData(md, now)
    if not md then return 0 end
    local timerSeconds = _getTimerSettingFromModData(md)
    if timerSeconds <= 0 then return 0 end
    now = now or getWorldAgeHours()
    local endHour = _getTimerEndFromModData(md)
    if not endHour then return 0 end
    local remaining = math.max(0, (endHour - now) * 3600)
    local running = timerSeconds - remaining
    if running < 0 then running = 0 end
    if running > timerSeconds then running = timerSeconds end
    return running
end

local function _getTimerRemainingSecondsFromModData(md, now)
    if not md then return 0 end
    local timerSeconds = _getTimerSettingFromModData(md)
    if timerSeconds <= 0 then return 0 end
    now = now or getWorldAgeHours()
    local endHour = _getTimerEndFromModData(md)
    if not endHour then return timerSeconds end
    local remaining = math.max(0, (endHour - now) * 3600)
    if remaining > timerSeconds then remaining = timerSeconds end
    return remaining
end

local function getWorldAgeHours()
    local gt = getGameTime and getGameTime()
    if gt and gt.getWorldAgeHours then
        return gt:getWorldAgeHours()
    end
    return 0
end

local function getHeatSetting(target)
    target = resolveModDataHolder(target)
    if not target then return DEFAULT_HEAT_SETTING end
    return _getHeatSettingFromModData(target:getModData())
end

local function setHeatSetting(target, value)
    local original = target
    target = resolveModDataHolder(target)
    if not target then return end
    local md = target:getModData()
    if not md then return end
    local normalized = _normalizeHeatSetting(value)
    local changed = false
    if md.RCHeatSetting ~= normalized then
        md.RCHeatSetting = normalized
        changed = true
    end
    if changed then
        if target.transmitModData then target:transmitModData() end
        if original and original ~= target and original.transmitModData then
            original:transmitModData()
        end
    end
end

local function getTimerSettingSeconds(target)
    target = resolveModDataHolder(target)
    if not target then return 0 end
    return _getTimerSettingFromModData(target:getModData())
end

local function setTimerSettingSeconds(target, info, value)
    local original = target
    target = resolveModDataHolder(target)
    if not target then return end
    local md = target:getModData()
    if not md then return end
    local normalized = _normalizeTimerSeconds(value)
    local now = getWorldAgeHours()
    local changed = false
    local wasActive = false
    if isPropaneHeater(info) then
        wasActive = md.RCPropaneActive == true
    elseif isElectricHeater(info) then
        wasActive = md.RCElectricActive == true
    end
    if md.RCHeatTimerSetting ~= normalized then
        md.RCHeatTimerSetting = normalized
        changed = true
    end
    if normalized <= 0 then
        if md.RCHeatTimerEnd ~= nil then
            md.RCHeatTimerEnd = nil
            changed = true
        end
    elseif wasActive then
        local running = _getTimerRunningForSecondsFromModData(md, now)
        local remaining = normalized - running
        if remaining < 0 then remaining = 0 end
        local newEnd = now
        if remaining > 0 then
            newEnd = now + (remaining / 3600)
        end
        if md.RCHeatTimerEnd ~= newEnd then
            md.RCHeatTimerEnd = newEnd
            changed = true
        end
    end
    if changed then
        if target.transmitModData then target:transmitModData() end
        if original and original ~= target and original.transmitModData then
            original:transmitModData()
        end
    end
end

local function getTimerRunningForSeconds(target)
    target = resolveModDataHolder(target)
    if not target then return 0 end
    return _getTimerRunningForSecondsFromModData(target:getModData(), getWorldAgeHours())
end

local function getTimerRemainingSeconds(target)
    target = resolveModDataHolder(target)
    if not target then return 0 end
    return _getTimerRemainingSecondsFromModData(target:getModData(), getWorldAgeHours())
end

local function updateTimerEndForActivation(target, info, active)
    local original = target
    target = resolveModDataHolder(target)
    if not target then return end
    local md = target:getModData()
    if not md then return end
    local changed = false
    if active then
        local timerSeconds = _getTimerSettingFromModData(md)
        if timerSeconds > 0 then
            local newEnd = getWorldAgeHours() + (timerSeconds / 3600)
            if md.RCHeatTimerEnd ~= newEnd then
                md.RCHeatTimerEnd = newEnd
                changed = true
            end
        elseif md.RCHeatTimerEnd ~= nil then
            md.RCHeatTimerEnd = nil
            changed = true
        end
    elseif md.RCHeatTimerEnd ~= nil then
        md.RCHeatTimerEnd = nil
        changed = true
    end
    if changed then
        if target.transmitModData then target:transmitModData() end
        if original and original ~= target and original.transmitModData then
            original:transmitModData()
        end
    end
end

local function getHeatOutput(target, info)
    if not info then return 0 end
    local heatSetting = getHeatSetting(target)
    return getHeatOutputForSetting(info, heatSetting)
end

local function getTimerRemainingHours(target)
    target = resolveModDataHolder(target)
    if not target then return math.huge end
    local md = target:getModData()
    if not md then return math.huge end
    local timerSeconds = _getTimerSettingFromModData(md)
    if timerSeconds <= 0 then return math.huge end
    local remaining = _getTimerRemainingSecondsFromModData(md, getWorldAgeHours())
    if remaining <= 0 then return 0 end
    return remaining / 3600
end

local function applyTimerAutoShutoff(target, info)
    local original = target
    target = resolveModDataHolder(target)
    if not target then return false end
    local md = target:getModData()
    if not md then return false end
    local endHour = _getTimerEndFromModData(md)
    if not endHour then return false end
    if getWorldAgeHours() < endHour then return false end
    if isPropaneHeater(info) then
        setHeaterActive(original, false)
    elseif isElectricHeater(info) then
        setElectricHeaterActive(original, false)
    end
    updateTimerEndForActivation(original, info, false)
    refreshHeaterAppearance(original, info)
    return true
end

local function hasElectricPowerAvailable(object, info)
    if not object or not isElectricHeater(info) then return false end
    local square = object.getSquare and object:getSquare()
    if not square then return false end
    if square.haveElectricity and square:haveElectricity() then
        return true
    end
    if square.hasGridPower and square:hasGridPower() then
        local room = square.getRoom and square:getRoom()
        if room then
            return true
        end
    end
    return false
end

local function isElectricHeaterActive(modDataHolder)
    modDataHolder = resolveModDataHolder(modDataHolder)
    if not modDataHolder then return false end
    local md = modDataHolder:getModData()
    return md and md.RCElectricActive == true
end

function setElectricHeaterActive(modDataHolder, active)
    local originalHolder = modDataHolder
    modDataHolder = resolveModDataHolder(modDataHolder)
    if not modDataHolder then return end
    local md = modDataHolder:getModData()
    if not md then return end
    local now = getWorldAgeHours()
    local changed = false

    if active then
        if md.RCElectricActive ~= true then
            md.RCElectricActive = true
            changed = true
        end
    else
        if md.RCElectricActive then
            md.RCElectricActive = nil
            changed = true
        end
    end

    if active then
        if md.RCElectricLastUpdate ~= now then
            md.RCElectricLastUpdate = now
            changed = true
        end
    elseif md.RCElectricLastUpdate ~= nil then
        md.RCElectricLastUpdate = nil
        changed = true
    end

    if changed then
        if modDataHolder.transmitModData then
            modDataHolder:transmitModData()
        end
        if originalHolder and originalHolder ~= modDataHolder and originalHolder.transmitModData then
            originalHolder:transmitModData()
        end
    end
end

function deactivateElectricHeaterDueToPowerLoss(target, info)
    if not target or not info or not isElectricHeater(info) then return false end
    if not isElectricHeaterActive(target) then return false end
    if hasElectricPowerAvailable(target, info) then return false end

    setElectricHeaterActive(target, false)
    updateTimerEndForActivation(target, info, false)
    return true
end

local function isHeaterActive(modDataHolder)
    modDataHolder = resolveModDataHolder(modDataHolder)
    if not modDataHolder then return false end
    local md = modDataHolder:getModData()
    return md and md.RCPropaneActive == true
end

local function hasPropaneTank(modDataHolder)
    modDataHolder = resolveModDataHolder(modDataHolder)
    if not modDataHolder then return false end
    local md = modDataHolder:getModData()
    return md and md.RCPropaneTankAttached == true
end

local function setPropaneTank(modDataHolder, attached, delta)
    local originalHolder = modDataHolder
    modDataHolder = resolveModDataHolder(modDataHolder)
    if not modDataHolder then return end
    local md = modDataHolder:getModData()
    if attached then
        md.RCPropaneTankAttached = true
        md.RCPropaneTankDelta = clamp01(delta or 1.0)
        md.RCPropaneActive = nil
        md.RCPropaneLastUpdate = getWorldAgeHours()
    else
        md.RCPropaneTankAttached = nil
        md.RCPropaneTankDelta = nil
        md.RCPropaneActive = nil
        md.RCPropaneLastUpdate = nil
    end
    if modDataHolder.transmitModData then
        modDataHolder:transmitModData()
    end
    if originalHolder and originalHolder ~= modDataHolder and originalHolder.transmitModData then
        originalHolder:transmitModData()
    end
end

local function getStoredTankDelta(modDataHolder)
    modDataHolder = resolveModDataHolder(modDataHolder)
    if not modDataHolder then return 1.0 end
    local md = modDataHolder:getModData()
    if md and md.RCPropaneTankDelta then
        return clamp01(md.RCPropaneTankDelta)
    end
    return 1.0
end

local function updateHeaterFuel(modDataHolder, info)
    if not isPropaneHeater(info) then return end
    local originalHolder = modDataHolder
    modDataHolder = resolveModDataHolder(modDataHolder)
    if not modDataHolder then return end
    local md = modDataHolder:getModData()
    if not md or not md.RCPropaneTankAttached then return end

    local now = getWorldAgeHours()
    local last = md.RCPropaneLastUpdate or now
    local elapsed = math.max(0, now - last)
    local changed = false
    local visualsChanged = false
    local delta = clamp01(md.RCPropaneTankDelta or 1.0)

    if elapsed > 0 then
        if md.RCPropaneActive and info and info.drainPerHour and info.drainPerHour > 0 then
            local newDelta = clamp01(delta - info.drainPerHour * elapsed)
            if newDelta ~= delta then
                delta = newDelta
                md.RCPropaneTankDelta = delta
                changed = true
            end
            if delta <= 0 and md.RCPropaneActive then
                md.RCPropaneActive = nil
                changed = true
                visualsChanged = true
            end
        end
        if md.RCPropaneLastUpdate ~= now then
            md.RCPropaneLastUpdate = now
            changed = true
        end
    else
        if md.RCPropaneTankDelta ~= delta then
            md.RCPropaneTankDelta = delta
            changed = true
        end
    end

    if changed then
        if modDataHolder.transmitModData then
            modDataHolder:transmitModData()
        end
        if originalHolder and originalHolder ~= modDataHolder and originalHolder.transmitModData then
            originalHolder:transmitModData()
        end
        if visualsChanged then
            refreshHeaterAppearance(modDataHolder, info)
            if originalHolder and originalHolder ~= modDataHolder then
                refreshHeaterAppearance(originalHolder, info)
            end
        end
    end
end

function setHeaterActive(modDataHolder, active)
    local originalHolder = modDataHolder
    modDataHolder = resolveModDataHolder(modDataHolder)
    if not modDataHolder then return end
    local md = modDataHolder:getModData()
    local now = getWorldAgeHours()
    local changed = false

    if active then
        if md.RCPropaneActive ~= true then
            md.RCPropaneActive = true
            changed = true
        end
    else
        if md.RCPropaneActive then
            md.RCPropaneActive = nil
            changed = true
        end
    end

    if md.RCPropaneLastUpdate ~= now then
        md.RCPropaneLastUpdate = now
        changed = true
    end

    if changed then
        if modDataHolder.transmitModData then
            modDataHolder:transmitModData()
        end
        if originalHolder and originalHolder ~= modDataHolder and originalHolder.transmitModData then
            originalHolder:transmitModData()
        end
    end
end

local function isHeaterVisuallyActive(target, info)
    if not info then return false end
    if isPropaneHeater(info) then
        return hasPropaneTank(target) and isHeaterActive(target)
    elseif isElectricHeater(info) then
        return isElectricHeaterActive(target)
    end
    return false
end

local function getHeaterStaticModel(info, hasTank, active)
    if not info then return nil end

    if hasTank then
        if active then
            return info.tankModelOn or info.tankModel or info.tankModelOff or info.baseModelOn or info.baseModel
        end
        return info.tankModel or info.tankModelOff or info.baseModel or info.baseModelOff or info.baseModelOn
    else
        if active then
            return info.baseModelOn or info.baseModel or info.baseModelOff
        end
        return info.baseModel or info.baseModelOff or info.baseModelOn
    end
end

local function applyInventoryModels(item, info)
    local active = isHeaterVisuallyActive(item, info)
    local hasTankAttached = isPropaneHeater(info) and hasPropaneTank(item)
    local desiredModel = getHeaterStaticModel(info, hasTankAttached, active)

    if desiredModel then
        if item.setStaticModel then
            item:setStaticModel(desiredModel)
        end
        if item.setWorldStaticModel then
            item:setWorldStaticModel(desiredModel)
        end
    end

    local pdata = getPlayerData(0)
    if pdata and pdata.playerInventory then
        pdata.playerInventory:refreshBackpacks()
        pdata.lootInventory:refreshBackpacks()
    end
end

local function applyWorldObjectModel(object, info)
    if not object or not info then return end
    if isElectricHeater(info) then
        deactivateElectricHeaterDueToPowerLoss(object, info)
    end
    local active = isHeaterVisuallyActive(object, info)
    local hasTankAttached = isPropaneHeater(info) and hasPropaneTank(object)
    local desiredModel = getHeaterStaticModel(info, hasTankAttached, active)

    local item = nil
    if object.getItem then
        local ok, result = pcall(object.getItem, object)
        if ok then
            item = result
        end
    end
    if item then
        applyInventoryModels(item, info)
    end
    if desiredModel and object.setStaticModel then
        if not object.getStaticModel or object:getStaticModel() ~= desiredModel then
            object:setStaticModel(desiredModel)
        end
    end
    local pdata = getPlayerData(0)
    if pdata and pdata.playerInventory then
        pdata.playerInventory:refreshBackpacks()
        pdata.lootInventory:refreshBackpacks()
    end
end

refreshHeaterAppearance = function(target, info)
    if not target or not info then return end
    if instanceof and instanceof(target, "InventoryItem") then
        applyInventoryModels(target, info)
    else
        applyWorldObjectModel(target, info)
        if target and target.transmitUpdatedSprite then
            target:transmitUpdatedSprite()
        end
        refreshModelByRotation(target)
    end
end

local function playerHasPropane(playerObj)
    if not playerObj then return false end
    local inv = playerObj:getInventory()
    if not inv or not inv.containsTypeRecurse then return false end
    return inv:containsTypeRecurse("PropaneTank")
end

local function removePropaneFromPlayer(playerObj)
    if not playerObj then return nil end
    local inv = playerObj:getInventory()
    if not inv or not inv.containsTypeRecurse then return nil end
    if not inv:containsTypeRecurse("PropaneTank") then return nil end
    local tankItem = inv:getFirstTypeRecurse("PropaneTank")
    if not tankItem then return nil end
    local delta = nil
    if tankItem.getCurrentUsesFloat then
        delta = tankItem:getCurrentUsesFloat()
    end
    if not delta and tankItem.getUsedDelta then
        delta = 1.0 - clamp01(tankItem:getUsedDelta())
    end
    if not delta then
        delta = tankItem.getUseDelta and tankItem:getUseDelta() or 1.0
    end
    delta = clamp01(delta)
    if playerObj:getPrimaryHandItem() == tankItem then
        playerObj:setPrimaryHandItem(nil)
    end
    if playerObj:getSecondaryHandItem() == tankItem then
        playerObj:setSecondaryHandItem(nil)
    end
    if inv.Remove then
        inv:Remove(tankItem)
    else
        inv:RemoveOneOf("Base.PropaneTank")
    end
    if sendRemoveItemFromContainer and inv and isClient and isClient() then
        sendRemoveItemFromContainer(inv, tankItem)
    end
    return delta
end

local function givePropaneToPlayer(playerObj, delta)
    if not playerObj then return end
    local inv = playerObj:getInventory()
    if not inv then return end
    delta = clamp01(delta or 0)
    local tankItem = inv:AddItem("Base.PropaneTank")
    if tankItem then
        if tankItem.setCurrentUsesFloat then
            tankItem:setCurrentUsesFloat(delta)
        elseif tankItem.setUsedDelta then
            tankItem:setUsedDelta(delta)
        end
        if sendAddItemToContainer and isClient and isClient() then
            sendAddItemToContainer(inv, tankItem)
        end
    else
        local square = playerObj:getSquare()
        if square then
            local worldItem = square:AddWorldInventoryItem("Base.PropaneTank", playerObj:getX() - square:getX(), playerObj:getY() - square:getY(), 0)
            if worldItem then
                local dropped = worldItem:getItem()
                if dropped and dropped.setCurrentUsesFloat then
                    dropped:setCurrentUsesFloat(delta)
                elseif dropped and dropped.setUsedDelta then
                    dropped:setUsedDelta(delta)
                end
            end
        end
    end
end

local function turnOnHeater(worldobjects, playerObj, object, info)
    if isPropaneHeater(info) then
        updateHeaterFuel(object, info)
        if not hasPropaneTank(object) then return end
        if getStoredTankDelta(object) <= 0 then return end
        setHeaterActive(object, true)
        updateTimerEndForActivation(object, info, true)
        refreshHeaterAppearance(object, info)
    elseif isElectricHeater(info) then
        if not hasElectricPowerAvailable(object, info) then return end
        setElectricHeaterActive(object, true)
        updateTimerEndForActivation(object, info, true)
        refreshHeaterAppearance(object, info)
    end
end

local function turnOffHeater(worldobjects, playerObj, object, info)
    if isPropaneHeater(info) then
        updateHeaterFuel(object, info)
        setHeaterActive(object, false)
        updateTimerEndForActivation(object, info, false)
        refreshHeaterAppearance(object, info)
    elseif isElectricHeater(info) then
        setElectricHeaterActive(object, false)
        updateTimerEndForActivation(object, info, false)
        refreshHeaterAppearance(object, info)
    end
end

local function createHeaterController(object, info, playerObj)
    if not object or not info then return nil end

    local controller = {
        object = object,
        info = info,
        playerObj = playerObj,
    }

    function controller:getHeatSetting()
        return getHeatSetting(self.object)
    end

    function controller:setHeatSetting(value)
        setHeatSetting(self.object, value)
    end

    function controller:getHeatSettingOptions()
        if RCHeaters and RCHeaters.getHeatSettingOptions then
            return RCHeaters.getHeatSettingOptions()
        end
        return {}
    end

    function controller:getHeatSettingDisplayText(value)
        if RCHeaters and RCHeaters.getHeatSettingDisplayText then
            return RCHeaters.getHeatSettingDisplayText(value)
        end
        return tostring(value)
    end

    function controller:getTimerSettingSeconds()
        return getTimerSettingSeconds(self.object)
    end

    function controller:setTimerSettingSeconds(value)
        setTimerSettingSeconds(self.object, self.info, value)
    end

    function controller:getTimerRunningForSeconds()
        return getTimerRunningForSeconds(self.object)
    end

    function controller:isActive()
        if isPropaneHeater(self.info) then
            return hasPropaneTank(self.object) and isHeaterActive(self.object)
        elseif isElectricHeater(self.info) then
            return isElectricHeaterActive(self.object)
        end
        return false
    end

    function controller:canTurnOn()
        if self:isActive() then return true end
        if isPropaneHeater(self.info) then
            return hasPropaneTank(self.object) and getStoredTankDelta(self.object) > 0
        elseif isElectricHeater(self.info) then
            return hasElectricPowerAvailable(self.object, self.info)
        end
        return false
    end

    function controller:setActive(active)
        if active then
            turnOnHeater(nil, self.playerObj, self.object, self.info)
        else
            turnOffHeater(nil, self.playerObj, self.object, self.info)
        end
    end

    function controller:getWorldPosition()
        if self.object and self.object.getX then
            local x = self.object:getX()
            local y = self.object.getY and self.object:getY() or 0
            local z = self.object.getZ and self.object:getZ() or 0
            return x, y, z
        end
        if self.playerObj then
            return self.playerObj:getX(), self.playerObj:getY(), self.playerObj:getZ()
        end
        return 0, 0, 0
    end

    function controller:getFacingTarget()
        return self.object
    end

    function controller:isValid()
        if not self.object then return false end
        if self.object.isDestroyed and self.object:isDestroyed() then return false end
        if self.object.isRemoved and self.object:isRemoved() then return false end
        return true
    end

    return controller
end

local function openHeaterUI(playerObj, object, info)
    if not playerObj or not object or not info then return end
    local controller = createHeaterController(object, info, playerObj)
    if not controller then return end

    local UI = require("RC_HeaterUI")
    if not UI then return end

    local player = playerObj:getPlayerNum()
    if UI.instance and UI.instance[player+1] then
        UI.instance[player+1].close:forceClick()
    end

    local ui = UI:new(0, 0, 430, 310, controller, playerObj)
    if not ui then return end
    ui:initialise()
    ui:addToUIManager()

    if JoypadState and JoypadState.players and JoypadState.players[player+1] then
        ui.prevFocus = JoypadState.players[player+1].focus
        setJoypadFocus(player, ui)
    end
end

local function openInventoryHeaterSettings(item, playerObj, info)
    openHeaterUI(playerObj, item, info)
end

local function openWorldHeaterSettings(worldobjects, playerObj, object, info)
    if object and object.getSquare and luautils and luautils.walkAdj then
        local square = object:getSquare()
        if square and not luautils.walkAdj(playerObj, square) then
            return
        end
    end
    local action = require("RC_HeaterUITimedAction")
    if not action then return end
    ISTimedActionQueue.add(action:new(playerObj, object, info))
end

local function attachPropaneToInventoryItem(item, playerObj, info)
    if hasPropaneTank(item) then return end
    local delta = removePropaneFromPlayer(playerObj)
    if not delta then return end
    setPropaneTank(item, true, delta)
    refreshHeaterAppearance(item, info)
    refreshModelByRotation(item)
    local worldItem = item.getWorldItem and item:getWorldItem()
    if worldItem then
        refreshHeaterAppearance(worldItem, info)
    end
end

local function detachPropaneFromInventoryItem(item, playerObj, info)
    if not hasPropaneTank(item) then return end
    updateHeaterFuel(item, info)
    local delta = getStoredTankDelta(item)
    setPropaneTank(item, false)
    refreshHeaterAppearance(item, info)
    refreshModelByRotation(item)
    local worldItem = item.getWorldItem and item:getWorldItem()
    if worldItem then
        refreshHeaterAppearance(worldItem, info)
    end
    givePropaneToPlayer(playerObj, delta)
end

local function attachPropaneToWorldObject(worldobjects, playerObj, object, info)
    if hasPropaneTank(object) then return end
    local delta = removePropaneFromPlayer(playerObj)
    if not delta then return end
    setPropaneTank(object, true, delta)
    refreshHeaterAppearance(object, info)
end

local function detachPropaneFromWorldObject(worldobjects, playerObj, object, info)
    if not hasPropaneTank(object) then return end
    updateHeaterFuel(object, info)
    setHeaterActive(object, false)
    local delta = getStoredTankDelta(object)
    setPropaneTank(object, false)
    refreshHeaterAppearance(object, info)
    givePropaneToPlayer(playerObj, delta)
end

local function addDisabledTooltip(option, text)
    if not option then return end
    option.notAvailable = true
    if ISInventoryPaneContextMenu and ISInventoryPaneContextMenu.addToolTip then
        local tooltip = ISInventoryPaneContextMenu.addToolTip()
        tooltip.description = text
        option.toolTip = tooltip
    elseif ISWorldObjectContextMenu and ISWorldObjectContextMenu.addToolTip then
        local tooltip = ISWorldObjectContextMenu.addToolTip()
        tooltip.description = text
        option.toolTip = tooltip
    end
end

function RCHeaters.onFillInventoryObjectContextMenu(playerNum, context, items)
    local playerObj = getSpecificPlayer(playerNum)
    if not playerObj then return end
    if not items or #items == 0 then return end
    if not ISInventoryPane or not ISInventoryPane.getActualItems then return end
    local actualItems = ISInventoryPane.getActualItems(items)
    if not actualItems or #actualItems == 0 then return end
    for _, item in ipairs(actualItems) do
        local info, fullType = getHeaterInfoByItem(item)
        if info then
            applyInventoryModels(item, info)
            local worldItem = item.getWorldItem and item:getWorldItem()
            if worldItem then
                local worldobjects = { worldItem }
                if isPropaneHeater(info) then
                    updateHeaterFuel(worldItem, info)
                    if hasPropaneTank(worldItem) then
                        local delta = getStoredTankDelta(worldItem)
                        local propaneActive = isHeaterActive(worldItem)
                        if propaneActive then
                            context:addOption(getTextOrDefault("ContextMenu_RCTurnOffHeater", "Turn Off Heater"), worldobjects, turnOffHeater, playerObj, worldItem, info)
                        elseif delta > 0 then
                            context:addOption(getTextOrDefault("ContextMenu_RCTurnOnHeater", "Turn On Heater"), worldobjects, turnOnHeater, playerObj, worldItem, info)
                        end
                        context:addOption(getTextOrDefault("ContextMenu_RCDetachPropaneTank", "Detach Propane Tank"), worldobjects, detachPropaneFromWorldObject, playerObj, worldItem, info)
                        context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), worldobjects, openWorldHeaterSettings, playerObj, worldItem, info)
                    else
                        local option = context:addOption(getTextOrDefault("ContextMenu_RCAttachPropaneTank", "Attach Propane Tank"), worldobjects, attachPropaneToWorldObject, playerObj, worldItem, info)
                        if not playerHasPropane(playerObj) then
                            addDisabledTooltip(option, getTextOrDefault("ContextMenu_NeedPropaneTank", "Requires a Propane Tank"))
                        end
                        context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), worldobjects, openWorldHeaterSettings, playerObj, worldItem, info)
                    end
                    return
                elseif isElectricHeater(info) then
                    local electricActive = isElectricHeaterActive(worldItem)
                    if electricActive then
                        context:addOption(getTextOrDefault("ContextMenu_RCTurnOffHeater", "Turn Off Heater"), worldobjects, turnOffHeater, playerObj, worldItem, info)
                    else
                        local option = context:addOption(getTextOrDefault("ContextMenu_RCTurnOnHeater", "Turn On Heater"), worldobjects, turnOnHeater, playerObj, worldItem, info)
                        if not hasElectricPowerAvailable(worldItem, info) then
                            addDisabledTooltip(option, getTextOrDefault("ContextMenu_RCNeedsElectricity", "Requires electricity"))
                        end
                    end
                    context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), worldobjects, openWorldHeaterSettings, playerObj, worldItem, info)
                    return
                end
            end
            if isPropaneHeater(info) then
                if hasPropaneTank(item) then
                    local option = context:addOption(getTextOrDefault("ContextMenu_RCDetachPropaneTank", "Detach Propane Tank"), item, detachPropaneFromInventoryItem, playerObj, info)
                    context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), item, openInventoryHeaterSettings, playerObj, info)
                    return
                else
                    local option = context:addOption(getTextOrDefault("ContextMenu_RCAttachPropaneTank", "Attach Propane Tank"), item, attachPropaneToInventoryItem, playerObj, info)
                    if not playerHasPropane(playerObj) then
                        addDisabledTooltip(option, getTextOrDefault("ContextMenu_NeedPropaneTank", "Requires a Propane Tank"))
                    end
                    context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), item, openInventoryHeaterSettings, playerObj, info)
                    return
                end
            elseif isElectricHeater(info) then
                context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), item, openInventoryHeaterSettings, playerObj, info)
                return
            end
        end
    end
end

function RCHeaters.onFillWorldObjectContextMenu(playerNum, context, worldobjects, test)
    local playerObj = getSpecificPlayer(playerNum)
    if not playerObj then return end
    local function findHeaterOnSquare(square)
        if not square then return nil, nil end
        local objects = square:getObjects()
        if not objects then return nil, nil end
        for i = 0, objects:size() - 1 do
            local object = objects:get(i)
            local objectInfo = select(1, getHeaterInfoByObject(object))
            if objectInfo then
                return object, objectInfo
            end
        end
        return nil, nil
    end
    local heaterObject
    local info
    local contextObjects = worldobjects
    if worldobjects then
        for _, object in ipairs(worldobjects) do
            local objectInfo = object and select(1, getHeaterInfoByObject(object)) or nil
            if objectInfo then
                heaterObject = object
                info = objectInfo
                break
            end
        end
    end
    local origSq
    if worldobjects and worldobjects[1] and worldobjects[1].getSquare then
        origSq = worldobjects[1]:getSquare()
    end
    if not heaterObject and origSq and instanceof(origSq, "IsoGridSquare") then
        heaterObject, info = findHeaterOnSquare(origSq)
    end
    if not heaterObject and origSq and instanceof(origSq, "IsoGridSquare") then
        local cell = origSq:getCell()
        if cell then
            local baseX, baseY, baseZ = origSq:getX(), origSq:getY(), origSq:getZ()
            for dx = -1, 1 do
                for dy = -1, 1 do
                    if not (dx == 0 and dy == 0) then
                        local adjSq = cell:getGridSquare(baseX + dx, baseY + dy, baseZ)
                        if adjSq then
                            heaterObject, info = findHeaterOnSquare(adjSq)
                            if heaterObject then
                                contextObjects = { heaterObject }
                                origSq = adjSq
                                break
                            end
                        end
                    end
                end
                if heaterObject then break end
            end
        end
    end
    if not heaterObject or not info then return end
    applyWorldObjectModel(heaterObject, info)
    local square = heaterObject:getSquare()
    if square and playerObj and playerObj:getSquare() ~= square then
        local dist = square:DistToProper(playerObj:getX(), playerObj:getY())
        if dist > 2 then
            return
        end
    end
    if isPropaneHeater(info) then
        updateHeaterFuel(heaterObject, info)
        if hasPropaneTank(heaterObject) then
            local delta = getStoredTankDelta(heaterObject)
            local propaneActive = isHeaterActive(heaterObject)
            if propaneActive then
                context:addOption(getTextOrDefault("ContextMenu_RCTurnOffHeater", "Turn Off Heater"), contextObjects, turnOffHeater, playerObj, heaterObject, info)
            elseif delta > 0 then
                context:addOption(getTextOrDefault("ContextMenu_RCTurnOnHeater", "Turn On Heater"), contextObjects, turnOnHeater, playerObj, heaterObject, info)
            end
            context:addOption(getTextOrDefault("ContextMenu_RCDetachPropaneTank", "Detach Propane Tank"), contextObjects, detachPropaneFromWorldObject, playerObj, heaterObject, info)
            context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), contextObjects, openWorldHeaterSettings, playerObj, heaterObject, info)
        else
            local option = context:addOption(getTextOrDefault("ContextMenu_RCAttachPropaneTank", "Attach Propane Tank"), contextObjects, attachPropaneToWorldObject, playerObj, heaterObject, info)
            if not playerHasPropane(playerObj) then
                addDisabledTooltip(option, getTextOrDefault("ContextMenu_NeedPropaneTank", "Requires a Propane Tank"))
            end
            context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), contextObjects, openWorldHeaterSettings, playerObj, heaterObject, info)
        end
        return
    elseif isElectricHeater(info) then
        local electricActive = isElectricHeaterActive(heaterObject)
        if electricActive then
            context:addOption(getTextOrDefault("ContextMenu_RCTurnOffHeater", "Turn Off Heater"), contextObjects, turnOffHeater, playerObj, heaterObject, info)
        else
            local option = context:addOption(getTextOrDefault("ContextMenu_RCTurnOnHeater", "Turn On Heater"), contextObjects, turnOnHeater, playerObj, heaterObject, info)
            if not hasElectricPowerAvailable(heaterObject, info) then
                addDisabledTooltip(option, getTextOrDefault("ContextMenu_RCNeedsElectricity", "Requires electricity"))
            end
        end
        context:addOption(getTextOrDefault("ContextMenu_RCHeaterSettings", "Heater Settings"), contextObjects, openWorldHeaterSettings, playerObj, heaterObject, info)
        return
    end
end

Events.OnPreFillInventoryObjectContextMenu.Add(RCHeaters.onFillInventoryObjectContextMenu)
Events.OnPreFillWorldObjectContextMenu.Add(RCHeaters.onFillWorldObjectContextMenu)

local function registerHeatSource()
    if RCHeaters._heatSourceId then return end
    if not RC_TempSim or not RC_TempSim.registerHeatSource then return end
    RCHeaters._heatSourceId = RC_TempSim.registerHeatSource({
        id = "rc_heat_propane_heaters",
        description = "Realistic Cold propane heaters",
        matchFn = function(obj)
            return select(1, getHeaterInfoByObject(obj)) ~= nil
        end,
        heatFn = function(obj)
            local info = select(1, getHeaterInfoByObject(obj))
            if not info then return 0 end
            local holder = resolveModDataHolder(obj)
            local md = holder and holder:getModData() or nil
            if not md then return 0 end
            if isPropaneHeater(info) then
                updateHeaterFuel(obj, info)
                if not (md.RCPropaneTankAttached and md.RCPropaneTankDelta and md.RCPropaneTankDelta > 0 and md.RCPropaneActive) then
                    return 0
                end
                if applyTimerAutoShutoff(obj, info) then
                    return 0
                end
                local heatOutput = getHeatOutput(obj, info)
                if heatOutput <= 0 then return 0 end
                return heatOutput
            elseif isElectricHeater(info) then
                if md.RCElectricActive and deactivateElectricHeaterDueToPowerLoss(obj, info) then
                    md = holder and holder:getModData() or md
                    refreshHeaterAppearance(obj, info)
                end
                if not (md.RCElectricActive and hasElectricPowerAvailable(obj, info)) then
                    return 0
                end
                if applyTimerAutoShutoff(obj, info) then
                    return 0
                end
                local heatOutput = getHeatOutput(obj, info)
                if heatOutput <= 0 then return 0 end
                return heatOutput
            end
            return 0
        end,
    })
end

registerHeatSource()
if Events and Events.OnGameStart then
    Events.OnGameStart.Add(registerHeatSource)
end

local function onRCLoadGridsquare(square)
    if not square then return end
    local objects = square.getObjects and square:getObjects()
    if not objects then return end
    for i = 0, objects:size() - 1 do
        local object = objects:get(i)
        local info = select(1, getHeaterInfoByObject(object))
        if info then
            refreshHeaterAppearance(object, info)
        end
    end
end

if Events and Events.LoadGridsquare then
    Events.LoadGridsquare.Add(onRCLoadGridsquare)
end

function RCHeaters.getRuntimeForObject(object)
    local info = select(1, getHeaterInfoByObject(object))
    if not info then return nil end
    local defaultHeat = getHeatOutputForSetting(info, DEFAULT_HEAT_SETTING)
    local runtime = {
        info = info,
        active = false,
        heatValue = defaultHeat,
        remainingHours = 0,
    }

    local holder = resolveModDataHolder(object)
    local md = holder and holder.getModData and holder:getModData() or nil
    if not md then return runtime end

    if isPropaneHeater(info) then
        updateHeaterFuel(object, info)
        if not (md.RCPropaneTankAttached and md.RCPropaneTankDelta and md.RCPropaneTankDelta > 0 and md.RCPropaneActive) then
            return runtime
        end
        local drain = info.drainPerHour or 0
        local delta = clamp01(md.RCPropaneTankDelta or 0)
        local remaining = 0
        if drain > 0 then
            remaining = delta / drain
        elseif delta > 0 then
            remaining = math.huge
        end
        local heatOutput = getHeatOutputForSetting(info, _getHeatSettingFromModData(md))
        if heatOutput <= 0 then
            runtime.heatValue = 0
            runtime.active = false
            runtime.remainingHours = 0
            return runtime
        end
        runtime.heatValue = heatOutput
        local timerRemaining = getTimerRemainingHours(object)
        runtime.active = true
        if timerRemaining < math.huge then
            runtime.remainingHours = math.min(remaining, timerRemaining)
        else
            runtime.remainingHours = remaining
        end
        return runtime
    elseif isElectricHeater(info) then
        local active = md.RCElectricActive and hasElectricPowerAvailable(object, info)
        runtime.active = active and true or false
        if runtime.active then
            local heatOutput = getHeatOutputForSetting(info, _getHeatSettingFromModData(md))
            if heatOutput <= 0 then
                runtime.active = false
                runtime.remainingHours = 0
                runtime.heatValue = 0
                return runtime
            end
            runtime.heatValue = heatOutput
            local timerRemaining = getTimerRemainingHours(object)
            if timerRemaining < math.huge then
                runtime.remainingHours = timerRemaining
            else
                runtime.remainingHours = math.huge
            end
        end
        return runtime
    end

    return runtime
end

function RCHeaters.getHeatSetting(object)
    return getHeatSetting(object)
end

function RCHeaters.setHeatSetting(object, value)
    setHeatSetting(object, value)
end

function RCHeaters.getTimerSettingSeconds(object)
    return getTimerSettingSeconds(object)
end

function RCHeaters.setTimerSettingSeconds(object, info, value)
    setTimerSettingSeconds(object, info, value)
end

function RCHeaters.getHeatSettingOptions()
    local options = {}
    for _, id in ipairs(HEAT_SETTING_ORDER) do
        local def = HEAT_SETTING_INFO[id]
        if def then
            local markerOffset = def.markerOffset
            if markerOffset then
                markerOffset = { x = markerOffset.x or 0, y = markerOffset.y or 0 }
            end

            options[#options + 1] = {
                id = id,
                name = def.name,
                text = getHeatSettingDisplayText(id),
                knobAngle = def.knobAngle,
                angle = def.knobAngle,
                markerAngle = def.markerAngle,
                markerRadius = def.markerRadius,
                markerRadiusOffset = def.markerRadiusOffset,
                markerOffsetX = def.markerOffsetX,
                markerOffsetY = def.markerOffsetY,
                markerOffset = markerOffset,
            }
        end
    end
    return options
end

function RCHeaters.getHeatSettingDisplayText(settingId)
    return getHeatSettingDisplayText(settingId)
end

function RCHeaters.getDefaultHeatSetting()
    return DEFAULT_HEAT_SETTING
end

function RCHeaters.getTimerRunningForSeconds(object)
    return getTimerRunningForSeconds(object)
end

function RCHeaters.openHeaterUI(playerObj, object, info)
    openHeaterUI(playerObj, object, info)
end

function RCHeaters.createHeaterController(object, info, playerObj)
    return createHeaterController(object, info, playerObj)
end

return RCHeaters
