--***********************************************************
--**                REALISTIC COLD - HEATER UI             **
--***********************************************************

require "ISUI/ISPanelJoypad"
require "ISUI/ISLabel"

local FONT_HGT_SMALL = getTextManager():getFontHeight(UIFont.Small)

local TEMP_MARKER_OFFSETS = {
    low = { x = 20, y = 15 },
    medium = { x = 10, y = 20 },
    high = { x = 0, y = 8 },
}

RCHeaterUI = ISPanelJoypad:derive("RCHeaterUI")
RCHeaterUI.instance = {}

function RCHeaterUI:initialise()
    ISPanel.initialise(self)

    local btnWid = 100
    local btnHgt = math.max(25, FONT_HGT_SMALL + 3 * 2)
    local padBottom = 10

    self.tempKnob = ISKnob:new(20, 40, self.knobTex, getTexture("media/ui/Knobs/KnobBGHeater.png"), getText("IGUI_Temperature"), self.playerObj)
    self.tempKnob:initialise()
    self.tempKnob:instantiate()
    -- The default knob background texture includes Fahrenheit numbers (250, 300, 350, etc.).
    -- Override the render function so we can keep the original positioning while skipping that
    -- texture draw, leaving only the Low/Medium/High markers we add below.
    function self.tempKnob:render()
        self:drawTextCentre(self.title, self.width / 2, 0, 1, 1, 1, 1, UIFont.Small)

        if self.valuesBg and not self.skipBackgroundTexture then
            if self.joypadFocused then
                self:drawTexture(self.valuesBg, 0, 20, 1, 1, 1, 1)
            else
                self:drawTexture(self.valuesBg, 0, 20, 1, 0.85, 0.85, 0.85)
            end
        end

        if self.selected > 0 and self.values[self.selected] then
            local centerX = self.width / 2
            local centerY = self.height / 2
            if self.valuesBg then
                centerY = centerY + 20
            end
            self:DrawTextureAngle(self.tex, centerX, centerY, self.values[self.selected].angle)
        end

        ISUIElement.render(self)
    end
    self.tempKnob.onMouseUpFct = RCHeaterUI.ChangeKnob
    self.tempKnob.target = self
    self.tempKnob.switchSound = "ToggleTemp"
    self:addChild(self.tempKnob)
    self.tempKnobMarkers = {}

    local texBG = getTexture("media/ui/Knobs/KnobBGOvenTimer.png")
    self.timerKnob = ISKnob:new(self.width - 20 - texBG:getWidthOrig(), 40, self.knobTex, texBG, getText("IGUI_Timer"), self.playerObj)
    self.timerKnob:initialise()
    self.timerKnob:instantiate()
    self.timerKnob.onMouseUpFct = RCHeaterUI.ChangeKnob
    self.timerKnob.target = self
    self:addChild(self.timerKnob)

    local buttonTop = self.tempKnob:getBottom() + 10
    self.ok = ISButton:new(10, buttonTop, btnWid, btnHgt, getText("ContextMenu_Turn_On"), self, RCHeaterUI.onClick)
    self.ok.internal = "OK"
    self.ok:initialise()
    self.ok:instantiate()
    self.ok.borderColor = { r = 1, g = 1, b = 1, a = 0.1 }
    if self.ok.setSound then
        self.ok:setSound("activate", "ToggleStove")
    end
    self:addChild(self.ok)

    self.closeButton = ISButton:new(self:getWidth() - btnWid - 10, buttonTop, btnWid, btnHgt, getText("UI_Close"), self, RCHeaterUI.onClick)
    self.closeButton.internal = "CLOSE"
    self.closeButton:initialise()
    self.closeButton:instantiate()
    self.closeButton.borderColor = { r = 1, g = 1, b = 1, a = 0.1 }
    self:addChild(self.closeButton)

    self:setHeight(self.closeButton:getBottom() + padBottom)

    self:addKnobValues()
    self:updateButtons()

    self:insertNewLineOfButtons(self.tempKnob, self.timerKnob)
    self:insertNewLineOfButtons(self.ok, self.closeButton)
end

function RCHeaterUI:ChangeKnob()
    if not self.controller then return end
    if self.tempKnob then
        self.controller:setHeatSetting(self.tempKnob:getValue())
    end
    if self.timerKnob then
        self.controller:setTimerSettingSeconds(self.timerKnob:getValue() * 60)
    end
end

function RCHeaterUI:update()
    ISPanelJoypad.update(self)
    self:updateButtons()

    if not self.controller or not self.controller:isValid() then
        self:close()
        return
    end

    local x, y = self.controller:getWorldPosition()
    if self.playerObj and self.controller:isActive() then
        local px, py = self.playerObj:getX(), self.playerObj:getY()
        local dx, dy = x - px, y - py
        local distSq = dx * dx + dy * dy
        if distSq > (3 * 3) then
            self:close()
            return
        end
    end
end

function RCHeaterUI:updateButtons()
    if not self.controller then return end

    local totalSeconds = self.controller:getTimerSettingSeconds() or 0
    local runningSeconds = self.controller:getTimerRunningForSeconds() or 0

    if self.timerKnob and not self.timerKnob.dragging then
        if runningSeconds > 0 and runningSeconds < totalSeconds then
            local minutesRemaining = math.ceil((totalSeconds - runningSeconds) / 60)
            if minutesRemaining < 0 then minutesRemaining = 0 end
            self.timerKnob:setKnobPosition(minutesRemaining)
        else
            self.timerKnob:setKnobPosition(totalSeconds / 60)
        end
    end

    if self.tempKnob and not self.tempKnob.dragging then
        self.tempKnob:setKnobPosition(self.controller:getHeatSetting())
    end

    if self.controller:isActive() then
        self.ok:setTitle(getText("ContextMenu_Turn_Off"))
    else
        self.ok:setTitle(getText("ContextMenu_Turn_On"))
    end

    local canToggle = self.controller:isActive() or self.controller:canTurnOn()
    self.ok:setEnable(canToggle)
end

function RCHeaterUI:addKnobValues()
    self.heatSettingOptions = {}
    local options = self.controller and self.controller.getHeatSettingOptions and self.controller:getHeatSettingOptions() or {}
    local count = #options
    if count == 0 then
        options = {
            { id = self.controller and self.controller:getHeatSetting() or 1 },
        }
        count = #options
    end

    local startAngle = 45
    local endAngle = 255
    local step = 0
    if count > 1 then
        step = (endAngle - startAngle) / (count - 1)
    end

    self:clearTempKnobMarkers()
    for index, option in ipairs(options) do
        local optionData = {}
        for key, optionValue in pairs(option) do
            optionData[key] = optionValue
        end

        local value = optionData.id or optionData.value or index
        if not optionData.text and self.controller and self.controller.getHeatSettingDisplayText then
            optionData.text = self.controller:getHeatSettingDisplayText(value)
        end

        local angle = optionData.angle or optionData.knobAngle
        if not angle then
            angle = startAngle + (index - 1) * step
        end

        local markerAngle = optionData.markerAngle or angle

        self.tempKnob:addValue(angle, value)

        optionData.angle = angle
        optionData.knobAngle = angle
        optionData.markerAngle = markerAngle

        self.heatSettingOptions[value] = optionData
        self:addTempKnobMarker(markerAngle, optionData.text, optionData)
    end

    self.timerKnob:addValue(0, 0)
    self.timerKnob:addValue(18, 1)
    self.timerKnob:addValue(36, 2)
    self.timerKnob:addValue(54, 3)
    self.timerKnob:addValue(72, 4)
    self.timerKnob:addValue(90, 5)
    self.timerKnob:addValue(108, 10)
    self.timerKnob:addValue(126, 15)
    self.timerKnob:addValue(144, 20)
    self.timerKnob:addValue(162, 25)
    self.timerKnob:addValue(180, 30)
    self.timerKnob:addValue(198, 40)
    self.timerKnob:addValue(216, 50)
    self.timerKnob:addValue(234, 60)
    self.timerKnob:addValue(252, 90)
    self.timerKnob:addValue(270, 120)
end

function RCHeaterUI:clearTempKnobMarkers()
    if not self.tempKnobMarkers then return end
    for _, label in ipairs(self.tempKnobMarkers) do
        self:removeChild(label)
    end
    self.tempKnobMarkers = {}
end

function RCHeaterUI:addTempKnobMarker(angle, text, option)
    if not text or text == "" then return end
    if not self.tempKnob then return end

    local label = ISLabel:new(0, 0, FONT_HGT_SMALL, text, 1, 1, 1, 1, UIFont.Small, true)
    label.center = true
    label:setWidthToName(FONT_HGT_SMALL)
    label:setHeightToFont(FONT_HGT_SMALL)

    local knobCenterX = self.tempKnob.x + self.tempKnob.width / 2
    local knobCenterY = self.tempKnob.y + self.tempKnob.height / 2
    if self.tempKnob.valuesBg then
        knobCenterY = knobCenterY + 10
    end

    local radius = math.min(self.tempKnob.width, self.tempKnob.height) / 2 - FONT_HGT_SMALL
    if option then
        if option.markerRadius then
            radius = option.markerRadius
        elseif option.markerRadiusOffset then
            radius = radius + option.markerRadiusOffset
        end
    end
    if radius < 0 then radius = 0 end

    local radians = math.rad(angle)
    local offsetX = math.sin(radians) * radius
    local offsetY = -math.cos(radians) * radius
    if option then
        local markerOffset = option.markerOffset
        local addX = option.markerOffsetX or (markerOffset and markerOffset.x) or 0
        local addY = option.markerOffsetY or (markerOffset and markerOffset.y) or 0
        offsetX = offsetX + addX
        offsetY = offsetY + addY
    end

    if option and option.name then
        local name = tostring(option.name):lower()
        local configuredOffset = TEMP_MARKER_OFFSETS[name]
        if configuredOffset then
            offsetX = offsetX + (configuredOffset.x or 0)
            offsetY = offsetY + (configuredOffset.y or 0)
        end
    end

    local x = knobCenterX + offsetX - label.width / 2
    local y = knobCenterY + offsetY - label.height / 2

    label:setX(x)
    label:setY(y)
    label.originalX = x
    label.originalY = y

    self:addChild(label)
    self.tempKnobMarkers = self.tempKnobMarkers or {}
    table.insert(self.tempKnobMarkers, label)
end

function RCHeaterUI:onClick(button)
    if button.internal == "CLOSE" then
        self:close()
        return
    end
    if button.internal == "OK" and self.controller then
        if self.controller:isActive() then
            self.controller:setActive(false)
        elseif self.controller:canTurnOn() then
            self.controller:setActive(true)
        end
        self:updateButtons()
    end
end

function RCHeaterUI:close()
    self:removeFromUIManager()
    self:setVisible(false)
    local player = self.playerNum or 0
    if RCHeaterUI.instance[player + 1] == self then
        RCHeaterUI.instance[player + 1] = nil
    end
    if JoypadState and JoypadState.players and JoypadState.players[player + 1] and self.prevFocus then
        setJoypadFocus(player, self.prevFocus)
    end
end

function RCHeaterUI:onGainJoypadFocus(joypadData)
    ISPanelJoypad.onGainJoypadFocus(self, joypadData)
    self.joypadIndexY = 1
    self.joypadIndex = 1
    self.joypadButtons = self.joypadButtonsY[self.joypadIndexY]
    if self.joypadButtons and self.joypadButtons[self.joypadIndex] then
        self.joypadButtons[self.joypadIndex]:setJoypadFocused(true)
    end
    self:setISButtonForA(self.ok)
    self:setISButtonForB(self.closeButton)
end

function RCHeaterUI:onJoypadDown(button)
    ISPanelJoypad.onJoypadDown(self, button)
    if button == Joypad.AButton then
        self:onClick(self.ok)
    elseif button == Joypad.BButton then
        self:onClick(self.closeButton)
    end
end

function RCHeaterUI:new(x, y, width, height, controller, character)
    local o = ISPanelJoypad:new(x, y, width, height)
    setmetatable(o, self)
    self.__index = self

    local player = character:getPlayerNum()
    if y == 0 then
        o.y = getPlayerScreenTop(player) + (getPlayerScreenHeight(player) - height) / 2
        o:setY(o.y)
    end
    if x == 0 then
        o.x = getPlayerScreenLeft(player) + (getPlayerScreenWidth(player) - width) / 2
        o:setX(o.x)
    end

    o.backgroundColor.a = 0.75
    o.width = width
    o.height = height
    o.playerObj = character
    o.character = character
    o.playerNum = player
    o.controller = controller
    o.knobTex = getTexture("media/ui/Knobs/KnobDial.png")
    o.anchorLeft = true
    o.anchorRight = true
    o.anchorTop = true
    o.anchorBottom = true
    o.moveWithMouse = true

    RCHeaterUI.instance[player + 1] = o

    return o
end

return RCHeaterUI
