require "ISUI/ISUIElement"
require "ISUI/ISAlarmClockDialog"

RC_DigitalWatchUI = ISUIElement:derive("RC_DigitalWatchUI")

local DISPLAY_COLOR = { r = 100 / 255, g = 200 / 255, b = 210 / 255, a = 1 }
local GHOST_COLOR   = { r = 40 / 255,  g = 40 / 255,  b = 40 / 255,  a = 128 / 255 }
local MARGIN_RIGHT  = 10
local TOP_PADDING   = 10

function RC_DigitalWatchUI:new(x, y)
    local width = 1
    local height = 1
    local o = ISUIElement.new(self, x, y, width, height)
    o.anchorLeft = false
    o.anchorRight = true
    o.anchorTop = true
    o.anchorBottom = false

    o.bLargeTextures = false
    o.background = nil
    o.digitsLarge = {}
    o.digitsSmall = {}
    o.colon = nil
    o.slash = nil
    o.minus = nil
    o.dot = nil
    o.tempC = nil
    o.tempF = nil
    o.tempE = nil
    o.texAM = nil
    o.texPM = nil
    o.alarmOn = nil
    o.alarmRinging = nil
    o.alarmIconRect = nil

    o.uxOriginal = 0
    o.uyOriginal = 0
    o.largeDigitSpacing = 0
    o.smallDigitSpacing = 0
    o.colonSpacing = 0
    o.ampmSpacing = 0
    o.alarmBellSpacing = 0
    o.decimalSpacing = 0
    o.degreeSpacing = 0
    o.slashSpacing = 0
    o.tempDateSpacing = 0
    o.dateOffset = 0
    o.minusOffset = 0
    o.amVerticalSpacing = 0
    o.pmVerticalSpacing = 0
    o.alarmBellVerticalSpacing = 0
    o.displayVerticalSpacing = 0
    o.decimalVerticalSpacing = 0

    o.digital = false
    o.isAlarmSet = false
    o.isAlarmRinging = false
    o.clockPlayer = nil
    o.visibleState = false

    o.overrideTimeOfDay = nil
    o.overrideTemperature = nil
    o.overrideDate = nil
    o.overrideDigital = nil
    o.overrideVisible = nil
    o.overrideAlarmSet = nil
    o.overrideAlarmRinging = nil

    o.vanillaClockHidden = false
    o.vanillaClockWasVisible = nil

    o.lastScreenWidth = nil
    o.lastScreenHeight = nil
    o.lastClockSize = nil

    return o
end

function RC_DigitalWatchUI:initialise()
    ISUIElement.initialise(self)
    self:assignTextures(getCore():getOptionClockSize() == 2)
    self:updateLayout()
end

function RC_DigitalWatchUI:update()
    ISUIElement.update(self)

    local uiVisible = self:isGlobalUIVisible()
    if self.overrideVisible ~= nil then
        uiVisible = uiVisible and self.overrideVisible
    end

    self:updateClockState()

    self:setVisible(uiVisible and self.visibleState)
    if self.background then
        self:setWidth(self.background:getWidth())
        self:setHeight(self.background:getHeight())
    end

    self:updateVanillaClockVisibility(self:isVisible())

    local core = getCore()
    local screenWidth = core and core:getScreenWidth() or 0
    local screenHeight = core and core:getScreenHeight() or 0
    local clockSize = core and core:getOptionClockSize() or nil

    if screenWidth ~= self.lastScreenWidth
            or screenHeight ~= self.lastScreenHeight
            or clockSize ~= self.lastClockSize then
        self:updateLayout()
    end
end

function RC_DigitalWatchUI:render()
    if not self:isVisible() or not self.background then
        return
    end

    self:assignTextures(getCore():getOptionClockSize() == 2)
    self:drawTexture(self.background, 0, 0, 0.75, 1, 1, 1)
    self:renderDisplay(true, GHOST_COLOR)
    self:renderDisplay(false, DISPLAY_COLOR)
    ISUIElement.render(self)
end

function RC_DigitalWatchUI:renderDisplay(ghost, colour)
    local x = self.uxOriginal
    local y = self.uyOriginal
    local timeDigits = self:timeDigits()

    for index = 0, 3 do
        local digitIndex = ghost and 8 or timeDigits[index]
        self:drawTinted(self.digitsLarge[digitIndex], x, y, colour)
        x = x + self.digitsLarge[0]:getWidth()
        if index == 1 then
            x = x + self.colonSpacing
            self:drawTinted(self.colon, x, y, colour)
            x = x + self.colon:getWidth() + self.colonSpacing
        elseif index < 3 then
            x = x + self.largeDigitSpacing
        end
    end

    x = x + self.ampmSpacing
    if not getCore():getOptionClock24Hour() or ghost then
        if ghost then
            self:drawTinted(self.texAM, x, y + self.amVerticalSpacing, colour)
            self:drawTinted(self.texPM, x, y + self.pmVerticalSpacing, colour)
        else
            if self:isAM() then
                self:drawTinted(self.texAM, x, y + self.amVerticalSpacing, colour)
            else
                self:drawTinted(self.texPM, x, y + self.pmVerticalSpacing, colour)
            end
        end
    end

    local alarmX = x + self.texAM:getWidth() + self.alarmBellSpacing
    local alarmTexture = self.alarmOn
    self.alarmIconRect = {
                x = alarmX,
                y = y + self.alarmBellVerticalSpacing,
                w = alarmTexture:getWidth(),
                h = alarmTexture:getHeight(),
            }
    if self.isAlarmRinging or ghost then
        self:drawTinted(self.alarmRinging, alarmX, y + self.alarmBellVerticalSpacing, colour)
    elseif self.isAlarmSet then
        self:drawTinted(self.alarmOn, alarmX, y + self.alarmBellVerticalSpacing, colour)
    end

    if self.digital or ghost then
        x = self.uxOriginal
        y = y + self.digitsLarge[0]:getHeight() + self.displayVerticalSpacing
        if not self.clockPlayer then
            x = x + self.dateOffset
        else
            local tempDigits = self:tempDigits()
            if tempDigits[0] == 1 or ghost then
                self:drawTinted(self.minus, x, y, colour)
            end
            x = x + self.minusOffset

            if tempDigits[1] == 1 or ghost then
                self:drawTinted(self.digitsSmall[1], x, y, colour)
            end
            x = x + self.digitsSmall[0]:getWidth() + self.smallDigitSpacing

            for index = 2, 4 do
                local digitIndex = ghost and 8 or tempDigits[index]
                self:drawTinted(self.digitsSmall[digitIndex], x, y, colour)
                x = x + self.digitsSmall[0]:getWidth()
                if index == 3 then
                    x = x + self.decimalSpacing
                    self:drawTinted(self.dot, x, y + self.decimalVerticalSpacing, colour)
                    x = x + self.dot:getWidth() + self.decimalSpacing
                elseif index < 4 then
                    x = x + self.smallDigitSpacing
                end
            end

            x = x + self.degreeSpacing
            self:drawTinted(self.dot, x, y, colour)
            x = x + self.dot:getWidth() + self.degreeSpacing

            if ghost then
                self:drawTinted(self.tempE, x, y, colour)
            elseif tempDigits[5] == 0 then
                self:drawTinted(self.tempC, x, y, colour)
            else
                self:drawTinted(self.tempF, x, y, colour)
            end
            x = x + self.digitsSmall[0]:getWidth() + self.tempDateSpacing
        end

        local dateDigits = self:dateDigits()
        for index = 0, 3 do
            local digitIndex = ghost and 8 or dateDigits[index]
            self:drawTinted(self.digitsSmall[digitIndex], x, y, colour)
            x = x + self.digitsSmall[0]:getWidth()
            if index == 1 then
                x = x + self.slashSpacing
                self:drawTinted(self.slash, x, y, colour)
                x = x + self.slash:getWidth() + self.slashSpacing
            elseif index < 3 then
                x = x + self.smallDigitSpacing
            end
        end
    end
end

function RC_DigitalWatchUI:drawTinted(texture, x, y, colour)
    if not texture then return end
    self:drawTexture(texture, x, y, colour.a, colour.r, colour.g, colour.b)
end

function RC_DigitalWatchUI:isAM()
    local time = self:getDisplayTimeOfDay()
    return time < 12.0
end

function RC_DigitalWatchUI:getDisplayTimeOfDay()
    if self.overrideTimeOfDay ~= nil then
        return self.overrideTimeOfDay
    end
    return getGameTime():getTimeOfDay()
end

function RC_DigitalWatchUI:getDisplayTemperature()
    if self.overrideTemperature ~= nil then
        return self.overrideTemperature
    end
    if not self.clockPlayer then
        return nil
    end
    local temperature

    if RC_TempSim and RC_TempSim.getSimulatedTemperatureForPlayer then
        local ok, value = pcall(function()
            return RC_TempSim.getSimulatedTemperatureForPlayer(self.clockPlayer)
        end)
        if ok then
            temperature = value
        end
    end

    if temperature == nil then
        temperature = getClimateManager():getAirTemperatureForCharacter(self.clockPlayer, false)
    end

    return temperature
end

function RC_DigitalWatchUI:getDisplayDate()
    if self.overrideDate ~= nil then
        return self.overrideDate
    end
    local gameTime = getGameTime()
    return { day = gameTime:getDay() + 1, month = gameTime:getMonth() + 1 }
end

function RC_DigitalWatchUI:timeDigits()
    local timeOfDay = self:getDisplayTimeOfDay()
    if timeOfDay < 0 then timeOfDay = 0 end
    local use24 = getCore():getOptionClock24Hour()
    if not use24 then
        if timeOfDay >= 13 then
            timeOfDay = timeOfDay - 12
        end
        if timeOfDay < 1 then
            timeOfDay = timeOfDay + 12
        end
    end
    local hours = math.floor(timeOfDay)
    local minutes = math.floor((timeOfDay - hours) * 60 + 0.5)
    if minutes >= 60 then
        minutes = minutes - 60
        hours = (hours + 1) % (use24 and 24 or 12)
        if not use24 and hours == 0 then hours = 12 end
    end

    local tensHour = math.floor(hours / 10)
    local onesHour = hours % 10
    local tensMinute = math.floor(minutes / 10)
    local onesMinute = minutes % 10

    self.cachedTimeDigits = self.cachedTimeDigits or {}
    self.cachedTimeDigits[0] = tensHour
    self.cachedTimeDigits[1] = onesHour
    self.cachedTimeDigits[2] = tensMinute
    self.cachedTimeDigits[3] = onesMinute
    return self.cachedTimeDigits
end

function RC_DigitalWatchUI:dateDigits()
    local date = self:getDisplayDate()
    local day = date.day
    local month = date.month
    local dayTens = math.floor(day / 10)
    local dayOnes = day % 10
    local monthTens = math.floor(month / 10)
    local monthOnes = month % 10
    local format = getCore():getOptionClockFormat()

    self.cachedDateDigits = self.cachedDateDigits or {}
    if format == 1 then
        self.cachedDateDigits[0] = monthTens
        self.cachedDateDigits[1] = monthOnes
        self.cachedDateDigits[2] = dayTens
        self.cachedDateDigits[3] = dayOnes
    else
        self.cachedDateDigits[0] = dayTens
        self.cachedDateDigits[1] = dayOnes
        self.cachedDateDigits[2] = monthTens
        self.cachedDateDigits[3] = monthOnes
    end
    return self.cachedDateDigits
end

function RC_DigitalWatchUI:tempDigits()
    self.cachedTempDigits = self.cachedTempDigits or {}
    local temperature = self:getDisplayTemperature()
    if temperature == nil then
        self.cachedTempDigits[0] = 0
        self.cachedTempDigits[1] = 0
        self.cachedTempDigits[2] = 0
        self.cachedTempDigits[3] = 0
        self.cachedTempDigits[4] = 0
        self.cachedTempDigits[5] = 0
        return self.cachedTempDigits
    end

    local unit = 0
    if not getCore():getOptionTemperatureDisplayCelsius() then
        temperature = temperature * 1.8 + 32.0
        unit = 1
    end

    local negative = 0
    if temperature < 0 then
        negative = 1
        temperature = -temperature
    end

    local hundreds = math.floor(temperature / 100)
    local tens = math.floor(temperature % 100 / 10)
    local ones = math.floor(temperature % 10)
    local tenths = math.floor(temperature * 10) % 10

    self.cachedTempDigits[0] = negative
    self.cachedTempDigits[1] = hundreds
    self.cachedTempDigits[2] = tens
    self.cachedTempDigits[3] = ones
    self.cachedTempDigits[4] = tenths
    self.cachedTempDigits[5] = unit
    return self.cachedTempDigits
end

function RC_DigitalWatchUI:updateClockState()
    self.visibleState = false
    self.digital = false
    self.isAlarmSet = false
    self.isAlarmRinging = false
    self.clockPlayer = nil

    local digitalOverride = self.overrideDigital
    local alarmSetOverride = self.overrideAlarmSet
    local alarmRingingOverride = self.overrideAlarmRinging

    if getSpecificPlayer(0) ~= nil then
        for playerIndex = 0, getNumActivePlayers() - 1 do
            local player = getSpecificPlayer(playerIndex)
            if player and not player:isDead() then
                if self:evaluateClockItems(player, player:getWornItems()) then
                    break
                end
                if self.clockPlayer then
                    break
                end
                local items = player:getInventory():getItems()
                for i = 0, items:size() - 1 do
                    local item = items:get(i)
                    if (item:isWorn() or item:isEquipped()) and self:isClockItem(item) then
                        self:applyClockItemState(player, item)
                    end
                end
                if self.clockPlayer then
                    break
                end
            end
        end
    end

    local forcedVisible = false
    if getDebug() and getDebugOptions():getBoolean("Cheat.Clock.Visible") then
        forcedVisible = true
        self.digital = true
    end

    if digitalOverride ~= nil then
        self.digital = digitalOverride
    end
    if alarmSetOverride ~= nil then
        self.isAlarmSet = alarmSetOverride
    end
    if alarmRingingOverride ~= nil then
        self.isAlarmRinging = alarmRingingOverride
    end

    local shouldBeVisible = false
    if self.clockPlayer then
        shouldBeVisible = self.digital
    end
    if forcedVisible then
        shouldBeVisible = true
    end

    self.visibleState = shouldBeVisible

    if self.overrideVisible ~= nil then
        self.visibleState = self.overrideVisible
    end

    self:assignTextures(getCore():getOptionClockSize() == 2)
end

function RC_DigitalWatchUI:evaluateClockItems(player, wornItems)
    for i = 0, wornItems:size() - 1 do
        local item = wornItems:getItemByIndex(i)
        if self:isClockItem(item) then
            self:applyClockItemState(player, item)
        end
    end
    return self.clockPlayer ~= nil
end

function RC_DigitalWatchUI:isClockItem(item)
    if not item then return false end
    return instanceof(item, "AlarmClock") or instanceof(item, "AlarmClockClothing")
end

function RC_DigitalWatchUI:applyClockItemState(player, item)
    if not self.clockPlayer then
        self.clockPlayer = player
    end
    if item.hasTag and item:hasTag("Digital") then
        self.digital = true
    elseif item.isDigital and item:isDigital() then
        self.digital = true
    end

    if item.isAlarmSet and item:isAlarmSet() then
        self.isAlarmSet = true
    end
    if item.isRinging and item:isRinging() then
        self.isAlarmRinging = true
    end
end

function RC_DigitalWatchUI:updateVanillaClockVisibility(shouldHide)
    if not UIManager or not UIManager.getClock then return end

    local vanillaClock = UIManager.getClock()
    if not vanillaClock then return end

    if shouldHide then
        if not self.vanillaClockHidden then
            self.vanillaClockWasVisible = vanillaClock:isVisible()
            self.vanillaClockHidden = true
        end

        if vanillaClock:isVisible() then
            UIManager.RemoveElement(vanillaClock)
        end
    elseif self.vanillaClockHidden then
        self.vanillaClockHidden = false
        if self.vanillaClockWasVisible == nil or self.vanillaClockWasVisible then
            vanillaClock:setVisible(true)
        end
        vanillaClock:resize()
        self.vanillaClockWasVisible = nil
    end
end

function RC_DigitalWatchUI:restoreVanillaClock()
    if self.vanillaClockHidden then
        self:updateVanillaClockVisibility(false)
    end
end

function RC_DigitalWatchUI:assignTextures(useLarge)
    if (not self.background) or self.bLargeTextures ~= useLarge then
        self.bLargeTextures = useLarge
        local backgroundPath = useLarge and "media/ui/ClockAssets/ClockLargeBackground.png" or "media/ui/ClockAssets/ClockSmallBackground.png"
        self.background = getTexture(backgroundPath)
        local largeSuffix = useLarge and "Large" or "Medium"
        local smallSuffix = useLarge and "Medium" or "Small"

        if useLarge then
            self:assignLargeOffsets()
        else
            self:assignSmallOffsets()
        end

        for i = 0, 9 do
            self.digitsLarge[i] = getTexture("media/ui/ClockAssets/ClockDigits" .. largeSuffix .. i .. ".png")
            self.digitsSmall[i] = getTexture("media/ui/ClockAssets/ClockDigits" .. smallSuffix .. i .. ".png")
        end
        self.colon = getTexture("media/ui/ClockAssets/ClockDivide" .. largeSuffix .. ".png")
        self.slash = getTexture("media/ui/ClockAssets/DateDivide" .. smallSuffix .. ".png")
        self.minus = getTexture("media/ui/ClockAssets/ClockDigits" .. smallSuffix .. "Minus.png")
        self.dot = getTexture("media/ui/ClockAssets/ClockDigits" .. smallSuffix .. "Dot.png")
        self.tempC = getTexture("media/ui/ClockAssets/ClockDigits" .. smallSuffix .. "C.png")
        self.tempF = getTexture("media/ui/ClockAssets/ClockDigits" .. smallSuffix .. "F.png")
        self.tempE = getTexture("media/ui/ClockAssets/ClockDigits" .. smallSuffix .. "E.png")
        self.texAM = getTexture("media/ui/ClockAssets/ClockAm" .. largeSuffix .. ".png")
        self.texPM = getTexture("media/ui/ClockAssets/ClockPm" .. largeSuffix .. ".png")
        self.alarmOn = getTexture("media/ui/ClockAssets/ClockAlarm" .. largeSuffix .. "Set.png")
        self.alarmRinging = getTexture("media/ui/ClockAssets/ClockAlarm" .. largeSuffix .. "Sound.png")
    end

    if self.background then
        self:setWidth(self.background:getWidth())
        self:setHeight(self.background:getHeight())
    end
end

function RC_DigitalWatchUI:assignSmallOffsets()
    self.uxOriginal = 3
    self.uyOriginal = 3
    self.largeDigitSpacing = 1
    self.smallDigitSpacing = 1
    self.colonSpacing = 1
    self.ampmSpacing = 1
    self.alarmBellSpacing = 1
    self.decimalSpacing = 1
    self.degreeSpacing = 1
    self.slashSpacing = 1
    self.tempDateSpacing = 5
    self.dateOffset = 33
    self.minusOffset = 0
    self.amVerticalSpacing = 7
    self.pmVerticalSpacing = 12
    self.alarmBellVerticalSpacing = 1
    self.displayVerticalSpacing = 2
    self.decimalVerticalSpacing = 6
end

function RC_DigitalWatchUI:assignLargeOffsets()
    self.uxOriginal = 3
    self.uyOriginal = 3
    self.largeDigitSpacing = 2
    self.smallDigitSpacing = 1
    self.colonSpacing = 3
    self.ampmSpacing = 3
    self.alarmBellSpacing = 5
    self.decimalSpacing = 2
    self.degreeSpacing = 2
    self.slashSpacing = 2
    self.tempDateSpacing = 8
    self.dateOffset = 65
    self.minusOffset = -2
    self.amVerticalSpacing = 15
    self.pmVerticalSpacing = 25
    self.alarmBellVerticalSpacing = 1
    self.displayVerticalSpacing = 5
    self.decimalVerticalSpacing = 15
end

function RC_DigitalWatchUI:updateLayout()
    if not self.background then return end

    local core = getCore()
    local screenWidth = core and core:getScreenWidth() or 0
    local screenHeight = core and core:getScreenHeight() or 0
    local clockSize = core and core:getOptionClockSize() or nil

    local vanillaClock = nil
    if UIManager and UIManager.getClock then
        vanillaClock = UIManager.getClock()
    end

    local x = nil
    local y = TOP_PADDING

    if vanillaClock then
        local vanillaRight = vanillaClock.getRight and vanillaClock:getRight() or nil
        y = vanillaClock:getY()

        local vanillaWidth = vanillaClock.getWidth and vanillaClock:getWidth() or nil
        local vanillaHeight = vanillaClock.getHeight and vanillaClock:getHeight() or nil
        if vanillaWidth and vanillaWidth > 0 then
            self:setWidth(vanillaWidth)
        else
            self:setWidth(self.background:getWidth())
        end
        if vanillaHeight and vanillaHeight > 0 then
            self:setHeight(vanillaHeight)
        else
            self:setHeight(self.background:getHeight())
        end
        if vanillaRight then
            local targetRight = screenWidth - MARGIN_RIGHT
            if math.abs(vanillaRight - targetRight) <= 1 then
                x = vanillaClock:getX()
            else
                x = targetRight - self:getWidth()
            end
        else
            x = screenWidth - self:getWidth() - MARGIN_RIGHT
        end
    else
        self:setWidth(self.background:getWidth())
        self:setHeight(self.background:getHeight())
        x = screenWidth - self:getWidth() - MARGIN_RIGHT
    end

    if not x then
        x = screenWidth - self:getWidth() - MARGIN_RIGHT
    end

    self:setX(x)
    self:setY(y)

    self.lastScreenWidth = screenWidth
    self.lastScreenHeight = screenHeight
    self.lastClockSize = clockSize
end

function RC_DigitalWatchUI:setTimeOverride(timeOfDay)
    self.overrideTimeOfDay = timeOfDay
end

function RC_DigitalWatchUI:setTemperatureOverride(temp)
    self.overrideTemperature = temp
end

function RC_DigitalWatchUI:setDateOverride(day, month)
    if day and month then
        self.overrideDate = { day = day, month = month }
    else
        self.overrideDate = nil
    end
end

function RC_DigitalWatchUI:getAlarmItem(player)
    local wornItems = player:getWornItems()
    if wornItems then
        for i = 0, wornItems:size() - 1 do
            local wornItem = wornItems:get(i)
            if wornItem then
                local item = wornItem:getItem()
                if item.isDigital and item:isDigital() then
                    return item
                end
            end
        end
    end

    return nil
end

function RC_DigitalWatchUI:setDigitalOverride(value)
    self.overrideDigital = value
end

function RC_DigitalWatchUI:setVisibleOverride(value)
    self.overrideVisible = value
end

function RC_DigitalWatchUI:setAlarmSetOverride(value)
    self.overrideAlarmSet = value
end

function RC_DigitalWatchUI:setAlarmRingingOverride(value)
    self.overrideAlarmRinging = value
end

function RC_DigitalWatchUI:toggleAlarm()
    local alarm = RC_DigitalWatchUI:getAlarmItem(self.clockPlayer)
    if not alarm or not self.clockPlayer then
        return
    end

    if not alarm.isAlarmSet or not alarm.setAlarmSet then
        return
    end

    local newState = not alarm:isAlarmSet()
    alarm:setAlarmSet(newState)

    if not newState and alarm.stopRinging then
        alarm:stopRinging()
    end

    self.isAlarmSet = newState
    self.isAlarmRinging = newState and alarm.isRinging and alarm:isRinging() or false
end

function RC_DigitalWatchUI:isGlobalUIVisible()
    if ISUIHandler and ISUIHandler.allUIVisible ~= nil then
        return ISUIHandler.allUIVisible
    end
    return true
end

function RC_DigitalWatchUI:onMouseUp(x, y)
    ISUIElement.onMouseUp(self, x, y)

    if not self.alarmIconRect then
        return
    end

    local withinX = x >= self.alarmIconRect.x and x <= (self.alarmIconRect.x + self.alarmIconRect.w)
    local withinY = y >= self.alarmIconRect.y and y <= (self.alarmIconRect.y + self.alarmIconRect.h)
    if withinX and withinY then
        self:toggleAlarm()
    end
end


function RC_DigitalWatchUI:onRightMouseUp(x, y)
    ISUIElement.onRightMouseUp(self, x, y)

    if not self:isVisible() then
        return
    end

    local player = self.clockPlayer or getSpecificPlayer(0)
    if not player then
        return
    end

    local alarm = RC_DigitalWatchUI:getAlarmItem(player)
    if not alarm then
        return
    end

    local modal = ISAlarmClockDialog:new(0, 0, 230, 160, player:getPlayerNum(), alarm)
    modal:initialise()
    modal:addToUIManager()
end

return RC_DigitalWatchUI
