-- Auto Room Lights (B42)
-- Turns lights on when entering a room, off when leaving it.
-- Client-side only; safe for SP and MP clients.

if not AutoRoomLights then AutoRoomLights = {} end

-- Track the last room per local player index
AutoRoomLights.lastRoomForPlayer = {}

-- Utility: safely toggle/activate all switches in a given IsoRoom
local function setRoomLights(room, turnOn)
    if not room or not room.hasLightSwitches or not room:getLightSwitches() then return end
    local switches = room:getLightSwitches()
    for i = 0, switches:size() - 1 do
        local sw = switches:get(i)        -- IsoLightSwitch
        -- Prefer setActive to force a given state. Fallback to toggle if needed.
        -- setActive returns boolean indicating if a change occurred.
        local changed = false
        -- Some switches respect "canSwitchLight" (no power, no bulb, etc.).
        -- We'll try to honor that first; if it passes, do setActive.
        if sw:canSwitchLight() then
            -- setActive(active, setActiveBoolOnly=false, ignoreSwitchCheck=false)
            changed = sw:setActive(turnOn, false, false)
        end
        -- If nothing changed (e.g., canSwitchLight false but state differs), try the 3-arg override:
        if not changed and sw.setActive then
            -- ignoreSwitchCheck=true to force the visual/state even when power/bulb rules block it.
            -- Comment the next line if you prefer strict realism.
            changed = sw:setActive(turnOn, false, true)
        end
        -- Optional: ensure underlying light state flips even if already active bool matched
        -- by calling switchLight; but generally setActive suffices.
        -- sw:switchLight(turnOn)
    end
end

-- Get the IsoRoom the player is currently in (nil means outside)
local function getPlayerRoom(player)
    if not player then return nil end
    local sq = player:getCurrentSquare()
    if not sq then return nil end
    return sq:getRoom()    -- IsoRoom or nil when outside
end

-- Called whenever the player is walking; lighter than OnPlayerUpdate.
local function onPlayerMove(character)
    if not character or not instanceof(character, "IsoPlayer") then return end

    local plrIndex = character:getPlayerNum()
    local currentRoom = getPlayerRoom(character)
    local lastRoom = AutoRoomLights.lastRoomForPlayer[plrIndex]

    -- Early out if still in the same room (including both nil/outside)
    if currentRoom == lastRoom then return end

    -- If we left a room, turn its lights off
    if lastRoom and lastRoom ~= currentRoom then
        setRoomLights(lastRoom, false)
    end

    -- If we entered a room, turn its lights on
    if currentRoom and currentRoom ~= lastRoom then
        setRoomLights(currentRoom, true)
    end

    AutoRoomLights.lastRoomForPlayer[plrIndex] = currentRoom
end

-- Also handle the first spawn so the initial room lights up
local function onCreatePlayer(playerIndex, playerObj)
    AutoRoomLights.lastRoomForPlayer[playerIndex] = getPlayerRoom(playerObj)
    if AutoRoomLights.lastRoomForPlayer[playerIndex] then
        setRoomLights(AutoRoomLights.lastRoomForPlayer[playerIndex], true)
    end
end

-- Clear when a player dies/respawns (optional robustness)
local function onPlayerDeath(playerObj)
    if not playerObj then return end
    AutoRoomLights.lastRoomForPlayer[playerObj:getPlayerNum()] = nil
end

Events.OnPlayerMove.Add(onPlayerMove)        -- fires while walking :contentReference[oaicite:0]{index=0}
Events.OnCreatePlayer.Add(onCreatePlayer)
Events.OnPlayerDeath.Add(onPlayerDeath)
