-- Trapping Tuner & Sync Fix (server-side core)
-- TT_TrappingTuner.lua
--
-- What this file does:
--   • Canonical bait: use self.trapBait as source of truth; keep alias self.bait in sync.
--   • calculTrap(): pre-sync alias bait so vanilla can attempt catch even when far chunks aren't hydrated.
--   • checkForAnimal(): canonical-bait gate, ambient cold-gate, and optional "near player" allowance.
--   • testForAnimal(): fallback gate (no double gating when called from checkForAnimal).
--   • Scale / cap animalAliveHour accumulation.
--   • Config: sandbox vars loaded on game start and when changed (no early read), with safe logging.
--
-- Server-only.
if isClient() then return end

----------------------------------------------------------------
-- Defaults (used until sandbox loads; then overridden)
----------------------------------------------------------------
local DEFAULTS = {
    HourlyIncreaseMultiplier = 1,    -- scale the +1h vanilla increment per in-game hour
    CapSimulatedHours        = false,
    MaxSimulatedHours        = 0,
    ColdCaptureModEnabled    = false,
    ColdTempThresholdC       = 0,    -- °C threshold for cold-gate
    ColdCapturePercent       = 10,   -- 0..100
    AllowCatchNearPlayer     = false,-- allow catch even when player is near (after anti-exploit check)
    UseAmbientGlobalTemp     = true, -- use ClimateManager:getTemperature() (campfire-proof)
    VerboseLogging           = true,
}

----------------------------------------------------------------
-- Runtime config (start with defaults; refresh later)
----------------------------------------------------------------
local CFG = {}
for k, v in pairs(DEFAULTS) do CFG[k] = v end

----------------------------------------------------------------
-- Logging helper
----------------------------------------------------------------
local function TT_LOG(msg)
    if CFG.VerboseLogging then
        print(string.format("[TrappingTuner] %s", tostring(msg)))
    end
end

----------------------------------------------------------------
-- Read sandbox options safely (no early concatenation with booleans)
----------------------------------------------------------------
local function readCfg()
    local sv = (SandboxVars and SandboxVars.TrappingTuner) or {}
    local cfg = {}
    cfg.HourlyIncreaseMultiplier = tonumber(sv.HourlyIncreaseMultiplier) or DEFAULTS.HourlyIncreaseMultiplier
    cfg.CapSimulatedHours        = (sv.CapSimulatedHours ~= nil) and sv.CapSimulatedHours or DEFAULTS.CapSimulatedHours
    cfg.MaxSimulatedHours        = tonumber(sv.MaxSimulatedHours) or DEFAULTS.MaxSimulatedHours

    cfg.ColdCaptureModEnabled    = (sv.ColdCaptureModEnabled ~= nil) and sv.ColdCaptureModEnabled or DEFAULTS.ColdCaptureModEnabled
    cfg.ColdTempThresholdC       = (sv.ColdTempThresholdC ~= nil) and sv.ColdTempThresholdC or DEFAULTS.ColdTempThresholdC
    cfg.ColdCapturePercent       = (sv.ColdCapturePercent ~= nil) and sv.ColdCapturePercent or DEFAULTS.ColdCapturePercent

    cfg.AllowCatchNearPlayer     = (sv.AllowCatchNearPlayer ~= nil) and sv.AllowCatchNearPlayer or DEFAULTS.AllowCatchNearPlayer
    cfg.UseAmbientGlobalTemp     = (sv.UseAmbientGlobalTemp ~= nil) and sv.UseAmbientGlobalTemp or DEFAULTS.UseAmbientGlobalTemp

    cfg.VerboseLogging           = (sv.VerboseLogging ~= nil) and sv.VerboseLogging or DEFAULTS.VerboseLogging
    return cfg
end

local function TT_RefreshConfig(reason)
    local new = readCfg()
    for k, v in pairs(new) do CFG[k] = v end
    TT_LOG(string.format("SandboxVars loaded (%s): Cold=%s Thr=%s %%=%s Near=%s Ambient=%s Verbose=%s",
        tostring(reason),
        tostring(CFG.ColdCaptureModEnabled),
        tostring(CFG.ColdTempThresholdC),
        tostring(CFG.ColdCapturePercent),
        tostring(CFG.AllowCatchNearPlayer),
        tostring(CFG.UseAmbientGlobalTemp),
        tostring(CFG.VerboseLogging)))
end

-- Manual console command (optional): > TT_reloadSandbox()
function TT_reloadSandbox() TT_RefreshConfig("manual") end

----------------------------------------------------------------
-- Helpers: canonical bait, temperature, scaled hours
----------------------------------------------------------------

-- Canonical bait from trapBait; alias 'bait' is transient.
local function _canonicalBaitType(self)
    local tb = self and self.trapBait
    if tb ~= nil and tb ~= "" then return tb end
    return nil
end
local function _hasCanonicalBait(self)
    return _canonicalBaitType(self) ~= nil
end

-- Ambient/global temperature (°C). Campfire-proof and chunk-agnostic.
local function getAmbientTempC()
    local cm = getClimateManager and getClimateManager()
    if not cm then return nil end
    if CFG.UseAmbientGlobalTemp and cm.getTemperature then
        local ok, t = pcall(function() return cm:getTemperature() end)
        if ok and type(t) == "number" then return t end
    end
    -- Fallbacks (rare): try any global API exposed by build
    if cm.getAirTemperature then
        local ok, t = pcall(function() return cm:getAirTemperature() end)
        if ok and type(t) == "number" then return t end
    end
    return nil
end

-- Cold gate using ambient temperature (one check per attempt).
local function coldGateAllowsCatch(self)
    if not CFG.ColdCaptureModEnabled then
        return true
    end
    local tC = getAmbientTempC()
    if tC == nil then
        TT_LOG("coldgate -> temp=nil -> ALLOW (no source)")
        return true
    end
    if tC < (CFG.ColdTempThresholdC or 0) then
        local roll = ZombRand(100)
        local pass = roll < (CFG.ColdCapturePercent or 0)
        TT_LOG(string.format("coldgate -> temp=%.2f°C [src=ambient], thr=%.2f°C, %%=%d, roll=%d -> %s",
            tC, CFG.ColdTempThresholdC, CFG.ColdCapturePercent, roll, pass and "ALLOW" or "DENY"))
        return pass
    end
    TT_LOG(string.format("coldgate -> temp=%.2f°C [src=ambient], thr=%.2f°C -> warm enough -> ALLOW",
        tC, CFG.ColdTempThresholdC))
    return true
end

-- Scale/cap alive-hour increments (vanilla adds +1 per in-game hour when animal is present).
local function addAliveHoursWithRules(self, deltaHours)
    local mult = math.max(0, CFG.HourlyIncreaseMultiplier or 1)
    local inc  = math.floor((deltaHours or 0) * mult)

    local before = tonumber(self.animalAliveHour) or 0
    local after  = before + inc

    if CFG.CapSimulatedHours then
        local capHours = math.max(0, CFG.MaxSimulatedHours or 0)
        if after > capHours then after = capHours end
    end

    self.animalAliveHour = after
    TT_LOG(string.format("aliveHours: delta=%d mult=%.2f before=%d after=%d",
        deltaHours or -1, mult, before, after))
end

----------------------------------------------------------------
-- Patch vanilla (idempotent)
----------------------------------------------------------------
local function applyPatch()
    if not STrapGlobalObject then
        TT_LOG("STrapGlobalObject not yet available; will try again later.")
        return
    end

    -- A) Reset alive timer when clearing animal (QoL)
    if not STrapGlobalObject._tt_clearAnimalAndChangeSprite then
        STrapGlobalObject._tt_clearAnimalAndChangeSprite = STrapGlobalObject.clearAnimalAndChangeSprite
        STrapGlobalObject.clearAnimalAndChangeSprite = function(self, player)
            local r = STrapGlobalObject._tt_clearAnimalAndChangeSprite(self, player)
            self.animalAliveHour = 0
            TT_LOG("clearAnimalAndChangeSprite: reset animalAliveHour=0")
            return r
        end
    end

	-- B) Gate capture attempts (canonical bait + ambient cold gate + near-player option)
	if STrapGlobalObject.checkForAnimal and not STrapGlobalObject._tt2_checkForAnimal then
		STrapGlobalObject._tt2_checkForAnimal = STrapGlobalObject.checkForAnimal
		-- Signature must remain (self, square)
		STrapGlobalObject.checkForAnimal = function(self, square)
			TT_LOG(("checkForAnimal started for trap (%s,%s,%s)"):format(tostring(self.x), tostring(self.y), tostring(self.z)))

			-- 0) Require canonical bait (don't trust transient 'bait')
			if not _hasCanonicalBait(self) then
				TT_LOG("checkForAnimal -> no canonical bait -> skip")
				return false
			else
				TT_LOG("checkForAnimal -> canonical bait -> continue")
			end

			-- 1) Vanilla early-return path when player is near (square ~= nil):
			--    keep anti-wall-exploit, and only bypass the early return if sandbox allows.
			if square then
				if self.checkForWallExploit and self:checkForWallExploit(square) then
					self:removeAnimal()
					self:removeBait()
					TT_LOG("checkForAnimal -> wall exploit -> cleared bait/animal -> return")
					return false
				end
				if not CFG.AllowCatchNearPlayer then
					TT_LOG("checkForAnimal -> player near -> vanilla early return")
					return
				end
				TT_LOG("checkForAnimal -> player near BUT sandbox allows -> continue as if square=nil")
			end

			-- 2) Ambient/global cold gate
			if not coldGateAllowsCatch(self) then
				TT_LOG("checkForAnimal -> coldgate DENY -> skip vanilla check")
				return false
			else
				TT_LOG("checkForAnimal -> coldgate ALLOW -> continue with vanilla check")
			end

			-- 3) Call original in protected mode; if near-player allowed, force square=nil
			--    NOTE: 'a and nil or b' is unsafe because 'nil' makes it always return 'b'.
			local argSquare = square
			local nearAllowed = (square ~= nil) and (CFG.AllowCatchNearPlayer == true)
			if nearAllowed then
				argSquare = nil
				TT_LOG("checkForAnimal -> forcing square=nil for vanilla call (near-player allowed)")
			end

			self._tt_gateInProgress = true -- so testForAnimal won't gate again
			local ok, ret = pcall(STrapGlobalObject._tt2_checkForAnimal, self, argSquare)
			self._tt_gateInProgress = nil
			if not ok then
				TT_LOG("checkForAnimal -> pcall error: " .. tostring(ret))
				return false
			end
			return ret
		end
		TT_LOG("hooked checkForAnimal (canonical bait + ambient cold gate + near-player sandbox)")
	end

	-- C) Fallback gate in testForAnimal (no double gate)
	if STrapGlobalObject.testForAnimal and not STrapGlobalObject._tt2_testForAnimal then
		STrapGlobalObject._tt2_testForAnimal = STrapGlobalObject.testForAnimal
		STrapGlobalObject.testForAnimal = function(self, ...)
			TT_LOG(("testForAnimal started for trap (%s,%s,%s)"):format(tostring(self.x), tostring(self.y), tostring(self.z)))
			-- If called from our checkForAnimal(), don't gate twice
			if self._tt_gateInProgress then
				return STrapGlobalObject._tt2_testForAnimal(self, ...)
			end
			-- Fallback gating (rare)
			if not _hasCanonicalBait(self) then
				TT_LOG("testForAnimal -> no canonical bait -> skip")
				return false
			else
				TT_LOG("testForAnimal -> canonical bait -> continue")
			end
			if not coldGateAllowsCatch(self) then
				TT_LOG("testForAnimal -> coldgate DENY -> skip vanilla test")
				return false
			else
				TT_LOG("checkForAnimal -> coldgate ALLOW -> continue with vanilla check")
			end
			return STrapGlobalObject._tt2_testForAnimal(self, ...)
		end
		TT_LOG("hooked testForAnimal (fallback gate, no double gating)")
	end

    -- D) calculTrap(): pre-sync alias bait, then scale/cap accumulation
    if STrapGlobalObject.calculTrap and not STrapGlobalObject._tt_calculTrap then
        STrapGlobalObject._tt_calculTrap = STrapGlobalObject.calculTrap
        STrapGlobalObject.calculTrap = function(self, square)
            TT_LOG("calculTrap started for trap (" .. tostring(self.x) .. ", " .. tostring(self.y) .. ", " .. tostring(self.z) .. ")")

            -- Sync transient alias: vanilla calculTrap() gates on self.bait
            local canonical = _canonicalBaitType(self)
            if self.bait ~= canonical then
				TT_LOG("calculTrap -> BEFORE alias was bait=" .. tostring(self.bait) .. " -> NOW synced to bait=" .. tostring(canonical))
				self.bait = canonical
            end

            local before = tonumber(self.animalAliveHour) or 0
            local hadAnimal = self.animal ~= nil

            local r = STrapGlobalObject._tt_calculTrap(self, square)

            local after = tonumber(self.animalAliveHour) or 0
            local vanillaDelta = after - before

			-- replace vanilla +1 with scaled increment (no double-add)
			if hadAnimal and vanillaDelta > 0 then
				-- reset baseline to 'before' and apply our scaled increment
				self.animalAliveHour = before
				addAliveHoursWithRules(self, 1) -- aplica mult/cap sobre 1h de vanilla
				TT_LOG(string.format("calculTrap: vanillaDelta=%d -> scaledAfter=%d", vanillaDelta, self.animalAliveHour))
			end
            return r
        end
        TT_LOG("hooked calculTrap (alias sync + scale/cap)")
    end

    -- E) toObject(): keep alias 'bait' consistent in modData for UIs/EveryDays
    if STrapGlobalObject.toObject and not STrapGlobalObject._tt_syncBaitInToObject then
        STrapGlobalObject._tt_syncBaitInToObject = STrapGlobalObject.toObject
        STrapGlobalObject.toObject = function(self, object, transmitData)
            self.bait = _canonicalBaitType(self)
            return STrapGlobalObject._tt_syncBaitInToObject(self, object, transmitData)
        end
        TT_LOG("hooked toObject (sync 'bait' alias from trapBait)")
    end
end

----------------------------------------------------------------
-- Apply patches and load sandbox at the right time
----------------------------------------------------------------
-- Try now (in case vanilla is already loaded)
applyPatch()

-- Refresh config & patch when the game starts
Events.OnGameStart.Add(function()
    TT_RefreshConfig("OnGameStart")
    applyPatch()
end)

-- Dedicated server start (if available)
if Events.OnServerStarted then
    Events.OnServerStarted.Add(function()
        TT_RefreshConfig("OnServerStarted")
        applyPatch()
    end)
end

-- Hot reload on sandbox options change (if exposed by the build)
if Events.OnSandboxOptionsChanged then
    Events.OnSandboxOptionsChanged.Add(function()
        TT_RefreshConfig("OnSandboxOptionsChanged")
        -- No need to re-apply hooks; they read CFG at runtime.
    end)
end

-- Quick sanity check after load (runs once; logs only if VerboseLogging=true)
-- Global function for Command Window debug checks
function TT_trapSanityCheck()
    -- Be safe if the trap system isn't ready yet
    if not STrapSystem or not STrapSystem.instance or not STrapSystem.instance.system then
        print("[TT] load check skipped (trap system not ready)")
        return
    end

    local count = STrapSystem.instance.system:getObjectCount()
    print(string.format("[TT] load check: %d traps", count))

    for i = 1, count do
        -- Vanilla uses the same access pattern (getObjectByIndex(...):getModData()).
        local L = STrapSystem.instance.system:getObjectByIndex(i-1):getModData()

        -- Log canonical bait vs alias to spot desyncs
        local tb = (L.trapBait ~= "" and L.trapBait) or ""
        local b  = L.bait

		-- English comments: show whether the cell/square is currently loaded.
		local sq = getSquare(L.x or -1, L.y or -1, L.z or 0)
		local cellLoaded = (sq ~= nil)

        print(string.format("[TT] trap %d @(%s,%s,%s), trapBait=%s, bait=%s, cellLoaded=%s",
            i,
            tostring(L.x or "?"), tostring(L.y or "?"), tostring(L.z or "?"),
            tostring(tb), tostring(b), cellLoaded and "YES" or "NO"))
    end
end

-- Avoid noisy logs unless explicitly enabled
if (CFG and CFG.VerboseLogging) then
	Events.OnGameStart.Add(TT_trapSanityCheck)
end											 