-- TT_TrapMenuSync.lua
--
-- Purpose:
--   Keep the client-side CTrapGlobalObject ("placedTrap") in sync with the
--   authoritative modData stored on the IsoTrap before the context menu is built.
--   This prevents stale 'placedTrap.bait' or 'placedTrap.animal' from making
--   ISTrapMenu show wrong options like "Discard bait" when there's no bait.
--
-- How:
--   We hook OnPreFillWorldObjectContextMenu(player, context, worldobjects, test),
--   find IsoTrap objects in 'worldobjects', look up the corresponding placedTrap
--   via CTrapSystem, and refresh a minimal set of fields from isoTrap:getModData():
--     - trapBait / baitDay  -> placedTrap.trapBait / placedTrap.bait / placedTrap.baitDay
--     - animal              -> placedTrap.animal (table with .type), not a plain string
--     - animalAliveHour     -> placedTrap.animalAliveHour (harmless mirror)
--
-- Notes:
--   * If placedTrap:fromModData(modData) exists, we prefer calling it (future-proof).
--   * Otherwise we do a manual minimal sync (keeping types as vanilla expects).
--   * We never change vanilla files; this is a local pre-menu refresh.
--

-- Guard against double-patching
if _G.__TT_TrapMenuSyncInstalled then return end
_G.__TT_TrapMenuSyncInstalled = true

local TT_MenuSync = {}
local debugLogs = true

local function dlog(msg)
    if debugLogs then print("[TT][MenuSync] " .. tostring(msg)) end
end

-- Return the client-side placedTrap (CTrapGlobalObject) at a square.
-- CTrapSystem.instance keeps Lua objects indexed by XYZ.
local function lookupPlacedTrap(square)
    local csys = CTrapSystem and CTrapSystem.instance
    if not csys or not square then return nil end
    return csys:getLuaObjectAt(square:getX(), square:getY(), square:getZ())
end

-- Map an animal 'type' (string) to the TrapAnimals table entry expected by UIs.
-- If not found, return an empty table (vanilla checks for .type).
local function mapAnimalTypeToTable(animalType)
    if type(animalType) == "table" then
        -- Already a table; vanilla expects .type to exist.
        return animalType
    end
    if not animalType or animalType == "" then
        return {} -- empty table -> "no animal"
    end
    if type(TrapAnimals) == "table" then
        for _, v in ipairs(TrapAnimals) do
            if v.type == animalType then
                return v
            end
        end
    end
    -- Fallback: minimal table with type field so ISTrapMenu conditions don't error.
    return { type = animalType }
end

-- Best-effort copy of authoritative fields from isoTrap.modData into placedTrap,
-- so the context menu will read up-to-date values.
function TT_MenuSync.syncPlacedTrapFromIso(placedTrap, isoTrap)
    if not placedTrap or not isoTrap then return end

    local md = isoTrap:getModData() or {}

    -- Prefer a vanilla refresh method if present
    if type(placedTrap.fromModData) == "function" then
        -- Many CGlobalObject implementations expose fromModData(modData)
        -- If present, this should refresh all derived fields (like .bait)
        local ok, err = pcall(function() placedTrap:fromModData(md) end)
        if not ok then dlog("fromModData failed: " .. tostring(err)) end
    else
        -- Manual minimal sync for the fields used by ISTrapMenu branching.

        -- Bait: keep canonical string in trapBait, and mirror to .bait as string-or-nil.
        placedTrap.trapBait = md.trapBait or ""
        -- Some builds store bait day as trapBaitDay; keep a fallback to baitDay if present.
        placedTrap.baitDay  = md.trapBaitDay or md.baitDay or 0
		placedTrap.trapBaitDay = md.trapBaitDay or placedTrap.trapBaitDay
        -- IMPORTANT: .bait should be string (full type) or nil, not boolean,
        -- because some UIs/extensions may expect the item type.
        if placedTrap.trapBait ~= nil and placedTrap.trapBait ~= "" then
            placedTrap.bait = placedTrap.trapBait
        else
            placedTrap.bait = nil
        end

        -- Animal: in modData, it can be string ("rabbit") or a table { type="rabbit", ... }.
        local animalVal = md.animal
        -- Normalize to TrapAnimals table entry (vanilla expects a table with .type).
        placedTrap.animal = mapAnimalTypeToTable(animalVal)

        -- Alive-hour mirror (harmless; used for spoil/death timers in UIs)
        if md.animalAliveHour ~= nil then
            placedTrap.animalAliveHour = md.animalAliveHour
        end
    end
end

-- This runs before vanilla builds the world-object context menu.
-- We take the opportunity to refresh placedTrap state from IsoTrap.modData
-- so ISTrapMenu sees the latest truth (bait present? animal present?).
-- SIGNATURE (very important): (player, context, worldobjects, test)
function TT_MenuSync.onPreFillWOCM(player, context, worldobjects, test)
    -- When 'test' is true, the engine is probing/estimating the menu size.
    -- We can skip work in that pass; the next real pass (test == false)
    -- will perform the actual sync.
    if test then return end
    if not worldobjects or #worldobjects == 0 then return end

    for i = 1, #worldobjects do
        local obj = worldobjects[i]
        -- Only care about IsoTrap here; other world objects are ignored.
        if obj and instanceof(obj, "IsoTrap") then
            local square = obj:getSquare()
            if square then
                local placed = lookupPlacedTrap(square)
                if placed then
                    TT_MenuSync.syncPlacedTrapFromIso(placed, obj)
                    dlog(string.format("synced trap @(%d,%d,%d)", square:getX(), square:getY(), square:getZ()))
                else
                    dlog("no placedTrap for streamed IsoTrap")
                end
            end
        end
    end
end

-- Register once.
Events.OnPreFillWorldObjectContextMenu.Add(TT_MenuSync.onPreFillWOCM)

return TT_MenuSync
