-- media/lua/shared/Computer/Computer_Persist.lua
-- Persistence + move restrictions for Computer mod (B42.11.00)
-- - Saves computer state (cd tray, inserted disk, installed games) onto the Moveable item when picked up
-- - Restores that state onto the placed world object
-- - Blocks moving a powered-on computer

-- Vanilla deps
require "Moveables/ISMoveableSpriteProps"
require "Moveables/ISMoveablesAction"

-- Local table
ComputerPersist = ComputerPersist or {}

-- Sprite names that identify this computer (on/off, all facings)
local SPRITES = {
  ["appliances_com_01_72"] = true, -- off S
  ["appliances_com_01_73"] = true, -- off E
  ["appliances_com_01_74"] = true, -- off N
  ["appliances_com_01_75"] = true, -- off W
  ["appliances_com_01_76"] = true, -- on S
  ["appliances_com_01_77"] = true, -- on E
  ["appliances_com_01_78"] = true, -- on N
  ["appliances_com_01_79"] = true, -- on W
}

local function isComputerObject(obj)
  if not obj or not obj.getSprite then return false end
  local spr = obj:getSprite(); if not spr or not spr.getName then return false end
  local n = spr:getName(); if not n then return false end
  return SPRITES[n] == true
end

local function isComputerSpriteName(name)
  if not name then return false end
  return SPRITES[name] == true
end

-- safe deep copy (tables only; primitives copied by value)
local function deepcopy(v)
  if type(v) ~= "table" then return v end
  local t = {}
  for k, val in pairs(v) do
    t[deepcopy(k)] = deepcopy(val)
  end
  return t
end

-- generate a simple id string (not cryptographically unique; fine for game use)
local function genId()
  local z = (ZombRand and ZombRand(2147483647) or math.random(2147483647))
  local t = os.time and os.time() or 0
  return tostring(t) .. "-" .. tostring(z)
end

-- Export current computer state from a world object
function ComputerPersist.captureStateFromObject(obj)
  if not obj or not obj.getModData then return nil end
  local md = obj:getModData()
  local cd = md.computerData or {}

  local state = {
    computerData = {
      cdOpen       = cd.cdOpen and true or false,
      cdDiskType   = cd.cdDiskType,
      cdDiskName   = cd.cdDiskName,
      installed    = deepcopy(cd.installed or {}),
      pendingUninstallName = cd.pendingUninstallName,
    },
    Computer_On = false, -- always force off when picked up
    Computer_UUID = md.Computer_UUID or genId(),
  }
  return state
end

-- Apply saved state to an InventoryItem (moveable)
function ComputerPersist.applyStateToItem(item, state)
  if not item or not item.getModData or not state then return end
  local imd = item:getModData()
  imd.computerData  = deepcopy(state.computerData or {})
  imd.Computer_On   = false
  imd.Computer_UUID = state.Computer_UUID or genId()
  -- (InventoryItem doesn't need transmitModData; handled by container ops / MP commands.)
end

-- Apply saved state to a world object
function ComputerPersist.applyStateToObject(obj, state)
  if not obj or not obj.getModData or not state then return end
  local md = obj:getModData()
  md.computerData    = deepcopy(state.computerData or {})
  md.Computer_On     = false
  md.Computer_UUID   = state.Computer_UUID or md.Computer_UUID or genId()
  if obj.transmitModData then obj:transmitModData() end
end

-- Pending placement map keyed by "x,y,z" -> state table (copied from placing item)
ComputerPersist._pendingBySquare = ComputerPersist._pendingBySquare or {}

local function sqKeyFromSquare(sq)
  if not sq then return nil end
  return tostring(sq:getX()) .. "," .. tostring(sq:getY()) .. "," .. tostring(sq:getZ() or 0)
end

-- === Hooks ===

-- 1) Block pickup when the computer is ON.
if not ComputerPersist._hooked_canPick then
  ComputerPersist._hooked_canPick = true
  local _orig_can = ISMoveableSpriteProps.canPickUpMoveableInternal
  ISMoveableSpriteProps.canPickUpMoveableInternal = function(self, _character, _square, _object, _isMulti)
    if _object and isComputerObject(_object) then
      local md = _object.getModData and _object:getModData() or nil
      if md and md.Computer_On then
        -- Disallow picking up a powered computer
        return false
      end
    end
    return _orig_can(self, _character, _square, _object, _isMulti)
  end
end

-- 2) On pickup: copy state from object -> inventory item.
if not ComputerPersist._hooked_pickup then
  ComputerPersist._hooked_pickup = true
  local _orig_pick = ISMoveableSpriteProps.pickUpMoveableInternal
  ISMoveableSpriteProps.pickUpMoveableInternal = function(self, _character, _square, _object, _sprInstance, _spriteName, _createItem, _rotating)
    local pickedState = nil
    if _object and isComputerObject(_object) then
      pickedState = ComputerPersist.captureStateFromObject(_object)
      -- also force it OFF in-world before removal (safety)
      if _object and _object.getModData then
        local md = _object:getModData()
        md.Computer_On = false
        if _object.transmitModData then _object:transmitModData() end
      end
    end

    local item = _orig_pick(self, _character, _square, _object, _sprInstance, _spriteName, _createItem, _rotating)

    if pickedState and item and item.getModData then
      ComputerPersist.applyStateToItem(item, pickedState)
    end
    return item
  end
end

-- 3) On place: stash state from the item so we can re-apply to the spawned IsoObject when OnObjectAdded fires.
if not ComputerPersist._hooked_place then
  ComputerPersist._hooked_place = true
  local _orig_place = ISMoveableSpriteProps.placeMoveableInternal
  ISMoveableSpriteProps.placeMoveableInternal = function(self, _square, _item, _spriteName)
    -- Only for our computer
    if _item and _item.getModData and isComputerSpriteName(_spriteName) then
      local key = sqKeyFromSquare(_square)
      if key then
        local state = {
          computerData  = deepcopy((_item:getModData().computerData) or {}),
          Computer_On   = false,
          Computer_UUID = (_item:getModData().Computer_UUID) or genId(),
        }
        ComputerPersist._pendingBySquare[key] = state
      end
    end

    _orig_place(self, _square, _item, _spriteName)
    -- The OnObjectAdded event will apply and clear the pending entry.
  end
end

-- 4) When a new object is added, apply any pending state if it matches our computer.
local function onObjectAdded(obj)
  if not obj or not isComputerObject(obj) then return end
  local sq = obj.getSquare and obj:getSquare() or nil
  if not sq then return end
  local key = sqKeyFromSquare(sq)
  if not key then return end

  local pending = ComputerPersist._pendingBySquare[key]
  if pending then
    ComputerPersist.applyStateToObject(obj, pending)
    ComputerPersist._pendingBySquare[key] = nil
  end
end

-- Install event handlers once
if not ComputerPersist._events then
  ComputerPersist._events = true
  Events.OnObjectAdded.Add(onObjectAdded)
end

-- Optional: also ensure that an already-placed computer is never considered furniture when ON.
-- (The move tool already queries canPickUpMoveableInternal which we patched.)
-- Nothing else needed here.

return ComputerPersist
