-- media/lua/client/Computer/patches/ISComputerPlayAction_Mood.lua
-- COMPUTER • Time-scaled mood patch (B42) • NO Mod Options
-- Effects apply PER *IN-GAME MINUTE* based on GameTime:getTimeOfDay().
-- Anti-drift: we apply deltas against a per-frame baseline so other scripts can't
-- nudge boredom/unhappiness/panic while the action is active.
-- No fatigue logic here.

require "Computer/TA/ISComputerPlayAction"

-- === CONSTANT RATES (per in-game minute) ===
-- Tuned to the values you approved:
local RATE = {
    boredom_per_min     = -0.7,  -- boredom (0..100)
    unhappiness_per_min = -0.3,  -- unhappiness (0..100)
    panic_per_min       = -0.3,  -- panic (0..100)
    stress_per_min      = -0.3,  -- stress accumulator 0..100 -> Stats:setStress(acc/100)
}

-- === Helpers ===
local function clamp(x, lo, hi) if x < lo then return lo elseif x > hi then return hi else return x end end
local function truncTowardsZero(x) if x >= 0 then return math.floor(x) else return math.ceil(x) end end
local function getTODHours() local gt = getGameTime(); if gt and gt.getTimeOfDay then return gt:getTimeOfDay() end; return 0.0 end

-- === Per-player state ===
ComputerMod = ComputerMod or {}
ComputerMod._PC = ComputerMod._PC or {
    active = {},
    lastTOD = {},
    acc = {},         -- fractional accumulators for b/u/p/s
    stressInt = {},   -- integer 0..100
    prev = {},        -- previous-frame baselines {b,u,p}
}

local function ensurePlayerState(player)
    if not player or not player.getPlayerNum then return nil end
    local id = player:getPlayerNum()
    local S = ComputerMod._PC
    if S.lastTOD[id] == nil then S.lastTOD[id] = getTODHours() end
    if S.acc[id] == nil then S.acc[id] = { b = 0.0, u = 0.0, p = 0.0, s = 0.0 } end
    if S.stressInt[id] == nil then
        local stats = player.getStats and player:getStats() or nil
        local s = stats and stats.getStress and stats:getStress() or 0
        S.stressInt[id] = math.floor((s or 0) * 100 + 0.5)
    end
    if S.prev[id] == nil then
        local body = player.getBodyDamage and player:getBodyDamage() or nil
        S.prev[id] = {
            b = body and body.getBoredomLevel and body:getBoredomLevel() or 0,
            u = body and body.getUnhappynessLevel and body:getUnhappynessLevel() or 0,
            p = player.getPanic and player:getPanic() or 0,
        }
    end
    return id
end

local function deltaInGameMinutes(id)
    local S = ComputerMod._PC
    local prevH = S.lastTOD[id] or getTODHours()
    local nowH  = getTODHours()
    local dH    = nowH - prevH
    if dH < -12 then dH = dH + 24 end
    if dH >  12 then dH = dH - 24 end
    local dMin = dH * 60.0
    S.lastTOD[id] = nowH
    if dMin < 0 then dMin = 0 end
    return dMin
end

local function applyScaled(player, dMin)
    if dMin <= 0 then return end
    local id = ensurePlayerState(player) ; if not id then return end
    local S = ComputerMod._PC
    local stats = player.getStats and player:getStats() or nil
    local body  = player.getBodyDamage and player:getBodyDamage() or nil
    local A     = S.acc[id]; if not A then return end
    local P     = S.prev[id]; if not P then return end

    -- accumulate fractional deltas
    A.b = A.b + RATE.boredom_per_min     * dMin
    A.u = A.u + RATE.unhappiness_per_min * dMin
    A.p = A.p + RATE.panic_per_min       * dMin
    A.s = A.s + RATE.stress_per_min      * dMin

    -- integer deltas (towards zero)
    local dB = truncTowardsZero(A.b) ; A.b = A.b - dB
    local dU = truncTowardsZero(A.u) ; A.u = A.u - dU
    local dP = truncTowardsZero(A.p) ; A.p = A.p - dP
    local dS = truncTowardsZero(A.s) ; A.s = A.s - dS

    -- apply against previous-frame baselines to cancel any third-party drift
    if body and body.getBoredomLevel and body.setBoredomLevel then
        local targetB = clamp(math.floor(P.b + dB + 0.0), 0, 100)
        body:setBoredomLevel(targetB)
        P.b = targetB
    end
    if body and body.getUnhappynessLevel and body.setUnhappynessLevel then
        local targetU = clamp(math.floor(P.u + dU + 0.0), 0, 100)
        body:setUnhappynessLevel(targetU)
        P.u = targetU
    end
    if player.getPanic and player.setPanic then
        local targetP = clamp(math.floor(P.p + dP + 0.0), 0, 100)
        player:setPanic(targetP)
        P.p = targetP
    end
    if stats and stats.setStress and stats.getStress then
        local acc = clamp((S.stressInt[id] or 0) + dS, 0, 100)
        S.stressInt[id] = acc
        stats:setStress(acc / 100)
    end
end

-- Hook lifecycle
local _orig_start = ISComputerPlayAction.start
function ISComputerPlayAction:start()
    if _orig_start then _orig_start(self) end
    local id = ensurePlayerState(self.character)
    if not id then return end
    ComputerMod._PC.active[id] = true
    ComputerMod._PC.lastTOD[id] = getTODHours()
    -- (re)seed baselines from actual current values
    local body = self.character and self.character.getBodyDamage and self.character:getBodyDamage() or nil
    local P = ComputerMod._PC.prev[id] or {}
    P.b = body and body.getBoredomLevel and body:getBoredomLevel() or 0
    P.u = body and body.getUnhappynessLevel and body:getUnhappynessLevel() or 0
    P.p = self.character and self.character.getPanic and self.character:getPanic() or 0
    ComputerMod._PC.prev[id] = P
end

local function clearFlags(self)
    if not self or not (self.character and self.character.getPlayerNum) then return end
    local id = self.character:getPlayerNum()
    ComputerMod._PC.active[id] = false
end

local _orig_stop = ISComputerPlayAction.stop
function ISComputerPlayAction:stop() clearFlags(self); if _orig_stop then _orig_stop(self) end end
local _orig_perform = ISComputerPlayAction.perform
function ISComputerPlayAction:perform() clearFlags(self); if _orig_perform then _orig_perform(self) end end

-- Driver
local function PC_OnPlayerUpdate(player)
    if not player then return end
    local id = (player.getPlayerNum and player:getPlayerNum()) or 0
    if not (ComputerMod and ComputerMod._PC and ComputerMod._PC.active and ComputerMod._PC.active[id]) then return end
    local dMin = deltaInGameMinutes(id)
    if dMin > 0 then applyScaled(player, dMin) end
end
Events.OnPlayerUpdate.Add(PC_OnPlayerUpdate)
