require "ISUI/ISPanel"
require "ISUI/ISButton"
require "ISUI/ISScrollingListBox"
require "Computer/UI/ISComputerInsertDialog"
require "Computer/Computer_Data"
require "Computer/Jobs/Computer_Jobs"
require "Computer/Computer_Light"
require "luautils"
require "Computer/TA/ISComputerPlayAction"
require "Computer/TA/ISComputerActivateAction"

ISComputerPanel = ISPanel:derive("ISComputerPanel")

-- power check (hydro, square flag, or generator)


function ISComputerPanel:new(x, y, w, h, playerObj, worldObj)
  local o = ISPanel:new(x, y, w, h)
  setmetatable(o, self)
  self.__index = self
  o.playerObj = playerObj
  o.playerIndex = playerObj and playerObj:getPlayerNum() or 0
  o.worldObj  = worldObj
  o.moveWithMouse = true
  o.background = true
  o.backgroundColor = { r=0, g=0, b=0, a=0.85 }
  o.borderColor = { r=1, g=1, b=1, a=1 }
  o.title = "Computer"
  o.rightTitle = "Games:"
  o.titlebarH = 30
  o.isOn = false
  o.cdOpen = false
  o.diskInserted = false
  o.insertedDiskType = nil
  o.insertedDiskName = nil
  -- cache
  o._lastPower = nil
  o._lastSelected = -1
  o._hadJob = false
  o._dotT = 0
  o._dotN = 0
  -- TA state
  o.activeAction = nil
  o.activeActionType = nil
  -- Insert dialog lock
  o.insertDlgOpen = false
  o._insertDlgRef = nil
  -- Tabs
  o._tabs = {"Games", "Apps", "Docs", "Audio", "Video"}
  o._tabKeys = {"games", "apps", "docs", "audio", "video"}
  o._activeTab = 1
  o.tabButtons = {}
  return o
end


function ISComputerPanel:initialise()
  ISPanel.initialise(self)

  local margin, gap, btnH = 10, 12, 28
  local leftW = 160
  local rightHeaderH = 28
  self._layout = {
    leftX = margin,
    leftY = self.titlebarH + margin,
    leftW = leftW,
    btnH = btnH,
    gap = gap,
    rightX = margin + leftW + margin,
    rightY = self.titlebarH + margin,
    rightW = self.width - (margin + leftW + margin*2),
    rightH = self.height - (self.titlebarH + margin*2),
    rightHeaderH = rightHeaderH,
  }

  if not self.closeBtn then
    self.closeBtn = ISButton:new(6, 6, 24, 24, "X", self, function() self:onClose() end)
    self.closeBtn.borderColor = {r=1, g=1, b=1, a=0.6}
    self.closeBtn:initialise()
    self:addChild(self.closeBtn)
  end

  local x, y = self._layout.leftX, self._layout.leftY
  self.btnPower = ISButton:new(x, y, leftW, btnH, "On", self, function() self:onTogglePower() end)
  self.btnPower:initialise(); self:addChild(self.btnPower); y = y + btnH + gap

  self.btnCd = ISButton:new(x, y, leftW, btnH, "Open CD-Rom", self, function() self:onToggleCD() end)
  self.btnCd:initialise(); self:addChild(self.btnCd); y = y + btnH + gap

  self.btnInsert = ISButton:new(x, y, leftW, btnH, "Insert CD", self, function() self:onInsertDisk() end)
  self.btnInsert:initialise(); self:addChild(self.btnInsert); y = y + btnH + gap

  self.btnEject = ISButton:new(x, y, leftW, btnH, "Eject CD", self, function() self:onEjectDisk() end)
  self.btnEject:initialise(); self:addChild(self.btnEject); y = y + btnH + gap

  self.btnInstall = ISButton:new(x, y, leftW, btnH, "Install", self, function() self:onInstallFromDisk() end)
  self.btnInstall:initialise(); self:addChild(self.btnInstall); y = y + btnH + gap

  self.btnUninstall = ISButton:new(x, y, leftW, btnH, "Uninstall", self, function() self:onRemoveSelected() end)
  self.btnUninstall:initialise(); self:addChild(self.btnUninstall); y = y + btnH + gap

  self.btnPlay = ISButton:new(x, y, leftW, btnH, "Play", self, function() self:onPlay() end)
  self.btnPlay:initialise(); self:addChild(self.btnPlay)

  -- Status (top) and Memory (bottom) bars under Play button
  do
    local x2 = self._layout.leftX
    local gap2 = self._layout.gap or 12
    local leftW2 = self._layout.leftW or 160
    local y2 = self.btnPlay.y + self.btnPlay.height + gap2 * 3

    self._ui = self._ui or {}
    self._ui.status = { x = x2, y = y2, w = leftW2, h = 24 }
    y2 = y2 + 24 + 4
    self._ui.memory = { x = x2, y = y2, w = leftW2, h = 22 }
  end

  -- Right list (with header)
  self.list = ISScrollingListBox:new(
    self._layout.rightX, self._layout.rightY + self._layout.rightHeaderH,
    self._layout.rightW, self._layout.rightH - self._layout.rightHeaderH
  )
  self.list:initialise()
  self.list.itemheight = 24
  self.list.doDrawItem = function(lst, y3, item, alt)
    if not item or not item.text then return y3 end
    local r, g, b = 1, 1, 1
    if lst.selected == item.index then r, g, b = 1, 1, 0.5 end
    lst:drawText(item.text, 8, y3+6, r, g, b, 1, UIFont.Small)
    return y3 + lst.itemheight
  end
  self:addChild(self.list)

  -- Tab strip on right header
  do
    local tm = getTextManager()
    local rx = self._layout.rightX
    local ry = self._layout.rightY
    local h  = self._layout.rightHeaderH - 4
    local xoff = rx + 2
    self.tabButtons = {}
    for i, label in ipairs(self._tabs) do
      local w = math.max(60, tm:MeasureStringX(UIFont.Small, label) + 12)
      local btn = ISButton:new(xoff, ry + 2, w, h, label, self, function(_, _) self:setActiveTab(i) end)
      btn:initialise()
      self:addChild(btn)
      table.insert(self.tabButtons, btn)
      xoff = xoff + w + 4
    end
  end

  -- restore states
  local md = self.worldObj and self.worldObj:getModData() or nil
  if md and md.Computer_On ~= nil then self.isOn = md.Computer_On and true or false end
  if md then
    md.computerData = md.computerData or {}
    if md.computerData.cdOpen ~= nil then self.cdOpen = md.computerData.cdOpen and true or false end
    if md.computerData.cdDiskType then
      self.diskInserted = true
      self.insertedDiskType = md.computerData.cdDiskType
      self.insertedDiskName = md.computerData.cdDiskName or "Game Disc"
    end
  end

  if not self:hasPower() and self.isOn then self:forcePowerOff() end

  ComputerJobs.ensureTracking(self.worldObj)
  self:setActiveTab(1) -- will populate list
  ComputerLight.refresh(self.worldObj)
  self:updateButtons()
end

-- is installed helper (distinct by fullType + name)

function ISComputerPanel:isInstalled(fullType, name)
  if not (fullType and self.list and self.list.items) then return false end
  for i=1, #self.list.items do
    local it = self.list.items[i]
    if it and it.item and it.item.id == fullType then
      if (not name) or (it.item.name == name) then return true end
    end
  end
  return false
end


function ISComputerPanel:updateButtons()
  if self._syncCdFromMd and self:_syncCdFromMd() then end
  local powerOK = self:hasPower()
  local job = ComputerJobs.getJob(self.worldObj)
  local _capMax = (ComputerData and ComputerData.MAX_GAMES) and ComputerData.MAX_GAMES or 20
  local _capCnt = (ComputerData and ComputerData.countInstalled and ComputerData.countInstalled(self.worldObj)) or ((self.list and self.list.items) and #self.list.items or 0)
  local _atCap = _capCnt >= _capMax

  -- selection present?
  local hasSelection = false
  if self.list and self.list.selected ~= nil and self.list.selected ~= -1 and self.list.items then
    local sel = self.list.items[self.list.selected]
    hasSelection = (sel ~= nil and sel.item ~= nil)
  end

  -- lock all during uninstall/tray
  if job and (job.type == "uninstall" or job.type == "tray") then
    local all = { self.btnPower, self.btnCd, self.btnInsert, self.btnEject, self.btnInstall, self.btnUninstall, self.btnPlay }
    for _, b in ipairs(all) do
      if b and b.setEnable then b:setEnable(false) end
    end
    if self.closeBtn and self.closeBtn.setEnable then self.closeBtn:setEnable(true) end
    return
  end
  if not powerOK then
    self.btnPower:setTitle(self.isOn and "Off" or "On")
    if self.btnPower and self.btnPower.setEnable then self.btnPower:setEnable(false) end
    if self.btnCd and self.btnCd.setEnable then self.btnCd:setEnable(false) end
    if self.btnInsert and self.btnInsert.setEnable then self.btnInsert:setEnable(false) end
    if self.btnEject and self.btnEject.setEnable then self.btnEject:setEnable(false) end
    if self.btnInstall and self.btnInstall.setEnable then self.btnInstall:setEnable(false) end
    if self.btnUninstall and self.btnUninstall.setEnable then self.btnUninstall:setEnable(false) end
    if self.btnPlay and self.btnPlay.setEnable then self.btnPlay:setEnable(false) end
    if self.closeBtn and self.closeBtn.setEnable then self.closeBtn:setEnable(true) end
    return
  end

  self.btnPower:setTitle(self.isOn and "Off" or "On")
  self.btnCd:setTitle(self.cdOpen and "Close CD-Rom" or "Open CD-Rom")

  if not self.isOn then
    self.btnPower:setEnable(true)
    self.btnCd:setEnable(false)
    self.btnInsert:setEnable(false)
    self.btnEject:setEnable(false)
    self.btnInstall:setEnable(false)
    self.btnUninstall:setEnable(false)
    self.btnPlay:setEnable(false)
    if self.closeBtn and self.closeBtn.setEnable then self.closeBtn:setEnable(true) end
    return
  end

  -- powered & on
  self.btnPower:setEnable(true)
  self.btnCd:setEnable(true)
  local j = job

  if j and j.type == "install" then
    self.btnInstall:setTitle("Cancel")
    self.btnInstall.onclick = function() self:onCancelInstall() end
    self.btnInstall:setEnable(true)
    self.btnCd:setEnable(false)
    self.btnInsert:setEnable(false)
    self.btnEject:setEnable(false)
    self.btnUninstall:setEnable(false)
    self.btnPlay:setEnable(false)
  elseif self.activeActionType == "play" then
    self.btnInstall:setTitle("Install")
    self.btnInstall.onclick = function() self:onInstallFromDisk() end
    self.btnInstall:setEnable(false)
    self.btnCd:setEnable(false)
    self.btnInsert:setEnable(false)
    self.btnEject:setEnable(false)
    self.btnUninstall:setEnable(false)
    self.btnPlay:setTitle("Cancel")
    self.btnPlay.onclick = function() self:onCancelPlay() end
    self.btnPlay:setEnable(true)
  else
    self.btnInstall:setTitle("Install")
    self.btnInstall.onclick = function() self:onInstallFromDisk() end
    local allowInstall = (not self.cdOpen) and self.diskInserted and (not self:isInstalled(self.insertedDiskType, self.insertedDiskName)) and (j == nil)
    -- install only in Games tab
    if not self:_isGamesTabActive() then allowInstall = false end
    -- disable "Insert CD" if dialog already open
    self.btnInsert:setEnable(self.cdOpen and (not self.diskInserted) and (not self.insertDlgOpen))
    self.btnEject:setEnable(self.cdOpen and self.diskInserted)
    if _atCap then
      allowInstall = false
    end
    self.btnInstall:setEnable(allowInstall)
    -- Uninstall/Play only meaningful on Games tab
    local enableByTab = self:_isGamesTabActive()
    self.btnUninstall:setEnable(enableByTab and hasSelection and (j == nil))
    self.btnPlay:setTitle("Play")
    self.btnPlay.onclick = function() self:onPlay() end
    self.btnPlay:setEnable(enableByTab and hasSelection and (j == nil))
  end
end

-- Internal do-ops (what actually happens once adjacent)
