local function Queue()
    return {
        first = 0,
        last = -1,
        data = {},

        push = function(self, value)
            self.last = self.last + 1
            self.data[self.last] = value
        end,

        pop = function(self)
            if self.first > self.last then return nil end
            local value = self.data[self.first]
            self.data[self.first] = nil
            self.first = self.first + 1
            return value
        end,

        empty = function(self)
            return self.first > self.last
        end,

        size = function(self)
            return self.last - self.first + 1
        end
    }
end

local function band(a, b)
    local result = 0
    local bitval = 1
    while a > 0 and b > 0 do
        local a_bit = a % 2
        local b_bit = b % 2
        if a_bit == 1 and b_bit == 1 then
            result = result + bitval
        end
        a = math.floor(a / 2)
        b = math.floor(b / 2)
        bitval = bitval * 2
    end
    return result
end

local function rshift(a, n)
    return math.floor(a / (2 ^ n))
end

local function GetRLightLevel(lightLevel)
    return rshift(band(lightLevel, 0xFF0000), 16)
end

local function GetGLightLevel(lightLevel)
    return rshift(band(lightLevel, 0xFF00), 8)
end

local function GetBLightLevel(lightLevel)
    return band(lightLevel, 0xFF)
end

local TestResultsBlockedInstance
local MoodleStackField

local MIN_DANGEROUS_LIGHT_LEVEL = 60
local cardinalDirections = {IsoDirections.N, IsoDirections.W, IsoDirections.S, IsoDirections.E}
local function isSunlit(player, scpPlayerData)
    local skyLightLevel = getGameTime():getSkyLightLevel()
    local RSkyLight = GetRLightLevel(skyLightLevel)
    local GSkyLight = GetGLightLevel(skyLightLevel)
    local BSkyLight = GetBLightLevel(skyLightLevel)
    local maxSkylight = math.max(RSkyLight, GSkyLight, BSkyLight)

    if getClimateManager():getDayLightStrength() <= 0 then return false end -- to disable damage during the night, regardless of light level (to mitigate cat's eyes trait, bright nights option, etc.)
    if maxSkylight < MIN_DANGEROUS_LIGHT_LEVEL then return false end

    local maxDistFromStart = math.max(math.floor((maxSkylight - MIN_DANGEROUS_LIGHT_LEVEL) / 16) - 1, 0) -- Max light level is 255, decays 16 per tile on the same z level

    local startSquare = player:getCurrentSquare()
    if not startSquare then return false end
    local visited = {}
    local queue = Queue()
    queue:push({ square = startSquare, dist = 0 })
    visited[startSquare] = true

    while not queue:empty() do
        local current = queue:pop()
        local square = current.square
        local dist = current.dist

        if square:getOpenAir() then
            scpPlayerData.currentSquareSunlight = maxSkylight - 16 * dist
            return true
        end

        if dist <= maxDistFromStart then
            for i = 1, #cardinalDirections do
                local dir = cardinalDirections[i]
                local result = square:testVisionAdjacent(dir:dx(), dir:dy(), 0, true, false)
                if result ~= TestResultsBlockedInstance then
                    local adjacent = square:getAdjacentSquare(dir)
                    if adjacent and not visited[adjacent] then
                        visited[adjacent] = true
                        queue:push({ square = adjacent, dist = dist + 1 })
                    end
                end
            end
        end
    end

    return false
end

local function doesFullyCoverHead(inventoryItem)
    local bodyLocation = inventoryItem:getBodyLocation()
    return bodyLocation == "FullSuitHead" or bodyLocation == "MaskFull" or bodyLocation == "FullHat"
end

local itemVisuals = ItemVisuals.new()
local coveredParts = ArrayList.new()
local function isBodyCovered(player)
    local uncovered = { }
    local uncoveredCount = 0
    for i = 0, BloodBodyPartType.MAX:index() - 1 do
        uncovered[i] = true
        uncoveredCount = uncoveredCount + 1
    end

    player:getItemVisuals(itemVisuals)
    for i = itemVisuals:size() - 1, 0, -1 do
        local itemVisual = itemVisuals:get(i)
        local inventoryItem = itemVisual:getInventoryItem()
        if inventoryItem then
            local bloodClothingTypeList = inventoryItem:getBloodClothingType()
            if bloodClothingTypeList then
                coveredParts:clear()
                BloodClothingType.getCoveredParts(bloodClothingTypeList, coveredParts)
                for a = 0, coveredParts:size() - 1 do
                    local coveredPart = coveredParts:get(a)
                    if coveredPart:index() == BloodBodyPartType.Head:index() then
                        if doesFullyCoverHead(inventoryItem) then
                            uncovered[BloodBodyPartType.Head:index()] = nil
                            uncoveredCount = uncoveredCount - 1
                        end
                    elseif uncovered[coveredPart:index()] then
                        uncovered[coveredPart:index()] = nil
                        uncoveredCount = uncoveredCount - 1
                    end
                end
            end
        end
    end
    return uncoveredCount == 0
end

local function getField(object, fieldName)
    local offset = string.len(fieldName)
    for i = 0, getNumClassFields(object) - 1 do
        local m = getClassField(object, i)
        if string.sub(tostring(m), -offset) == fieldName then
            return m
        end
    end
    return nil
end

--[[local weatherShaderData
local function getWeatherShader()
    local floorRenderShader = IsoCell.m_floorRenderShader
    local ShaderMapField = getField(floorRenderShader, "ShaderMap")
    local ShaderMap = getClassFieldVal(floorRenderShader, ShaderMapField)
    local shaders = ArrayList.new(ShaderMap:values())
    for i = 0, shaders:size() - 1 do
        local shader = shaders:get(i)
        if string.find(tostring(shader), "WeatherShader") then
            return shader
        end
    end
    return nil
end]]

--[[local function getWeatherShaderData()
    local shader = getWeatherShader()

    local shaderProgramSpecificField = getField(IsoCell.m_floorRenderShader, "m_shaderProgramSpecific")
    local shaderProgramSpecific = getClassFieldVal(shader, shaderProgramSpecificField)

    local shaderProgramField = getField(shaderProgramSpecific, "m_shaderProgram")
    local shaderProgram = getClassFieldVal(shaderProgramSpecific, shaderProgramField)

    local shaderProgramIDField = getField(shaderProgram, "m_shaderID")
    local shaderID = tonumber(getClassFieldVal(shaderProgram, shaderProgramIDField))

    local uniformsByNameField = getField(shaderProgram, "uniformsByName")
    local uniformsByName = getClassFieldVal(shaderProgram, uniformsByNameField)
    local drunkFactorUniform = uniformsByName:get("DrunkFactor")
    local blurFactorUniform = uniformsByName:get("BlurFactor")

    local uniformLocationField = getField(drunkFactorUniform, "loc")
    local drunkFactorUniformLocation = tonumber(getClassFieldVal(drunkFactorUniform, uniformLocationField))
    local blurFactorUniformLocation = tonumber(getClassFieldVal(blurFactorUniform, uniformLocationField))

    local data = {}
    data.ShaderID = shaderID
    data.DrunkFactorUniformLocation = drunkFactorUniformLocation
    data.BlurFactorUniformLocation = blurFactorUniformLocation
    return data
end]]

local MAX_TICKS_TO_LETHAL = 60 -- ticks are 100ms apart
local MIN_TICKS_TO_LETHAL = 10
local function calculateBodyDamageFromSunlight(sunlight)
    if sunlight <= MIN_DANGEROUS_LIGHT_LEVEL then return 0 end
    local norm = (sunlight - MIN_DANGEROUS_LIGHT_LEVEL) / (255 - MIN_DANGEROUS_LIGHT_LEVEL)
    local ticksToLethal = MAX_TICKS_TO_LETHAL - norm * (MAX_TICKS_TO_LETHAL - MIN_TICKS_TO_LETHAL)
    return 100.0 / ticksToLethal
end

local function getMoodleByType(player, moodleType)
    local moodleStack = getClassFieldVal(player:getMoodles(), MoodleStackField)
    for i = 0, moodleStack:size() - 1 do
        local moodle = moodleStack:get(i)
        if moodle:getMoodleType() == moodleType then
            return moodle
        end
    end
    return nil
end

local function startLiquefy(player, scpPlayerData)
    scpPlayerData.liquefying = true

    scpPlayerData.liquefyingStartTimestampMs = getTimestampMs()

    local drunkenness = player:getStats():getDrunkenness()
    scpPlayerData.originalDrunkenness = drunkenness

    scpPlayerData.drunkMoodle = getMoodleByType(player, MoodleType.Drunk)

    local soundName = "burningFlesh1"
    local sound = getSoundManager():PlaySound(soundName, false, 0)
    getSoundManager():PlayAsMusic(soundName, sound, false, 0)
    sound:setVolume(0.3)
    scpPlayerData.sound = sound

    player:getStats():setDrunkenness(100.0)
end

local function stopLiquefy(player, scpPlayerData)
    scpPlayerData.liquefying = false

    scpPlayerData.drunkMoodle = nil

    local drunkenness = scpPlayerData.originalDrunkenness
    player:getStats():setDrunkenness(drunkenness)
    if scpPlayerData.sound then
        scpPlayerData.sound:stop()
        scpPlayerData.sound = nil
    end
end

local function liquefy(player, scpPlayerData)
    player:getStats():setDrunkenness(100.0)

    local timestampMs = getTimestampMs()
    if timestampMs - scpPlayerData.liquefyingStartTimestampMs >= 1000 and timestampMs - scpPlayerData.lastDamageDoneTimestampMs >= 100 then
        player:getBodyDamage():ReduceGeneralHealth(calculateBodyDamageFromSunlight(scpPlayerData.currentSquareSunlight))
        scpPlayerData.lastDamageDoneTimestampMs = timestampMs
    end
end

local function shouldLiquefy(player, scpPlayerData)
    if scpPlayerData.needsClothingReload then
        scpPlayerData.needsClothingReload = false
        scpPlayerData.isBodyCovered = isBodyCovered(player)
    end

    if scpPlayerData.isBodyCovered then return false end

    if not isSunlit(player, scpPlayerData) then return false end

    return true
end

local isInitialized = false
local function initialize(player)
    local square = getCell():getGridSquare(player:getX(), player:getY(), 0)
    TestResultsBlockedInstance = square:testVisionAdjacent(-2, 0, 0, true, false) -- always returns LosUtil.TestResults.Blocked instance
    MoodleStackField = getField(player:getMoodles(), "moodleStack")
end

local scpDataPerPlayer = {}
local function OnPlayerUpdate(player)
    if not isInitialized then
        initialize(player)
        isInitialized = true
    end

    local scpPlayerData = scpDataPerPlayer[player]
    if not scpPlayerData then
        scpPlayerData = {}
        scpPlayerData.isBodyCovered = false
        scpPlayerData.needsClothingReload = true
        scpPlayerData.liquefying = false
        scpPlayerData.liquefyingStartTimestampMs = 0
        scpPlayerData.originalDrunkenness = 0.0
        scpPlayerData.drunkMoodle = nil
        scpPlayerData.currentSquareSunlight = 0
        scpPlayerData.lastDamageDoneTimestampMs = 0
        scpPlayerData.sound = nil
        scpDataPerPlayer[player] = scpPlayerData
    end

    if shouldLiquefy(player, scpPlayerData) then
        if scpPlayerData.liquefying then
            liquefy(player, scpPlayerData)
        else
            startLiquefy(player, scpPlayerData)
        end
    elseif scpPlayerData.liquefying then
        stopLiquefy(player, scpPlayerData)
    end
end
Events.OnPlayerUpdate.Add(OnPlayerUpdate)

local function OnClothingUpdated(player)
    local scpPlayerData = scpDataPerPlayer[player]
    if not scpPlayerData then return end
    scpPlayerData.needsClothingReload = true
end
Events.OnClothingUpdated.Add(OnClothingUpdated)

local function OnInitWorld()
    local WARM,NORMAL,CLOUDY = 0, 1, 2
    local SUMMER,FALL,WINTER,SPRING = 0, 1, 2, 3

    local climateManager = getClimateManager()

    -- ###################### Dawn ######################
    climateManager:setSeasonColorDawn(WARM,SUMMER,0.14,0.10,0.90,0.82,true)
    climateManager:setSeasonColorDawn(NORMAL,SUMMER,0.16,0.11,0.85,0.80,true)
    climateManager:setSeasonColorDawn(CLOUDY,SUMMER,0.18,0.13,0.80,0.77,true)

    climateManager:setSeasonColorDawn(WARM,FALL,0.14,0.10,0.88,0.81,true)
    climateManager:setSeasonColorDawn(NORMAL,FALL,0.16,0.11,0.83,0.79,true)
    climateManager:setSeasonColorDawn(CLOUDY,FALL,0.18,0.13,0.78,0.77,true)

    climateManager:setSeasonColorDawn(WARM,WINTER,0.15,0.10,0.86,0.79,true)
    climateManager:setSeasonColorDawn(NORMAL,WINTER,0.17,0.12,0.81,0.77,true)
    climateManager:setSeasonColorDawn(CLOUDY,WINTER,0.19,0.14,0.76,0.76,true)

    climateManager:setSeasonColorDawn(WARM,SPRING,0.14,0.10,0.89,0.81,true)
    climateManager:setSeasonColorDawn(NORMAL,SPRING,0.16,0.11,0.84,0.79,true)
    climateManager:setSeasonColorDawn(CLOUDY,SPRING,0.18,0.13,0.79,0.77,true)

    -- ###################### Day ######################
    climateManager:setSeasonColorDay(WARM,SUMMER,0.00,0.00,1.00,0.85,true)
    climateManager:setSeasonColorDay(NORMAL,SUMMER,0.05,0.05,0.95,0.83,true)
    climateManager:setSeasonColorDay(CLOUDY,SUMMER,0.08,0.08,0.92,0.81,true)

    climateManager:setSeasonColorDay(WARM,FALL,0.00,0.00,1.00,0.85,true)
    climateManager:setSeasonColorDay(NORMAL,FALL,0.05,0.05,0.95,0.83,true)
    climateManager:setSeasonColorDay(CLOUDY,FALL,0.08,0.07,0.90,0.81,true)

    climateManager:setSeasonColorDay(WARM,WINTER,0.00,0.00,1.00,0.85,true)
    climateManager:setSeasonColorDay(NORMAL,WINTER,0.06,0.06,0.94,0.82,true)
    climateManager:setSeasonColorDay(CLOUDY,WINTER,0.10,0.10,0.90,0.80,true)

    climateManager:setSeasonColorDay(WARM,SPRING,0.00,0.00,1.00,0.85,true)
    climateManager:setSeasonColorDay(NORMAL,SPRING,0.05,0.05,0.96,0.83,true)
    climateManager:setSeasonColorDay(CLOUDY,SPRING,0.09,0.08,0.93,0.81,true)

    -- ###################### Dusk ######################
    climateManager:setSeasonColorDusk(WARM,SUMMER,0.14,0.10,0.90,0.82,true)
    climateManager:setSeasonColorDusk(NORMAL,SUMMER,0.16,0.11,0.85,0.80,true)
    climateManager:setSeasonColorDusk(CLOUDY,SUMMER,0.18,0.13,0.80,0.77,true)

    climateManager:setSeasonColorDusk(WARM,FALL,0.14,0.10,0.88,0.81,true)
    climateManager:setSeasonColorDusk(NORMAL,FALL,0.16,0.11,0.83,0.79,true)
    climateManager:setSeasonColorDusk(CLOUDY,FALL,0.18,0.13,0.78,0.77,true)

    climateManager:setSeasonColorDusk(WARM,WINTER,0.15,0.10,0.86,0.79,true)
    climateManager:setSeasonColorDusk(NORMAL,WINTER,0.17,0.12,0.81,0.77,true)
    climateManager:setSeasonColorDusk(CLOUDY,WINTER,0.19,0.14,0.76,0.76,true)

    climateManager:setSeasonColorDusk(WARM,SPRING,0.14,0.10,0.89,0.81,true)
    climateManager:setSeasonColorDusk(NORMAL,SPRING,0.16,0.11,0.84,0.79,true)
    climateManager:setSeasonColorDusk(CLOUDY,SPRING,0.18,0.13,0.79,0.77,true)
end
Events.OnInitWorld.Add(OnInitWorld)

local function OnPlayerDeath(player)
    local scpPlayerData = scpDataPerPlayer[player]
    if not scpPlayerData or not scpPlayerData.liquefying then return end
    stopLiquefy(player, scpPlayerData)
end
Events.OnPlayerDeath.Add(OnPlayerDeath)

local function OnTick()
    for i = 1, getNumActivePlayers() do
        local player = getSpecificPlayer(i - 1)
        if player then
            local scpPlayerData = scpDataPerPlayer[player]
            if scpPlayerData and scpPlayerData.liquefying then
                local drunkMoodle = scpPlayerData.drunkMoodle
                if drunkMoodle then
                    local playerStats = player:getStats()
                    playerStats:setDrunkenness(scpPlayerData.originalDrunkenness)
                    drunkMoodle:Update()
                end
            end
        end
    end
end
Events.OnTick.Add(OnTick)