require "ISUI/ISPanel"

NB_FilterBar = ISPanel:derive("NB_FilterBar")

local FONT_HGT_SMALL = getTextManager():getFontHeight(UIFont.Small)
local FONT_HGT_MEDIUM = getTextManager():getFontHeight(UIFont.Medium)

-- ----------------------------------------------------------------------------------------------------- --
-- initialise
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:initialise()
    ISPanel.initialise(self)
end

function NB_FilterBar:new(x, y, width, height, BuildingPanel)
    local o = ISPanel:new(x, y, width, height)
    setmetatable(o, self)
    self.__index = self
    
    o.BuildingPanel = BuildingPanel
    o.player = BuildingPanel.player
    o.logic = BuildingPanel.logic
    o.padding = BuildingPanel.padding
    o.buttonsize = math.floor(FONT_HGT_SMALL * 1.2)
    o.scrollBarWidth = BuildingPanel.scrollBarWidth
    o.dragging = false
    
    -- Layout icons
    o.GridStyleIcon = getTexture("media/ui/Neat_Building/Icon/Icon_CraftGridStyle.png")
    o.ListStyleIcon = getTexture("media/ui/Neat_Building/Icon/Icon_CraftListStyle.png")
    
    -- Search functionality
    o.searchText = ""
    o.searchMode = "RecipeName"
    o.clearIcon = getTexture("media/ui/Neat_Building/ICON/Icon_Close.png")
    o.UIElementBGTextures = {
        left = getTexture("media/ui/NeatUI/Button/Button_FULL_L.png"),
        middle = getTexture("media/ui/NeatUI/Button/Button_FULL_M.png"),
        right = getTexture("media/ui/NeatUI/Button/Button_FULL_R.png")
    }

    return o
end

-- ----------------------------------------------------------------------------------------------------- --
-- Calculate Layout
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:calculateLayout(_preferredWidth, _preferredHeight)
    local width = math.max(_preferredWidth or 0, 0)
    local height = math.max(_preferredHeight or 0, 0)

    local buttonY = math.floor((height + self.padding*1.5 - self.buttonsize) / 2)
    local buttonCount = 4

    -- Sort Button
    if self.sortButton then
        self.sortButton:setX(0)
        self.sortButton:setY(buttonY)
        self.sortButton:setWidth(self.buttonsize)
        self.sortButton:setHeight(self.buttonsize)
    end

    -- Search Mode Button
    if self.searchModeButton then
        local searchModeButtonX = self.buttonsize + self.padding
        self.searchModeButton:setX(searchModeButtonX)
        self.searchModeButton:setY(buttonY)
        self.searchModeButton:setWidth(self.buttonsize)
        self.searchModeButton:setHeight(self.buttonsize)
    end

    -- Search Text Box
    local minSearchWidth = self.buttonsize * 4
    if self.searchBox then
        local searchBoxX = self.buttonsize * 2 + self.padding * 2
        local searchBoxWidth = width - searchBoxX - self.buttonsize * 2 - self.padding * 2
        
        self.searchBox:setX(searchBoxX)
        self.searchBox:setY(buttonY)
        self.searchBox:setWidth(searchBoxWidth)
        self.searchBox:setHeight(self.buttonsize)

        -- Clear Button
        if self.clearButton then
            local clearButtonX = searchBoxX + searchBoxWidth - self.buttonsize - self.padding/2
            self.clearButton:setX(clearButtonX)
            self.clearButton:setY(buttonY)
            self.clearButton:setWidth(self.buttonsize)
            self.clearButton:setHeight(self.buttonsize)
        end
    end

    -- Layout Toggle Button
    if self.layoutToggleButton then
        local layoutButtonX = width - self.buttonsize * 2 - self.padding
        self.layoutToggleButton:setX(layoutButtonX)
        self.layoutToggleButton:setY(buttonY)
        self.layoutToggleButton:setWidth(self.buttonsize)
        self.layoutToggleButton:setHeight(self.buttonsize)
    end

    if self.closeButton then
        local closeButtonX = width - self.buttonsize
        self.closeButton:setX(closeButtonX)
        self.closeButton:setY(buttonY)
        self.closeButton:setWidth(self.buttonsize)
        self.closeButton:setHeight(self.buttonsize)
    end

    width = math.max(width, buttonCount * (self.buttonsize + self.padding) + minSearchWidth)
    
    self:setWidth(width)
    self:setHeight(height)
end

-- ----------------------------------------------------------------------------------------------------- --
-- createChildren
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:createChildren()
    -- Sort Button
    self.sortButton = NI_SquareButton:new(0, 0, 10, getTexture("media/ui/Neat_Building/Icon/Icon_CraftSort.png"), self, self.onSortButtonClick)
    self.sortButton:initialise()
    self:addChild(self.sortButton)

    -- Search Mode Button
    local searchIcon = getTexture("media/ui/Neat_Building/ICON/Icon_SearchItem.png")
    self.searchModeButton = NI_SquareButton:new(0, 0, 10, searchIcon, self, self.onSearchModeButtonClick)
    self.searchModeButton:initialise()
    self:addChild(self.searchModeButton)

    -- Search Text Entry Box
    self.searchBox = ISTextEntryBox:new("", 0, 0, 10, 10)
    self.searchBox:initialise()
    self.searchBox.font = UIFont.Small
    self.searchBox.onTextChange = function()
        self:onSearchTextChanged()
    end
    self.searchBox.backgroundColor.a = 0
    self.searchBox.borderColor.a = 0
    self.searchBox.prerender = function(searchBox)
        NeatTool.ThreePatch.drawHorizontal(
            searchBox,
            -self.padding/2, 0, 
            searchBox.width + self.padding, searchBox.height,
            self.UIElementBGTextures.left,
            self.UIElementBGTextures.middle,
            self.UIElementBGTextures.right,
            1, 0.4, 0.4, 0.4
        )
    end
    self:addChild(self.searchBox)

    -- Clear Button
    self.clearButton = ISButton:new(0, 0, 10, 10, "", self, self.onClearButtonClick)
    self.clearButton:initialise()
    self.clearButton.borderColor.a = 0
    self.clearButton.backgroundColor.a = 0
    self.clearButton.backgroundColorMouseOver.a = 0
    self.clearButton:setVisible(false)
    
    self.clearButton.prerender = function(button)
        if button:isVisible() then
            local alpha = button:isMouseOver() and 1.0 or 0.7
            button:drawTextureScaled(
                self.clearIcon,
                0, 0,
                button.width, button.height,
                alpha, 0.8, 0.8, 0.8
            )
        end
    end
    self:addChild(self.clearButton)

    -- Layout Toggle Button
    self.layoutToggleButton = NI_SquareButton:new(0, 0, 10, self.ListStyleIcon, self, self.onLayoutToggleButtonClick)
    self.layoutToggleButton:initialise()
    self:addChild(self.layoutToggleButton)

    -- Close Button
    self.closeButton = NI_SquareButton:new(0, 0, 10, self.clearIcon, self, self.onCloseButtonClick)
    self.closeButton:initialise()
    self.closeButton:setActive(true)
    self.closeButton:setActiveColor(0.8, 0.2, 0.2)
    self:addChild(self.closeButton)

    self:updateSortMode()
    self:updateLayoutButton()
    self:updateSearchModeButton()
    self:updateClearButtonVisibility()
end

function NB_FilterBar:onCloseButtonClick()
    if self.BuildingPanel and self.BuildingPanel.close then
        self.BuildingPanel:close()
    end
end

-- ----------------------------------------------------------------------------------------------------- --
-- Sort Button Functions
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:onSortButtonClick()
    local context = ISContextMenu.get(0, self.sortButton:getAbsoluteX(), self.sortButton:getAbsoluteY() + self.sortButton:getHeight())
    local option1 = context:addOption(getText("IGUI_SortType_RecipeName") or "Recipe Name", self, self.updateSortMode, "RecipeName")
    option1.iconTexture = getTexture("media/ui/Neat_Building/ICON/Grey.png")
    
    local option2 = context:addOption(getText("IGUI_SortType_LastUsed") or "Last Used", self, self.updateSortMode, "LastUsed")
    option2.iconTexture = getTexture("media/ui/Neat_Building/ICON/Blue.png")
    
    local option3 = context:addOption(getText("IGUI_SortType_MostUsed") or "Most Used", self, self.updateSortMode, "MostUsed")
    option3.iconTexture = getTexture("media/ui/Neat_Building/ICON/Orange.png")
end

function NB_FilterBar:updateSortMode(sortmode)
    if not self.sortButton then return end
    if not sortmode then 
        sortmode = self.logic:getRecipeSortMode() and self.logic:getRecipeSortMode() or "RecipeName"
    end

    self.logic:setRecipeSortMode(sortmode)

    if sortmode == "RecipeName" then
        self.sortButton:setActiveColor(0.2, 0.2, 0.2)
        self.sortButton:setActive(false)
    elseif sortmode == "LastUsed" then
        self.sortButton:setActiveColor(0.2, 0.4, 0.8)
        self.sortButton:setActive(true)
    elseif sortmode == "MostUsed" then
        self.sortButton:setActiveColor(0.8, 0.5, 0.2)
        self.sortButton:setActive(true)
    end

    if self.logic.sortRecipeList then
        self.logic:sortRecipeList()
    end
end

-- ----------------------------------------------------------------------------------------------------- --
-- Layout Toggle Functions
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:onLayoutToggleButtonClick()
    local currentStyle = self.logic:getSelectedRecipeStyle()
    local newStyle = (currentStyle == "grid") and "list" or "grid"
    
    if self.logic.setSelectedRecipeStyle then
        self.logic:setSelectedRecipeStyle(newStyle)
    end
    self:updateLayoutButton()

    if self.BuildingPanel.recipeListPanel then
        local isGridLayout = (newStyle == "grid")
        self.BuildingPanel.recipeListPanel:setLayoutMode(isGridLayout)
    end
end

function NB_FilterBar:updateLayoutButton()
    if not self.layoutToggleButton then return end

    local currentStyle = self.logic:getSelectedRecipeStyle()

    if currentStyle == "grid" then
        self.layoutToggleButton:setIcon(self.GridStyleIcon)
    else
        self.layoutToggleButton:setIcon(self.ListStyleIcon)
    end
end

-- ----------------------------------------------------------------------------------------------------- --
-- Search Functions
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:onSearchTextChanged()
    self.searchText = self.searchBox:getInternalText()
    self:updateClearButtonVisibility()
    self:performSearch()
end

function NB_FilterBar:performSearch()
    local searchString = self.searchText or ""

    if self.searchMode and self.searchMode ~= "RecipeName" and searchString ~= "" then
        searchString = searchString .. "-@-" .. self.searchMode
    end

    self.BuildingPanel:onSearchTextChanged(searchString)
end

function NB_FilterBar:onSearchModeButtonClick()
    local context = ISContextMenu.get(
        0, 
        self.searchModeButton:getAbsoluteX(), 
        self.searchModeButton:getAbsoluteY() + self.searchModeButton:getHeight()
    )
    
    local option1 = context:addOption(getText("IGUI_FilterType_RecipeName") or "Recipe Name", self, self.setSearchMode, "RecipeName")
    option1.iconTexture = getTexture("media/ui/Neat_Building/ICON/Grey.png")
    
    local option2 = context:addOption(getText("IGUI_FilterType_InputName") or "Input Name", self, self.setSearchMode, "InputName")
    option2.iconTexture = getTexture("media/ui/Neat_Building/ICON/Blue.png")
end

function NB_FilterBar:setSearchMode(mode)
    self.searchMode = mode
    self:updateSearchModeButton()

    if self.searchText and self.searchText ~= "" then
        self:performSearch()
    end
end

function NB_FilterBar:onClearButtonClick()
    self:clearSearch()
    self:updateClearButtonVisibility()
end

function NB_FilterBar:updateClearButtonVisibility()
    if self.clearButton then
        local hasText = self.searchText and self.searchText ~= ""
        self.clearButton:setVisible(hasText)
    end
end

function NB_FilterBar:updateSearchModeButton()
    if not self.searchModeButton then return end

    if self.searchMode == "RecipeName" then
        self.searchModeButton:setActiveColor(0.2, 0.2, 0.2)
        self.searchModeButton:setActive(false)
    elseif self.searchMode == "InputName" then
        self.searchModeButton:setActiveColor(0.2, 0.4, 0.8)
        self.searchModeButton:setActive(true)
    end
end

-- ----------------------------------------------------------------------------------------------------- --
-- Search Utility Functions
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:getSearchText()
    return self.searchText
end

function NB_FilterBar:getSearchMode()
    return self.searchMode
end

function NB_FilterBar:setSearchText(text)
    if self.searchBox then
        self.searchBox:setText(text or "")
        self.searchText = text or ""
        self:updateClearButtonVisibility()
    end
end

function NB_FilterBar:clearSearch()
    self:setSearchText("")
    self:performSearch()
end

-- ----------------------------------------------------------------------------------------------------- --
-- Filter Manager
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:getFilteredRecipes()
    local allRecipes = self.logic:getRecipeList()
    local filteredRecipes = {}
    
    if not allRecipes then return filteredRecipes end

    for i = 0, allRecipes:size() - 1 do
        local recipe = allRecipes:get(i)
        table.insert(filteredRecipes, recipe)
    end
    
    return filteredRecipes
end

function NB_FilterBar:onMouseDown(x, y)
    self.dragging = true
    self.BuildingPanel:bringToTop()
    return true
end

function NB_FilterBar:onMouseMove(dx, dy)
    if self.dragging then
        self.BuildingPanel:setX(self.BuildingPanel:getX() + dx)
        self.BuildingPanel:setY(self.BuildingPanel:getY() + dy)
    end
    return true
end

function NB_FilterBar:onMouseUp(x, y)
    self.dragging = false
    return true
end

function NB_FilterBar:onMouseMoveOutside(dx, dy)
    if self.dragging then
        self.BuildingPanel:setX(self.BuildingPanel:getX() + dx)
        self.BuildingPanel:setY(self.BuildingPanel:getY() + dy)
    end
    return true
end

function NB_FilterBar:onMouseUpOutside(x, y)
    self.dragging = false
    return true
end

-- ----------------------------------------------------------------------------------------------------- --
-- Render
-- ----------------------------------------------------------------------------------------------------- --
function NB_FilterBar:prerender()
    local TitleBG = NinePatchTexture.getSharedTexture("media/ui/Neat_Building/Panel/MainTitle_BG.png")
    if TitleBG then
        TitleBG:render(self:getAbsoluteX() - self.scrollBarWidth - 1, self:getAbsoluteY(), self.width + self.scrollBarWidth + self.padding + 1, self.height, 0.1, 0.1, 0.1, 1)
    end
end

return NB_FilterBar