require "ISUI/ISPanel"

NB_BuildingPanel = ISTableLayout:derive("NB_BuildingPanel")

local FONT_HGT_SMALL = getTextManager():getFontHeight(UIFont.Small)
local FONT_HGT_MEDIUM = getTextManager():getFontHeight(UIFont.Medium)

-- ----------------------------------------------------------------------------------------------------- --
-- initialise
-- ----------------------------------------------------------------------------------------------------- --
function NB_BuildingPanel:initialise()
    ISTableLayout.initialise(self)
end

function NB_BuildingPanel:new(x, y, width, height, player, isoObject, recipeQuery, contextRecipe)
    local o = ISTableLayout:new(x, y, width, height)
    setmetatable(o, self)
    self.__index = self
    
    -- Main Data
    o.player = player
    o.playerNum = player:getPlayerNum()
    o.isoObject = isoObject
    o.recipeQuery = recipeQuery or "AnySurfaceCraft"
    o.buildEntity = nil
    o.contextRecipe = contextRecipe

    o.tempAdjWidth = 0
    o.tempAdjHeight = 0
    o:setWantKeyEvents(true)

    o.isCollapse = false
    o.pin = true
    
    -- Main Config
    o.padding = math.floor(FONT_HGT_SMALL * 0.4)
    o.scrollBarWidth = FONT_HGT_SMALL * 0.6
    o.scrollViewSpacing = 2
    
    -- Building specific config
    o.filterbarHeight = math.floor(FONT_HGT_SMALL * 1.6) + o.padding*2
    
    -- Category and filter state
    o._categoryString = ""
    o._filterString = ""

    o.logic = BuildLogic.new(o.player, o.craftBench, o.isoObject)
    o.logic:addEventListener("onUpdateContainers", o.onLogicUpdateContainers, o)
    o.logic:addEventListener("onRecipeChanged", o.onLogicRecipeChanged, o)
    o.logic:addEventListener("onUpdateRecipeList", o.onLogicUpdateRecipeList, o)
    -- o.logic:addEventListener("onShowManualSelectChanged", o.onShowManualSelectChanged, o)
    o.logic:addEventListener("onStopCraft", o.onStopCraft, o)
    o.lastPlayerMovingState = nil
    
    return o
end


-- ----------------------------------------------------------------------------------------------------- --
-- createChildren
-- ----------------------------------------------------------------------------------------------------- --
function NB_BuildingPanel:createChildren()
    self:updateContainers()
    self:setLogicRecipes()

    self:addRowFill(nil)

    self.categoryColumn = self:addColumn(nil)
    
    self.recipeListColumn = self:addColumnFill(nil)

    local spacingCloumn = self:addColumn(nil)
    spacingCloumn.minimumWidth = self.padding

    self:createCellRecipeList()
    self:createResizeWidget()
    self:createCategoryPanel()
    self.filterBar = self.cellRecipeList.filterBar
    self.recipeListPanel = self.cellRecipeList.recipeListPanel

    self:performInitialUpdate()
end

function NB_BuildingPanel:createCategoryPanel()
    self.categoryPanel = NB_BuildingCategoryPanel:new(0, 0, 10, 10, self)
    self.categoryPanel:initialise()
    self:setElement(self.categoryColumn:index(), 0, self.categoryPanel)
end

function NB_BuildingPanel:createCellRecipeList()
    self.cellRecipeList = NB_CellRecipeList:new(0, 0, 10, 10, self)
    self.cellRecipeList:initialise()
    self:setElement(self.recipeListColumn:index(), 0, self.cellRecipeList)
end

function NB_BuildingPanel:createResizeWidget()
    local resizeSize = self.padding
    self.resizeWidget = ISResizeWidget:new(self.width - resizeSize-2, self.height - resizeSize-2, resizeSize, resizeSize, self, false)
    self.resizeWidget.anchorRight = true
    self.resizeWidget.anchorBottom = true
    self.resizeWidget:initialise()
    self.resizeWidget:instantiate()
    self.resizeWidget:setAlwaysOnTop(true)
    self.resizeWidget.prerender = function(widget)
        local alpha = widget.mouseOver and 0.8 or 0.6
        widget:drawTextureScaledAspect(getTexture("media/ui/NeatUI/Resize/ResizeIcon.png"), 0, 0, widget.width, widget.height, alpha, 1, 1, 1)
    end
    self.resizeWidget.resizeFunction = function(target, newWidth, newHeight)
        target:calculateLayout(newWidth, newHeight)
    end
    ISPanel.addChild(self, self.resizeWidget)
end

-- make sure everything initialise
function NB_BuildingPanel:performInitialUpdate()
    local currentRecipe = self.logic:getRecipe()
    -- force choose Recipe again
    if currentRecipe then
        self:onLogicRecipeChanged()
    end

    if self.recipeListPanel then
        self:updateRecipeDisplay()
    end

    if self.contextRecipe then
        self.logic:setRecipe(self.contextRecipe)
        self.recipeListPanel:scrollToRecipe(self.contextRecipe)
        self.contextRecipe = nil
    end
end

-- ----------------------------------------------------------------------------------------------------- --
-- Logic Event listener
-- ----------------------------------------------------------------------------------------------------- --
function NB_BuildingPanel:onCategoryChanged(categoryValue)
    self._categoryString = categoryValue
    self:filterRecipeList()
end

function NB_BuildingPanel:filterRecipeList()
    local filterString = self._filterString or ""
    if self.recipeListPanel and self.recipeListPanel._filterMode and filterString ~= "" then
        filterString = filterString .. "-@-" .. self.recipeListPanel._filterMode
    end
    self.logic:filterRecipeList(filterString, self._categoryString)
end

function NB_BuildingPanel:setLogicRecipes()
    local list = self.logic:getAllBuildableRecipes()
    self.logic:setRecipes(list)
end

-- setContainers
function NB_BuildingPanel:onLogicUpdateContainers()
    -- self:updateRecipeDisplay()
end

-- sortRecipeList:filterRecipeList()setRecipes()
function NB_BuildingPanel:onLogicUpdateRecipeList()
    self:updateRecipeDisplay()
end

function NB_BuildingPanel:startBuild()
    self:createBuildIsoEntity()
end

function NB_BuildingPanel:onStopCraft()
    self:updateContainers()
    self.logic:sortRecipeList()
    self.logic:refresh()
    self:updateRecipeDisplay()
    -- recreate build entity for quick sequential building by player  
    self:createBuildIsoEntity()
end

function NB_BuildingPanel:onSearchTextChanged(searchString)
    self._filterString = searchString
    self:filterRecipeList()
end

function NB_BuildingPanel:onFilterChanged()
    self:updateRecipeDisplay()
end


function NB_BuildingPanel:updateRecipeDisplay()
    if not self.filterBar or not self.recipeListPanel then
        return
    end

    local filteredRecipes = self.filterBar:getFilteredRecipes()

    self.recipeListPanel:updateRecipeList(filteredRecipes)
end

-- setRecipe
function NB_BuildingPanel:onLogicRecipeChanged()

    self:updateContainers()

    self:updateRecipeDisplay()

    local recipe = self.logic:getRecipe()
    if recipe then
        self:showBuildingInfoPanel(recipe)
    else
        self:closeBuildingInfoPanel()
    end
end

-- ----------------------------------------------------------------------------------------------------- --
-- build Entity
-- ----------------------------------------------------------------------------------------------------- --

function NB_BuildingPanel.SetDragItem(item, playerNum)
    local windowKey = "BuildWindow"
    if not ISEntityUI or not ISEntityUI.players[playerNum] or not ISEntityUI.players[playerNum].windows[windowKey] or not ISEntityUI.players[playerNum].windows[windowKey].instance then return end
    
    local instance = ISEntityUI.players[playerNum].windows[windowKey].instance
    
    if item then
        instance.pin = false
        instance.isCollapse = true
        if instance.recipeListColumn then
            instance.recipeListColumn:setVisible(false)
        end
        if instance.spacingCloumn then
            instance.spacingCloumn:setVisible(false)
        end
        if instance.resizeWidget then
            instance.resizeWidget:setVisible(false)
        end
        if instance.BuildingInfoPanel then
            instance.BuildingInfoPanel:setVisible(false)
        end
        instance.tempadjWidth = instance:getWidth()
        instance.tempadjHeight = instance:getHeight()
        instance:calculateLayout(instance.categoryListWidth, instance.tempadjHeight)
    else
        instance.pin = true
    end
end

function NB_BuildingPanel:createBuildIsoEntity(dontSetDrag)
    local player = self.player
    local info = self.logic:getSelectedBuildObject()
    local recipe = self.logic:getRecipe()
    
    if info ~= nil and recipe ~= nil then
        if self.buildEntity == nil or self.buildEntity.objectInfo ~= info then
            local containers = ISInventoryPaneContextMenu.getContainers(self.player)
            self.buildEntity = ISBuildIsoEntity:new(player, info, 1, containers, self.logic)
            self.buildEntity.dragNilAfterPlace = false
            self.buildEntity.blockAfterPlace = true
            self.buildEntity.player = player:getPlayerNum()
    
            local inventory = player:getInventory()
            
            local function getTool(toolInfo, inventory)
                if toolInfo then
                    local inputScript = toolInfo
                    local entryItems = inputScript:getPossibleInputItems()
                    
                    for m = 0, entryItems:size() - 1 do
                        local itemType = entryItems:get(m):getFullName()
                        local result = inventory:getAllTypeEvalRecurse(itemType, ISBuildIsoEntity.predicateMaterial)
                        if result:size() > 0 then
                            return result:get(0):getFullType()
                        end
                    end
                end
                return nil
            end
            
            self.buildEntity.equipBothHandItem = getTool(recipe:getToolBoth(), inventory)
            self.buildEntity.firstItem = getTool(recipe:getToolRight(), inventory)
            self.buildEntity.secondItem = getTool(recipe:getToolLeft(), inventory)
        end

        local canBuild = self.logic:canPerformCurrentRecipe() or self.player:isBuildCheat()
        
        if self.logic:isCraftActionInProgress() then
            canBuild = false
        end

        self.buildEntity.blockBuild = not canBuild

        if not dontSetDrag then
            getCell():setDrag(self.buildEntity, player:getPlayerNum())
        end
    else
        self.buildEntity = nil
        getCell():setDrag(nil, player:getPlayerNum())
    end
end

-- ----------------------------------------------------------------------------------------------------- --
-- Key Function
-- ----------------------------------------------------------------------------------------------------- --

function NB_BuildingPanel:isKeyConsumed(key)
    return key == Keyboard.KEY_ESCAPE
end

function NB_BuildingPanel:onKeyRelease(key)
    if self.cellRecipeList:isVisible() and self:isVisible() and key == Keyboard.KEY_ESCAPE then
        self:close()
        return true
    end

    if not self.cellRecipeList:isVisible() and self.categoryPanel:isVisible() and key == Keyboard.KEY_ESCAPE then
        getCell():setDrag(nil, self.player:getPlayerNum())
        return true
    end

    return false
end

function NB_BuildingPanel:onKeyPressed(key)
    if key == Keyboard.KEY_ESCAPE or getCore():isKey("Crafting UI", key) or getCore():isKey("Build UI", key) then
        return true
    end
    return false
end

-- ----------------------------------------------------------------------------------------------------- --
-- Update
-- ----------------------------------------------------------------------------------------------------- --
function NB_BuildingPanel:update()
    ISTableLayout.update(self)

    self:updateCollapseState()

    if self.BuildingInfoPanel then
        local mouseOverInfoPanel = self.BuildingInfoPanel:isMouseOver()
        local mouseOverBuildingPanel = self:isMouseOver()
        
        if (mouseOverInfoPanel or mouseOverBuildingPanel) and not self.isCollapse then
            self.BuildingInfoPanel:setVisible(true)
        else
            self.BuildingInfoPanel:setVisible(false)
        end
    end

    local currentMovingState = self.player:isPlayerMoving()

    if self.lastPlayerMovingState == true and currentMovingState == false then
        self.lastPlayerMovingState = currentMovingState
        self:updateContainers()
        self:updateRecipeDisplay()
    end

    if self.lastPlayerMovingState ~= currentMovingState then
        self.lastPlayerMovingState = currentMovingState
    end
end

function NB_BuildingPanel:updateContainers()
    local containers = ISInventoryPaneContextMenu.getContainers(self.player)
    self.logic:setContainers(containers)
end


function NB_BuildingPanel:updateResizeWidgetPosition()
    if self.resizeWidget then
        local resizeSize = self.padding
        local newX = self.width - resizeSize - 2
        local newY = self.height - resizeSize - 2
        self.resizeWidget:setX(newX)
        self.resizeWidget:setY(newY)
    end
end

function NB_BuildingPanel:updateCollapseState()
    local shouldExpand
    
    if self.pin then
        shouldExpand = true
    else
        shouldExpand = self:isMouseOver() or self.BuildingInfoPanel:isMouseOver()
    end

    if shouldExpand and self.isCollapse then
        self.isCollapse = false
        if self.recipeListColumn then
            self.recipeListColumn:setVisible(true)
        end
        if self.spacingCloumn then
            self.spacingCloumn:setVisible(true)
        end
        if self.resizeWidget then
            self.resizeWidget:setVisible(true)
        end
        self:calculateLayout(self.tempadjWidth, self.tempadjHeight)
        
    elseif not shouldExpand and not self.isCollapse then
        self.isCollapse = true
        if self.recipeListColumn then
            self.recipeListColumn:setVisible(false)
        end
        if self.spacingCloumn then
            self.spacingCloumn:setVisible(false)
        end
        if self.resizeWidget then
            self.resizeWidget:setVisible(false)
        end
        self.tempadjWidth = self:getWidth()
        self.tempadjHeight = self:getHeight()
        self:calculateLayout(self.categoryListWidth, self.tempadjHeight)
    end
end
-- ----------------------------------------------------------------------------------------------------- --
-- Panel Manager
-- ----------------------------------------------------------------------------------------------------- --
function NB_BuildingPanel:close()

    self:closeBuildingInfoPanel()
    ISEntityUI.OnCloseWindow(self)
    getCell():setDrag(nil, self.player:getPlayerNum())
    
    if JoypadState.players[self.playerNum+1] then
        if isJoypadFocusOnElementOrDescendant(self.playerNum, self) then
            setJoypadFocus(self.playerNum, nil)
        end
    end

    if self.isoObject and getCore():getOptionDoContainerOutline() then
        self.isoObject:setOutlineHighlight(false);
        self.isoObject:setOutlineHlAttached(false);
    end
    
    self:removeFromUIManager()
end

function NB_BuildingPanel:showBuildingInfoPanel()
    
    if not (self.BuildingInfoPanel and self.BuildingInfoPanel:isReallyVisible()) then
        self.BuildingInfoPanel = NB_BuildingInfoPanel:new(0, 0, 50, 50, self)
        self.BuildingInfoPanel:initialise()
        self.BuildingInfoPanel:addToUIManager()
        self.BuildingInfoPanel:setVisible(false)
        self.BuildingInfoPanel:onRecipeChanged()
    else
        self.BuildingInfoPanel:onRecipeChanged()
    end
end

function NB_BuildingPanel:closeBuildingInfoPanel()
    if self.BuildingInfoPanel then
        self.BuildingInfoPanel:close()
        self.BuildingInfoPanel = nil
    end
end

-- ----------------------------------------------------------------------------------------------------- --
-- Render
-- ----------------------------------------------------------------------------------------------------- --
function NB_BuildingPanel:prerender()

    local panelBG = NinePatchTexture.getSharedTexture("media/ui/Neat_Building/Panel/MainPanelBG_RoundTop.png")
    if panelBG and not self.isCollapse then
        panelBG:render(self:getAbsoluteX(), self:getAbsoluteY(), self.width, self.height, 0.15, 0.15, 0.15, 1)
    end
end

Events.SetDragItem.Remove(ISBuildPanel.SetDragItem)
Events.SetDragItem.Add(NB_BuildingPanel.SetDragItem)